/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * interp.c - interpret escape sequences as VGTS commands
 *
 * Bill Nowicki September 1982
 */


#include <Vioprotocol.h>
#include "Vtermagent.h"
#include "Vgts.h"
#include "chars.h"
#include "sdf.h"
#include "vgt.h"
#include "pad.h"
#include "interp.h"

extern short Debug, RowOrder;

/* Arguments to AddItem, etc. */
#define u_sdf	 	(u->arg[0])
#define u_item	 	(u->arg[1])
#define u_xmin	 	(u->arg[2])
#define u_xmax	 	(u->arg[3])
#define u_ymin	 	(u->arg[4])
#define u_ymax	 	(u->arg[5])
#define u_typedata  	(u->arg[6])
#define u_type	 	(u->arg[7])

struct ParseEntry
  {
  	/*
	 * Table for parsing the escape sequences into VGTS requests.
	 * THIS MUST BE KEPT UP TO DATE with the arguments
	 * of the corrsponding routines!
	 */
    enum VgtsRequest req;	/* request type for double-checking */
    short numargs;		/* number of numeric arguments */
    char stringFlag;		/* true if last arg is a string */
  } ParseTable[] =
  {
 /*    Request code  	     # args   string? */
    VgtsCreateSDF, 		0, 	0,
    VgtsDeleteSDF, 		1, 	0,
    VgtsOldDefineSymbol,	2,	1,/* for backward compatibility only! */
    VgtsOldEndSymbol,		3,	0,/* for backward compatibility only! */
    VgtsAddItem,		8,	1,
    VgtsAddCall,		5,	0,
    VgtsDeleteItem,		2,	0,
    VgtsInquireItem,		2,	0,
    VgtsInquireCall,		2,	0,
    VgtsChangeItem,		8,	1,
    VgtsOldEditSymbol,		2,	0,/* for backward compatibility only! */
    VgtsDeleteSymbol,		2,	0,
    VgtsCreateVGT,		3,	1,
    VgtsDeleteVGT,		2,	0,
    VgtsOldDisplayItem,		3,	0,/* for backward compatibility only! */
    VgtsCreateView,		8,	0,
    VgtsGetMouseClick,		0,	0,	/* All these are "special" */
    VgtsFindSelectedObject,	5,	0,
    VgtsPopup,			0,	0,
    VgtsDefineFont,		1,	1,
    VgtsSetMode,		2,	0,
    VgtsDefaultView,		7,	0,
    VgtsDefineSymbol,		2,	1,
    VgtsEndSymbol,		3,	0,
    VgtsEditSymbol,		2,	0,
    VgtsDisplayItem,		3,	0,
    (enum VgtsRequest)-1, -1,	0
  };

#ifdef LITTLE_ENDIAN
extern unsigned char BitReverse[256];
#endif LITTLE_ENDIAN

struct InterpUser *CreateInterp(vgt)
  {
      /*
       * Initialize the VGTS interpreter for the given vgt,
       * and return a pointer to it.
       */
    register struct InterpUser *u;
    
    u = (struct InterpUser *)malloc(sizeof(struct InterpUser));
    if (u) 
      {
        u->vgt = vgt;
	u->pad = TtyPadList + vgt;
        u->state = Transparent;
      }
     return(u);
  }


int InterpPutString( u, s, length, mode )
    register struct InterpUser *u;	/* pointer to Interp user structure */
    char *s;				/* pointer to string */
    int  length; 			/* string length */
    int  mode;				/* mode of cooking */
  {
  	/*
	 * Interpret the character according to our state.
	 * Typical finite state machine interpreter, driven
	 * by our input character, and the Parse table to
	 * determine the next state action.
	 *
	 * Returns the ACTUAL number of bytes interpreted; may be less
	 * than length because the string may fill the pad in PageOutput mode.
	 */

    register int strCount = 0;
    register int strLength = length;
    register unsigned char c;

    if (u==NULL) return(0);
        
    if (Debug & DebugVgtsVgtp) dprintf("Interp got %d bytes\n", strLength);

    while(strCount < strLength)
      {
        c = *s++;
        switch (u->state)
         {
	  case NotThere:
	  	return(0);

	  case Transparent:
	    if (c == InterpEscape)
	      {
	        u->state = GetRequest;
	        break;
	      }
	    else if (!TransparentChar(u->pad, mode, c))
		return(strCount);
	    break;

	  case GetRequest:
	    u->req = (enum VgtsRequest)c;
	    u->state = GetParams;
	    u->count = 0;
	    if ((int)u->req>(int)VgtsMaxCode || (int)u->req < 0 )
	      {
	         if (Debug & DebugVgtsVgtp) 
			dprintf("Error- request code %d, max is %d\n",
				u->req, VgtsMaxCode );
	         u->state = Transparent;
		 if (!TransparentChar(u->pad, mode, InterpEscape))
		   {
		     strCount--; /* ??? */
		     return (strCount);
		   }
		 if (!TransparentChar(u->pad, mode, c))
		     return (strCount);
		 break;
	      }
	    if (ParseTable[c].numargs>0) break;
	    CallVgts(u);
	    break;

 	  case GetParams:
	    u->state = GetHalfParam;
	    u->arg[u->count] = c << 8;
	    break;

	  case GetHalfParam:
	    u->arg[u->count] += c;
	    if (Debug & DebugVgtsVgtp)
	        dprintf("arg[%d]=%d ", u->count, u->arg[u->count]);
	    u->count++;
	    u->state = GetParams;
	    if (u->count<ParseTable[(int)u->req].numargs) break;
	    if (ParseTable[(int)u->req].stringFlag) u->state = GetLength;
	    else CallVgts(u);
	    break;
	    
	  case GetLength4:
	    u->state = GetLength3;
	    u->length = c << 24;
	    break;

    	  case GetLength3:
	    u->state = GetLength2;
	    u->length += c << 16;
	    break;
	    
	  case GetLength2:
	    u->state = GetLength1;
	    u->length += c << 8;
	    break;
	    
	  case GetLength:
	    if ( c==MAGIC4 ) 
	      {
	    	u->state = GetLength4;
	    	break;
	      }
	    u->length = 0;

	/* NOTE: case GetLength falls through to case GetLength1 */

	  case GetLength1:
	    u->length += c;
	    u->count = 0;
	    u->state = GetString;
	    if (u->length == 0) 
	      {
	        u->string = NULL;
	        CallVgts(u);
		break;
	      }
	    u->string = (char *)malloc(u->length+1);
/*
printf( "malloc in interp.c line 221: %8x: %d bytes, req=%d\n\r",
(unsigned int) u->string, u->length+1, u->req );
*/
	    if ((Debug & DebugVgtsVgtp) && u->length> 255)
	        dprintf("Getting a string of length %d, request %d\n",
			u->length,u->req);
	    if (u->string==NULL)
	    	u->state = SkipString;
	    else if (u->req == VgtsAddItem
	      && (u_type == SDF_RASTER) && RowOrder == 1)
	      {
		u->state = GetStringRotate;
		u->u.work[0] = 0;	/* offset into string */
		u->u.work[1] = 0;	/* current row number */
		u->u.work[2] =		/* row width in bytes - column width */
		    (((u_xmax - u_xmin + 1 + 15) >> 4) << 1) - 2;
		u->u.work[3] = u_ymax - u_ymin + 1; /* height */
	      }
	    break;

	  case SkipString:
	    u->count++;
	    if (u->count >= u->length )
		CallVgts(u);
	    break;

	  case GetString:
	    u->string[u->count++] = c;
	    if (u->count >= u->length )
	      {
	        u->string[u->count] = 0;
		if (Debug & DebugVgtsVgtp) dprintf("string=%s ", u->string);
		CallVgts(u);
	      }
	    break;

	  case GetStringRotate:
	    u->count++;
	    u->string[u->u.work[0]++] =
#ifndef LITTLE_ENDIAN
		c;
#else LITTLE_ENDIAN
		BitReverse[c];
#endif LITTLE_ENDIAN
	    if (u->count >= u->length )
	      {
	        u->string[u->count] = 0;
		if (Debug & DebugVgtsVgtp) dprintf("string=%s ", u->string);
		CallVgts(u);
	      }
	    else if ((u->count & 1) == 0)
	      {
		u->u.work[0] += u->u.work[2];	/* pos += row width-2 */
		u->u.work[1]++;			/* increment row number */
		if (u->u.work[1] >= u->u.work[3]) /* test for new column */
		  {u->u.work[1] = 0; u->u.work[0] -= u->length-2; }
	      }
	    break;

	  case GetMenuLength:
	    u->length = c;
	    u->count = 0;
	    u->state = GetMenuString;
	    if (u->length==0) 
	      {
	      	  /*
		   * we actually do the pop up when
		   * we get a string of length zero
		   */
	        u->u.menu[u->menuCount].string = NULL;
	        u->u.menu[u->menuCount].menuNumber = 0;
	        ReturnShort( u->vgt, popup(u->u.menu), 1 );
		ReleaseMenuStrings(u->u.menu);
		u->state = Transparent;
		break;
	      }
	    u->string = (char *)malloc(u->length+1);
	    if (u->string==NULL)
	    	u->state = SkipString;
	    break;

	  case GetMenuString:
	    u->string[u->count++] = c;
	    if (u->count >= u->length )
	      {
	        u->string[u->count] = 0;
		u->u.menu[u->menuCount].string = u->string;
		u->state = GetMenuNumber;
	      }
	    break;
	    
	  case GetMenuNumber:
	    if (Debug & DebugVgtsVgtp) 
	        dprintf("Menu item %d is %s\r\n", c, u->string);
	    u->u.menu[u->menuCount++].menuNumber = c;
	    u->state = GetMenuLength;
	    break;
	  }  /* end switch */

	strCount++;
      } /* end loop on characters */
  return(strCount);
  }


BOOLEAN TransparentChar(pad, mode, c)
    register TtyPadType *pad;
    int  mode;				/* mode of cooking */
    char c;				/* the character */
  {
  	/*
	 * Try to send a character to the pad.  
	 * If PageOutput mode is on and pad is full, put continuation prompt
	 * in banner and return FALSE; else send the character and return TRUE.
	 */
    if (mode & PageOutput)
      {
	if ( (pad->newlinesLeft <= 0) &&
	   ( (c=='\n') || (c=='\r') || (pad->curColumn > pad->width) ) )
	  {
	    int vgtindex = pad - TtyPadList;

	    SetBannerState(vgtindex, DisplayPageMessage);
	    return(FALSE);
	  }
      }
      
    PadPutChar(c, pad);
    if ( (c == '\n') && (mode & LF_Output) ) PadPutChar('\r', pad);
    return(TRUE);
  }


static CallVgts(u)
    register struct InterpUser *u;
  {
	/*
	 * call one of the Vgts routines
	 */
    short item, xmin, ymin, xmax, ymax, buttons;
    LISTTYPE FindSelectedObject(), list;
    char type, typedata;
    short vgtindex = u->vgt;
    short sdf;

    if ((int)u->req > (int)VgtsMaxCode)
      {
         if (Debug & DebugVgtsVgtp) 
	    dprintf("Error- request code %d, max is %d\n",
			u->req, VgtsMaxCode );
         u->state = Transparent;
	 return;
      }

    if (Debug & DebugVgtsVgtp)
        dprintf(" call, code %d for vgt %d\n", u->req, vgtindex );

    switch (u->req)
      {
	case VgtsDefineSymbol:
		DefineSymbol(u_sdf, u_item, u->string);
    		break;

	case VgtsEndSymbol:
		EndSymbol(u_sdf, u_item, u_xmin);
    		break;

	case VgtsEditSymbol:
		EditSymbol(u_sdf, u_item);
    		break;

	case VgtsDisplayItem:
		DisplayItem(u_sdf, u_item, u_xmin);
    		break;

	case VgtsCreateSDF:
		sdf = CreateSDF();
		ReturnShort(vgtindex,sdf,1);
    		break;

	case VgtsDeleteSDF:
		sdf = DeleteSDF(u_sdf,TRUE);
		ReturnShort(vgtindex,sdf,1);
    		break;

	case VgtsAddItem:
		item = AddItem(u_sdf, u_item,
		    u_xmin, u_xmax, u_ymin, u_ymax,
		    u_typedata, u_type, u->string);
    		break;

	case VgtsDeleteItem:
		FreeString(u_sdf, u_item);
		item = DeleteItem(u_sdf, u_item);
    		break;

	case VgtsInquireItem:
		item = InquireItem(u_sdf, u_item,&xmin,&xmax,
			&ymin, &ymax, &typedata, &type, NULL);
		ReturnShort(vgtindex,xmin,0);
		ReturnShort(vgtindex,xmax,0);
		ReturnShort(vgtindex,ymin,0);
		ReturnShort(vgtindex,ymax,0);
		ReturnShort(vgtindex,typedata,0);
		ReturnShort(vgtindex,type,0);
		ReturnShort(vgtindex,item,1);
    		break;

	case VgtsInquireCall:
		item = InquireCall(u_sdf, u_item);
		ReturnShort(vgtindex,item,1);
    		break;

	case VgtsChangeItem:
		if (u->string)
		    FreeString(u_sdf, u_item);
		item = ChangeItem(u_sdf, u_item, u_xmin, u_xmax,
			 u_ymin, u_ymax, u_typedata, u_type, u->string);
    		break;

	case VgtsDeleteSymbol:
		item = DeleteSymbol(u_sdf, u_item);
		ReturnShort(vgtindex,item,1);
    		break;

	case VgtsCreateVGT:
	      {
		short vgtindex1 = CreateVGT(u_sdf, u_item, u_xmin, u->string);
		ReturnShort(vgtindex,vgtindex1,1);
		if ( u->string ) free( u->string );
		u->string = NULL;
		if (vgtindex1<0) break;
		ClientTable[vgtindex1].master = vgtindex;
		ClientTable[vgtindex1].owner = ClientTable[vgtindex].owner;
		ClientTable[vgtindex1].vgt = vgtindex1;
    		break;
	      }

	case VgtsDeleteVGT:
	      {
		short vgtindex1 = DeleteVGT(u_sdf, u_item);
		ReturnShort(vgtindex,vgtindex1,1);
		ClientTable[vgtindex1].master = -1;
    		break;
	      }

	case VgtsCreateView:
		item = CreateView(u_sdf, u_item, u_xmin, u_xmax,
		 u_ymin, u_ymax, u_typedata, u_type>>8, u_type & 0377);
		ReturnShort(vgtindex,item,1);
    		break;

	case VgtsGetMouseClick:
		WaitForEvent(vgtindex);
                u->state = Transparent;
    		break;

	case VgtsFindSelectedObject:
		list = FindSelectedObject(u_sdf, u_item, u_xmin,
			u_xmax, u_ymin);
		if (list.NumOfElements == 0)
		  {
		    ReturnShort(vgtindex,0,1);
		  }
		else
		  {
		    ReturnShort(vgtindex,1,0);
		    ReturnShort(vgtindex,list.Header->item,0);
		    ReturnShort(vgtindex,list.Header->edgeset,1);
		  }
    		break;
		
	case VgtsPopup:
		  /*
		   * these arguments are handled specially;
		   * there can be an arbitrary number.
		   */
		u->menuCount = 0;
		u->state = GetMenuLength;
		return;
		
	case VgtsDefineFont:
		if (u->string) 
		    item = DefineFont(u->string, u->string+u_sdf);
		else item = -1;
		if ( u->string ) free( u->string );
		u->string = NULL;
		ReturnShort(vgtindex,item,1);
		break;

	case VgtsSetMode:
		SetPadMode(vgtindex, (u_sdf<<16)|(u_item) );
		break;

	case VgtsDefaultView:
		DefaultView(u_sdf, u_item, u_xmin, u_xmax,
			u_ymin, u_ymax, u_typedata, &xmin, &ymin );
		ReturnShort(vgtindex,xmin,FALSE);
		ReturnShort(vgtindex,ymin,TRUE);
		break;

	case VgtsAddCall:
		item = AddCall(u_sdf, u_item, u_xmin, u_xmax,
			 u_ymin);
    		break;

	case VgtsOldDefineSymbol:
		item = DefineSymbol(u_sdf, u_item, u->string);
		ReturnShort(vgtindex,item,1);
    		break;

	case VgtsOldEndSymbol:
		item = EndSymbol(u_sdf, u_item, u_xmin);
		ReturnShort(vgtindex,item,1);
    		break;

	case VgtsOldEditSymbol:
		item = EditSymbol(u_sdf, u_item);
		ReturnShort(vgtindex,item,1);
    		break;

	case VgtsOldDisplayItem:
		item = DisplayItem(u_sdf, u_item, u_xmin);
		ReturnShort(vgtindex,item,1);
    		break;
     }
    u->state = Transparent;

  }


static ReleaseMenuStrings(p)
    PopUpEntry *p;
  {
  	/*
	 * free up the string space used in the given pop-up menu
	 */
    for (;p->string;p++)
      free(p->string);
  }


ReturnShort(user,n,pushFlag)
 short user;
 short n;
 char pushFlag;
  {
  	/*
	 * send back a return value to the indicated user
	 * as an escape character followed by two bytes.
	 * if pushFlag is set, the packet is "pushed" out.
	 */
    register struct Client *client = ClientTable + user;

    UserPut(client,InterpEscape);
    UserPut(client,n>>8);
    if (pushFlag)
        UserPush(client,n&0377);
     else
        UserPut(client,n&0377);
    if (Debug & DebugVgtsVgtp) 
        dprintf("%s value %d to user %d\n", 
    		pushFlag ? "Pushed" : "Returned", n, user);
  }




ReturnToMaster(master,xw,yw,buttons,vgtindex)
  {
    ReturnShort(master,xw,0);
    ReturnShort(master,yw,0);
    ReturnShort(master,buttons,0);
    ReturnShort(master,vgtindex,1);
  }


static WaitForEvent(user)
  {
	/*
	 * The interp client want to wait for an event --
	 * get one off the queue, or else set the waiting flag.
	 */
    ClientTable[user].requestFlag |= SlaveReq;
    ClientTable[user].mode |= ReportClick;
    EventToUser(ClientTable + user);
  }
