/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* handlers.c - handlers for various message-based requests
 *
 * Kenneth Brooks, September 1984
 *
 * This file was invented to keep the compile time on mux.c from reaching
 * infinity.
 *
 * Gus Fernandez, May 18, 1985 - Added sdftrace hack to debug.
 */

#include <Vioprotocol.h>
#include <Vtermagent.h>
#include <Vgts.h>
#include <Vgtp.h>
#include "sdf.h"
#include "vgt.h"
#include "pad.h"
#include "interp.h"

#define DEBUG_TRACE 0x3141		/* magic number for debug trace hack */

extern short Debug;
extern TtyPadType *DebugPad;

extern short InputVGT;

extern short MouseX, MouseY, MouseButtons;
extern struct InterpUser *CreateInterp();
extern struct Buffer *CreateBuffer();

short TtySDF;		/* SDF for Tty VGT */

/* the following typedef must match, exactly, the one in
 * .../libc/vgts/mi/openpad.c.  I know this is a hack, but it is
 * a low impact hack */

typedef struct
  {
    unsigned		fillA[ 2 ];
#ifndef LITTLE_ENDIAN
    short	lines;
    short	columns;
    short	x;
    short	y;
#else   LITTLE_ENDIAN
    short	columns;
    short	lines;
    short	y;
    short	x;
#endif  LITTLE_ENDIAN
    unsigned	flag;	/* very important, distinguishes
			 * between OpenPad and OpenAndPositionPad */
    unsigned	fillB[ 3 ];
  } CreateInstanceUnspecifiedFields;

HandleCreate(req, pid)
  register CreateInstanceRequest *req;
  ProcessId pid;
  {
  	/*
	 * Return appropriate stuff to create an instance.
	 * This creates a new pad, unless the Directory bit is set,
	 * in which case it just returns the Directory instance.
	 */
    int readMode = (req->filemode & FBASIC_MASK) == FREAD;
    short lines, columns, width, height, xCorner, yCorner;
    register Term1 *term1;
    unsigned short vgtnumber;
    struct InterpUser *interp;
    char name[128];
    Message msg;
    register CreateInstanceReply *reply;
    char forcePosition;
    CreateInstanceUnspecifiedFields *unspecified;
    
    
    /* Save a copy of the request message. */
    Copy(msg, req, sizeof(Message));
    reply = (CreateInstanceReply *) req;
    req = (CreateInstanceRequest *) msg;
    unspecified = (CreateInstanceUnspecifiedFields *) msg;
    
    if (req->filemode & FDIRECTORY)
      {
        reply->fileid = DirectoryInstance;
	reply->fileserver = VgtsPid;
        reply->blocksize = VgtDescriptorSize();
	reply->filetype = READABLE+FIXED_LENGTH;
	reply->filelastblock = DirectoryInstance;
	reply->filelastbytes = VgtDescriptorSize();
	reply->filenextblock = 0;
	reply->replycode = OK;
	if (Debug & DebugVgtsMessages) 
		dprintf("Directory create- OK\n");
	return;
      }

    forcePosition = ( unspecified->flag ==
	( CREATE_INSTANCE ^ req->unspecified[ 0 ] ^ req->unspecified[ 1 ] ));
    if ( forcePosition )
      {
	lines = unspecified->lines;
	columns = unspecified->columns;
      }
    else
      {
	lines = req->unspecified[0];
	columns = req->unspecified[1];
      }
    req->filename += req->filenameindex;
    req->filenamelen -= req->filenameindex;
    if (req->filenamelen>0)
      { 
        if (req->filenamelen>=VGTnameLength)
	    req->filenamelen = VGTnameLength - 1;
        MoveFrom( pid, name, req->filename, req->filenamelen);
	name[req->filenamelen] = 0;
      }
     else strcpy(name, "no name");

    if (lines<2 || lines>96) lines = PadHeight;
    if (columns<10 || columns>127) columns = PadWidth;
    width = ViewWidth(columns);
    height = ViewHeight(lines);
    vgtnumber = NewTextVGT(TtySDF, name, lines, columns);
    if (vgtnumber <= 0 ||
    	vgtnumber>= MaxVGT ||
	(interp = CreateInterp(vgtnumber)) == 0) 
      {
        reply->replycode = NO_SERVER_RESOURCES;
        return;
      }

    term1 = Term1Table + vgtnumber;
    term1->owner = pid;
    term1->reader = 0;
    term1->requestFlag = 0;
    term1->BlockNumber = 0;
    term1->TotalBytesWritten = 0;
    term1->interp = interp;
    term1->vgt = vgtnumber;
    term1->writerPid = 0;
    term1->termPid = 0;
    term1->master = -1;
    term1->isexec = FALSE;
    term1->lineEditBuf = CreateBuffer();
    term1->bytesWritten = 0;
    term1->mode = CR_Input+LF_Output+Echo+LineBuffer
    		   +PageOutput+PageOutputEnable;
    term1->outputFlag = 0;
       
    reply->replycode = OK;
    reply->fileserver = VgtsPid;
    reply->fileid = vgtnumber;
    if (readMode)
      {
        reply->blocksize = IO_MSG_BUFFER;
	reply->filetype = READABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	reply->filelastblock = MAXUNSIGNED;
	reply->filelastbytes = MAXUNSIGNED;
	reply->filenextblock = 0;
      }
    else 
      {
        reply->blocksize = BlockSize;
        reply->filetype = WRITEABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	reply->filelastblock = 0;
	reply->filelastbytes = BlockSize;
	reply->filenextblock = 0;
      }

    TtyPutString("Creating another Pad, upper left corner at B: ");
    if ( forcePosition )
      {
	CreateView( vgtnumber, unspecified->x, unspecified->y,
		    unspecified->x + width, unspecified->y + height,
        	    0, 0, 0, 0);
	TtyPutString("\r\n");
	return;
      }
    SetPadCursor();
    WaitForDownButton();
    xCorner = MouseX;
    yCorner = MouseY;
    FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    while (MouseButtons!=0)
      {
        ReadMouseInput();

	FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
	if (MouseButtons==7)
          {
            SetMainCursor();
	    TtyPutString(" Aborted\r\n");
            reply->replycode = RETRY;
	    return;
      	 }
        xCorner = MouseX;
        yCorner = MouseY;
	CursorUpdate();
	FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    }
    FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    TtyPutString("\r\n");
    SetMainCursor();

    CreateView( vgtnumber, xCorner, yCorner, 
        	xCorner+width, yCorner+height, 0, 0, 0, 0);
  }

/*
 * Return appropriate stuff to release an instance.
 * msg becomes the reply message.
 *
 * If we are deleting the current input pad, we need
 * to find another one.
 */
HandleRelease(msg)
  register CreateInstanceReply *msg;
  {
    register vgt = msg->fileid;

    if (Debug & DebugVgtsMessages) dprintf("Release of vgt %d\n", vgt );
    msg->replycode = OK;
    DeleteTextVGT(vgt, TRUE);

    FlushEvents(Term1Table + vgt);
    if (Term1Table[vgt].interp)
      {
        free(Term1Table[vgt].interp);
        Term1Table[vgt].interp = 0;
      }
    Term1Table[vgt].owner = 0;
    if ( Term1Table[ vgt ].lineEditBuf )
	free( Term1Table[ vgt ].lineEditBuf );
    Term1Table[ vgt ].lineEditBuf = NULL;
    MakeNotCurrent(vgt);
  }


MakeNotCurrent(pad)
    short pad;
  {
	/*
	 * checks if we are referring to the current input pad.
	 * If so, select some other likely candidate.
	 * Returns the new input pad.
	 */
    register Term1 *term1;
    register int i;

    if (InputVGT==pad)
      {
        for (i=1, term1=Term1Table+1; i<MaxVGT; i++, term1++)
	  if (term1->vgt>0 && term1->requestFlag && term1->reader)
	      {
	        if (term1->owner && ValidPid(term1->owner))
		  {
		    SelectForInput(i);
	            return;
		  }
		/* we've found a dead one, so delete it */
		DeleteTextVGT(i, TRUE);
		FlushEvents(term1);
		  {
		    IoReply m;
		    m.replycode = END_OF_FILE;
		    Reply(&m, term1->reader);
		  }
		term1->reader = 0;
		term1->owner = 0;
		term1->master = -1;
		term1->isexec = FALSE;
		if (term1->interp)
		    free(term1->interp);
		term1->interp = NULL;
		if ( term1->lineEditBuf )
		    free( term1->lineEditBuf );
		term1->lineEditBuf = NULL;
	      }

        for (i=1, term1=Term1Table+1; i<MaxVGT; i++, term1++)
        if (term1->vgt>0 && term1->reader)
	    {
	        SelectForInput(i);
	        return(i);
	    }
      }
  return(i);
  }



HandleQuery(msg, pid)
  register CreateInstanceReply *msg;
  {
  	/*
	 * Return appropriate stuff to a query instance.
	 * msg becomes the reply message.
	 */
    QueryInstanceRequest *req = (QueryInstanceRequest *)msg;
    int readMode = ( (req->filemode & FBASIC_MASK) == FREAD );
    register Term1 *term1 = Term1Table + msg->fileid;
    
    if (Debug & DebugVgtsMessages) 
        dprintf("Query on id %d, mode=0x%x, readMode=%d\n", msg->fileid,
		 ((QueryInstanceRequest *)msg) ->filemode,
		 readMode);
    msg->replycode = OK;
    msg->fileserver = VgtsPid;
    
    if (msg->fileid == DirectoryInstance)
      {        
	msg->fileserver = VgtsPid;
        msg->blocksize = VgtDescriptorSize();
	msg->filetype = READABLE+WRITEABLE;
	msg->filelastblock = MaxVGT-1;
	msg->filelastbytes = 0;
	msg->filenextblock = 0;
	msg->replycode = OK;
	return;
      }
    if (term1->owner==0)
	term1->owner = pid;      
    if (term1->vgt <= 0)
	term1->vgt = msg->fileid;
    if (term1->interp==0)
        term1->interp = CreateInterp(msg->fileid);

    if (readMode)
      {
        msg->blocksize = IO_MSG_BUFFER;
	msg->filetype = READABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	msg->filelastblock = MAXUNSIGNED;
	msg->filelastbytes = MAXUNSIGNED;
	msg->filenextblock = 0;
      }
    else 
      {
        msg->blocksize = BlockSize;
        msg->filetype = WRITEABLE+STREAM+VARIABLE_BLOCK+INTERACTIVE;
	msg->filelastblock = term1->BlockNumber;
	msg->filelastbytes = BlockSize;
	msg->filenextblock = 0;
      }
  }

HandleQueryFile(msg)
  register struct ModifyMsg *msg;
  {
  	/*
	 * Return appropriate stuff to a query file.
	 * msg becomes the reply message.
	 */
    int index = msg->fileid;
    register Term1 *term1 = Term1Table + index;
    register struct InterpUser *interp = term1->interp;
    register TtyPadType *pad = interp->pad;
    
    if (index < MaxVGT)
      {
	/* Note: do NOT return the internal value of the PageOutputEnable 
	 * flag, because this might cause the caller to inadvertantly alter it 
	 * on a subsequent call to ModifyPad().
	 */
        msg->mode = (term1->mode) & ~PageOutputEnable;
	msg->lines = pad->length + 1;
	msg->columns = pad->width + 1;
	msg->requestcode = OK;
      }
  }


HandleModify(msg)
  register struct ModifyMsg *msg;
  {
   	/*
	 * Set stuff for a Modify file request.
	 * Right now this is just the degree of cooking.
	 * msg becomes the reply message.
	 */

    int index = msg->fileid;
    register Term1 *term1 = Term1Table + index;
    register Term2 *term2 = term2Ptr(term1);
    register struct InterpUser *interp = term1->interp;
    register TtyPadType *pad = interp->pad;

    if (index<MaxVGT)
      {
	if (Debug & DebugVgtsMessages)
		dprintf("ModifyPad request: mode 0x%x for pad %d\n", 
			msg->mode, index);

	/* RSF - If you find the following code as confusing as I did, you 
	 * might want to read the file "pagedoutput.doc".  This contains a 
	 * copy of a message from Craig Dunwoody (the perpetrator of this 
	 * hack), explaining the reason for this dirty deed.
	 */
	if (msg->mode & PageOutputEnable)
	  {
	    if ( !(msg->mode & PageOutput) ) msg->mode &= ~PageOutputEnable;
	  }
	else if (term1->mode & PageOutputEnable) msg->mode |= PageOutputEnable;
	else msg->mode &= ~PageOutput;

        term1->mode = msg->mode;

	if (term1->mode & PageOutput)
	  {
	    pad->newlinesLeft = pad->length;
	    pad->pageLength = pad->length; 
	  }
	else if (term2->writerPid)
	    term2->outputFlag |= PageOutputShutdown;

	if ( (term1->mode & LineBuffer) && !(term2->lineEditBuf) )
	  {
	    term2->lineEditBuf = CreateBuffer();
	    if (term2->lineEditBuf == NULL)
	      {
		msg->mode &= ~LineBuffer;
		term1->mode = msg->mode;
		msg->requestcode = NO_SERVER_RESOURCES;
		return;
	      }
          }
	else if ( !(term1->mode & LineBuffer) && (term2->lineEditBuf) )
	  {
	    DestroyBuffer(term2->lineEditBuf);
	    term2->lineEditBuf = NULL;
	  }

	msg->requestcode = OK;
      }
  }


HandleSetBreak(msg)
    SetBreakRequest *msg;
  {
  	/*
	 * The break process is destroyed when the Kill Program
	 * command is selected, or the user types the break character.
	 */
    register Term1 *term1 = &Term1Table[msg->fileid];
    register Term2 *term2 = term2Ptr(term1);

    term2->termPid = msg->breakprocess;
    msg->requestcode = OK;
  }


HandleSetOwner(msg)
    register IoRequest *msg;
  {
  	/*
	 * Change the "owner" of an instance
	 */
    Term1Table[msg->fileid].owner = msg->instanceowner;
    msg->requestcode = OK;
  }


HandleGetRawIO(msg)
  register RawIOReply *msg;
  {
    msg->inserver = stdin->fileserver;
    msg->infile = stdin->fileid;
    msg->outserver = stdout->fileserver;
    msg->outfile = stdout->fileid;
    msg->replycode = OK;
  }


HandleBanner(msg, pid)
    IoRequest *msg;
    ProcessId pid;
  {
#define BannerStringMax 512
    char string[BannerStringMax];
    SystemCode status;
    int len = BannerStringMax-1;

    if (msg->bytecount < BannerStringMax-1)
        len = msg->bytecount;
    status = MoveFrom(pid, string, msg->bufferptr, len);
    string[len] = '\0';
    SetBanner(msg->fileid, string);
    msg->requestcode = OK;
  }


SetPadMode(pad,mode)
  {
  	/*
	 * Procedural hook for the interpeter
	 */

    if (pad<MaxVGT)
      {
        Term1Table[pad].mode = mode;
	if (Debug & DebugVgtsMessages)
		dprintf("Pad %d set to mode %d\n", pad, mode );
      }
  }


SelectForInput(i)
 short i;
  {
  	/*
	 * select the given pad for input.
	 * Zero means do not change.
	 * Bad values give an error message.
	 */

     if (i<1)
       {
         TtyBlinkError("Commands are control up-arrow followed by:");
	 TtyBlinkError("  1, 2, 3, etc to select");
	 return;
       }
    if (i>=MaxVGT || Term1Table[i].owner==0) return;
    if (Term1Table[i].interp==0)
      {
	  /* if this is a graphics VGT instead of a TTY,
	   * then just bring it to the top
	   */
	MakePadTop(i);
	return;
      }
    PadCursorOn(InputVGT);
    PadRedrawLine(InputVGT);
    if (InputVGT != i ) 
      {
        InputVGT = i;
        FixBanners(i);
      }
    MakePadTop(i);
  }


FixBanners(new)
   short new;
  {
	/*
	 * Set the banner to be Highlighted or Normal depending
	 * on the change in input.
	 */
    register View *w;
    register char *s;
    short master;

    for ( w = ViewList; w; w = w->next)
      if ( *w->banner & 0200)
        {
		/*
		 * Banner is now ON.  Check if it is about
		 * to go off, and redraw if so.
		 */
	 if (IsInput(w->vgt)) continue;
	 for ( s = w->banner; *s;)
              *s++ = (*s & 0177);
	 RedrawBanner(w);
	}
      else
	{
		/*
		 * Banner is now OFF.  Check if it is about
		 * to go on, and redraw if so.
		 */
	 if ( !IsInput(w->vgt)) continue;
	 for ( s = w->banner; *s;)
              *s++ = *s | 0200;
	 RedrawBanner(w);
	}
  }


IsInput(vgt)
  {
  	/*
	 * return true if this VGT is accepting input
	 */
   if (vgt==InputVGT) return 1;
   vgt = Term1Table[vgt].master;
   return (vgt==InputVGT);
  }

#define abs(a) (a>0?a:-a)
HandleDebug(req)
  register DebugServerRequest *req;
  {

    if (req->debugcode == DEBUG_TRACE)
      {
	SdfTrace(VGTtable[abs(req->fileid)].topSymbol,req->fileid>=0?0:-1,1);
      }
    else 
      {
        if (req->fileid >= 0 &&
    	    req->fileid < MaxVGT &&
	    VGTtable[req->fileid].type == TTY)
            DebugPad = TtyPadList + req->fileid;
        else  DebugPad = NULL;
        Debug = req->debugcode;

        if (Debug) dprintf("Debugging on\n");
        else  dprintf("Debugging off\n");
      }
    return(OK);
  }


HandleSwitchInput(index, pid)
  ProcessId pid;
  short index;
  {
    register Term1 *term1 = &Term1Table[InputVGT];
    register Term2 *term2 = term2Ptr(term1);

    /* don't do it unless they're friends */
    if (SameTeam(pid, term1->owner) ||
	(term2 && SameTeam(pid, term2->termPid) ) ||
	SameTeam(pid, ACTIVE_PROCESS))	/* for keyboard */
	SelectForInput(index);
  }


