#include <Vioprotocol.h>
#include <rasterops.h>
#include "Vgts.h"
#include "sdf.h"
#include "arc.h"
#include "vgt.h"

Seg *segtable[1024];  /* one per scan line, for bucket sorting */
char leftstate[1024];	/* whether we are inside the polygon at left clip */

Seg *segfree = NULL;	/* freelist */
Seg *segtrash;		/* list of segs to be freed at end of this polyarc */
Seg *horizontals;	/* special list of horizontal edges */
Seg *seglist;		/* list of segments before bucket sorting */
Seg *MakeSeg();

/* Type "mixed" operations */
/* int / int -> mix */
#define MixedQuotient(a, b)  (((a) << 16) / (b))

/* The trashlist is used because of edge shortening.  We must be sure that
 * an edge record, even if deleted by clipping, does not get reused during
 * this polygon, because it might get shortened based on its pointer stored
 * in oldedge or firstedge.
 */
#define FreeSeg(e) \
  { \
    e->next = segtrash; \
    segtrash = e; \
  }

extern Arcdata *TransformArcData();

/* aim values for remembering which way we are proceeding */
#define UP 1
#define DOWN 2

Seg *firstseg;
short firstaim;


/* The FillPolyarc drawing routine.  Derived from the FillArea algorithm
 * from Foley & Van Dam, pp. 457-460, with changes to handle curved boundary
 * segments as well as straight ones.  Wherever the text says "edge", we
 * here say "segment", meaning arc or line segment.  The fill is inclusive;
 * any point deemed to be "on" the boundary will be filled, regardless of which
 * side of that boundary is being filled in.
 *
 * Input to this routine is an array of npoints Arcdata elements, of
 * which the last must be the same as the first (else this routine will
 * probably crash!)  All x,y coordinates stored in that array are relative to
 * the (xoffset, yoffset) passed to the function.
 */
FillPolyarc(npoints, indata, pattern, xoffset, yoffset, zoom, clip)
  short npoints, xoffset, yoffset, zoom;
  Arcdata indata[];
  unsigned pattern[];  /* array of 16 shorts, the fill pattern */
  register struct SubView *clip;
  {
    Arcdata *data;
    register Arcdata *p;
    register Seg *seg, *oldseg, *temp;
    short aim, oldaim;
    register int i;

    /* for zoom, and eventually rotate, etc. we make a transformed copy	
     * of the input data array.
     */
    if (zoom == 0) data = indata;
    else data = TransformArcData(npoints, indata, zoom);
    if (!data) return;

    horizontals = seglist = segtrash = NULL;
    for (i=clip->ScreenYmin; i <= clip->ScreenYmax; i++)
	segtable[i] = NULL;

    oldaim = 0;  /* neither up nor down, prevents edge linkage 1st time */
    firstaim = 0;

    /* sort the edges into segtable, with each bucket sorted on xmin */
    for (p = data; p < data + npoints - 1; p++)
      {
	if (p->y > (p+1)->y)
	  {
	    seg = MakeSeg(p, p+1, xoffset, yoffset);
	    aim = DOWN;
	  }
	else if (p->y < (p+1)->y)
	  {
	    seg = MakeSeg(p, p+1, xoffset, yoffset);
	    aim = UP;
	  }
	else  /* horizontal, special treatment.  Short tabs of arc may get
		 in here too. */
	  {
	    StoreHorizontal(yoffset - p->y, xoffset + p->x, xoffset + (p+1)->x,
			    clip);
	    continue;
	  }

	if (seg->type == VESTIGE)  /* a very small arc, too small to draw */
	  {
	    FreeSeg(seg);
	    continue;
	  }

	if (!seg) return;  /* malloc failed, can't display */
	if (firstaim == 0)  /* remember the first non-horizontal segment */
	  {
	    firstseg = seg;
	    firstaim = aim;
	  }

	/* edge linkage to avoid double-counting at vertices */
	if (oldaim == DOWN && aim == DOWN)
	  {
	    oldseg->link = seg;
	    seg->linked = 1;
	  }
	else if (oldaim == UP && aim == UP)
	  {
	    seg->link = oldseg;
	    oldseg->linked = 1;
	  }
	oldaim = aim;
	oldseg = seg;

	/* store in seglist (they end up in reverse order; who cares?) */
	seg->next = seglist;
	seglist = seg;
      } /* end loop on points */

    /* edge linkage: last edge to first edge */
    if (aim == DOWN && firstaim == DOWN)
      {
	seg->link = firstseg;
	firstseg->linked = 1;
      }
    else if (aim == UP && firstaim == UP)
      {
	firstseg->link = seg;
	seg->linked = 1;
      }

    /* clip edges and store in hash table */
    for (seg = seglist; seg; seg = temp)
      {
	temp = seg->next;  /* seg will get transferred to another list */
	/* clip above as needed */
	if (seg->y0 < clip->ScreenYmin)
	  {
	    /* clip if entirely above */
	    if (seg->y1 < clip->ScreenYmin)  { FreeSeg(seg); }
	    else /* fine clipping at top edge */
	      {
		TopClip(seg, clip);
		StoreSeg(seg);
	      }
	  }
	/* clip if entirely below */
	else if (seg->y0 > clip->ScreenYmax)  {FreeSeg(seg);}
	/* segs that are partially below will take care of themselves */
	else  StoreSeg(seg);
      }

    ScanFill(pattern, clip);/*Uses data stored in segtable to draw the image*/

    FreeWholeList(segtrash);
    if (zoom) free(data);
  }


/* Transform the arcdata array to account for a zoom factor.  This same
 * routine should someday support general scaling and rotation.
 */
Arcdata *TransformArcData(npoints, indata, zoom)
  short npoints, zoom;
  register Arcdata *indata;
  {
    register Arcdata *outdata, *outarray;
    register int i;
#define RoundShift(a, sh) 	( ((a) >> sh) + (((a) >> (sh -1)) & 1) )

    outarray = outdata = (Arcdata *) malloc(npoints * sizeof(Arcdata));
    if (!outdata) return(NULL);

    if (zoom > 0)
      {
	for (i=0; i < npoints; i++)
	  {
	    outdata->type = indata->type;
	    outdata->r = indata->r << zoom;
	    outdata->x = indata->x << zoom;
	    outdata->y = indata->y << zoom;
	    outdata->cx = indata->cx << zoom;
	    outdata->cy = indata->cy << zoom;
	    indata++;
	    outdata++;
	  }
      }
    else
      {
	zoom = -zoom;
	for (i=0; i < npoints; i++)
	  {
	    outdata->type = indata->type;
	    outdata->r = RoundShift(indata->r, zoom);
	    outdata->x = RoundShift(indata->x, zoom);
	    outdata->y = RoundShift(indata->y, zoom);
	    outdata->cx = RoundShift(indata->cx, zoom);
	    outdata->cy = RoundShift(indata->cy, zoom);
	    indata++;
	    outdata++;
	  }
      }
    return(outarray);
  }



/* For speed, we store in each segment record the pointer to the appropriate
 * function to calculate xleft and xright and advance it to the next y.
 */
extern AdvanceLine(), AdvanceUL(), AdvanceUR(), AdvanceLR(), AdvanceLL();

function *advancefns[8] = {
	NULL, AdvanceLine, NULL, NULL,
	AdvanceUL, AdvanceUR, AdvanceLR, AdvanceLL };

#define lseg ((struct lineseg *) seg)


Seg *MakeSeg(p1, p2, xoffset, yoffset)
  register Arcdata *p1, *p2;
  register short xoffset, yoffset;  /* must declare exactly 2 data registers */
  {
    register Seg *seg;
    /* declare no data registers, or MixedQuotient will break! */

    if (!segfree)
      {
	seg = (Seg *) malloc(sizeof(Seg));
	if (!seg)  return(NULL);
      }
    else
      {
	seg = segfree;
	segfree = seg->next;
      }

    seg->type = p1->type;
    seg->link = NULL;
    seg->linked = 0;
    seg->advance = advancefns[seg->type];

    /* ordering: y0 is always the lower y value (higher on the screen) */
    if (p1->y > p2->y)
      {
	seg->x0 = xoffset + p1->x;
	seg->y0 = yoffset - p1->y;
	seg->x1 = xoffset + p2->x;
	seg->y1 = yoffset - p2->y;
      }
    else
      {
	seg->x0 = xoffset + p2->x;
	seg->y0 = yoffset - p2->y;
	seg->x1 = xoffset + p1->x;
	seg->y1 = yoffset - p1->y;
      }

    if (seg->type >= ULARC)  /* an arc */
      { register short r;

	seg->cx = xoffset + p1->cx;
	seg->cy = yoffset - p1->cy;
	r = p1->r;

	/* zoom protection: if r <= 1 make it a line segment */
	if (r <= 1)
	  {
	    if (p1->y == p2->y)
	      {
		seg->type = VESTIGE;  /* to be ignored */
		return(seg);
	      }

	    seg->type = LINE;
	    seg->advance = AdvanceLine;
	    goto lineseg;
	  }
	
	seg->r = r;
	/* make rsq here so we can use it for clipping */
	seg->rsq = r*r + r;  /* radius approx. r + 1/2 */

	/* center adjustment: prevent rounding error from putting the
	 * start point outside the stated radius
	 */
	if (seg->type <= URARC)  /* upper left or upper right */
	  {
	    if (seg->cy - seg->y0 > r)  seg->cy = seg->y0 + r;
	  }
	else		/* lower left or lower right */
	  {
	    if (seg->y1 - seg->cy > r)  seg->cy = seg->y1 - r;
	  }
      }
    else
      {
lineseg:
	lseg->x = lseg->x0;
	lseg->xlo = 0x8000;  /* fraction 1/2 to counteract truncation */
	lseg->slope = MixedQuotient(p2->x - p1->x, p1->y - p2->y);
      }
    return(seg);
  }


TopClip(seg, clip)
  register Seg *seg;
  register struct SubView *clip;
  {
    register short x, y;

    /* break linkage, upper connecting edge has been clipped away */
    seg->linked = 0;
    if (seg->type >= ULARC)  /* it's an arc */
      {
	seg->y0 = clip->ScreenYmin;
	y = seg->y0 - seg->cy;
	x = isqrt(seg->rsq - y*y);
	if (seg->type == URARC || seg->type == LRARC)
	    seg->x0 = seg->cx + x;
	else seg->x0 = seg->cx - x;
      }
    else /* it's a straight line */
      { register mixed t;

	t = lseg->slope * (clip->ScreenYmin - lseg->y0);
	lseg->x0 += (t >> 16);
	lseg->x = lseg->x0;
	lseg->xlo = t & 0xffff;
	lseg->y0 = clip->ScreenYmin;
      }
  }


/* Bucket sort on y0; with each bucket, insertion sort on x0.
 * Linked edges do not actually get stored, because they will be found
 * by linkage at the proper time.  But we initialize arc-computation
 * parameters for all arc segments here.
 */
StoreSeg(seg)
  register Seg *seg;
  {
    Seg **bucket;
    register Seg *p;
    register short x, y;
    extern int debug;

    if (seg->type >= ULARC)  /* it's an arc */
      {
	x = seg->x0 - seg->cx;
	seg->x = x;
	seg->xsq = x*x;
	y = seg->y0 - seg->cy;
	seg->y = y;
	seg->ysq = y*y;
	/* sometimes rounding error makes r2 < x2 + y2: here we correct */
	if (seg->rsq < seg->xsq + seg->ysq)
	    seg->rsq = seg->xsq + seg->ysq;
      }
/*  if (debug) PrintSeg(seg);  -- not currently linked in  */
    if (seg->linked) return;

    bucket = &segtable[seg->y0];
    if (*bucket == NULL || seg->x0 < (*bucket)->x0)
      {
	seg->next = *bucket;
	*bucket = seg;
      }
    else 
      {
	for (p = *bucket; p->next && seg->x0 > p->next->x0; p = p->next) ;
	seg->next = p->next;
	p->next = seg;
      }
  }


#define swap(a, b)  (t=a, a=b, b=t)

/* Clipping and storing of horizontal edges */
StoreHorizontal(y, x0, x1, clip)
  register short y, x0, x1;	/* screen coordinates */
  register struct SubView *clip;
  {
    register short t;
    register Seg *seg;

    /* gross clipping */
    if (y < clip->ScreenYmin || y > clip->ScreenYmax ||
	    x1 < clip->ScreenXmin || x0 > clip->ScreenXmax)
	return;
    if (x0 > x1) swap(x0, x1);

    /* fine clipping */
    if (x0 < clip->ScreenXmin) x0 = clip->ScreenXmin;
    if (x1 > clip->ScreenXmax) x1 = clip->ScreenXmax;

    if (!segfree)
      {
	seg = (Seg *) malloc(sizeof(Seg));
	if (!seg)  return;	/* no space, can't store it */
      }
    else
      {
	seg = segfree;
	segfree = seg->next;
      }
    seg->y0 = y;
    seg->x0 = x0;
    seg->x1 = x1;

    seg->next = horizontals;
    horizontals = seg;
  }

/* Advance functions: for line segment, and for each of the four quadrant
 * types of arc.  Computes xleft and xright, in screen coordinates, for the
 * current row, and increments y to prepare for the next row.
 */
AdvanceLine(line)
  register struct lineseg *line;
  {
    register mixed t;

    line->xleft = line->xright = line->x;
    t = line->xlo + (line->x << 16) + line->slope;
    line->xlo = t & 0xffff;
    line->x = t >> 16;
  }

AdvanceUL(arc)
  register Seg *arc;
  {
    register short x = arc->x;
    register unsigned xsq = arc->xsq;
    register unsigned limsq = arc->rsq - arc->ysq;
    register unsigned savesq = xsq;

    arc->xright = x + arc->cx;
    while (xsq <= limsq)
      {
	savesq = xsq;
	xsq += (-x<<1) + 1;
	x--;
      }
    xsq = savesq;
    x++;
    arc->ysq += (arc->y<<1) + 1;
    arc->y++;
    arc->xleft = x + arc->cx;
    arc->x = x;
    arc->xsq = xsq;
  }

AdvanceUR(arc)
  register Seg *arc;
  {
    register short x = arc->x;
    register unsigned xsq = arc->xsq;
    register unsigned limsq = arc->rsq - arc->ysq;
    register unsigned savesq = xsq;

    arc->xleft = x + arc->cx;
    while (xsq <= limsq)
      {
	savesq = xsq;
	xsq += (x<<1) + 1;
	x++;
      }
    xsq = savesq;
    x--;
    arc->ysq += (arc->y<<1) + 1;
    arc->y++;
    arc->xright = x + arc->cx;
    arc->x = x;
    arc->xsq = xsq;
  }


AdvanceLR(arc)
  register Seg *arc;
  {
    register short x = arc->x;
    register unsigned xsq = arc->xsq;
    register unsigned newysq;

    arc->xright = x + arc->cx;
    newysq = arc->ysq + (arc->y<<1) + 1;
    while (newysq + xsq > arc->rsq && x > 0)
      {
	xsq += (-(arc->x)<<1) + 1;
	x--;
      }
    arc->xleft = x + arc->cx;
    if (arc->xleft < arc->x1)  arc->xleft = arc->x1;  /* clipped final row */
    arc->ysq = newysq;
    arc->y++;
    arc->xsq = xsq;
    arc->x = x;
  }

AdvanceLL(arc)
  register Seg *arc;
  {
    register short x = arc->x;
    register unsigned xsq = arc->xsq;
    register unsigned newysq;

    arc->xleft = x + arc->cx;
    newysq = arc->ysq + (arc->y<<1) + 1;
    while (newysq + xsq > arc->rsq && x < 0)
      {
	xsq += (arc->x<<1) + 1;
	x++;
      }
    arc->xright = x + arc->cx;
    if (arc->xright > arc->x1)  arc->xright = arc->x1;  /* clipped, final row */
    arc->ysq = newysq;
    arc->y++;
    arc->xsq = xsq;
    arc->x = x;
  }


/* Pattern writing function, see framebuf.h.  Where pattern bit is on,
 * writes black; where pattern bit is off, leaves alone.
 */
#define GXandInvPattern (GXDEST & ~GXMASK)
extern short fill_area_color, fill_area_opacity;

Seg *active;  /* Head of the active edge table */
Seg BogusLeftEdge = {NULL, NULL, NULL, -32768, -32768}; /* left of everything */

ScanFill(pattern, clip)
  unsigned pattern[];
  register struct SubView *clip;
  {
    register Seg *seg, *p, *q;
/* newseg and temp are logically separate variables from p and q, but I
 * need them to occupy the same registers with p and q! */
#define temp p
#define newseg q
    register short y, x1, x2;
    register int t;
    short on;

    /* set the framebuffer function once and for all */
    SetFillFunction(fill_area_color, fill_area_opacity);

    active = &BogusLeftEdge;  /* simplifies insertion */
    active->next = NULL;
    for (y = clip->ScreenYmin; y <= clip->ScreenYmax; y++)
      {
	/* Edge linkage: an old one hands off to a new one.  If the old
	 * one was going to generate several pixels of edge on this row,
	 * we just make a horizontal segment of it.
	 */
	for (seg = active; seg->next; seg = seg->next)
	  {
	    if (seg->next->y1 <= y && seg->next->link)
	      {
		p = seg->next;
		p->advance(p);  /* compute its final row */
		if (p->xleft != p->xright && p->y1 == y)
		    StoreHorizontal(y, p->xleft, p->xright, clip);
		p->link->next = p->next;
		seg->next = p->link;
		FreeSeg(p);
	      }
	  }

	/* remove outdated edges */
	for (seg = active; seg->next; )
	  {
	    if (seg->next->y1 < y)  /* a local minimum */
	      {
		temp = seg->next;
		seg->next = seg->next->next;
		temp->next = segfree;
		segfree = temp;
	      }
	    else  seg = seg->next;
	  }

	/* update x values and make sure active edge list stays sorted in x */
	seg = active;
	p = seg->next;
	if (!p)  goto newsegs;  /* no edges to sort */
	p->advance(p);
	q = p->next;
	while(q)
	  {
	    q->advance(q);
	    if (p->xleft > q->xleft)  /* time for a swap */
	      {
		seg->next = q;
		p->next = q->next;
		q->next = p;
		seg = q;
	      }
	    else
	      {
		seg = p;
		p = q;
	      }
	    q = p->next;
	  }

	/* insert new edges that we may have come to */
newsegs:
	seg = active;
	newseg = segtable[y];  /* most common case: empty */
	while (newseg)
	  {
	    newseg->advance(newseg);  /* compute first xleft, xright */
	    /* insert it into the list at the right place */
	    while (seg->next && seg->next->xleft < newseg->xleft)
		seg = seg->next;
	    temp = newseg->next;
	    newseg->next = seg->next;
	    seg->next = newseg;
	    newseg = temp;
	  }
	
	/* The drawing itself */
	on = 0;
	for (seg = active->next; seg && seg->xright < clip->ScreenXmin; 
		seg = seg->next)
	    on ^= 1;
	if (on)
	  {
	    x1 = clip->ScreenXmin;
	    x2 = seg->xright;
	    if (x2 > clip->ScreenXmax)  x2 = clip->ScreenXmax;
	    FillRow(y, x1, x2, pattern);
	    seg = seg->next;
	  }
	while (seg && seg->xleft <= clip->ScreenXmax)
	  {
	    x1 = seg->xleft;
	    if (x1 < clip->ScreenXmin)  x1 = clip->ScreenXmin;
	    seg = seg->next;
	    x2 = seg->xright;
	    if (x2 > clip->ScreenXmax)  x2 = clip->ScreenXmax;
	    FillRow(y, x1, x2, pattern);
	    seg = seg->next;
	  }
      }  /* end loop on scan lines */
    FreeWholeList(active->next);

    /* now we draw in all the horizontals */
    for (seg = horizontals; seg; seg = seg->next)
	FillRow(seg->y0, seg->x0, seg->x1, pattern);
    FreeWholeList(horizontals);
  } /* end ScanFill */


FreeWholeList(head)
  Seg *head;
  {
    register Seg *seg;

    if (!head) return;
    for (seg = head; seg->next; seg = seg->next) ;
    seg->next = segfree;
    segfree = head;
  }
