/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* edit.c - Line Editing functions
 *
 *
 * Bill Nowicki September 1983
 *
 * Per Bothner September 1983
 * Added EditEndLine (^E), left and right arrow (<Esc>[D and <Esc>[C)
 *
 * Renamed EditRelease to EditReleaseLine. Added new EditRelease. PB Oct 83
 */

#include <Vioprotocol.h>
#include <Vtermagent.h>
#include <Vgts.h>
#include <Vgtp.h>
#include "sdf.h"
#include "vgt.h"
#include "pad.h"
#include "chars.h"

extern short Debug;
enum EditKind KeyTable[], EscKeyTable[];

/*
 * converts a possible control character to printing version
 */
#define CtrlToPrinting(c) ( c | 0x40 )

struct Buffer *CreateBuffer()
  {
    register struct Buffer *b;
    
    b = (struct Buffer *) malloc( sizeof(struct Buffer) );
    if (b)
      {
        b->column = 0;
	b->count = 0;
	b->escapeMode = 0;
      }
    return(b);
  }


DestroyBuffer(b)
    struct Buffer *b;
  {
    if (b)
      free(b);
  }

HandleShoveInput(term1,msg,pid)
    register Term1 *term1;
    register IoRequest *msg;
  {
  	/*
	 *  The client wants to shove some input into the line buffer
	 */
     register struct Buffer *b = term2Ptr(term1)->lineEditBuf;
     TtyPadType *pad = padPtr(term1);
     int i;

      if (b)
        {
	 if (msg->bytecount + b->count >= LineBufferSize-1) 
	 	msg->bytecount = LineBufferSize-1 - b->count;
	 msg->requestcode = OK;
	 MoveFrom(pid, b->c+b->column, msg->bufferptr, msg->bytecount);
	 if (term1->mode & Echo)
	     for (i=msg->bytecount;i>0;i--)
	       PadPutChar(b->c[b->column++],pad); 
	 b->count += msg->bytecount;
	}
      else msg->requestcode = NO_SERVER_RESOURCES;
      Reply(msg,pid);
  }


LineEdit(term1,c)
    register Term1 *term1;
    char c;
  {
  	/*
	 * Perform line editing on the current line buffer
	 */
    register struct Buffer *b = term2Ptr(term1)->lineEditBuf;
    TtyPadType *pad = padPtr(term1);
    register enum EditKind action;

    if (b==NULL) return;

    switch (b->escapeMode)
      {
	case 0: action = KeyTable[c]; break;

	case 1: action = EscKeyTable[c]; break;

	default:
	    switch (c)
	      {
		case 'C': action = EditForwardChar; break;
		case 'D': action = EditBackwardChar; break;
		default: action = EditNop; break;
	      }
      }

    switch (action)
      {
        case EditEscape:
	    b->escapeMode++;
	    return;

        case EditInsert:
	    InsertInLine(b);	    
	    if (b->column >= LineBufferSize-1) break;
	    b->c[b->column++] = c;
	    if (term1->mode & Echo) 
	      {
	    	PadInsertChar(pad);
		PadPutChar(c,pad);
	      }
	    break;

        case EditInsertTab:
	    do
	      {
		InsertInLine(b);	    
		if (b->column >= LineBufferSize-1) break;
		b->c[b->column++] = ' ';
		if (term1->mode & Echo) 
		  {
		    PadInsertChar(pad);
		    PadPutChar(' ',pad);
		  }
	      }
	    while ((b->column & 7) != 0);
	    break;

        case EditReleaseLine:
	    FlushLineBuffer(term1);
	    PadPutChar(CR,pad);
	    PadPutChar(LF,pad);
	    if (term1->mode & CR_Input)
	      UserPush(term1, LF );
	    else
	      UserPush(term1, c );
	    break;

	case EditRelease:
	    while (b->column < b->count)
	      {
	        b->column++;
	        PadCursorForward(pad);
	      }
	    b->c[b->count++] = c;
	    FlushLineBuffer(term1);
	    EventToUser(term1);
	    break;

        case EditDeleteChar:
	    if (b->column <= 0) break;
	    DeleteInLine(b);
	    b->column--;
	    PadPutChar(BS, pad);
            PadDeleteChar(pad);
	    break;

        case EditDeleteWord:
		/*
		 * Delete any spaces, then any non-spaces.
		 */
	    while (b->column > 0 && !WordChar(b->c[b->column-1]) )
	      {
	        DeleteInLine(b);
	        b->column--;
	        PadPutChar(BS, pad);
                PadDeleteChar(pad);
	      }
	    if (b->column <= 0) break;
	    while (b->column > 0 && WordChar(b->c[b->column-1]) )
	      {
	        DeleteInLine(b);
	        b->column--;
	        PadPutChar(BS, pad);
                PadDeleteChar(pad);
	      }
	    break;

        case EditDeleteNextChar:
	    if (b->column >= b->count) break;
	    b->column++;
	    DeleteInLine(b);
	    b->column--;
            PadDeleteChar(pad);
	    break;

        case EditDeleteNextWord:
		/*
		 * Delete any spaces, then any non-spaces.
		 * but in the forward direction!
		 */
	    while (b->column < b->count && !WordChar(b->c[b->column]) )
	      {
	    	b->column++;
	    	DeleteInLine(b);
	    	b->column--;
                PadDeleteChar(pad);
	      }
	    if (b->column >= b->count) break;
	    while (b->column < b->count && WordChar(b->c[b->column]) )
	      {
	    	b->column++;
	    	DeleteInLine(b);
	    	b->column--;
                PadDeleteChar(pad);
	      }
	    break;

        case EditBackwardChar:
	    if (b->column <= 0) break;
	    b->column--;
	    PadPutChar(BS, pad);
	    break;

        case EditBackwardWord:
		/*
		 * skip any spaces, then any non-spaces.
		 */
	    while (b->column > 0 && !WordChar(b->c[b->column-1]) )
	      {
	        b->column--;
	        PadPutChar(BS, pad);
	      }
	    if (b->column <= 0) break;
	    while (b->column > 0 && WordChar(b->c[b->column-1]) )
	      {
	        b->column--;
	        PadPutChar(BS, pad);
	      }
	    break;

        case EditForwardChar:
	    if (b->column >= b->count) break;
	    b->column++;
	    PadCursorForward(pad);
	    break;

        case EditForwardWord:
		/*
		 * Skip any spaces, then any non-spaces.
		 * but in the forward direction!
		 */
	    while (b->column < b->count && !WordChar(b->c[b->column]) )
	      {
	        PadCursorForward(pad);
		b->column++;
	      }
	    if (b->column >= b->count) break;
	    while (b->column < b->count && WordChar(b->c[b->column]) )
	      {
	        PadCursorForward(pad);
		b->column++;
	      }
	    break;

        case EditDeleteNextLine:
	    while (b->column < b->count) 
	      {
	        b->column++;
	        DeleteInLine(b);
	        b->column--;
	        PadDeleteChar(pad);
	      }
	    break;

        case EditDeleteLine:
	    while (b->column > 0) 
	      {
	        b->column--;
	        DeleteInLine(b);
	        PadPutChar(BS, pad);
	        PadDeleteChar(pad);
	      }
	    break;

        case EditBeginLine:
	    while (b->column > 0)
	      {
	        b->column--;
	        PadCursorBackward(pad);
	      }
	    break;

        case EditEndLine:
	    while (b->column < b->count)
	      {
	        b->column++;
	        PadCursorForward(pad);
	      }
	    break;

        case EditTransposeChar:
	    if (b->column > 1)
	      {
	        char temp = b->c[b->column-2];
		b->c[b->column-2] = b->c[b->column-1];
		b->c[b->column-1] = temp;
	        PadCursorBackward(pad);
	        PadCursorBackward(pad);
		PadPutChar(b->c[b->column-2],pad);
		PadPutChar(b->c[b->column-1],pad);
	      }
	    break;

        case EditKillBreak:
	    FlushLineBuffer(term1);
	    EventToUser(term1);
	    if (term2Ptr(term1)->termPid==0) break;
	    if (term1->mode & Echo) 
	      {
	        PadPutChar('^',pad);
	        PadPutChar( CtrlToPrinting(c), pad);
	      }
	    DestroyProcess(term2Ptr(term1)->termPid);
	    term2Ptr(term1)->termPid = 0;
	    break;

        case EditEOF:
	    FlushLineBuffer(term1);
	    if (term1->mode & Echo) 
	      {
	        PadPutChar('^',pad);
	        PadPutChar( CtrlToPrinting(c), pad);
	      }
	    PutEndEvent(term1);
	    break;

        case EditBeep:
	    PadPutChar(BELL,pad);
	    break;

        case EditNop:
	    break;
      }
    b->escapeMode = FALSE;
    if (Debug & DebugVgtsLineEdit)
      {
        int i;

        dprintf("Action: %d Line: col=%d count=%d: ", action,
		b->column, b->count );
	for (i=0;i<b->count;i++)
	  dputchar( b->c[i]);
	dputchar('\n');	
      }
  }


InsertInLine(b)
    register struct Buffer *b;
  {
    register char *from, *to;
    register char *end;
    
    end = b->c + b->column;
    to = b->c + b->count;
    for (from = to - 1 ;to > end; )
      *to-- = *from--;
    if (b->count<LineBufferSize-1)
      b->count++;
  }


DeleteInLine(b)
    register struct Buffer *b;
  {
    register char *from, *to;
    register char *end;
    
    end = b->c + b->count;
    from = b->c + b->column;
    for (to = from - 1 ;to < end; )
      *to++ = *from++;
    b->count--;
  }

WordChar(c)
    char c;
  {
  	/*
	 * return true if the character is considered part of a word.
	 */
    if ( 'A' <= c && c<= 'Z' ) return (TRUE);
    if ( 'a' <= c && c<= 'z' ) return (TRUE);
    if ( '0' <= c && c<= '9' ) return (TRUE);
    return(FALSE);
  }


FlushLineBuffer(term1)
    register Term1 *term1;
  {
    register struct Buffer *b = term2Ptr(term1)->lineEditBuf;
    TtyPadType *pad = padPtr(term1);

    if (b==NULL) return;
    for (b->column = 0; b->column < b->count; b->column++)
	      UserPut(term1, b->c[b->column] );
    b->column = 0;
    b->count = 0;
  }
