/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1984			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * File: attrib.c
 *
 * PHIGS-style attribute handling for the VGTS
 *
 */

#include "Vgts.h"
#include "sdf.h"

/* attribute variables */
short fill_pattern;
short fill_area_opacity;
short fill_border_style;
short fill_area_color;
short marker_type;
short marker_color;
short polyline_color;
short text_color;
short font_index;

/* Set the attributes to their initial values, as we begin an sdf traversal
 * to redraw a VGT.
 */
InitAttributes()
  {
    fill_pattern = 1;		/* default: solid (black) */
    fill_area_opacity = 0;	/* default: transparent */
    fill_border_style = 0;	/* default 0 means no border */
    fill_area_color = 0;	/* default: black */
    marker_type = 1;		/* default: minimal dot */
    marker_color = 0;		/* default: black */
    polyline_color = 0;	/* default: black */
    text_color = 0;		/* default: black */
    font_index = 0;		/* default: the SIMPLE_TEXT font */
  }


/* An attribute stack is used to push attributes as we descend the sdf
 * structure, maintaining the rule that a subordinate symbol cannot by any
 * means affect the attributes of its calling symbol.  If calls proceed
 * beyond the limit of 20, this will not be true, and things will work very
 * strangely indeed.
 */
typedef struct AttribStackRecord {
	short fill_pattern;
	short fill_area_opacity;
	short fill_border_style;
	short fill_area_color;
	short marker_type;
	short marker_color;
	short polyline_color;
	short text_color;
	short font_index;
	}  AttribStackRecord;

#define ATTRIB_STACK_DEPTH 20
AttribStackRecord attribStack[ATTRIB_STACK_DEPTH];
short attribStackPointer = 0;

PushAttributes()
  {
    register AttribStackRecord *r = &attribStack[attribStackPointer];

    if (attribStackPointer >= ATTRIB_STACK_DEPTH) return;

    r->fill_pattern = fill_pattern;
    r->fill_area_opacity = fill_area_opacity;
    r->fill_border_style = fill_border_style;
    r->fill_area_color = fill_area_color;
    r->fill_pattern = fill_pattern;
    r->marker_type = marker_type;
    r->marker_color = marker_color;
    r->polyline_color = polyline_color;
    r->text_color = text_color;
    r->font_index = font_index;
    attribStackPointer++;
  }

PopAttributes()
  {
    register AttribStackRecord *r;

    if (attribStackPointer <= 0) return;
    r = &attribStack[--attribStackPointer];

    fill_pattern = r->fill_pattern;
    fill_area_opacity = r->fill_area_opacity;
    fill_border_style = r->fill_border_style;
    fill_area_color = r->fill_area_color;
    fill_pattern = r->fill_pattern;
    marker_type = r->marker_type;
    marker_color = r->marker_color;
    polyline_color = r->polyline_color;
    text_color = r->text_color;
    font_index = r->font_index;
  }


/* Attribute definition tables, which are by SDF, allow the user to change
 * the definitions of some attribute values, such as marker types and fill
 * patterns.  InitAttributeTable sets up these tables to their default states,
 * at the creation of a new sdf.
 */
extern short *markerbits[];
extern short *Patterns[];

InitAttributeTables(sdf)
  register SdfTableType *sdf;
  {
    register int i;

    for (i=0; i<MAX_MARKER_TYPE_INDEX; i++)
	sdf->markers[i] = markerbits[i];
    for (i=0; i<MAX_FILL_PATTERN_INDEX; i++)
	sdf->patterns[i] = Patterns[i];
    for (i=0; i<MAX_FONT_INDEX; i++)
	sdf->fonts[i] = 0;
  }


DefineMarkerType(sdf, index, m)
  register SdfTableType *sdf;
  int index;
  short *m;
  {
    register short *dst;

    if (index < 0 || index >= MAX_MARKER_TYPE_INDEX)  return(0);
    sdf->markers[index] = m;
    return(1);
  }


DefinePattern(sdf, index, p)
  register SdfTableType *sdf;
  int index;
  short *p;
  {
    register short *dst;

    if (index < 0 || index >= MAX_FILL_PATTERN_INDEX)  return(0);
    sdf->patterns[index] = p;
    return(1);
  }
