/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Top-level routines for the V team server.
 */


#include "team.h"
#include <Vquerykernel.h>
#include <Vgroupids.h>

TeamsEntry Teams[MaxTeams] = {0};
TeamsEntry *TeamsFreeList;	/* Stack of free team entry records. */
TeamsEntry *TeamsInUseList;	/* Circular, doubly-linked list of active 
				   team entry records. */

int LockId = 0;

SystemCode CheckHostStatus();			/*ms*/

SystemCode LoadTeamSpace();
SystemCode TerminateTeams();
SystemCode InvokeDebugger();
SystemCode UpdateHostStatusReq();
SystemCode CheckOnUpdateStatus();
SystemCode SetTeamPriorityReq();
SystemCode GetTeamPriorityReq();
static SystemCode CreateInstance();
static SystemCode QueryInstance();
static SystemCode ReadInstance();
static SystemCode HandleReadDescriptor();
static SystemCode HandleQueryName();
static SystemCode HandleGetName();
static SystemCode HandleGetContextId();
static SystemCode HandleRemoveFile();
SystemCode MigrationRequest();
SystemCode TransferHostRequest();
SystemCode DestroyHostRequest();
SystemCode LookupName(), MapCid();

extern char *malloc();
extern ProcessId LoadTeamFromFileInstance();

#define min(a,b) (((a) < (b)) ? (a) : (b))


/*
 * InitTeamServer:
 * Starts up the team server if it doesn't already exist.
 */

InitTeamServer(newProcess)
    int newProcess;		/* True if TeamServer should run as a 
				   separate process. */
  {
    ProcessId timerPid, validatorPid;
    int TeamServer(), TeamTimerProcess(), ValidatorProcess();
    int UpdaterProcess();

    if (newProcess)
      {
	TeamServerPid = Create(3, TeamServer, 5000);
	if (TeamServerPid == 0)
	  {
	    printf("ERROR - Failed to create team server process.\n");
	    exit();
	  }
	if (!Ready(TeamServerPid, 0))
	  {
	    printf("ERROR - Failed to ready team server process.\n");
	    exit();
	  }
      }
    else
	TeamServerPid = GetPid(ACTIVE_PROCESS, LOCAL_PID);

    /* Set the first team's priority to REAL_TIME3. */
    SetTeamPriority(TeamServerPid, DEFAULT_TEAM_PRIORITY + 0x300);

    /* Start up timer process. */
    timerPid = Create(4, TeamTimerProcess, 3000);
    if (timerPid == 0)
      {
        printf("ERROR - Failed to create team server timer process.\n");
	exit();
      }
    if (!Ready(timerPid, 0))
      {
        printf("ERROR - Failed to ready team server timer process.\n");
	exit();
      }

    /* Start up validator process. */
    validatorPid = Create(4, ValidatorProcess, 2000);
    if (validatorPid == 0)
      {
        printf("ERROR - Failed to create team server validator process.\n");
	exit();
      }
    if (!Ready(validatorPid, 0))
      {
        printf("ERROR - Failed to ready team server validator process.\n");
	exit();
      }

    if (!newProcess)
	TeamServer();
  }


/*
 * TeamServer:
 * Top level of the team server.
 */

TeamServer()
  {
    Message msg;
    LoadTeamRequest *reqMsg = (LoadTeamRequest *) msg;
    LoadTeamReply *repMsg = (LoadTeamReply *) msg;
    ProcessId reqPid;
    ProcessId exceptionServerPid;
    RegHandlerRequest *regReq = (RegHandlerRequest *) msg;
    SystemCode status;
    ProcessId groupId;
    static char segbuf[MAX_APPENDED_SEGMENT+1];
    unsigned segsize;

    /* Join team server and naming groups. */
    groupId = (TeamServerPid & LOGICAL_HOST_PART) + LTEAM_SERVER_GROUP;
    if ( (status = JoinGroup(groupId, TeamServerPid)) != OK ||
	 (status = JoinGroup(VCSNH_SERVER_GROUP, TeamServerPid)) != OK ||
	 (status = JoinGroup(VTEAM_SERVER_GROUP, TeamServerPid)) != OK )
      {
	fprintf(stderr, "ERROR - team server can't join group: %s\n",
		ErrorString(status));
	exit(status);
      }

    /* Initialize team server's internal data structures. */
    InitServerDataStructures();

    /* Register the team server as the exception handler of last resort. */
    exceptionServerPid = GetPid(EXCEPTION_SERVER, LOCAL_PID);
    if (exceptionServerPid)
      {
	regReq->requestcode = REGISTER_HANDLER;
	regReq->regFlag = 1;
	regReq->handlerPid = TeamServerPid;
	regReq->regPid = regReq->handlerPid;
	exceptionServerPid = Send(msg, exceptionServerPid);
	if (exceptionServerPid == 0)
	  {
	    fprintf(stderr, 
	       "ERROR - couldn't register as exception server of last resort");
	    exit();
	  }
      }

    /* Loop on requests for service. */
    for (;;)
      {
	segsize = MAX_APPENDED_SEGMENT;
	reqPid = ReceiveWithSegment(reqMsg, segbuf, &segsize);
	switch (reqMsg->requestcode)
	  {
	    case CREATE_INSTANCE:
	    case CREATE_INSTANCE_RETRY:
		repMsg->replycode = 
			CreateInstance(reqMsg, reqPid, segbuf, segsize);
		break;
	    case READ_INSTANCE:
		repMsg->replycode = ReadInstance(reqMsg, reqPid);
		break;
	    case QUERY_INSTANCE:
		repMsg->replycode = QueryInstance(reqMsg, reqPid);
		break;
	    case RELEASE_INSTANCE:
	    case SET_INSTANCE_OWNER:
		repMsg->replycode = OK;
		break;
	    case READ_DESCRIPTOR:
	    case NREAD_DESCRIPTOR:
		repMsg->replycode = HandleReadDescriptor(reqMsg, reqPid);
		break;
	    case GET_CONTEXT_ID:
	        repMsg->replycode =
			HandleGetContextId(reqMsg, reqPid, segbuf, segsize);
		break;
	    case QUERY_NAME:
	        repMsg->replycode =
			HandleQueryName(reqMsg, reqPid, segbuf, segsize);
		break;
	    case GET_ABSOLUTE_NAME:
	    case GET_CONTEXT_NAME:
	    case GET_FILE_NAME:
	        repMsg->replycode =
			HandleGetName(reqMsg, reqPid, segbuf, segsize);
		break;
	    case TeamLoadRequest:
	        repMsg->replycode = LoadTeamSpace(reqMsg, reqPid);
		break;
	    case TERMINATE_TEAM:
	    case DeadTeamNotice:
	        repMsg->replycode = TerminateTeams(reqMsg, reqPid);
		break;
	    case REMOVE_FILE:
	        repMsg->replycode =
			HandleRemoveFile(reqMsg, reqPid, segbuf, segsize);
		break;
	    case EXCEPTION_REQUEST:
	        repMsg->replycode = InvokeDebugger(reqMsg, reqPid);
		break;
	    case UpdateRequest:
		repMsg->replycode = UpdateHostStatusReq(reqMsg, reqPid);
	        break;
	    case SERVICE_SERVER_ALIVE:
		repMsg->replycode = CheckOnUpdateStatus(reqMsg, reqPid);
		break;
      	    case HOST_STATUS:
	    	repMsg->replycode = CheckHostStatus (reqMsg, reqPid);
		break ;
	    case CHANGE_TEAM_PRIORITY:
		repMsg->replycode = SetTeamPriorityReq(reqMsg, reqPid);
		break;
	    case GET_TEAM_PRIORITY:
		repMsg->replycode = GetTeamPriorityReq(reqMsg, reqPid);
		break;
	    case MIGRATION_REQUEST:
	    case REDO_MIGRATION_REQUEST:
		repMsg->replycode = MigrationRequest(reqMsg, reqPid);
		break;
	    case TRANSFER_HOST_REQUEST:
		repMsg->replycode = TransferHostRequest(reqMsg, reqPid);
		break;
	    case DESTROY_HOST:
		repMsg->replycode = DestroyHostRequest(reqMsg, reqPid);
		break;
	    default:
	        repMsg->replycode = REQUEST_NOT_SUPPORTED;
		break;
	  }
	if (repMsg->replycode != NO_REPLY)
	  {
	    Reply(repMsg, reqPid);
	  }
      }
  }


/*
 * LoadTeamSpace:
 * Loads a team from an already opened file which is specified by a file
 * instance (server, fileid).  Replies to the requestor with the processId
 * of the team's root process and also forwards the team to the requestor so
 * that the he can complete setting up the team (e.g. arguments).
 *
 * %%% Note: Should modify to be a NameRequest so that the team server can
 *     be specified by name!  TPM
 */

SystemCode LoadTeamSpace(reqMsg, reqPid)
    LoadTeamRequest *reqMsg;
    ProcessId reqPid;
  {
    LoadTeamReply *repMsg = (LoadTeamReply *) reqMsg;
    TeamsEntry *teamEntryPtr;
    SystemCode error;
    char buffer[MaxArgsLength];
    char *curPtr, *t;
    int len;
    short nameLength;
    char *name, *teamend;
    ProcessId newPid;
    ProcessId lhost;

    /* Check if this is a remote request and the host is not available
       for such. */
    if (!RequestAllowed(reqMsg, reqPid, NULL))
      {
	return(NO_PERMISSION);
      }

    /* Find an unused entry in the Teams table. */
    teamEntryPtr = PopStack(TeamsFreeList);
    if (teamEntryPtr == NULL)
      {
	return(NO_SERVER_RESOURCES);
      }
    
    /* Get parameters associated with the loaded team. */
    if (reqMsg->bytecount > MaxArgsLength)
      {
        PushStack(TeamsFreeList, teamEntryPtr);
        return(BAD_ARGS);
      }
    error = MoveFrom(reqPid, buffer, reqMsg->bufferptr, reqMsg->bytecount);
    if (error != OK)
      {
        PushStack(TeamsFreeList, teamEntryPtr);
	return(error);
      }
    curPtr = buffer;		/* Points to file name. */
    nameLength = 1 + strlen(curPtr); /* Don't forget the \0 terminator */
    name = malloc(nameLength);
    if (name == NULL)
      {
        PushStack(TeamsFreeList, teamEntryPtr);
        return(NO_MEMORY);
      }
    strcpy(name, curPtr);	/* Assumes that file name is terminated with
				   a \0. */
    curPtr += nameLength;	/* Points past \0 */
    
    /* Try to create and load a team from the specified file instance. */
    if (Remote(reqPid))
      {
	/* Create a new logical host for this team. */
	lhost = 0;
      }
    else
      {
	/* Put the team on the same logical host as its (local) requestor. */
	lhost = reqPid;
      }
    newPid = LoadTeamFromFileInstance(lhost, reqMsg->fileserver, 
		reqMsg->fileid, &error, &teamend);
    if (newPid == NULL)
      {
        free(name);
        PushStack(TeamsFreeList, teamEntryPtr);
        return(error);
      }
    /* Set user number to same as requestor. */
    SetUserNumber( newPid, User(reqPid) );    

    /* Initialize Teams table entry. */
    Lock(LockId);
    AddCList(TeamsInUseList, TeamsInUseList, teamEntryPtr);
    if (reqMsg->noOwner)
      {
	teamEntryPtr->ownerPid = TeamServerPid;
        teamEntryPtr->runPriority = NONRUNNING_BG_TEAM_PRIORITY;
      }
    else
      {
	teamEntryPtr->ownerPid = reqPid;
        teamEntryPtr->runPriority = NONRUNNING_FG_TEAM_PRIORITY;
      }
    if (Remote(reqPid))
      {
	teamEntryPtr->runPriority = NONRUNNING_GUEST_TEAM_PRIORITY;
      }
    teamEntryPtr->inUse = 1;
    Unlock(LockId);
    SetTeamPriority(newPid, teamEntryPtr->runPriority);
    teamEntryPtr->rootPid = newPid;
    teamEntryPtr->name = name;
    teamEntryPtr->loadServer = reqMsg->fileserver;
    teamEntryPtr->loadFile = reqMsg->fileid;

    /* The remainder of the segment contains two "RootMessage"s, which we 
     * may have to byte-swap:
     */
    if (DifferentByteOrder(reqPid))
      {
	ByteSwapRootMessage((RootMessage *)curPtr);
	ByteSwapRootMessage((RootMessage *)(curPtr + sizeof(RootMessage)));
      }
    t = (char *) &(teamEntryPtr->debugRtMsg);
    len = sizeof(RootMessage);
    curPtr += len;		/* Skip over the rtMsg that is currently still
				   being passed along. */
    while (len-- > 0)
	*t++ = *curPtr++;
    
    /* Reply to the requestor with the team's root process. */
    repMsg->replycode = OK;
    repMsg->rootPid = newPid;
    repMsg->teamend = teamend;
    Reply(repMsg, reqPid);
    
    /* Forward the team to the requestor so that he can complete setting it
     *  up.  The contents of the message is unimportant, but it must be a 
     *  valid request (not reply) message. */
    reqMsg->requestcode = NO_OPERATION;
    Forward(reqMsg, newPid, reqPid);

    return(NO_REPLY);
  }


/*
 * TerminateTeams:
 * Destroys the specified team(s).  Also destroys any associated postmortem
 * debuggers.  If the termination
 * request is for a postmortem debugger, then the associated team it was
 * invoked on is also destroyed.
 * Reclamation of teams owned by the team(s) to destroy is also done.
 *
 * %%% Note: could implement terminating teams by name.  TPM
 */

SystemCode TerminateTeams(reqMsg, reqPid)
    ExitTeamRequest *reqMsg;
    ProcessId reqPid;
  {
    TeamsEntry *teamEntryPtr, *tmpPtr;
    ProcessBlock pb;
    ProcessId rootPid;

    if (reqMsg->requestType == SpecificTeam)
      {
        /* Find the root process pid for the specified team. */
	rootPid = reqMsg->rootPid;
	if (reqMsg->requestcode == TERMINATE_TEAM)
	  {
	    if (QueryProcessState(rootPid, &pb) != rootPid)
	      {
		return(BAD_ARGS);
	      }
	    rootPid = pb.team_root;
	  }
	
	/* Find the Teams table entry for the specified team. */
	for (teamEntryPtr = FirstCList(TeamsInUseList);
		teamEntryPtr != NULL;
		teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr))
	    if (rootPid == teamEntryPtr->rootPid)
		break;
	if (teamEntryPtr == NULL)
	  {
	    return(NOT_FOUND);
	  }
	
	/* Terminate and reclaim the teams for the table entry. */
	TerminateSingleTeam(teamEntryPtr, reqMsg->requestcode);
      }
    else
	/* Terminate resp. all teams or all teams of the specified 
	   requestor. */
      {
        teamEntryPtr = FirstCList(TeamsInUseList);
	while (teamEntryPtr != NULL)
	  {
	    if ((reqMsg->requestType == AllTeams) ||
 	    	    (reqPid == teamEntryPtr->ownerPid))
	      {
		tmpPtr = IterCList(TeamsInUseList, teamEntryPtr);
				/* NOTE: This statement must precede the
				   termination routine so that the next ptr
				   is still valid. */
	        /* Terminate and reclaim the teams for this table entry. */
		TerminateSingleTeam(teamEntryPtr, 
				reqMsg->requestcode);
		teamEntryPtr = tmpPtr;
	      }
	    else
	      {
		teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr);
	      }
	  }
      }
    return(OK);
  }


/*
 * HandleRemoveFile:
 * Another way of terminating a specific team.
 */
static SystemCode HandleRemoveFile(req, pid, segbuf, segsize)
  register NameRequest *req;
  ProcessId pid;
  char *segbuf;
  unsigned segsize; 
  {
    SystemCode r;
    ContextPair ctx;
    TeamsEntry *team;

    r = LookupName(req, pid, segbuf, segsize, NULL, &ctx, &team);
    if (r != OK) return r;
    if (team == NULL) return NO_PERMISSION;

    TerminateSingleTeam(team, TERMINATE_TEAM);
    return OK;
  }



/*
 * CreateInstance:
 * Permit reading a context directory.  Nothing is actually
 *   created here, since the teamserver maintains a single, permanent
 *   instance for each context.
 */

static SystemCode CreateInstance(req, reqPid, segbuf, segsize)
    register CreateInstanceRequest *req;
    ProcessId reqPid;
    char *segbuf;
    unsigned segsize;
  {
    register CreateInstanceReply *rep = (CreateInstanceReply *) req;
    SystemCode r;
    ContextPair ctx;
    TeamsEntry *team;

    if (req->requestcode == CREATE_INSTANCE_RETRY &&
        IgnoreRetry(req, reqPid, segbuf, &segsize, TeamServerPid))
      {
	return DISCARD_REPLY;
      }

    r = LookupName(req, reqPid, segbuf, segsize, NULL, &ctx, &team);
    if (r != OK) return r;
    rep->fileid = ctx.cid;

    switch (ctx.cid)
      {
        case GLOBAL_ROOT_CONTEXT:
	case TEAM_SERVER_CONTEXT:
    	    r = QueryInstance(req, reqPid);
	    if (r == OK && IsGroupId(Forwarder(reqPid)))
	        return MORE_REPLIES;
	    else
	        return r;

	case DEFAULT_CONTEXT:
    	    return QueryInstance(req, reqPid);

	default:
	    return INVALID_CONTEXT;
      }
  }


/*
 * QueryInstance:
 * Return parameters of a context directory instance.
 */

static SystemCode QueryInstance(req, pid)
    QueryInstanceRequest *req;
    ProcessId pid;
  {
#define reply ((QueryInstanceReply *) req)

    reply->fileserver = TeamServerPid;
    reply->blocksize = sizeof(ArbitraryDescriptor);
    reply->filetype = READABLE|FIXED_LENGTH;
    reply->filenextblock = 0;

    switch (req->fileid)
      {
	case GLOBAL_ROOT_CONTEXT:
	    reply->filelastblock = 0;
	    reply->filelastbytes = reply->blocksize;
	    break;
	
	case TEAM_SERVER_CONTEXT:
	    reply->filelastblock = 0;
	    reply->filelastbytes = reply->blocksize;
	    break;

	case DEFAULT_CONTEXT:
    	    reply->filelastblock = MaxTeams-1;
	    reply->filelastbytes = reply->blocksize;
	    break;

	default:
	    return INVALID_FILE_ID;
      }

    return OK;

#undef reply
  }


/*
 * ReadInstance:
 * Read entries from a context directory.
 */

static SystemCode ReadInstance(req, pid)
    register IoRequest *req;
    ProcessId pid;
  {
    unsigned reqlen, len, block;
#define reply ((IoReply *) req)
    ArbitraryDescriptor desc;
    SystemCode error;
    int i, last;
    TeamsEntry *teamEntryPtr;

    reqlen = req->bytecount;
    block = req->blocknumber;
    reply->bytecount = 0;
    last = 0;

    if (reqlen > sizeof(ArbitraryDescriptor))
	return(BAD_BYTE_COUNT);

    switch (req->fileid)
      {
	case GLOBAL_ROOT_CONTEXT:
	  if (block > 0)
	      return END_OF_FILE;
	  last++;
	  len = BuildDescriptor(&desc, NULL, TEAM_SERVER_CONTEXT, pid);
	  break;

	case TEAM_SERVER_CONTEXT:
	  if (block > 0) return END_OF_FILE;
	  last++;
	  len = BuildDescriptor(&desc, NULL, DEFAULT_CONTEXT, pid);
	  break;

	case DEFAULT_CONTEXT:
	  /* Directory of teams */
	  teamEntryPtr = FirstCList(TeamsInUseList); 
          for (i = 0; (i < block) && (teamEntryPtr != NULL); i++)
	    {
	      teamEntryPtr = IterCList(TeamsInUseList, teamEntryPtr);
	    }
	  if (teamEntryPtr == NULL)	
	      return(END_OF_FILE);
	  if (teamEntryPtr->next == NULL) last++;

	  /* Set up descriptor to return */
	  len = BuildDescriptor(&desc, teamEntryPtr,
	  			       teamEntryPtr->rootPid, pid);
	  break;

	default:
	  return INVALID_FILE_ID;
      }

    /* Do the actual data movement and reply-to-read. */
    reply->bytecount = len;
    if (len > MAX_APPENDED_SEGMENT)
      {
	/* This can't happen since all descriptors are shorter than
	 *  MAX_APPENDED_SEGMENT!
	 */
	error = MoveTo(pid, req->bufferptr, &desc, len);
	if (error != OK)
	  {
	    reply->bytecount = 0;
	    return (error);
	  }
	else if (last)
	    return (END_OF_FILE);  /* successful read to EOF */
	else
	    return (OK);
      }
    else
      {
	if (last)
	    reply->replycode = END_OF_FILE;  /* successful read to EOF */
	else
            reply->replycode = OK;
	
	ReplyWithSegment(reply, pid, &desc, req->bufferptr, len);
        return(NO_REPLY);
      }
#undef reply
  }


/*
 * HandleReadDescriptor:
 * Handles READ_DESCRIPTOR or NREAD_DESCRIPTOR
 * Read the descriptor of a single object, specified by instance id or name.
 *
 */

static SystemCode HandleReadDescriptor(req, pid)
    register DescriptorRequest *req;
    ProcessId pid;
  {
#define reply ((DescriptorReply *) req)
    ArbitraryDescriptor desc;
    unsigned len;
    TeamsEntry *teamEntryPtr;
    char *name = (char *) &desc;
    SystemCode status;
    ContextPair ctx;

    
    if (req->requestcode == NREAD_DESCRIPTOR)
      {
        /* Find named object */
        status = 
	    LookupName(req, pid, NULL, 0, NULL, &ctx, &teamEntryPtr);
        if (status != OK) return status;
      }
    else
      {
	/* File id is the same as the context id */
	/* The descriptor returned really describes the basic
	 *  object at the same node as the context directory
	 *  we are reading, not the context dir itself.
	 */
	ctx.cid = req->fileid;
	status = MapCid(&ctx, &teamEntryPtr);
	if (status != OK || teamEntryPtr != NULL)
	    return INVALID_FILE_ID;
      }

    /* Build a descriptor for it */
    len = BuildDescriptor(&desc, teamEntryPtr, ctx.cid, pid);
    if (req->segmentlen - req->dataindex < len)
	return (BAD_BUFFER);

    /* Do the actual data movement. */
    if (len > MAX_APPENDED_SEGMENT)
      {
	return(MoveTo(pid, req->segmentptr + req->dataindex,
	    &desc, len));
      }
    else
      {
        reply->replycode = OK | REPLY_SEGMENT_BIT;
	reply->segmentptr = req->segmentptr + req->dataindex;
	reply->segmentlen = len;
	ReplyWithSeg(reply, pid, &desc);
        return(NO_REPLY);
      }
#undef reply
  }    
    


static SystemCode HandleQueryName(req, pid, segbuf, segsize)
  register ContextRequest *req;
  ProcessId pid;
  char *segbuf;
  unsigned segsize;
  {
    char *suffix;
    ContextPair ctx;
    TeamsEntry *team;
#define reply ((ContextReply *) req)
    extern char NameBuffer[];
    register SystemCode r;

    r = LookupName(req, pid, segbuf, segsize, &suffix, &reply->context, &team);
    if (r != OK) return r;
    
    reply->nameindex += suffix - NameBuffer;
    return OK;

#undef reply
  }


static SystemCode HandleGetContextId(req, pid, segbuf, segsize)
  ContextRequest *req;
  ProcessId pid;
  char *segbuf;
  unsigned segsize;
  {
    TeamsEntry *team;
#define reply ((ContextReply *) req)

    return ( LookupName(req, pid, segbuf, segsize, 
    		        NULL, &(reply->context), &team) );

#undef reply
  }


static SystemCode HandleGetName(req, pid, segbuf, segsize)
  register ContextRequest *req;
  ProcessId pid;
  char *segbuf;
  unsigned segsize;
  {
    char *suffix;
    ContextPair ctx;
    TeamsEntry *team;
#define reply ((ContextReply *) req)
    SystemCode r;
    extern char NameBuffer[];

    switch (req->requestcode)
      {
	case GET_ABSOLUTE_NAME:
    	  r = LookupName(req, pid, segbuf, segsize, &suffix, &ctx, &team);
          /* No code to guess at the name if not found, because
	   *  this server does not create things by name.
	   */
	  if (r != OK) return r;
	  reply->context = ctx;
	  break;

	case GET_CONTEXT_NAME:
	  ctx.cid = req->context.cid;
	  r = MapCid(&ctx, &team);
	  if (r != OK) return r;
	  break;

	case GET_FILE_NAME:
	  ctx.cid = req->instanceid;
	  r = MapCid(&ctx, &team);
	  if (r != OK || team != NULL) return INVALID_FILE_ID;
	  break;
      }

    /* We have the context id and also the teams table pointer,
     *  if this is a team.
     */

    switch (ctx.cid)
      {
	case GLOBAL_ROOT_CONTEXT:
	  strcpy(NameBuffer, "[");
	  break;
	case TEAM_SERVER_CONTEXT:
	  strcpy(NameBuffer, "[team/");
	  break;
	case DEFAULT_CONTEXT:
	  strcpy(NameBuffer, "[team/");
	  strcat(NameBuffer, hostname());
	  strcat(NameBuffer, "]");
	  break;
	default:
	  if (team == NULL) return NOT_FOUND;  /* should not happen */
	  strcpy(NameBuffer, "[team/");
	  strcat(NameBuffer, hostname());
	  strcat(NameBuffer, "]");
	  strcat(NameBuffer, team->name);
	  break;
      }
   
    
    reply->replycode = OK | REPLY_SEGMENT_BIT;
    reply->nameptr = req->nameptr;
    reply->namelength = strlen(NameBuffer) + 1;
    ReplyWithSeg(reply, pid, NameBuffer);

    return NO_REPLY;

#undef reply
  }
