/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* edprimitives.c - Primitives for VGTS line editing
 *
 *
 * Craig Dunwoody December 1983
 *
 */


#include "edit.h"


/*	Principle: there is no such thing as a non-printing character.	*/
/*	Control characters are shown in "^A" style, while eighth-bit	*/
/*	characters are shown in "\200" style.				*/


int Charsize(ch)

  unsigned char ch;

  {

	/*
	 * This routine returns the width of the visual representation of the
	 * given character.
	 */

    if ( ch < MinPrintingChar || ch == DEL ) return(2);

    else if (ch <= MaxPrintingChar) return(1);

    else return(4);

  }
 

unsigned char *Charstring(ch)

  unsigned char ch;

  {

	/*
	 * This routine returns the string used as the visual representation
	 * of the given character.
	 */

    static unsigned char chstring[5];

    switch ( Charsize(ch) )

      {

	case 1:
	    chstring[0] = ch; 
	    break;

	case 2:
	    chstring[0] = '^';
	    if (ch == DEL) chstring[1] = '?';
	    else chstring[1] = ch + '@';
	    break;

	case 4:
	    chstring[0] = '\\';
	    sprintf( chstring+1, "%3o", ch );
	    break;

      }

    return(chstring);

  }


/* Findend: given a straytext pointer, return a Mark to the end of text. */
Mark  Findend(chunk)  register Chunk chunk;
{ 
  if (chunk == NULL)  ErrorMsg("Findend of NULL");
  for (; chunk->next; chunk = chunk->next);  /* seek last chunk */
  return( Makemark(chunk, Chunkend(chunk)) );
  }



/*	Functions to insert and delete text 		*/

/* This is a good place to document policy on marks.  A Permanent Mark is a
variable of type Mark whose value endures from command to command; A
Normal Mark is a permanent mark to which a pointer is kept on the Marklist:
hence it will be corrected in the standard way after an insertion or deletion.
curmark and endmark and regionmark are normal marks; beginmark is not.
beginmark is changed only by special test, and the first chunk is seldom
changed or relinquished.
*/

static MarkLink marklinkp;

/* TextInsertChar: the textual (as opposed to visual, or graphical) aspect of
   inserting a character.  All normal marks are left pointing to the character
   they used to point to - a mark on the insertion spot ends up before it.
   Returns Nominal if successful. */

EditStatusCode TextInsertChar(insertMarkp, ch)  Mark *insertMarkp; char ch;
{ register Chunk ichunk;
  register char *icp,*p,*q;
  char *zcp;
  Mark *m;
  int yes;

  ichunk = insertMarkp->chunk;  icp = insertMarkp->cp;  zcp=Chunkend(ichunk);

  if (ichunk->length == CHUNKSIZE) {
    if ((icp - ichunk->text) <= CHUNKSPLITPOINT) { /* 70% rule */
	if (SplitChunk(ichunk,icp+1) != Nominal)
	    return(Status); /*handles all marks*/
	zcp = icp+1;
	}
    else {
	if (SplitChunk(ichunk,ichunk->text + CHUNKSPLITPOINT) != Nominal)
	    return(Status);
	icp = ichunk->next->text + ((icp - ichunk->text) - CHUNKSPLITPOINT);
	ichunk = ichunk->next;
	zcp = Chunkend(ichunk);
	}
    }

  for (p=zcp, q=zcp+1; p > icp; ) *--q = *--p;
  ichunk->length++;
  *icp = ch;
/* A mark sitting after the insertion point gets adjusted by 1 */
  for (m = Firstmark(ichunk,&yes); yes; m = Nextmark(ichunk, &yes) ) {
    if (m->cp > icp) m->cp++;
    }
  if (Atend(*insertMarkp)) ebuf->endmark.cp++;
  if (ebuf->endmark.cp != Chunkend(ebuf->endmark.chunk)) NewMsg("Inconsistent endmark");

  return(Status = Nominal);
  }


/* TextDeleteForward: the textual aspect of deleting a character.  There is
   no DeleteBackward - you move backward then delete forward if need be.
   Returns Nominal if successful.  */

EditStatusCode TextDeleteForward(deleteMarkp)

  Mark *deleteMarkp;

{ register Chunk dchunk;
  register char *p,*q;
  char *dcp, *zcp;
  Mark *m;
  int yes, dcflag;

  if (Atend(*deleteMarkp))  return(Status = AtTextEnd);

  dchunk = deleteMarkp->chunk;  dcp = deleteMarkp->cp;  zcp = Chunkend(dchunk);
  if (dcp == zcp)
    {
      NewMsg("TextDeleteForward error: at end of chunk!");
      return(Status = AtChunkEnd);
    }

  for (p = dcp, q = dcp+1; q < zcp; )  *p++ = *q++;
  dchunk->length--;  zcp--;
  if ((dchunk->length == 0) && (dchunk->prev || dchunk->next)) {
    DeleteChunk(dchunk);
    dcflag = 1;
    }
  else {
    for (m = Firstmark(dchunk,&yes); yes; m = Nextmark(dchunk,&yes) ) {
	if (m->cp > dcp) m->cp--;
	else if (m->cp == zcp && m->chunk->next) {
	    m->chunk = m->chunk->next;
	    m->cp = m->chunk->text;
	    }
	}
    dcflag = 0;
    }
  if ((!dcflag) && dchunk->prev)  Scavenge(dchunk->prev);
  return(Status = Nominal);
  }


/* BlockInsert: (purely textual) Given a Mark and a pointer to the first
chunk of a piece of straytext, inserts that text here.  Usually "Display" is
the appropriate graphical counterpart.  "straytext" has the following
properties:  It has the same chunk structure as buffer text, but there are
no row descriptors for it.  Its last chunk->next is NULL.  It is not linked
into the main text.  There are (by current plans) no permanent marks pointing
into it.  BlockInsert is intended to work whether imark is in a chunk, on a
chunk boundary, at beginning or end of text, or in a null text.  The nchunk
pointer may not be null, but it may point to a null text.  Since an insertion
moves all Marks on the insertion point to the end of the insertion,
BlockInsert returns the Mark of the beginning of the insertion. Sets Status to
Nominal if successful.  */

Mark  BlockInsert(imark, nchunk)  Mark imark;  Chunk nchunk;
{ Chunk ichunk,echunk,lastchunk;
  register char *icp;
  Mark *m;

  ichunk = imark.chunk;  icp = imark.cp;
  
 {
  /* find the end of the block being inserted */
    for (lastchunk = nchunk; lastchunk->next; lastchunk = lastchunk->next);

  /* prepare the main text to receive it: make sure imark is on a chunk
     boundary */
    if (icp == ichunk->text){echunk = ichunk; ichunk = ichunk->prev;}
    else if (!Atend(imark))
      {
	if (SplitChunk(ichunk,icp) != Nominal) return(imark);
	echunk = ichunk->next;
      }
    else echunk = NULL;   /* when we insert at end of text */

/* link in the new text */
    if (Atstart(imark)) {
	ebuf->headmark.chunk = nchunk; ebuf->headmark.cp = nchunk->text;
        ebuf->promptmark = ebuf->inputmark = ebuf->backupmark = ebuf->headmark;
	}
    else { ichunk->next = nchunk; nchunk->prev = ichunk; }

    if (Atend(imark)) {
        for (marklinkp = ebuf->markList; marklinkp; marklinkp = marklinkp->next)
	{
	    m = marklinkp->pmark;
	    if (m->cp == imark.cp) { /* in particular, fix endmark */
		m->chunk = lastchunk;  m->cp = Chunkend(lastchunk);
		}
	    }
	if (echunk)
	  {
	    free(echunk);
	    echunk = NULL;
	  }
	}
    else { lastchunk->next = echunk;  echunk->prev = lastchunk; }

  /* tempmark0 is on the Marklist so it will survive Scavenges */
    tempmark0 = Makemark(nchunk, nchunk->text);
    if (ichunk) {
	Scavenge(ichunk);
	if (ichunk->prev) Scavenge(ichunk->prev);
	}
    if (echunk) {
	Scavenge(echunk);
	if (echunk->prev) Scavenge(echunk->prev);
	}
    Status = Nominal;
    return(tempmark0);
    }
  } /* end BlockInsert */


/* BlockDelete: given a pair of marks (which must be in order!) delete the
   text between them and return it as a sparetext - stripped of 
   marks.  For large deletes this involves the creation of two new chunks:
   the chunks containing the beginning and end of the block are each split.
   Sets Status to Nominal if successful.  */

Chunk  BlockDelete(bmark,emark)  Mark bmark,emark;
{ Chunk bchunk,	/* beginning of deletion - the part that remains */
        echunk, /* end of deletion - the part that remains */
	nchunk, /* beginning of deletion - the deleted part */
	lchunk, /* last chunk of the deleted part */
	chunk;
  Mark *m;
  char *bcp,*ecp,*zcp;
  int yes, dcflag;

  if (MarkEQ(bmark,emark))
      { Status = NullText; return(NULL);}

  bchunk = bmark.chunk; echunk = emark.chunk;
  bcp = bmark.cp; ecp = emark.cp;  zcp = Chunkend(bchunk);

       /* simple version: part of a single chunk */
  if (bchunk == echunk) {
    if ( !(nchunk = CutSubchunk(bchunk,bcp,ecp)) ) return(NULL);
    }

  else {  /* multichunk version */
    if ( !(nchunk = CutSubchunk(bchunk,bcp,zcp)) ) return(NULL);
    if ( !(lchunk = CutSubchunk(echunk,echunk->text,ecp)) )
      {
	free(nchunk);
	return(NULL);
      }
    nchunk->next = bchunk->next;  nchunk->next->prev = nchunk;
    nchunk->prev = NULL;
    lchunk->prev = echunk->prev;  lchunk->prev->next = lchunk;
    lchunk->next = NULL;
    if (lchunk->length == 0) { /* no empty tail on sparetext */
	lchunk = lchunk->prev;  free(lchunk->next);
	lchunk->next = NULL;
	}
    bchunk->next = echunk;  echunk->prev = bchunk;

  /* clean the marks out of any interior chunks */
    if (nchunk != lchunk) {
      for (chunk = nchunk->next; chunk; chunk = chunk->next)
	for (m = Firstmark(chunk,&yes); yes; m = Nextmark(chunk,&yes) )
	  if (m->chunk == chunk) {m->chunk = echunk;  m->cp = echunk->text;}
      }

  /* move any marks on deletion spot to head of echunk */
    for (m = Firstmark(bchunk,&yes); yes; m = Nextmark(bchunk,&yes) ){
	if (m->cp == bcp){m->chunk = echunk;  m->cp = echunk->text;}
	}
    } /* end multichunk version */

  /* delete an empty chunk - unless it's the only one */
  dcflag = 0;
  if (bchunk->prev || bchunk->next) 
    if (bchunk->length == 0) {DeleteChunk(bchunk); dcflag = 1;}
  if (echunk != bchunk && (echunk->prev || echunk->next))
    if (echunk->length == 0) DeleteChunk(echunk);

  if (!dcflag) {
    Scavenge(bchunk);
    Scavenge(bchunk->prev);
    }

  Status = Nominal;
  return(nchunk);
  }


/* SplitChunk: given the chunk to split, and the cp at which to split it,
   splits it into two chunks, properly linked.  Handles all marks on the
   Marklist properly.  Not currently usable on unlinified text.
   Returns Nominal if successful.  */
EditStatusCode SplitChunk(chunk,cp)  Chunk chunk;  char *cp;
{ register char *p,*q,*zcp;
  Chunk nchunk;
  int k,yes;
  Mark *m;
  
  zcp = Chunkend(chunk);
  k = zcp - cp;
  if (k==0) {
	NewMsg("Inappropriate SplitChunk at beginning of chunk");
	return(Status = AtChunkStart);}

  else if (k==chunk->length) {
	NewMsg("Inappropriate SplitChunk at end of chunk");
	return(Status = AtChunkEnd);}

  if ( !(nchunk = MakeChunk(chunk, k)) ) return(Status);

  for (p=cp, q = nchunk->text; p < zcp; ) *q++ = *p++;
  chunk->length -= k;
  nchunk->next = chunk->next;
  if (chunk->next) nchunk->next->prev = nchunk;
  chunk->next = nchunk;
 
  for (m = Firstmark(chunk,&yes); yes; m = Nextmark(chunk,&yes) ) 
    if (m->cp >= cp) {
	m->chunk = nchunk;
	m->cp = nchunk->text + (m->cp - cp);
	}

  return(Status = Nominal);
  }



Chunk MakeChunk(prev, length)

  Chunk prev;
  int length;
  {

	/*
	 * This routine tries to allocate and initialize a new Chunk.
	 * If the allocation succeeds, the Chunk is initialized by setting its
	 * prev and length fields to the given values and setting its next field
	 * to NULL, and a pointer to the new chunk is returned.  If the 
	 * allocation fails, the routine prints an error message, sets Status
	 * to MakeChunkFailure, and returns NULL.
	 */

    register Chunk chunk;

    if (chunk = (Chunk) malloc(sizeof(struct chunk)))
      {
	chunk->prev = prev;
	chunk->length = length;
	chunk->next = NULL;
	Status = Nominal;
      }

    else
      {
	ErrorMsg("Operation aborted:  out of buffer space.");
	ErrorMsg("Use ^D or DEL to free some space.");
	Status = MakeChunkFailure;
      }

    return(chunk);

  }


/* DeleteChunk: handles pointer-pushing and mark-fixing when we delete a
   chunk.  The chunk should already be empty or otherwise unwanted before
   DeleteChunk is called.  Eventually the freeing task will be passed to
   Scavenger. */

DeleteChunk(dchunk)  Chunk dchunk;
{ Chunk pchunk;
  Mark *m;
  int yes;

  if (dchunk == ebuf->headmark.chunk) {
    if (!dchunk->next) {ErrorMsg("DeleteChunk: solitary chunk!"); return;}
    ebuf->headmark.chunk = dchunk->next;
    ebuf->headmark.cp = ebuf->headmark.chunk->text;
    ebuf->headmark.chunk->prev = 0;
    for (m = Firstmark(dchunk,&yes); yes; m = Nextmark(dchunk,&yes) ) {
	m->chunk = ebuf->headmark.chunk;  m->cp = ebuf->headmark.cp;
	}
    free(dchunk);
    return;
    }
  else if (!dchunk->prev) {ErrorMsg("DeleteChunk: losing head of chain!");
    return;
    }  
  pchunk = dchunk->prev;
  pchunk->next = dchunk->next;
  if (dchunk->next) dchunk->next->prev = pchunk;
  
  if (pchunk->next)
    for (m = Firstmark(dchunk,&yes); yes; m = Nextmark(dchunk,&yes) ) {
	m->chunk = pchunk->next;
	m->cp = m->chunk->text;
	}
  else
    for (m = Firstmark(dchunk,&yes); yes; m = Nextmark(dchunk,&yes) ) {
	m->chunk = pchunk;
	m->cp = Chunkend(pchunk);
	}
  free(dchunk);
  }


/* CutSubchunk: chop out a section of an old chunk and return it as a new
   one.  Any marks collapse into the deletion point: the new chunk contains
   no marks. Sets Status to Nominal if successful. */
Chunk  CutSubchunk(chunk,bcp,ecp)  Chunk chunk;  char *bcp,*ecp;
{ Chunk nchunk;
  Mark *m;
  register char *p,*q;  char *zcp;
  int k,yes;

  k = ecp - bcp;  zcp = Chunkend(chunk);
  if ( !(nchunk = MakeChunk(NULL, k)) ) return(NULL);
  for (p = bcp, q = nchunk->text;  p < ecp; )  *q++ = *p++;
  for (p = bcp, q = ecp;  q < zcp; )  *p++ = *q++;
  chunk->length -= k;

  if (ecp == zcp && chunk->next) {
    for (m = Firstmark(chunk,&yes); yes; m = Nextmark(chunk,&yes) )
	if (m->cp >= bcp) {m->chunk = chunk->next; m->cp = m->chunk->text;}
    }
  else
    for (m = Firstmark(chunk,&yes); yes; m = Nextmark(chunk,&yes) )
      if (m->cp > bcp) {
   	if (m->cp < ecp) m->cp = bcp;
    	else m->cp -= k;
    	}
  Status = Nominal;
  return(nchunk);
  }


/***************************  Mark list subroutines *************************/

/* Mark loop helpers.  Firstmark should be called once, followed by calls to
   Nextmark, until yes == 0.  They first retrieve any marks from the Marklist
   which point into this chunk, then any row descriptors which do so.  The 
   first two fields in a RowRec look just like a Mark, so it is safe to coerc
   a RowRec pointer into a (Mark *) type pointer and fool the outside 
   routine.  */

static MarkLink marklp;
static int	markrow;
static int	donemarks;

Mark * Firstmark(chunk, yes)  Chunk chunk;  int *yes;
{ Mark *pmark;

  donemarks = 0;
  for (marklp = ebuf->markList; marklp != NULL; marklp = marklp->next) {
    if (marklp->pmark->chunk == chunk) {
	*yes = 1;
	return(marklp->pmark);
	}
    }

  *yes = 0;
  donemarks = 1;
  for ( markrow=headrow ; markrow < Nrows && ebuf->rows[markrow].exists ;
	markrow++ ) 
      if ( ebuf->rows[markrow].chunk == chunk ) {*yes = 1; break;}
  if (*yes)
    {
      pmark = (Mark *) &ebuf->rows[markrow];
      return(pmark);
    }
  }

Mark *Nextmark(chunk, yes)  Chunk chunk;  int *yes;
{ Mark *pmark;

  if (donemarks)  { markrow++;  goto A;}
  for (marklp = marklp->next; marklp != NULL; marklp = marklp->next) {
    if (marklp->pmark->chunk == chunk) {
	*yes = 1;
	return(marklp->pmark);
	}
    }

  donemarks = 1;
  for ( markrow=headrow ; markrow < Nrows && ebuf->rows[markrow].exists &&
       !( ebuf->rows[markrow].chunk == chunk ) ; markrow++ );

A:*yes = (markrow < Nrows && ebuf->rows[markrow].exists &&
	  ebuf->rows[markrow].chunk == chunk );
  pmark = (Mark *) &ebuf->rows[markrow];
  return(pmark);
  }


EditStatusCode AddMark(markList, pmark)

  MarkLink	*markList;
  Mark		*pmark;

  {

	/*
	 * If Status is Nominal, this routine tries to add a marklink, with 
	 * a pointer to the given Mark, to the head of a given linked list of
	 * marklinks.  If successful, returns Nominal; if unsuccessful, returns
	 * OutOfMemory.
	 */

    struct markLink *newLink;

    if (Status != Nominal) return(Status);

    if ( newLink = (struct markLink *) malloc( sizeof(struct markLink) ) )

      {
	newLink->next = *markList;
	newLink->pmark = pmark;
	*markList = newLink;
	return(Status = Nominal);
      }

    else return(Status = OutOfMemory);

  }



/* Scavenge: checks a chunk to see if it is small to be merged with the next 
   chunk, and if so, merges it.  Prevents editing from producing a
   progressively less efficient structure.  */

Scavenge(chunk)  Chunk chunk;
{ Chunk nextchunk;
  Mark *m;
  char *p,*q;
  int yes,k,n,offset;

  if (chunk == NULL) return;

  nextchunk = chunk->next;
  if (nextchunk) n = chunk->length + nextchunk->length;
  if (nextchunk && n < CHUNKFILLPOINT ) {
    k = nextchunk->length;
    for (p = chunk->text + chunk->length, q = nextchunk->text;
		q < Chunkend(nextchunk); )
	*p++ = *q++;
    chunk->length += k;
    offset = q - p;
	/* Yes, I subtracted pointers in different chunks! It works. */
    
    for (m = Firstmark(nextchunk,&yes); yes; m = Nextmark(nextchunk,&yes) ) {
	m->chunk = chunk;
	m->cp -= offset;  /* directly corrects the character pointer */
	}

    DeleteChunk(nextchunk);
    }
  }


/*	All routines involved in changing the display and		*/
/*	the row descriptors.  The screen and the row descriptors are	*/
/*	always changed closely in parallel.  Only the chunk and cp	*/
/* 	fields are changed anywhere else than in this file.		*/

/* Row descriptors are maintained in an array of Nrows records, one	*/
/* for each row on the screen.						*/
/* They contain:							*/
/* 	chunk and cp	the Mark of the first character in this row	*/
/*	endcol		the column of the last character in this row	*/
/*	exists		0 when row is beyond end of file, else 1	*/
/*	continues	this row is a long line continuation		*/
/*	continued	this line is continued onto the next row	*/
/* Note that rows[0].continues is never true: we guarantee that the 	*/
/* screen starts at the start of a logical line.			*/

/* Display: display the material between two marks, starting at the	*/
/* given position.  If the text hasn't been changed, don't start	*/
/* redisplay at the previous line or allow scrolling (these should	*/
/* be used only for incremental updates.)				*/
/* In general, wherever there is displaying,				*/
/* there is fiddling with the row descriptors, and nowhere else (except */
/* to update chunk and cp when the text is altered).			*/

Display(start, end, startpos, changed)

  Mark start, end;
  Pos *startpos; 
  BOOLEAN changed;

{ register int row, col;
  register int prevcol, fixstartpos=0;
  register char ch;
  register RowRec *r;
  Mark	   m, backmark;
  BOOLEAN  quitFlag = FALSE;

  m = start;

	/* Back off one character if we have an overrunning line which	*/
	/* may no longer overrun.					*/

  if ( changed && (startpos->col == 0) && ( Retract(&m) == Nominal ) )
    {
      *startpos = Markpos(m);
      fixstartpos = 1;
    }


  row = startpos->row;
  col = startpos->col;
  r = ebuf->rows + row;


	/* Move startpos to the next row if it is at the end of a full	*/
	/* line to which an overrun has newly been added.		*/

  if ( changed && startpos->col + Charsize(*m.cp) > Ncols )
    {
      startpos->row++;
      startpos->col = 0;
    }

  if ( col == ( (row == headrow) ? headcol : 0 ) )
    {
      r->chunk = m.chunk;
      r->cp = m.cp;
      r->exists = 1;
    }

  DisplayPosition( pad, row, col );


  while ( !Atend(m) && !quitFlag )

    {

      prevcol = col;
      if ( MarkEQ(m, end) ) quitFlag = TRUE;
      ch = *m.cp;
      Advance(&m);			/* take care of chunk-crossings */	
      col = ShowChar(ch,row,col,changed); 	/* display, update col */
      if (col <= 0) {
        r->endcol = prevcol - 1;
        if (row >= Nrows-1)
	  {
	    if (!changed) break;
	    else HandleScroll();
	  }
        else {r++; row++;}
        r->exists = 1;
        backmark = m;
        Retract(&backmark);
        r->chunk = backmark.chunk;  r->cp = backmark.cp;
        col = -col;
      }

    }


  if ( Atend(m) )
    {
      r->endcol = ( Atstart(m) ) ? headcol : col - 1;
      DisplayClearToEOL(pad);
      for (; ++row < Nrows && (++r)->exists ; r->exists = 0)
	{
	  DisplayCRLF(pad);
	  DisplayClearToEOL(pad);
	}
    }

  if (fixstartpos)
    {
      *startpos = Markpos(start);
      fixstartpos = 0;
    }


  }


/* ShowChar: given a char and the current row & column, display it & return the
   resulting column.  Column<0 means this is a
   char which displays as multiple chars and didn't fit; it had to be
   displayed on the next line. */
int ShowChar(ch, row, col,scrollok)
  unsigned /* it matters! */ char ch;
  register int row, col;
  BOOLEAN scrollok;
{
  register int csize, i, oldcol;
  unsigned char *chptr;

  if (ch == '\n')  {
    for ( i = col; i < Ncols-1; i++ ) DisplaySend(pad, ' ');
    if (row < Nrows-1 || scrollok) DisplayCRLF(pad);
    return(0);
    }
  else if (ch == '\t') {
    if (col == Ncols) {
	col = -8;
	csize = 8;
	}
    else {
	csize = 1 + (col | 07) - col;
	if (col + csize > Ncols) {
	    col = Ncols;
	    csize = Ncols - col;
	    }
	else col += csize;
	}
    for (i=0; i<csize; i++)  DisplaySend(pad,' ');
    }
  else  { 
    csize = Charsize(ch);
    chptr = Charstring(ch);
    oldcol = col;
    col += csize;
    if (col > Ncols) {
	for (i=oldcol; i<Ncols; i++) DisplaySend(pad, ' ');
	if (!scrollok) return(-csize);
	}

    for (i=0; i<csize ; i++, chptr++)

      {
	DisplaySend(pad, *chptr);
      }

    }

  if (col <= Ncols)  return(col);
  else
  return(-csize);
  }

/* CountWidth: the counting aspect of ShowChar without any displaying */
int CountWidth(ch, col)  unsigned char ch;  int col;
{
  if (ch == '\n') return(0);
  else if (ch == '\t') {
    if (col == Ncols) return(-8);
    col = 1 + (col | 07);
    if (col > Ncols) col = Ncols;
    }
  else col += Charsize(ch);

  if (col > Ncols) col = - Charsize(ch);
  return(col);
  }


/* Advance: move a Mark forward one character. */

EditStatusCode Advance(mp)

  register Mark *mp;

  {

    if ( mp->cp >= Chunkend(mp->chunk) ) return(Status = AtTextEnd);

    else if ( ++mp->cp >= Chunkend(mp->chunk) )

      {

        if (mp->chunk->next == NULL) mp->cp = Chunkend(mp->chunk);

        else
	  {
	    mp->chunk = mp->chunk->next;
	    mp->cp = mp->chunk->text;
	  }

      }

    return(Status = Nominal);

  }


/* Retract: move a Mark back one character. */

EditStatusCode Retract(mp)  register Mark *mp;
{
  if (mp->cp == mp->chunk->text) {
    if (mp->chunk->prev == NULL)  return(Status = AtTextStart);
    mp->chunk = mp->chunk->prev;
    mp->cp = Chunkend(mp->chunk) - 1;
    }
  else mp->cp--;
  return(Status = Nominal);
  }


/* Setcursor: presumes the caller knows both the position and the mark of the
   desired cursor.  If not so, the functions PosSetcursor, MarkSetcursor, and
   LineSetcursor will handle the necessary preprocessing. */
Setcursor(mark, pos)  Pos pos;  Mark mark;
{
  ebuf->curpos = pos;
  ebuf->curmark = mark;
  DisplayPosition(pad, ebuf->curpos.row, ebuf->curpos.col);
  }

PosSetcursor(pos)  Pos pos;
{ Mark mark;
  mark = Posmark(&pos);
  Setcursor(mark, pos);
  }


/*
HandleScroll: The screen has just scrolled, so shift each row descriptor up
one row and adjust headpos and curpos.
*/

HandleScroll()

  {

  register RowRec *r1, *r2;

  if (headrow > 0) headrow--;

  if (currow > 0) currow--;
  else
    {
      if (ebuf->selectionexists) Deselect();
      PosSetcursor( Makepos(0,0) );
    }

  for ( r1 = ebuf->rows, r2 = ebuf->rows+1; r2 < ebuf->rows+Nrows; r1++, r2++ )
      *r1 = *r2;

  }

/* MarkGT: comparison of Marks.  This is one of the disoptimized operations
   in this data structure system, and takes time proportional to the number
   of chunks in the buffer.  Needed only when a Region operation is to be
   done.  Returns 1 if the first argument follows the second, else 0.
   -1 if the two are in disjoint spaces (error!) and cannot be compared.
*/
int MarkGT(m1, m2)  Mark m1, m2;
{ Chunk chunk;
/* 1. same chunk */
  if (m1.chunk == m2.chunk) {
    if (m1.cp > m2.cp) return(1);
    else return(0);
    }
/* 2. look for m2 in a chunk after m1 */
  for (chunk = m1.chunk->next; chunk; chunk = chunk->next)
    if (chunk == m2.chunk)  return(0);
/* 3. look for m2 in a chunk before m1 */
  for (chunk = m1.chunk->prev; chunk; chunk = chunk->prev)
    if (chunk == m2.chunk)  return(1);
/* error: m2 is nowhere to be found from m1 */
  ErrorMsg("MarkGT on unrelated marks");
  return(-1);
  }


/* elementary routines to convert between the (chunk,cp) 	*/
/* representation and the (row,col) representation of a place in the	*/
/* text.  Various other routines for computing places in the text.	*/

/* Makepos: construct a Pos out of two integers */
Pos Makepos(r,c)  int r,c;
{ Pos p;
  p.row = r;  p.col = c;
  return(p);
  }

/* Makemark: construct a Mark out of a chunk pointer and a char pointer */
Mark Makemark(chunk,cp)  Chunk chunk;  char *cp;
{ Mark m;
  m.chunk = chunk;  m.cp = cp;
  return(m);
  }

/* Rowmark: given a row number, find the Mark of its first character.	*/
/* warning: do not call Rowmark on a nonexistent row.			*/
Mark Rowmark(row) register int row;
{ Mark m;
  m.chunk = ebuf->rows[row].chunk;
  m.cp = ebuf->rows[row].cp;
  return(m);
  }



/* Rowsend: given a row number, find the Mark at the end of the row:	*/

Mark Rowsend(row) register int row;

{
  Mark m;

  if (row < headrow) return(ebuf->headmark);

  else if (row+1 >= Nrows || !(ebuf->rows[row+1].exists)) return(ebuf->endmark);

  else
    {
      m = Rowmark(row + 1);
      Retract(&m);
      return(m);
    }

  }


/* Markcol: given a row and a Mark (which must be in that row) find 	*/
/* the column position of that Mark.					*/
int Markcol(row, mark)  register int row; Mark mark;
{ register int col = (row == headrow) ? headcol : 0;
  register char ch;
  Mark m;

  m = Rowmark(row);
  while(MarkNEQ(m,mark)) {
    ch = *m.cp;
    Advance(&m);
    col = CountWidth(ch,col);
    if (col < 0) {
	return(0);
	}
    }
  return(col);
  }


/* Markpos: given a Mark, which must be on the screen somewhere, return the
   position of that character on the screen. */

Pos Markpos(mark)  Mark mark;
{ Pos p;
  Chunk chunk = ebuf->rows[headrow].chunk;
  register int row = headrow;

  /* find first row containing our chunk */

  for ( ; ; row++ )

    {
      while (TRUE)
	{
	  if ( chunk == mark.chunk ||
	       (row+1 < Nrows &&
		ebuf->rows[row+1].exists &&
		ebuf->rows[row+1].chunk == chunk) ) break; 

	  if ( !(chunk = chunk->next) ) return(ebuf->headpos);
	}

      if (chunk == mark.chunk) break;
    }

  /* find our row */

  while ( row + 1 < Nrows &&
	  ebuf->rows[row+1].exists &&
 	  mark.chunk == ebuf->rows[row+1].chunk &&
	  mark.cp >= ebuf->rows[row+1].cp ) row++;

  p.col = Markcol(row,mark);
  p.row = row;
  return(p);
  }


/* Posmark: given a position on the screen, return the Mark of the character
   nearest to it.  If there is no match, ALTER the Pos to match the truth. */
Mark Posmark(ppos) register  Pos *ppos;
{ Mark m;
  register int row, col, prevcol;
  register char ch;


  row = ppos->row;
  if ( (row < headrow) || ( (row == headrow) && ppos->col <= headcol) ) { 
    row = headrow;
    col = headcol;
    m = ebuf->headmark;
    }

  else if ( (row >= Nrows) || !(ebuf->rows[row].exists) ) /* beyond text end */
    {
      *ppos = Markpos(ebuf->endmark);
      row = ppos->row;
      col = ppos->col;
      m = ebuf->endmark;
    }

  else if ( ppos->col > ebuf->rows[row].endcol )
    {  /* beyond end of line */
    m = Rowsend(row);
    col = Markcol(row, m);
    }

  else {
    m = Rowmark(row);
    col = (row == headrow) ? headcol : 0;
    while (col <= ppos->col) {
	prevcol = col;
	ch = *m.cp;
	Advance(&m);
	col = CountWidth(ch,col); 
	}
    col = prevcol;
    Retract(&m);
    }

  ppos->row = row;
  ppos->col = col;
  return(m);
  }


/* Backrows: find the line n rows back from a given line.  If that would */
/* land you in the middle of an overflowed line, go to the beginning of  */
/* that line.  Calculations are done in terms of tabs: text lying after	 */
/* the last-seen tab is of a known length, hence "solid".  Text lying    */
/* before the last-seen tab is of uncertain length, hence "fluid".	 */
/* The calculation is faulty in one respect: it assumes that all tabs	 */
/* fill out a field of length 8, calculates the natural length of the	 */
/* line, and divides by Ncols.  However, when Ncols mod 8 != 0 a tab at	 */
/* the end of the line may fill out a field of length <8.		 */
Mark Backrows(m,n) Mark m;  int n;
{ register int rowcount, solid, fluid, tabseen;
  register Chunk chunk;
  register char *cp;
  register char ch;
  Mark newmark;

  chunk = m.chunk;  cp = m.cp;
  rowcount = -1;

  while (rowcount < n) {
    solid = fluid = tabseen = 0;
    cp--;
    if (cp < chunk->text) {
	if (!chunk->prev)  return(Makemark(chunk, chunk->text));
	else {
	    chunk = chunk->prev;
	    cp = Chunkend(chunk) - 1;
	    }
	}
    ch = *cp;
    if (ch == '\t') {
	if (tabseen) solid += (fluid & 07) + 1;
	else solid = fluid;
	tabseen = 1;  fluid = 0;
	}
    else if (ch == '\n') {
	if (tabseen) solid += (fluid & 07) + 1;
	else solid = fluid;
	if (solid > 0)  rowcount += 1 + ((solid-1)/Ncols);
	else rowcount++;
	solid = fluid = tabseen = 0;
	}
    else fluid += Charsize(ch);	/* all chars of predictable size */
    }
  newmark = Makemark(chunk, cp+1); /* forward from the \n */
  Advance(&newmark);
  return(newmark);
  }


#ifndef STS

/* MapMouse: given world coordinates, return a pos.	*/

MapMouse(x,y,ppos)  register short x, y;  register Pos *ppos;
{
  PadFindPoint(pad, Nrows-1, x, y, &(ppos->row), &(ppos->col) );

  if (ppos->row > 255) ppos->row = 0; /* unsigned: I really mean < 0 */
  if (ppos->col > 255) ppos->col = 0;

  }

#endif


/********************  Interaction window routines  *******************/

NewMsg(s)  register char *s;
  {
    TtyPutString(s);
    DisplayCRLF(VgtsPad);
  }

#ifdef DEBUG

/* FormatMsg: write a formatted string to the interaction window */
FormatMsg(s, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9, arg10)
  register char *s;
  int arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8,arg9, arg10;
{
  char lineBuf[256];

  sprintf(lineBuf, s, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9,
	  arg10);
  TtyPutString(lineBuf);
  DisplayCRLF(VgtsPad);
  }
  
#endif


/**************  Routines for ansi standard terminal interface  **************/

DisplayString(p, string)

  Padtype p;
  register char *string;

  {
    register char c;

    while (c = *string++)  DisplaySend(p, c);
  }

DisplayCRLF(p)  Padtype p;
{ 
  DisplayClearToEOL(p);
  DisplayReturn(p);
  DisplayNewLine(p);
  }
