/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* edops.c - Basic operations for VGTS line editing
 *
 *
 * Craig Dunwoody December 1983
 *
 */


#include "edit.h"

Padtype pad;		/* current pad's output file descriptor */

short Nrows;	 /* number of rows in current pad */
short Ncols;	 /* number of columns in current pad */

struct Client	*client;
EditBuffer	*ebuf;

Mark	tempmark0;
Mark	tempmark1,tempmark2;	/* used by BlockInsert and BlockDelete, also
				 * usable by anything that just uses single-char
				 * inserts and deletes
				 */

EditStatusCode Status;

Pos mousepos;

Pos cursorPos;	/* REAL cursor position.  Necessary in environments	*/
		/* where we must use relative cursor positioning (i.e,	*/
		/* STS).						*/

/*********************  Multi-buffering functions  **************************/

/* FetchBuffer: Set global variables for current client. */

FetchBuffer(curClient)  register struct Client *curClient;
{ 
  client = curClient;
  ebuf = client->lineEditBuf;
  pad = (Padtype) TtyPadList + client->vgt;  
  PadGetSize(pad, &Nrows, &Ncols);
  }


/* SelfInsert: insert this character at the cursor, and advance the cursor.
   Returns Nominal if successful. */

EditStatusCode SelfInsert(ch) unsigned char ch;
{ 
  char *dcp;

  if (TextInsertChar(&ebuf->curmark, ch) != Nominal) return(Status);
  Display(ebuf->curmark, ebuf->endmark, &ebuf->curpos, TRUE);
  Setcursor(ebuf->curmark, ebuf->curpos);
  Forespace();
  return(Status);
  }


/* DeleteForward: Deletes the next character.  Returns Nominal if successful. */

EditStatusCode DeleteForward()

{ 
  TextDeleteForward(&ebuf->curmark);
  Display(ebuf->curmark, ebuf->endmark, &ebuf->curpos, TRUE);
  Setcursor(ebuf->curmark,ebuf->curpos);
  return(Status);
  }


/* Forespace: move the cursor forward by 1 character. Returns Nominal if
   successful. */

EditStatusCode Forespace()
{ char ch = *ebuf->curmark.cp;
  if (Advance(&ebuf->curmark) != Nominal) return(Status);
  if (ch >= MinPrintingChar && ch <= MaxPrintingChar &&
      curcol < ebuf->rows[currow].endcol)
    {  
      DisplayCursorForward(pad);
      curcol++;
    }
  else MarkSetcursor(ebuf->curmark);
  ebuf->wishcol = curcol;
  return(Status = Nominal);
  }


/* Backspace: backspace the cursor by 1 character. Returns Nominal if
   successful. */

EditStatusCode Backspace()
{ char ch;
  if (Retract(&ebuf->curmark) != Nominal) return(Status);
  ch = *ebuf->curmark.cp;
  if (ch >= MinPrintingChar && ch <= MaxPrintingChar && curcol > 0)
    {  
      DisplayCursorBackward(pad);
      curcol--;
    }
  else
    {
      ebuf->curpos = Markpos(ebuf->curmark);
      DisplayPosition(pad, currow, curcol);
    }
  ebuf->wishcol = curcol;
  return(Status = Nominal);
  }

/* ForwardWord: find the next end of word, forward */

EditStatusCode ForwardWord(mp)  Mark *mp;
{
  char ch;

  if (Atend(*mp)) return(Status = AtTextEnd);

  ch = *mp->cp;
  while (!wordchar(ch)) {
    if ( Advance(mp) != Nominal ) return(Status);
    ch = *mp->cp;
    }
  while (wordchar(ch)) {
    if ( Advance(mp) != Nominal ) return(Status = Nominal);
    ch = *mp->cp;
    }
  return(Status = Nominal);
  }


/* BackWord: find the next beginning of word, backward */

EditStatusCode BackWord(mp)  Mark *mp;
{
  char ch;

  do {
    if ( Retract(mp) != Nominal ) return(Status);
    ch = *mp->cp;
    }
   while (!wordchar(ch));

  while (wordchar(ch)) {
    if ( Retract(mp) != Nominal ) return(Status = Nominal);
    ch = *mp->cp;
    }
  Advance(mp);
  return(Status = Nominal);
  }


/* AtWordStart:  returns TRUE if given mark is at the start of a word. */

BOOLEAN AtWordStart(m)

  Mark m;

  {

    if ( Atend(m) || !wordchar(*m.cp) ) return(FALSE);
    return( (Retract(&m) == Nominal) ? !wordchar(*m.cp) : TRUE );

  }


/* AtWordEnd:  returns TRUE if given mark is at the end of a word. */

BOOLEAN AtWordEnd(m)

  Mark m;

  {

    return( (Advance(&m) == Nominal) ? !wordchar(*m.cp) : TRUE );

  }


/* Downaline: the down-arrow function. */
EditStatusCode Downaline()
{
  if ( currow >= Nrows-1 || !ebuf->rows[currow+1].exists )
      return(Status = AtTextEnd);
  currow++;
  PosSetcursor(Makepos(currow, ebuf->wishcol));
  return(Status = Nominal);
  }

/* Upaline: the up-arrow function. */
EditStatusCode Upaline()
{
  if (currow == headrow) return(Status = AtTextStart);
  currow--;
  PosSetcursor(Makepos(currow,ebuf->wishcol));
  return(Status = Nominal);
  }



/*	Functions for killbuffer and straytext operations.	*/

Chunk killbuffer;


/* Kill: Replace the killbuffer with the text between the two given Marks. */
/* Returns Nominal if successful.					   */

EditStatusCode Kill(startMark, endMark)

  Mark startMark, endMark;

  {

    Chunk nchunk;

    if ( !(nchunk = BlockDelete(startMark, endMark)) )
	return(Status);
    Display(ebuf->headmark, ebuf->endmark, &ebuf->headpos, TRUE);
    MarkSetcursor(ebuf->curmark);
    FreeText(killbuffer);
    killbuffer = nchunk;
    return(Status = Nominal);

  }
    
  
/* FreeText: given a pointer to the start of a straytext, frees its storage */
FreeText(chunk)  Chunk chunk;
{ Chunk temp;
  while (chunk) {
    temp = chunk->next;
    free(chunk);
    chunk = temp;
    }
  }


/* StringText: copy a string into a new sparetext. 			*/
Chunk StringText(s, len)
  char *s;
  int len;
{
  Chunk chunk, texthead;
  register char *p, *cp, *ecp;
  char *endstr = s + len;

  if ( !(texthead = chunk = MakeChunk(NULL, CHUNKFILLPOINT)) ) return(NULL);
  cp = chunk->text;  ecp = cp + CHUNKFILLPOINT;
  p = s;

  while (p < endstr) {
    *cp++ = *p++;
    if (cp >= ecp) {
	if ( !(chunk->next = MakeChunk(chunk,CHUNKFILLPOINT)) )return(texthead);
	chunk = chunk->next;  cp = chunk->text;
	ecp = cp + CHUNKFILLPOINT;
	}
    }
  chunk->length = cp - chunk->text;
  return(texthead);
  }

/* TextString: copy a section out of the buffer into a string of maximum */
/* length len.  Returns the actual length.				 */
int TextString(m1, m2, s, len)
  Mark m1, m2;
  char *s;
  int len;
{
  char *limit = s + len;
  register char *p = s;

  Status = Nominal;

  while (MarkNEQ(m1, m2) && p < limit && Status == Nominal) {
    *p++ = *m1.cp;
    Advance(&m1);
    }
  if (p < limit) *p = '\0';
  return(p - s);
  }


#ifndef STS

/* KillSelection: delete the current selection to the killbuffer
   Returns Nominal if successful.	  */
EditStatusCode KillSelection()
{ 
  Chunk killText;

  if (!ebuf->selectionexists) return(Status = NoSelection);
  if ( !(killText = BlockDelete(ebuf->curmark, ebuf->mousemark)) )
      return(Status);
  FreeText(killbuffer);
  killbuffer = killText;
  Display(ebuf->curmark, ebuf->endmark, &ebuf->curpos, TRUE);
  client->mode &= ~NoCursor;
  ebuf->selectionexists = 0;
  ebuf->curKeyTable = GetKeyTable(ebuf->keyTableList, ClientKeyTable);
  Setcursor(ebuf->curmark, ebuf->curpos);
  return(Status = Nominal);
  }

/* InsertSelected: take a piece of sparetext and insert it at curmark,
   making it the new selection.  */
InsertSelected(newtext)  Chunk newtext;
{ 
  if (!newtext) return;
  ebuf->mousemark = ebuf->curmark;
  ebuf->curmark = BlockInsert(ebuf->curmark,newtext);
  Display(ebuf->curmark, ebuf->endmark, &ebuf->curpos, TRUE);
  MarkSetcursor(ebuf->curmark);
  Select(ebuf->curmark, ebuf->mousemark);
  }



/* DynamicSelect: handles selection with mouse-tracking inverse video to
   display it.  Begins with the mouse click that started it,
   returns the pos of that click and of the point of release.  */
DynamicSelect(initxw, inityw, ppos1,ppos2)
    short initxw, inityw;
    Pos *ppos1,*ppos2;
{ Pos clickpos, oldpos, mousepos;
  short xw, yw, onPad;
  
  MapMouse(initxw, inityw, &clickpos);
  oldpos = clickpos;

  do {
    ReadMouseInput();
    CursorUpdate();
    onPad = ( client->vgt == FindTopVGT(MouseX, MouseY) );
    if (onPad) {
	FindMouseClick(MouseX, MouseY, &xw, &yw);
        MapMouse(xw, yw, &mousepos);
	if (PosEQ(mousepos,oldpos)) ;  /* no motion - no action */
	else if (PosGT(mousepos, oldpos)) {  /* forward motion */
	    if (PosGE(oldpos, clickpos))  Darken(oldpos,mousepos);
	    else if (PosLE(mousepos, clickpos)) Lighten(oldpos,mousepos);
	    else {Lighten(oldpos,clickpos);  Darken(clickpos,mousepos);}
	    PadRedraw(padIndex);
	    }
	else {		/* backward motion */
	    if (PosGE(mousepos,clickpos)) Lighten(mousepos,oldpos);
	    else if (PosLE(oldpos,clickpos)) Darken(mousepos,oldpos);
	    else {Darken(mousepos,clickpos);  Lighten(clickpos,oldpos);}
	    PadRedraw(padIndex);
	    }
	oldpos = mousepos;
	}  /* end if (onPad) */
    }
     while (MouseButtons == LeftButton && onPad);
    
  if (MouseButtons != 0 || !onPad) {  /* mouse went off edge of pad 
  				         or several buttons were depressed */
    if (PosGT(mousepos, clickpos)) Lighten(clickpos, mousepos);
    else if (PosLT(mousepos, clickpos)) Lighten(mousepos, clickpos);
    *ppos1 = *ppos2 = clickpos;
    return;
    }

  *ppos1 = clickpos; /* pos where mouse was first clicked */
  *ppos2 = oldpos;  /* last pos when the mouse was in this pad */
  }

/* Lighten and Darken: Turn the region between two positions to inverse or
   normal video.  */
Lighten(firstpos,lastpos)  Pos firstpos,lastpos;
{ Mark firstmark, lastmark;
  firstmark = Posmark(&firstpos);
  lastmark = Posmark(&lastpos);
  if ( MarkNEQ(firstmark, lastmark) ) Retract(&lastmark);
  DisplayNormal(pad);
  Display(firstmark, lastmark, &firstpos, FALSE);
  }

Darken(firstpos,lastpos)  Pos firstpos,lastpos;
{ Mark firstmark, lastmark;
  firstmark = Posmark(&firstpos);
  lastmark = Posmark(&lastpos);
  if ( MarkNEQ(firstmark, lastmark) ) Retract(&lastmark);
  DisplayInverse(pad);
  Display(firstmark, lastmark, &firstpos, FALSE);
  DisplayNormal(pad);
  }


/* Select: darken and record a selection.  */
Select(firstmark, lastmark)  Mark firstmark, lastmark;
{
  ebuf->mousemark = lastmark;  mousepos = Markpos(ebuf->mousemark);
  MarkSetcursor(firstmark);
  Retract(&lastmark);
  DisplayInverse(pad);
  Display(firstmark, lastmark, &ebuf->curpos, FALSE);
  DisplayNormal(pad);
  PadCursorOff(padIndex);
  client->mode |= NoCursor;
  ebuf->selectionexists = 1;
  ebuf->curKeyTable = GetKeyTable(ebuf->keyTableList, SelectionKeyTable);
  }

/* Deselect: undo any selection present, leaving a normal cursor at the end of
   the selected area.  */
Deselect()
{ DisplayNormal(pad);
  Display(ebuf->curmark, ebuf->mousemark, &ebuf->curpos, FALSE);
  ebuf->curmark = ebuf->mousemark;
  client->mode &= ~NoCursor;
  ebuf->selectionexists = 0;
  ebuf->curKeyTable = GetKeyTable(ebuf->keyTableList, ClientKeyTable);
  }

#endif



#ifdef DEBUG

/*************************** Debugging Subroutines **************************/

DumpChunks(chunk)  Chunk chunk;
{ for ( ; chunk!=NULL; chunk=chunk->next)
    FormatMsg("Chunk %x (%c%c%c%c): %d chars, end %x, prev %x, next %x",
        chunk,chunk->text[0],chunk->text[1],
	chunk->text[2],chunk->text[3],chunk->length,
	chunk->text+chunk->length,chunk->prev,chunk->next);
  }

/* SixRows: dumps the data for the first 6 RowRecs			*/
SixRows()
{ register int i;
  DisplayCRLF(VgtsPad);
  for (i=0; i<6; i++) DumpRow(i);
  }

LastRows()
{ register int i;
  DisplayCRLF(VgtsPad);
  for (i=Nrows-6; i<Nrows; i++) DumpRow(i);
  }

DumpRow(n)  int n;
{ register RowRec *r = ebuf->rows + n;
  register char ch;
  FormatMsg("Row %d  ", n);
  if (r->exists) {
    ch = *r->cp;
    if (ch == '\n') ch = 01;	/* control-A, a down arrow */
    FormatMsg("'%c'  chunk %x cp %x  endcol %d ",ch,r->chunk,r->cp,r->endcol);
    }
  else DisplaySend(VgtsPad,'-');
  DisplayCRLF(VgtsPad);
  }

#endif
