/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* edinterface.c - Program and client interfaces for VGTS line editing
 *
 *
 * Craig Dunwoody December 1983
 *
 */


#include "edit.h"

static PopUpEntry HardMenu[] =
  {
	"[context]",	1,
	"[device]",	2,
	"[public]",	3,
	"[internet]",	4,
	"[vgts]",	5,
	"[team]",	6,
	"[service]",	7,
	"[home]",	8,
	"[bin]",	9,
	"[screen]",	10,
	"[diablo]",	11,
	0,		0
  };


KeyTableEntry DefaultMainKeyTable [] = 

{

MinPrintingChar,	KeyTableBeginRange,		0,
MaxPrintingChar,	EditSelfInsert,			ClientKeyTable,
Control('I'),		EditSelfInsert,			ClientKeyTable,
'\n',			EditSelfInsert,			ClientKeyTable,
DEL,			EditDeleteChar,			ClientKeyTable,
Control('H'),		EditDeleteChar,			ClientKeyTable,
Control('D'),		EditDeleteNextChar,		ClientKeyTable,
ESC,			EditNop,			EscKeyTable,
Control('F'),		EditForwardChar,		ClientKeyTable,
CadlincRightArrow,	EditForwardChar,		ClientKeyTable,
Control('B'),		EditBackwardChar,		ClientKeyTable,
CadlincLeftArrow,	EditBackwardChar,		ClientKeyTable,
Control('T'),		EditTransposeChars,		ClientKeyTable,
Control('W'),		EditKillWord,			ClientKeyTable,
Control('U'),		EditKillLine,			ClientKeyTable,
Control('A'),		EditBeginLine,			ClientKeyTable,
Control('E'),		EditEndLine,			ClientKeyTable,
Control('K'),		EditKillToEnd,			ClientKeyTable,
Control('Y'),		EditUnKill,			ClientKeyTable,
Control('P'),		EditUpLine,			ClientKeyTable,
CadlincUpArrow,		EditUpLine,			ClientKeyTable,
Control('N'),		EditDownLine,			ClientKeyTable,
CadlincDownArrow,	EditDownLine,			ClientKeyTable,
Control('L'),		EditRedisplay,			ClientKeyTable,
Control('V'),		EditForwardPage,		ClientKeyTable,
Control('G'),		EditReleaseAbort,		ClientKeyTable,
'\r',			EditRelease,			ClientKeyTable,
Control('C'),		EditKillBreak,			ClientKeyTable,
Control('Z'),		EditEOF,			ClientKeyTable,
Control('Q'),		EditNop,			QuoteKeyTable,
Control('\\'),		EditNop,			HiquoteKeyTable,
0,			KeyTableEnd,			ClientKeyTable

};


KeyTableEntry DefaultSelectionKeyTable [] =

{

Control('L'),		EditRedisplay,			SelectionKeyTable,
MinChar,		KeyTableBeginRange,		0,
MaxChar,		EditDeselect,			ClientKeyTable,
0,			KeyTableEnd,			ClientKeyTable

};


KeyTableEntry DefaultEscKeyTable [] =

{

'[',			EditNop,			ANSIKeyTable,
DEL,			EditKillWord,			ClientKeyTable,
'h',			EditKillWord,			ClientKeyTable,
'b',			EditBackwardWord,		ClientKeyTable,
'\b',			EditBackwardWord,		ClientKeyTable,
'd',			EditKillNextWord,		ClientKeyTable,
'f',			EditForwardWord,		ClientKeyTable,
't',			EditTransposeWords,		ClientKeyTable,
',',			EditBeginInput,			ClientKeyTable,
'.',		   	EditEndBuffer,			ClientKeyTable,
'<',			EditBeginBuffer,		ClientKeyTable,
'>',		   	EditEndBuffer,			ClientKeyTable,
'v',		   	EditBackPage,			ClientKeyTable,
'O',			EditNop,			PFKeyTable,
Control('G'),		EditAbort,			ClientKeyTable,
'1',			EditDumpCursor,			ClientKeyTable,
'2',			EditDumpCurrentChunk,		ClientKeyTable,
'3',			EditDumpMarklist,		ClientKeyTable,
'4',			EditDumpChunks,			ClientKeyTable,
'5',			EditDumpFirstRows,		ClientKeyTable,
'6',			EditDumpLastRows,		ClientKeyTable,
'7',			EditDumpKillbuffer,		ClientKeyTable,
'8',			EditDumpMajorMarks,		ClientKeyTable,
MinChar,		KeyTableBeginRange,		0,
MaxChar,		KeyTableUseNext,		ClientKeyTable,
0,			KeyTableEnd,			ClientKeyTable

};


KeyTableEntry DefaultANSIKeyTable [] =

{

'D',			EditBackwardChar,		ClientKeyTable,
'C',			EditForwardChar,		ClientKeyTable,
'A',			EditUpLine,			ClientKeyTable,
'B',			EditDownLine,			ClientKeyTable,
'H',			EditBeginBuffer,		ClientKeyTable,
Control('G'),		EditAbort,			ClientKeyTable,
MinChar,		KeyTableBeginRange,		0,
MaxChar,		KeyTableUseNext,		ClientKeyTable,
0,			KeyTableEnd,			ClientKeyTable

};


KeyTableEntry DefaultPFKeyTable [] =

{

'P',			EditScrollUp,			ClientKeyTable,
'Q',			EditScrollDown,			ClientKeyTable,
'R',			EditForwardPage,		ClientKeyTable,
'S',			EditBackPage,			ClientKeyTable,
MinChar,		KeyTableBeginRange,		0,
MaxChar,		KeyTableUseNext,		ClientKeyTable,
0,			KeyTableEnd,			ClientKeyTable

};


KeyTableEntry DefaultQuoteKeyTable [] =

{

MinChar,		KeyTableBeginRange,		0,
MaxChar,		EditSelfInsert,			ClientKeyTable,
0,			KeyTableEnd,			ClientKeyTable

};


KeyTableEntry DefaultHiquoteKeyTable [] =

{

MinChar,		KeyTableBeginRange,		0,
MaxChar,		EditSelfInsertHigh,		ClientKeyTable,
0,			KeyTableEnd,			ClientKeyTable

};


EditStatusCode AddKeyTable(keyTableList, keyTableId, keyTable)

  KeyTableRec	**keyTableList;
  KeyTableId	keyTableId;
  KeyTableEntry	*keyTable;

  {

	/*
	 * If Status is Nominal, this routine tries to add a KeyTableRec, with 
	 * the given keyTable and keyTableId, to the head of a given linked list
	 * of KeyTableRecs.  If successful, returns Nominal; if unsuccessful,
	 * returns OutOfMemory.
	 */

    KeyTableRec *newRec;

    if (Status != Nominal) return(Status);

    if ( newRec = (KeyTableRec *) malloc( sizeof(struct keyTableRec) ) )

      {
	newRec->next = *keyTableList;
	newRec->keyTableId = keyTableId;
	newRec->keyTable = keyTable;
	*keyTableList = newRec;
	return(Status = Nominal);
      }

    else return(Status = OutOfMemory);

  }


KeyTableRec *GetKeyTable(keyTableList, keyTableId)

  KeyTableRec	*keyTableList;
  KeyTableId	keyTableId;

  {

	/*
	 * This routine searches the given keyTableList for the first
	 * KeyTableRec whose keyTableId field matches the given keyTableId.
	 * The routine returns this KeyTableRec if found; otherwise it returns
	 * the KeyTableRec on the given keyTableList whose keyTableId has the
	 * value MainKeyTable.
	 */

    KeyTableRec *listPtr = keyTableList;

    for (; listPtr && (listPtr->keyTableId != keyTableId);
	 listPtr = listPtr->next);

    return( (listPtr) ? listPtr : GetKeyTable(keyTableList, MainKeyTable) );

  }


OpCode GetOp(inChar, curKeyTable, keyTableList)

  unsigned char	inChar;
  KeyTableRec	**curKeyTable;
  KeyTableRec	*keyTableList;

  {

	/*
	 * This routine looks up the given character in the keytable pointed to
	 * by the KeyTableRec curKeyTable.
	 *
	 * If no match is found, the routine sets curKeyTable to the KeyTableRec
	 * in keyTableList whose keyTableId field matches the nextKeyTableId
	 * field of the entry which terminates the keytable, and the routine
	 * returns the opCode value EditNop.
	 * 
	 * If a match is found, the routine first sets curKeyTable to the
	 * KeyTableRec in keyTableList whose keyTableId field matches the
	 * nextKeyTableId field of the matching keytable entry.
	 *
	 * In the case of a match, if the opCode of the matching keytable entry
	 * has the value KeyTableUseNext, the routine calls itself
	 * recursively and returns the opCode resulting from a lookup of the
	 * given character in the next keytable.  Otherwise, the routine simply
	 * returns the opCode of the matching keytable entry.
	 */

    KeyTableEntry *entryPointer = (*curKeyTable)->keyTable;

    for ( ; entryPointer->opCode != KeyTableEnd ; entryPointer++ )

	if ( (entryPointer->opCode == KeyTableBeginRange &&
	      inChar >= entryPointer->charCode &&
	      (inChar <= (++entryPointer)->charCode ||
	       entryPointer->opCode == KeyTableEnd) ) ||

	     (inChar == entryPointer->charCode) ) break;


    if ( (*curKeyTable)->keyTableId != entryPointer->nextKeyTableId )
	*curKeyTable = GetKeyTable(keyTableList, entryPointer->nextKeyTableId);

    if (entryPointer->opCode == KeyTableUseNext)
	return( GetOp(inChar, curKeyTable, keyTableList) );

    else if (entryPointer->opCode == KeyTableEnd) return(EditNop);

    else return(entryPointer->opCode);

  }


EditBuffer *CreateBuffer(index)

  short index;

  {
	/*
	 * This routine allocates and initializes a descriptor for an
	 * editing buffer, and returns a pointer to it.  If there is
	 * not enough memory, NULL is returned.
	 */

    if (!(ebuf = (EditBuffer*) malloc( sizeof(EditBuffer)))) return(NULL);

    /*
     * Initialize variables so we can safely call DestroyBuffer on a partially
     * allocated buffer if necessary.
     */

    ebuf->markList = NULL;
    ebuf->keyTableList = NULL;
    ebuf->headmark = Makemark(NULL, NULL);

    /* create the row descriptor array */

    PadGetSize(TtyPadList + index, &Nrows, &Ncols);

    ebuf->rows = (RowRec *) calloc( Nrows, sizeof(struct rowRec) );

    if (ebuf->rows == NULL)
      {
	free(ebuf);
	return(NULL);
      }

    /* create the markList and keyTableList */

    Status = Nominal;

    AddMark( &ebuf->markList, &ebuf->curmark	);
    AddMark( &ebuf->markList, &ebuf->endmark	);
    AddMark( &ebuf->markList, &ebuf->mousemark	);
    AddMark( &ebuf->markList, &ebuf->promptmark	);
    AddMark( &ebuf->markList, &ebuf->inputmark	);
    AddMark( &ebuf->markList, &ebuf->backupmark	);
    AddMark( &ebuf->markList, &tempmark0	);
    AddMark( &ebuf->markList, &tempmark1	);
    AddMark( &ebuf->markList, &tempmark2	);

    AddKeyTable(&ebuf->keyTableList,HiquoteKeyTable,  DefaultHiquoteKeyTable  );
    AddKeyTable(&ebuf->keyTableList,QuoteKeyTable,    DefaultQuoteKeyTable    );
    AddKeyTable(&ebuf->keyTableList,SelectionKeyTable,DefaultSelectionKeyTable);
    AddKeyTable(&ebuf->keyTableList,PFKeyTable,       DefaultPFKeyTable       );
    AddKeyTable(&ebuf->keyTableList,ANSIKeyTable,     DefaultANSIKeyTable     );
    AddKeyTable(&ebuf->keyTableList,EscKeyTable,      DefaultEscKeyTable      );
    AddKeyTable(&ebuf->keyTableList,MainKeyTable,     DefaultMainKeyTable     );
  
    if (Status != Nominal)
      {
	DestroyBuffer(ebuf);
	return(NULL);
      }	

    BufInit(ebuf);
    return(ebuf);

  }


DestroyBuffer(ebuf)

  EditBuffer *ebuf;

  {

	/*
	 * This routine destroys an editing buffer by deallocating the
	 * data spaces associated with it.
	 */

    MarkLink	mListPtr, mNextPtr = ebuf->markList;
    KeyTableRec *kListPtr, *kNextPtr = ebuf->keyTableList;

    FreeText(ebuf->headmark.chunk);
    free(ebuf->rows);
    for ( ; mListPtr = mNextPtr ; mNextPtr = mListPtr->next, free(mListPtr) );
    for ( ; kListPtr = kNextPtr ; kNextPtr = kListPtr->next, free(kListPtr) );
    free(ebuf);

  }


BufInit(ebuf)

  EditBuffer *ebuf;

  {
	/*
	 * This routine initializes the given buffer structure to empty.
	 */

    Chunk chunk;
    register RowRec *r;

    chunk = MakeChunk( NULL, 0 );	/* allocate the first chunk */

    ebuf->headmark.chunk = chunk;
    ebuf->headmark.cp = chunk->text;
    ebuf->endmark = ebuf->curmark = ebuf->mousemark =  ebuf->headmark;
    ebuf->promptmark = ebuf->inputmark = ebuf->backupmark = ebuf->headmark;
    ebuf->headpos = ebuf->curpos = Makepos(0,0);
    ebuf->wishcol = curcol;
    ebuf->selectionexists = ebuf->displayed = 0;
    ebuf->curKeyTable = GetKeyTable(ebuf->keyTableList, ClientKeyTable);
    ebuf->menu = HardMenu;
    ebuf->pagesToKeep = 4;
    ebuf->prompt = NULL;

    for ( r = ebuf->rows ; r < ebuf->rows + Nrows ; (r++)->exists = 0 );
    Display(ebuf->headmark, ebuf->endmark, &ebuf->headpos, TRUE);
  }


LineEdit(curClient, ch)

  register struct Client *curClient;
  register unsigned char ch;

  {

	/*
	 * Command interpreter for line editing.
	 */

    OpCode opCode;
    char *cp;
    Chunk chunk;
    Mark m;
    MarkLink marklp;
    struct Event *event;
    register int i;

    FetchBuffer(curClient);

    switch( opCode = GetOp(ch, &(ebuf->curKeyTable), ebuf->keyTableList) )

      {


      /********** Motion commands **********/

      case EditBackwardChar:
    	  Backspace();
	  break;

      case EditForwardChar:
	  Forespace();
	  break;

      case EditBackwardWord:
	  BackWord(&ebuf->curmark);
	  MarkSetcursor(ebuf->curmark);
	  break;

      case EditForwardWord:
	  ForwardWord(&ebuf->curmark);
	  MarkSetcursor(ebuf->curmark);
	  break;

      case EditBeginLine:
	  MarkSetcursor( Rowmark(currow) ); ebuf->wishcol = 0;
	  break;

      case EditEndLine:
	MarkSetcursor( Rowsend(currow) ); ebuf->wishcol = Ncols;
	break;

      case EditUpLine:
	  Upaline();
	  break;

      case EditDownLine:
	  Downaline();
	  break;

      case EditBeginBuffer:
	  Setcursor(ebuf->headmark, ebuf->headpos);  ebuf->wishcol = curcol;
	break;

      case EditEndBuffer:
	  MarkSetcursor(ebuf->endmark);  ebuf->wishcol = curcol;
	  break;


      /********** Text changing commands **********/

      case EditSelfInsert:
	  SelfInsert(ch);
 	  break;

      case EditSelfInsertHigh:
	  SelfInsert(ch | 0x80);
	  break;

      case EditDeleteChar:
	  if ( Backspace() != Nominal ) break;
	  DeleteForward();
	  break;

      case EditDeleteNextChar:
	  DeleteForward();
	  break;

      case EditKillWord:
	  tempmark1 = ebuf->curmark;
	  BackWord(&ebuf->curmark);
	  Kill(ebuf->curmark, tempmark1);
	  break;

      case EditKillNextWord:
	  tempmark1 = ebuf->curmark;
	  ForwardWord(&ebuf->curmark);
	  Kill(tempmark1, ebuf->curmark);
	  break;

      case EditKillLine:
	  Setcursor(ebuf->headmark, ebuf->headpos);

      /* FALLTHRU */

      case EditKillToEnd:
	  Kill(ebuf->curmark, ebuf->endmark);	
	  break;

      case EditUnKill:
	  if (killbuffer == NULL || killbuffer->length == 0) break;
	  m = BlockInsert(ebuf->curmark, killbuffer);
	  killbuffer = NULL;
	  Display(m, ebuf->endmark, &ebuf->curpos, TRUE);
	  MarkSetcursor(ebuf->curmark);
	  break;

      case EditTransposeChars:
	  tempmark2 = ebuf->curmark;
	  Retract(&tempmark2);
	  tempmark1 = tempmark2;
	  Retract(&tempmark1);
	  if (Status != Nominal) break;
	  ch = *tempmark2.cp;
	  *tempmark2.cp = *tempmark1.cp;
	  *tempmark1.cp = ch;
	  Display(tempmark1, ebuf->endmark, &Markpos(tempmark1), TRUE);
	  Setcursor(ebuf->curmark, ebuf->curpos);
	  break;

      case EditTransposeWords:
	  tempmark2 = ebuf->curmark;
	  if (!AtWordStart(tempmark2)) BackWord(&tempmark2);
	  tempmark1 = tempmark2;
	  BackWord(&tempmark1);
	  if (Status != Nominal) break;
	  ebuf->curmark = tempmark2;
	  ForwardWord(&ebuf->curmark);
	  ebuf->mousemark = tempmark1;
	  ForwardWord(&ebuf->mousemark);
	  chunk = BlockDelete(tempmark1, ebuf->mousemark);	  
	  BlockInsert(tempmark2, chunk);
	  chunk = BlockDelete(tempmark2, ebuf->curmark);
	  tempmark1 = BlockInsert(tempmark1, chunk);
	  Display( ebuf->headmark, ebuf->endmark, &ebuf->headpos, TRUE );
	  MarkSetcursor(ebuf->curmark);
	  break;	  

#ifndef STS

      /********** Selection Commands **********/

      case EditDeselect:
	  Deselect();
	  MarkSetcursor(ebuf->curmark);
	  LineEdit(curClient, ch);	/* now do the original command */
	  break;

#endif


      /********** Miscellaneous Commands **********/

      case EditAbort:
	  Beep;
	  break;

      case EditRedisplay:
	  Display(ebuf->headmark, ebuf->endmark, &ebuf->headpos, TRUE);
	  MarkSetcursor(ebuf->curmark);
	  if (ebuf->selectionexists) Select(ebuf->curmark, ebuf->mousemark);
	  break;

      case EditRelease:
      case EditReleaseAbort:
      case EditEOF:
      case EditKillBreak:

	  if (opCode == EditKillBreak && !client->termPid) break;

	  MarkSetcursor(ebuf->endmark);
	  if (opCode == EditReleaseAbort)
	      TextInsertChar( &ebuf->curmark, Control('G') );
	  else if (opCode == EditRelease) TextInsertChar(&ebuf->curmark, '\n');

	  for ( chunk = ebuf->inputmark.chunk ; chunk && chunk->length > 0 ;
	  	chunk = chunk->next )

	    {
	      if ( !(event = FreeEvents) ) break;
	      FreeEvents = event->next;
	      if (client->eventTail) client->eventTail->next = event;
	      if (client->eventQ == NULL) client->eventQ = event;
	      client->eventTail = event;
	      event->next = NULL;
	      for ( i = 0 ; i < chunk->length ; i++ )
		  event->shortbuffer[i] = chunk->text[i];
	      event->bytecount = chunk->length;
	      event->code=(ebuf->displayed) ? ReportEditedText:ReportCharacter;
	    }


	  if (opCode == EditEOF)
	    {
	      DisplayInverse(pad);
	      DisplayString(pad, "<EOF>");
	      DisplayNormal(pad);
	      DisplayCRLF(pad);
	      PutEndEvent(client);
	    }

	  else if (opCode == EditKillBreak)
	    {
	      EventToUser(client);
	      DestroyProcess(client->termPid);
	      DisplayInverse(pad);
	      DisplayString(pad, "<BREAK>");
	      DisplayNormal(pad);
	      DisplayCRLF(pad);
	      client->termPid = 0;
	    }

	  else
	    {
	      EventToUser(client);
	      if (opCode == EditRelease) DisplayCRLF(pad);
	    }

	  BufInit(ebuf);

	  break;


#ifdef DEBUG

      /********** Debugging Commands **********/

      case EditDumpCursor:
	  FormatMsg("cursor: chunk=%x  cp=%x  row=%d  col=%d  ch=%c,  wishcol=%d",
		ebuf->curmark.chunk,ebuf->curmark.cp,currow,curcol,*ebuf->curmark.cp,ebuf->wishcol);
	  break;

      case EditDumpCurrentChunk:
	  DisplayCRLF(VgtsPad);
	  for (cp = ebuf->curmark.chunk->text; cp < Chunkend(ebuf->curmark.chunk); cp++)
		DisplaySend(VgtsPad, *cp);
	  DisplaySend(VgtsPad, 11);	/* control-K, a vertical bar */
	  DisplayCRLF(VgtsPad);
	  break;

      case EditDumpMarklist:
	  DisplayCRLF(VgtsPad);
	  for (marklp = ebuf->markList; marklp; marklp = marklp->next)
	    {
	      FormatMsg("<chunk %x, cp %x>  ",marklp->pmark->chunk,
		marklp->pmark->cp);
	    }
	  DisplayCRLF(VgtsPad);
	  break;

      case EditDumpChunks:
	  DisplayCRLF(VgtsPad);
	  DumpChunks(ebuf->headmark.chunk);
	  break;

      case EditDumpFirstRows:
	  SixRows();
	  break;

      case EditDumpLastRows:
	  LastRows();
	  break;

      case EditDumpKillbuffer:
	  NewMsg("killbuffer:");
	  DumpChunks(killbuffer);
	  for (chunk = killbuffer; chunk; chunk = chunk->next)
	    for (cp = chunk->text; cp < Chunkend(chunk); cp++)
		  DisplaySend(VgtsPad, *cp);
	  break;

      case EditDumpMajorMarks:
	  FormatMsg("headmark = (%x, %x)  endmark = (%x, %x)  regionmark = (%x, %x)",
		ebuf->headmark.chunk, ebuf->headmark.cp, ebuf->endmark.chunk, ebuf->endmark.cp,
		ebuf->regionmark.chunk, ebuf->regionmark.cp);
	  break;

#endif

      }  /* end switch */

  }


DisplayInput(curClient)

  register struct Client *curClient;

  {
	/*
	 * This routine displays the line edit buffer for the given client,
	 * starting at the current cursor position.
	 */

    register RowRec *r;

    FetchBuffer(curClient);    
    PadGetCursor(pad, &headrow, &headcol);
    headrow--;		/* pos is zero-based */
    headcol--;
    cursorPos = ebuf->headpos;
    for ( r = ebuf->rows ; r < ebuf->rows + Nrows ; (r++)->exists = 0 );
    ebuf->displayed = (client->mode & Echo) ? 1 : 0;
    Display(ebuf->headmark, ebuf->endmark, &ebuf->headpos, TRUE);
    MarkSetcursor(ebuf->endmark);
    ebuf->wishcol = curcol;

  }


HandleShoveInput(curClient, msg, pid)

  register struct Client *curClient;
  register IoRequest *msg;

  {

  	/*
	 *  The client wants to shove some input into the line buffer.
	 */

    register int nChars;
    Chunk newText;
    char lineBuf[256];

    FetchBuffer(curClient);

    if (client->mode & LineBuffer)

      {

	nChars = min(msg->bytecount, 256);
	MoveFrom(pid, lineBuf, msg->bufferptr, nChars);
	newText = StringText(lineBuf, nChars);

	if (newText)
	  {
	    FreeText(ebuf->headmark.chunk);
	    BufInit(ebuf);
	    ebuf->headmark = BlockInsert(ebuf->headmark, newText);
	    Display(ebuf->headmark, ebuf->endmark, &ebuf->headpos, TRUE);
	    msg->requestcode = OK;
	  }

	else msg->requestcode = NO_SERVER_RESOURCES;

      }

    else msg->requestcode = ILLEGAL_REQUEST;
    Reply(msg,pid);

  }

#ifndef STS

BOOLEAN LineEditGraphics(curClient, x, y, buttons, uptrans)

    register struct Client *curClient;
    short x, y, buttons;
    register int uptrans;

{ Pos clickpos, releasepos;
  Mark clickmark, releasemark;
  short entry;
  int index;


    FetchBuffer(curClient);

    if (!ebuf->displayed) return;

/*** Button 1 ***/
    if (buttons == LeftButton) {
	if (ebuf->selectionexists) Deselect();
	PadCursorOff(padIndex);
	client->mode |= NoCursor;
	DynamicSelect(x, y, &clickpos, &releasepos);
	clickmark = Posmark(&clickpos);
	releasemark = Posmark(&releasepos);
	if (MarkEQ(clickmark,releasemark)) { /* simply place the cursor */
	    Setcursor(clickmark, clickpos);
	    client->mode &= ~NoCursor;
	    }
	else {  /* we have a selection */
	    ebuf->selectionexists = 1;
	    ebuf->curKeyTable=GetKeyTable(ebuf->keyTableList,SelectionKeyTable);
	    if (PosGT(releasepos,clickpos)) {
		mousepos = releasepos;  ebuf->mousemark = releasemark;
		MarkSetcursor(clickmark);
		}
	    else {
		mousepos = clickpos;  ebuf->mousemark = clickmark;
		MarkSetcursor(releasemark);
		}
	    }
	ebuf->wishcol = curcol;
	return(TRUE);
	}

/*** Button 2 ***/
    else if ( (buttons == MiddleButton) && uptrans )

      {

	if (ebuf->selectionexists)
	  {
	    KillSelection();
	    ebuf->wishcol = curcol;
	  }

	else if (killbuffer)
	  {
	    InsertSelected(killbuffer);
	    killbuffer = NULL;
	  }

	return(TRUE);

      }

/*** Button 3 ***/
    else if ( (buttons == RightButton) && uptrans && ebuf->menu )

      {
	if ( (entry = popup(ebuf->menu)) >= 0 )

	  {
	    char *stringPtr;
	    int nChars;
	    Chunk newText;
	    Mark m;

	    for (index = 0 ; ebuf->menu[index].menuNumber != entry ; index++);
	    stringPtr = ebuf->menu[index].string;
	    for (nChars = 0 ; *(stringPtr++) != '\0' ; nChars++);
	    newText = StringText(ebuf->menu[index].string, nChars);
	    m = BlockInsert(ebuf->curmark, newText);
	    Display( m, ebuf->endmark, &ebuf->curpos, TRUE);
	    MarkSetcursor(ebuf->curmark);
	  }

      }

    else return(FALSE);

  }

#endif
