/*
 * Server for Microvax loading
 *
 * Originally by Steve Tepper (greep@su-pescadero.ARPA), January 1985
 * Modified   by Thomas Maslen(maslen@su-pescadero.ARPA) 14-Feb-85
 *		 - add code for transfer address (entry point),
 *		   including -T option
 *		 - fix calls to fprintf
 *	      NOTE: I'm not sure whether this contained specifically LITTLE-
 *		    ENDIAN code before, but it certainly does now.
 * Modified by greep March 19, 1985: if there is an error in reading or
 *          loading the file, it loads the standard file and stores the
 *          error message at _end.
 * Modified by greep March 20, 1985:
 *          change -T to look at next argument for number
 *          read load address as well as program name from secndboot
 * See RCS logs for further history.
 * This program isn't the @i(most) unsavoury one around, but it's not really
 * nice.
 *
 * The following information is from the "Microvax I CPU technical
 *  description", page 2-32:
 *
 * microvax sends multicast "program request" message
 * server sends "assistance volunteer" message
 * (microvax sends all further messages to the server which responded,
 * rather than multicast)
 * microvax sends another "program request" message
 * server sends "memory load" message with the first chunk of the program
 * microvax sends "request memory load" to acknowledge the load and to
 *  request the next chunk
 * previous two steps are repeated until all of program is loaded
 * (if server doesn't get acknowledgement, it retransmits the "memory
 *  load" message)
 * server sends "parameter load plus transfer address" message
 * microvax starts executing at the specified address
 *
 * The formats of the messages are described in the manual with the
 * following unwieldy title:
 *  DECnet Digital Network Architecture
 *  Phase IV
 *  Maintenance Operations
 *  Functional Specifications
 */

/* Microvax I loads programs starting at BASE, and offsets all load and transfer
 *   addresses by BASE.
 */
#define BASE 0x7000     /* Microvax I adds this to load address */

#define MVAX_PKT_TYPE (ushort) 0x0160  /* Packet type used in MOP */

#define RETRY_COUNT  5  /* Number of times to retry in main load loop */

/* Priorities >= 2 are "high", meaning that a given packet will be given
   to at most one process. */
#define ENETFILTERPRIORITY ((unsigned char) 2)

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/in.h>
#include <netinet/if_ether.h>

#include "configboot.h"

#include <sys/enet.h>
#include <sys/enetdefs.h>
#include <signal.h>
#include <stdio.h>
#include <a.out.h>

extern char *sys_errlist[];
extern int errno, sys_nerr;


struct Packet {
    struct ether_addr DstAddr;   /* Address of destination host */
    struct ether_addr SrcAddr;   /* Address of source host */
    unsigned short PacketType;  /* Packet type */
    unsigned short PacketLen;   /* Length of data (everything past here) */
    unsigned char MOPcode;      /* MOP op code */
    union {
	struct {
	    unsigned char dev;              /* Device type */
	    unsigned char format;           /* Format */
	    unsigned char type;             /* Program request type */
	} PgmReq;                   /* Program request message */
	struct {
	    unsigned char loadnum;      /* Load number */
	    unsigned char loadaddr[4];  /* Load address */
	    char loaddata[1];           /* Data to be loaded */
	} MemLoad;                  /* Memory load */
	struct {
	    unsigned char loadnum;      /* Load number */
	    unsigned char error;        /* Error flag */
	} ReqMemLoad;               /* Request memory load */
	struct {
	    unsigned char loadnum;      /* Load number */
	    char params[1];             /* Parameters and transfer address */
	} ParamLoad;                /* Paramet load with transfer address */
	char PacketData[1500];      /* General; bigger than any Ethernet  */
				    /*   packet we could receive	  */
    } un;                       /* Data */
};

#ifndef PGM
>>>>>>> PGM should be defined (use '-DPGM="pathname"') to be the pathname
>>>>>>> of the default bootstrap program (Vload).
#endif  PGM

int Verbose;
int Debug;
int doAltEtherAddr;	    /* If non-zero and we can't find a config file */
			    /*   for a given ethernet address, then search */
			    /*   all the config files for an 		   */
			    /*   "alt-ether-addr" entry with the right addr*/
int noConfigFileNeeded;	    /* Even if we can't find any config file, boot */
			    /*   any requesting machine using defaults.    */

char *BannerAddr;           /* Place to write arg into */
struct enfilter Filter1;    /* Filter for initial (multicast) request */
struct enfilter Filter2;    /* Filter for this load */
struct ether_addr OurAddr;   /* Out ethernet address */
struct ether_addr UserAddr;  /* Ethernet address of cpu requesting load */
struct ether_addr MulticastAddr = {{0xab, 0x00, 0x00, 0x01, 0x00, 0x00}};
int MyPid;                  /* Current process id */
long int MvaxBase  = BASE;  /* Default if no -T given */
long int StartAddr = 0x0000;
int EnetFD;
struct endevp devparams;

/* List of ethernets to try */
char *EnetDevs[] = {"/dev/enet", "/dev/eneta", "/dev/enetb", 0};

char *EnetDev;      /* Device being used */

int catch_sigs();

main(argc, argv)
int argc;
char *argv[];
{
    struct stat statbuf;
    int i;

    for (i=1; i<NSIG; i++)
	if ( i != SIGCHLD  /* Want the default action (ignore) for this	*/
	  && i != SIGCONT  /* Ditto, otherwise some things (e.g. trying */
			   /*   to ignore SIGHUPS) don't work too well	*/
	  && i != SIGKILL  /* Can't do anything with this		*/
	  && i != SIGSTOP  /* Can't do anything with this either	*/
	  && i != SIGHUP   /* Want to ignore this - see below		*/
	    )
	    if ((int)signal(i, catch_sigs) == -1)
	      {
		char s[50];
		sprintf(s, "Error trying to catch signal %d", i);
		perror(s);
		exit(1);
	      }
    if ((int)signal(SIGHUP, SIG_IGN) == -1)
      {
	perror("Error trying to ignore SIGHUP");
	exit (1);
      }
    if ((int)signal(SIGCHLD, SIG_DFL) == -1)
      {
	perror("Error trying to use default for SIGCHLD");
	exit (1);
      }

    MyPid = getpid();
    setpgrp(MyPid, MyPid);	/* Put us in a separate process group (and  */
				/*   our pid seems like a good number for a */
				/*   process group) so we don't receive     */
				/*   stray signals from other processes     */

    ccDefaultProgram = PGM;

    setbuf(stdout, NULL);	/* Make stdout unbuffered, so we see things */
				/*   as soon as possible (if at all)	    */

    for (i = 1; i < argc; i++)
    {
	if (argv[i][0] == '-')
	{
	    switch(argv[i][1])
	    {

	      case 'v':
		Verbose = 1;
		break;

	      case 'd':
		Debug = 1;
		break;

	      case 'a':
		doAltEtherAddr = 1;
		break;

	      case 'n':
		noConfigFileNeeded = 1;
		break;

	      case 'e':                   /* Ethernet device to use */
		EnetDev = argv[++i];
		break;

	      case 'c':
		if ( i+1 < argc )
		    ccConfigDir = argv[++i];
		else
		  {
		    fprintf(stderr,
		"Argument after '-c' should be directory for config files\n");
		    exit(1);
		  }
		break;

	      case 'T':
		if ( sscanf(argv[++i], "%X", &MvaxBase) == 1 ) {
		    /* Successfully read hex value. */
		    break;
		} else {
		    fprintf(stderr,"Invalid hex offset %s\n", argv[i]);
		    exit(1);
		}

	      default:
		fprintf(stderr,"Invalid flag %s\n", argv[i]);
		exit(1);
	    }
	} else                            /* Program to load */
	{
	    /* Copy name, since arg will get overwritten */
	    ccDefaultProgram = (char *)malloc(strlen(argv[i])+1);
	    strcpy(ccDefaultProgram, argv[i]);
	    BannerAddr = argv[i];       /* Write into this location */
	}
    }

    ccVerbose = Verbose;
    ccDebug   = Debug;

    if ( !FindBootFile(ccDefaultProgram) )
	/* Maybe noone cares, because every config file has a */
	/*   "bootfile:" entry.				      */
	fprintf(stderr,"Warning: couldn't find default bootfile \"%s\"\n",
		ccDefaultProgram);

    if ( ccConfigDir == NULL || *ccConfigDir == '\0' ||
	 stat(ccConfigDir, &statbuf) == -1 ||
	 (statbuf.st_mode & S_IFMT) != S_IFDIR )
	if (noConfigFileNeeded)
	  {
	    if (Verbose || Debug)
		fprintf(stderr,
		     "Warning: couldn't find config-file directory \"%s\"\n",
		     ccConfigDir);
	  }
	else
	  {
	    fprintf(stderr, "Couldn't find config directory \"%s\" - quit\n",
		 ccConfigDir);
	    exit(1);
	  }

    if (Verbose)
      {
	printf("Config-file directory: %s\n", ccConfigDir);
	printf("Default boot file    : %s\n", ccDefaultProgram);
      }

    if (Verbose)
	printf("Listener process is %d\n", MyPid);
    EnetFD = find10mbEnet();
    if (EnetFD < 0)
	exit(1);

    /* Accept packets of type 0160 (hex) */
    /* (that's what the microvax sends out) */
    efinit(&devparams, &Filter1, ENETFILTERPRIORITY, MVAX_PKT_TYPE);

    Filter2 = Filter1;  /* Will use Filter2 for specific requests */

    /* Have the filter check for the multicast address as the destination */
    /* Note that this assumes 48-bit addresses (won't work with 3Meg net) */
    /* Packet offsets (third argument) are relative to data past header */
    /* (header consists of dst addr, src addr, and type) */
    {
	struct Packet dummypacket;
	if (eflginsert(&devparams, &Filter1,        /* Check first 4 bytes */
	    -((int)&dummypacket.PacketLen - (int)&dummypacket.DstAddr),
	    *(long *)&MulticastAddr.ether_addr_octet[0]) ||
	  efwdinsert(&devparams, &Filter1,        /* Check last 2 bytes */
	    -((int)&dummypacket.PacketLen - (int)&dummypacket.DstAddr) + 4,
	    *(short *)&MulticastAddr.ether_addr_octet[4]) ||
	  efAND(&Filter1) || /* Form the logical AND of these checks */
	  efAND(&Filter1))
	{
	    fprintf(stderr, "Can't add to filter\n");
	    exit(1);
	}
	if (ensetfilt(EnetFD, &Filter1))
	{
	    fprintf(stderr, "Can't set filter\n");
	    exit(1);
	}
    }

    if (Debug) {
	int i;
	printf("Filter1 priority: %d\n", Filter1.enf_Priority & 0xff);
	for (i = 0; i < Filter1.enf_FilterLen; i++)
	    printf("%04x\n", Filter1.enf_Filter[i]);
    }

    /* Sit and listen for load requests */
    for(;;)
	getrequest();
}


/* Use the device name specified if any, otherwise go through standard list */
find10mbEnet()
{
    char **devptr;
    int fd;

    if (EnetDev)    /* Was a particular device specified? */
    {               /* If so, try it */
	if ((fd = open10mbEnet(EnetDev)) >= 0)
	    return fd;
    }
      else          /* Otherwise try a standard list */
    {
	for (devptr = EnetDevs; *devptr != 0; devptr++)
	{
	    EnetDev = *devptr;
	    if ((fd = open10mbEnet(EnetDev)) >= 0)
		return fd;
	}

    }
    if (Verbose)
	printf("Can't find 10meg net\n");
    return -1;
}


/* Try to open an ethernet and see if it's a 10-megabit net */
open10mbEnet(dev)
char *dev;
{
    struct endevp recvparam;
    int fd;

    if (Verbose)
	printf("Trying %s\n", dev);
    fd = our_enopen(dev, &devparams);  /* Get an Ethernet channel */
    if (fd < 0)
    {
	if (Verbose)
	    perror(dev);
	return -1;
    }
    /* Find out the Ethernet address and type */
    ioctl(fd, EIOCDEVP, &recvparam);
    if (Debug)
	printf("enet type: %d\n", recvparam.end_dev_type);
    if (recvparam.end_dev_type != ENDT_10MB)
    {
	close(fd);
	return -1;      /* Not a 10-megabit net */
    }

    if (Verbose) {
	int i;

	printf("My address:");
	for (i = 0; i < 6; i++)
	    printf(" %02x", recvparam.end_addr[i]&0xff);
	printf("\n");
	printf("Broadcast address:");
	for (i = 0; i < 6; i++)
	    printf(" %02x", recvparam.end_broadaddr[i]&0xff);
	printf("\n");
    }

    bcopy(recvparam.end_addr, &OurAddr, 6);
    return fd;          /* Return the file descriptor */
}

catch_sigs(sig, code, scp)
    int sig, code;
    struct sigcontext *scp;
  {
    if (sig == SIGTERM)
      {
	printf("Killed by a SIGTERM\n");
	exit(100);
      }
    fprintf(stderr, "Process %d (parent %d) got sig %d, code %d\n",
			getpid(), getppid(), sig, code);
    fprintf(stderr, "  onstack=%x, mask=%x, sp=%x, pc=%x, ps=%x\n",
			scp->sc_onstack, scp->sc_mask, scp->sc_sp,
			scp->sc_pc, scp->sc_ps);
    fprintf(stderr, "  Let's try to dump core by doing a SIGQUIT\n");
    signal(SIGQUIT, SIG_DFL);
    kill(getpid(), SIGQUIT);
    fprintf(stderr, "  Aaargh - what are we doing here? - exit\n");
    exit(200);
  }

static int request_num;	/* Just used to tag printed messages; since we   */
			/*   intermix messages on stderr and stdout from */
			/*   a number of processes, it gets confusing.   */

getrequest()
{
    /* Main work is done here */

    struct Packet inpacket;     /* Input packet */
    int newpid;                 /* New process id */
    struct HostTableEntry *ht;	/* Config file data */

    request_num++;

    if (Verbose)
	printf("[%d]Waiting for request\n", request_num);
    
    if (readpacket(&inpacket))      /* Read a packet; filter is Filter1 */
	return;
    /*
     * This should be a program request (8) message addressed to the
     * multicast address
     * (The filter should be checking for the multicast address but
     *  do it here anyway.)
     */
    if (!(inpacket.MOPcode == 8 &&     /* Check for program request */
	bcmp(&inpacket.DstAddr, &MulticastAddr, sizeof MulticastAddr) == 0))
      {
	if (Verbose)
	  {
	    if (inpacket.MOPcode != 8)
		printf("Got MOP code=%d, not program load request\n",
		    inpacket.MOPcode);
	    else
		printf("Got non-multicast program load request\n");
	  }
	return;                 /* Otherwise ignore it */
      }

    /* Remember who this user is */
    UserAddr = inpacket.SrcAddr;

    if (!( ((ht = HostLookup(&UserAddr)					) ||
	    (ht = ReadConfigFileByAddress(&UserAddr)			) ||
	    (doAltEtherAddr && (ht = ReadConfigFileWithAwk(&UserAddr))	) ||
	    (noConfigFileNeeded && (ht = UseDefaults(&UserAddr))	)
	   ) 
	    && ht->bootpath
	    && *(ht->bootpath)
	 ) )
	/*
	 * Then either there isn't a config file for this host (and we're not
	 * servicing requests unless we have a config file), or the config
	 * file explicitly says not to service boot requests from it, so...
	 */
	return;

    /* Make the child an orphan so it will be adopted by init */
    /* (If you don't understand this, find a unix wizard to explain) */
    fflush(stdout);
    fflush(stderr);
    newpid = fork();
    if (newpid == 0)
      {
	int i;
	char *p;
	int bannerlen;

	newpid = fork();
	if (newpid != 0)
	    exit(0);        /* Always exit, even on failure */

	signal(SIGALRM, SIG_DFL);	/* Use the default handler, i.e. die*/
					/*   quietly if we ever get an alarm*/

	/* If there's space, write the client host address for ps to see so
	   people tell this is not the multicast listener and know what
	   machine to look at if this process hangs */
	if (BannerAddr && (bannerlen = strlen(BannerAddr)) > 12)
	  {
	    for (p = BannerAddr, i = 0; i < 6; i++)
	      {
		sprintf(p, "%02x", UserAddr.ether_addr_octet[i]);
		p += 2; /* Each byte takes two positions in hex */
	      }
	    while (p < &BannerAddr[bannerlen])  /* Fill out with blanks */
		*p++ = ' ';
	  }

	close(EnetFD);      /* Have to get a new file descriptor */
	EnetFD = our_enopen(EnetDev, &devparams);  /* Get an Ethernet channel */
	if (EnetFD < 0) {
	    perror(EnetDev);
	    exit(1);
	}
	MyPid = getpid();
	loadpgm(ht->bootpath);
	exit(0);

      }
    else if (newpid != -1)
      {
	wait(0);            /* Wait for intermediate process */
	return;
      }
    /* If the fork failed, tough. */
}


loadpgm(bootpath)
    char *bootpath;		/* Name of boot file */
{
    struct Packet inpacket;     /* Input packet */
    struct Packet outpacket;    /* Output packet */
    int loadnum;                /* Chunk number */
    struct exec   proghdr;	/* a.out header */
    int len;                    /* Amount of data read from file */
    char buf[256];              /* Input buffer for file */
    long int loadaddr;          /* Address currently loading into */
    long int loadlen;           /* Number of bytes to load */
    FILE *FileFD;               /* Input stream pointer */

    if (Verbose) {
	int i;
	printf("[%d]Load request from host ", request_num);
	for (i = 0; i < 6; i++)
	    printf("%02x ", UserAddr.ether_addr_octet[i]&0xff);
	printf("handled by process %d\n", MyPid);
    }

    /* Look for packets from this user only */
    if (eflginsert(&devparams, &Filter2,        /* Check first 4 bytes */
	-((int)&inpacket.PacketLen - (int)&inpacket.SrcAddr),
	*(long *)&UserAddr.ether_addr_octet[0]) ||
      efwdinsert(&devparams, &Filter2,        /* Check last 2 bytes */
	-((int)&inpacket.PacketLen - (int)&inpacket.SrcAddr) + 4,
	*(short *)&UserAddr.ether_addr_octet[4]) ||
      efAND(&Filter2) || /* Form the logical AND of these checks */
      efAND(&Filter2))   /* Need one more AND for the packet type */
    {
	fprintf(stderr, "Can't add to filter 2\n");
	exit(1);
    }
    if (ensetfilt(EnetFD, &Filter2))
    {
	fprintf(stderr, "Can't set filter 2\n");
	exit(1);
    }

    if (Debug) {
	int i;
	printf("[%d]Filter2 priority: %d\n", 
		request_num, Filter2.enf_Priority & 0xff);
	for (i = 0; i < Filter2.enf_FilterLen; i++)
	    printf("%04x\n", Filter2.enf_Filter[i]);
    }

    /* Offer to help this user */
    outpacket.DstAddr = UserAddr;
    outpacket.MOPcode = 3;  /* Assistance volunteer */
    outpacket.PacketLen = 1;          /* 1 byte of data (op code) */
    writepacket(&outpacket);

    alarm(60);  /* Wait 60 seconds max for response. */

    /*
     * Now wait for another program request with format=1 and
     * type=2, directed specifically to us (not multicast).
     * (These numbers are from the MOP manual.)
     *
     * We have to time out after a finite period since this program
     * may be running on more than one machine, or there may be more
     * one copy running on a single machine, and the microvax will
     * respond to only one.
     */

    if (Debug)
	printf("[%d]Waiting for second load request\n", request_num);
    do {
	if (readpacket(&inpacket))
	    return;
    } while ((!(inpacket.MOPcode == 8 &&
	bcmp(&inpacket.DstAddr, &OurAddr, sizeof OurAddr) == 0 &&
	/* Filter should be checking source address but do it here anyway */
	bcmp(&inpacket.SrcAddr, &UserAddr, sizeof UserAddr) == 0 &&
	inpacket.un.PgmReq.format == 1 &&
	inpacket.un.PgmReq.type == 2 )));
    alarm(0);

    if (Debug)
	printf("[%d]Got second load request\n", request_num);

retry:
    FileFD = fopen(bootpath, "r");
    if (FileFD == NULL) {
	/* Shouldn't happen, unless someone blows away the file while we're */
	/*   not looking.  Can't really do much.  Maybe we should open the  */
	/*  file before we volunteer to help				    */
	perror(bootpath);
	return;
    }

    if ( fread(&proghdr, 1, sizeof proghdr, FileFD) != sizeof proghdr ||
	 N_BADMAG(proghdr) ) 
	/* Maybe we should check for this in FindBootFile() */
    {
	fprintf(stderr, "%s: not executable or invalid magic number\n",
		bootpath);
	fclose(FileFD);
	return;
    }
    /*
     * Load only the text and data -- not the symbol table and relocation
     * information.
     */
    loadlen = proghdr.a_text + proghdr.a_data;
    fseek(FileFD, (long)N_TXTOFF(proghdr), 0);

    if (Verbose)
	printf("[%d]Loading %s at %x + %x\n",
		request_num,bootpath, StartAddr, MvaxBase);

    /* Read in the file and send it out */
    outpacket.MOPcode = 2;              /* Memory load */
    loadaddr = StartAddr;               /* Starting load address */
    loadnum = 0;
    if (Debug)
	printf("[%d]Total of %x+%x=%x bytes\n", request_num, proghdr.a_text,
	      proghdr.a_data, loadlen);
    while (loadlen > 0) {
	int readlen;
	readlen = sizeof buf;     /* Read min(loadlen, sizeof buf) bytes */
	if (loadlen < sizeof buf)
	    readlen = loadlen;
	/* Read one block of the file */
	if ((len = fread(buf, 1, readlen, FileFD)) <= 0) {
	    fprintf(stderr, "Read error\n");
	    fclose(FileFD);
	    return;
	}
	if (Debug)
	    printf("[%d]Loading 0x%x bytes at 0x%x\n",
		   request_num, len, loadaddr);
        outpacket.DstAddr = UserAddr;
	outpacket.un.MemLoad.loadnum = loadnum; /* Load number */
	bcopy(&loadaddr, outpacket.un.MemLoad.loadaddr, 4); /* Address */
	bcopy(buf, outpacket.un.MemLoad.loaddata, len);
	outpacket.PacketLen = len + 6;
	/* Send it to the microvax */
	if (sendload(&outpacket, loadnum) != 0) {
	    fclose(FileFD);
	    return;
	}
	loadlen -= len;     /* Decrement amount remaining to read */
	loadnum++;          /* This is the number the microvax uses */
	loadaddr += len;    /* Increment address for next load */
	if (Verbose)
	{
	    printf(".");    /* Show that progress is being made */
	    fflush(stdout);
	}
    }
    fclose(FileFD);
    if (Verbose)
	printf("\n");

    {
	char *p;

	/* Send "parameter load with transfer address" message */
	outpacket.MOPcode = 20;
	outpacket.un.ParamLoad.loadnum = loadnum;   /* Load number */
	p = outpacket.un.ParamLoad.params;
	/* Parameters can be inserted here */
	*p++ = 0;                           /* Mark end of parameters */
	{
	    unsigned long int i;

	    proghdr.a_entry &= ~0x80000000;
	    i = proghdr.a_entry - MvaxBase; /* Microvax I will add 0x7000 */
	    bcopy(&i, p, 4);                /* Address to start executing at */
	}
	p += 4;
	outpacket.PacketLen = p - (char *)&outpacket.MOPcode;
	outpacket.DstAddr = UserAddr;
	writepacket(&outpacket);
    }
    if (Verbose) {
	printf("[%d]Program loading finished\n", request_num);
	fflush(stdout);
    }
    return;
}


struct Packet *resend_packet;
int Num_Retries;   ;
int timeout_rtn();

sendload(packet, loadnum)
struct Packet *packet;
int loadnum;
{
    struct Packet inpacket;

    resend_packet = packet;     /* Put in global variable for interrupt rtn */
    Num_Retries = 0;
    writepacket(packet);
    if ((int)signal(SIGALRM, timeout_rtn) == -1) {
	perror("Trying to catch SIGALRM for retransmissions");
	exit(1);
    }
    alarm(3);       /* Call interrupt routine after a while */
    do {
	if (Debug)
	    printf("[%d]Waiting for ACK on load number %d\n",
		    request_num, loadnum);
	
	if (readpacket(&inpacket))  /* Filter is (or should be) Filter2 */
	    return;
    } while
       (!(inpacket.MOPcode == 10 &&
	  bcmp(&inpacket.DstAddr, &OurAddr, sizeof OurAddr) == 0 &&
	  bcmp(&inpacket.SrcAddr, &UserAddr, sizeof UserAddr) == 0 &&
	  (inpacket.un.ReqMemLoad.loadnum & 0xff) == 
		((loadnum + 1) & 0xff)));
    /*
     * This message acts both as an acknowledgement and as a request
     * for the next load.  The "load number" is the number of the
     * next load, not the one being acknowledged.
     */
    alarm(0);       /* Cancel timeout */
    if (Debug)
	printf("[%d]Got ACK\n", request_num);
    if (inpacket.un.ReqMemLoad.error) {
	fprintf(stderr, "Microvax reported load error\n");
	return 1;
    }
    return 0;
}

timeout_rtn()
{
    if (Debug)
	printf("[%d]Timeout\n", request_num);
    writepacket(resend_packet);
    if (++Num_Retries > RETRY_COUNT)
    {
	if (Verbose)
	    printf("[%d]No response\n", request_num);
	exit(1);
    }
    /*
     * We don't need this for 4.2BSD (the signal handler stays enabled until
     *   we ask for a change) but this may nail one portability bug.
     */
    if ((int)signal(SIGALRM, timeout_rtn) == -1) {
	perror("Trying to re-catch SIGALRM for retransmissions");
	exit(1);
    }
    alarm(3);               /* Reset alarm clock */
}


int our_enopen(dev, devpars)
char *dev;
struct endevp *devpars;
{
    struct eniocb eniocb;
    int fd;

    fd = enopen(dev, devpars);
    if (fd >= 0)
    {
	eniocb.en_rtout = 0;		/* Set timeout to 0 ("no timeout"):  */
	ioctl(fd, EIOCSETP, &eniocb);	/* enopen() calls open(), which gets */
    }					/* it right, then calls enflush()    */
					/* which sets the timeout to -1      */
    return(fd);
}

int
readpacket(packet)
struct Packet *packet;
{
    short int len;
    char *p;
    int count;

    /* The ethernet driver often returns an error even when nothing
       appears to be wrong, so give it several chances. ??!! */
    for (count = 0; count < 100; count++)
    {
	/* We assume that the filter and timeout have already been set; */
	/*   ensetfilt() sets the filter, and the timeout is set to 0   */
	/*   ("no timeout") by our_enopen(), which is what we want (?). */
	/* Hence we can just use enreadquick().				*/
	if (enreadquick(EnetFD, &devparams, &packet->PacketLen) == 0)
	    goto goodread;      /* Success */
	if (Verbose)
	    printf("[%d]Ethernet read error\n", request_num);
    }
    return 1;               /* No good */

goodread:
    if (Debug)
    {
	int i;

	printf("[%d]Read(%d):\n", request_num, MyPid);
	bcopy(&packet->PacketLen, &len, 2);
	len += 16;
	for (i=0, p=(char *)packet; i < len; i++) {
	    if (i%16 == 0 && i > 0) /* Print 16 bytes per line */
		printf("\n");
	    printf(" %02x", (*p++)&0xff);
	}
	printf("\n");
    }
    return 0;
}


writepacket(packet)
struct Packet *packet;
{
    /*
     * Write the packet.  Enwrite will fill in the source and
     * destination addresses and the packet type, so the address
     * given is the beginning of the length field.  The length
     * is incremented by 2 to account for the length field itself,
     * and then rounded up to an even value.
     */
    packet->SrcAddr = OurAddr;
    if (enwrite(&UserAddr, MVAX_PKT_TYPE, EnetFD, &devparams,
	&packet->PacketLen, ((packet->PacketLen + 2)+1)&~1) != 0) {
	perror("enwrite");
	exit(1);
    }
    if (Debug) {
	int i;
	unsigned short len;
	char *p;

	printf("[%d]Written(%d):\n", request_num, MyPid);
	bcopy(&packet->PacketLen, &len, 2);
	len += 16;
	for (i=0, p=(char *)packet; i < len; i++) {
	    if (i%16 == 0 && i > 0) /* Print 16 bytes per line */
		printf("\n");
	    printf(" %02x", (*p++)&0xff);
	}
	printf("\n");
    }
}
