/*			Memory File Server
 *
 * File manipulation - requests and subroutines
 */

#include <Vstorage.h>
#include <Vgroupids.h>
#include "memserv.h"

extern char NameBuffer[];

SystemCode HandleCreateFile(req, id, segbuf, segsize)
    FSRequest *req;
    ProcessId id;
    char *segbuf;
    unsigned segsize;
  {
    Filedesc	*fd;
    char 	*suffix;
    SystemCode  r;

    if ((r = LookupFile(req, id, &fd, &suffix, segbuf, segsize)) == OK)
	r = DUPLICATE_NAME;
    if (IsGroupId(fd->ctxpair.pid))
	r = MULTI_MANAGER;

    if (r == NOT_FOUND)
      {
        r = NewFile(fd, suffix, NULL);
      }

    return( r );
  }



UnlinkFromTree(fd)
  Filedesc *fd;
  {
    Filedesc *bro;

    /* Unlink fd from tree */
    bro = fd->father->son;
    if (fd == bro)
      {
	fd->father->son = fd->brother;
      }
    else
      {
	while (bro->brother != fd) bro = bro->brother;
	bro->brother = fd->brother;
      }
  }

SystemCode HandleRemoveFile(req, id, segbuf, segsize)
  FSRequest *req;
  ProcessId id;
  char *segbuf;
  unsigned segsize;
  {
    Filedesc     *fd;
    SystemCode   r;

    if ((r = LookupFile(req, id, &fd, NULL, segbuf, segsize)) != OK )
    	return r;
    if (IsGroupId(fd->ctxpair.pid))
    	return MULTI_MANAGER;
    if (FindInstance(fd))
    	return BUSY;

    RemoveBlocksFromFile(fd, fd->nblocks);
    UnlinkFromTree(fd);
    fd->start = NULL;	/* sign of an empty file descriptor */

    return( OK );
  }


/*   Rename a file.  The old name and new name are concatenated (separated
 *   by a NULL) and pointed to by req->filename.  req->filenamelen is the
 *   the length of the total string.  req->arg1 is the length of the first
 *   name (not incl. the NULL char.).
 */
SystemCode HandleRenameFile(req, id, segbuf, segsize)
    register NameRequest *req;
    ProcessId id;
    char *segbuf;
    unsigned segsize;
  {
    Filedesc *fd, *father, *newplace;
    SystemCode r;
    unsigned tmplength;
    char *newname, *suffix;

    if((r = LookupFile(req, id, &fd, &suffix, segbuf, segsize)) != OK)
    	return(r);
    if (IsGroupId(fd->ctxpair.pid)) return MULTI_MANAGER;

    /* Advance past first name in segment in preparation for
     *  lookup of second name.
     * Also set requestcode to 0 to flag this request as non-
     *  forwardable by LookupFile.
     */
    req->requestcode = 0;
    tmplength = req->nameindex + (suffix - NameBuffer) + 1;
    segbuf += tmplength;
    segsize = (segsize > tmplength) ? (segsize - tmplength) : 0;
    req->nameptr += tmplength;
    req->namelength = (req->namelength > tmplength) ?
		      (req->namelength - tmplength) : 0;
    req->nameindex = 0;
    req->namecontextid = DEFAULT_CONTEXT;	/* ?? */
 
    /* Unlink the node from its old location now, so that the lookup
     *  will fail if we are trying to make it a descendant of itself.
     */
    father = fd->father;
    UnlinkFromTree(fd);

    /* Find new location for node */
    r = LookupFile(req, id, &newplace, &newname, segbuf, segsize);
    if (r == OK && newplace->son == NULL &&
    		  !IsGroupId(newplace->ctxpair.pid))
      {
	/* There is an existing file with the new name and no 
	 *  substructure.  Delete it.
	 */
	RemoveBlocksFromFile(newplace, newplace->nblocks);
	UnlinkFromTree(newplace);
	newplace->start = NULL;		/* sign of an empty file descriptor */
	newname = newplace->name;
	newplace = newplace->father;
      }
    else if (r != NOT_FOUND || !Legalname(newname))
      {
	/* The name is illegal in some way. */

	/* Put fd back under its old father */
	fd->father = father;
	fd->brother = father->son;
	father->son = fd;

	if (r != NOT_FOUND || r != OK) return r;
	if (newplace->son != NULL) return HAS_SUBSTRUCTURE;
	if (IsGroupId(newplace->ctxpair.pid)) return MULTI_MANAGER;
	return ILLEGAL_NAME;
      }

    /* Change the name component stored in the node */
    strncpy(fd->name, newname, MAX_NAME_LENGTH);

    /* Put fd in its new place */
    fd->father = newplace;
    fd->brother = newplace->son;
    newplace->son = fd;

    /* Assign new context ids to fd and its descendants
     *  Iterative algorithm to walk the tree in preorder
     *  stolen from the V kernel.
     */
    while (fd->son) fd = fd->son;
    do
      {
        fd->ctxpair.cid =
	    (fd->ctxpair.cid & 0xffff0000) + 0x18000 + (fd - fdtable);

        if (fd->brother)
          {
            fd = fd->brother;
            while (fd->son) fd = fd->son;
          }
        else fd = fd->father;
      }
    while (fd != newplace);

    return( OK );
  }


/***********************  File-Related Subroutines  *************************/

/* Create a new file by the name specified, and make it a son
 *  of the fd specified.  If fdpointer is non-NULL,
 *  return a pointer to the new file's descriptor in *fdpointer.
 */
SystemCode NewFile(fatherfd, name, fdpointer)
  char *name;
  Filedesc *fatherfd;
  Filedesc **fdpointer;
  {
    register int i;
    SystemCode r;
    Filedesc *fd;

    if (!Legalname(name)) return(ILLEGAL_NAME);

    /* we have the name, now get a file descriptor */
    fd = GetFreeFd();
    if (!fd) return(NO_SERVER_RESOURCES);
    strncpy(fd->name, name, MAX_NAME_LENGTH);

    /* fd will describe an empty file, located at the top of memory */
    fd->start = (mem *) memtop;
    fd->nblocks = 0;
    fd->size = 0;
    fd->timestamp = GetTime(NULL);
    fd->ctxpair.pid = mypid;
    fd->ctxpair.cid =
	(fd->ctxpair.cid & 0xffff0000) + 0x18000 + (fd - fdtable);
    fd->father = fatherfd;
    fd->brother = fatherfd->son;
    fd->son = NULL;
    fatherfd->son = fd;

    if (fdpointer) *fdpointer = fd;
    return(OK);
  }


/* Find or create a free file descriptor */
Filedesc *GetFreeFd()
  {
    register Filedesc *fd, *fdmax;

    fdmax = fdtable + nfiles;
    for (fd = fdtable+2; fd < fdmax; fd++)
	if (fd->start == NULL) return(fd);

    /* we're going to have to create a new one */
    nfiles++;
    if (nfiles * sizeof(Filedesc) > fdfile->nblocks * BLOCKSIZE)
      { /* allocate a new block of file descriptors */
	if (!AddBlocksToFile(fdfile, 1))
	  {
	    nfiles--;
	    return(NULL);
	  }
      }

    /* fd is positioned at the new file descriptor */
    fdfile->size += sizeof(Filedesc);
    fd->start = NULL;  /* make it look empty */
    return(fd);
  }
