/*			Memory File Server
 *
 * File block allocation and manipulation.
 */

#include "memserv.h"


/* Allocate n new blocks at the top of memory.  Returns pointer to first
 * new block, or NULL if unavailable
 */
mem *AddBlocks(n)
  unsigned n;
  {
    unsigned newsize, actual;
    mem *newspace = (mem *) memtop;

    newsize = memtop + n * BLOCKSIZE;
    if (newsize > memlimit) return(NULL);
    actual = SetTeamSize(mypid, newsize);
    if (actual != newsize)
      { /* give back anything you did get */
	SetTeamSize(mypid, memtop);
	return(NULL);
      }
    
    memtop = newsize;
    return(newspace);
  }


/* Allocate n new blocks at the top of memory, or cause a fatal error
 * if we fail.  (Used only in initialization.)
 */
mem *RequireBlocks(n)
  unsigned n;
  {
    mem *newspace;
    if (!(newspace = AddBlocks(n))) 
	Fatal("Required memory is not available");
    return(newspace);
  }


/* Add n blocks to the end of a file.  Note that only the allocated size
 * of the file (nblocks), not the logical size of the file (size) is
 * changed.
 */
int AddBlocksToFile(fd, n)
  register Filedesc *fd;
  int n;
  {
    mem *oldtop, *addpoint;
    register Filedesc *f;
    unsigned shift = n * BLOCKSIZE;

    oldtop = AddBlocks(n);
    if (!oldtop) return(0);	/* failure */
    
    addpoint = fd->start + fd->nblocks * BLOCKSIZE;
    CopyUp(addpoint, oldtop, memtop);
    fd->nblocks += n;

    /* Careful: adding to 0-length files can be dangerous, since there
     * may be >1 files with the same f->start.  We shift up all of them
     * except the one being added to.
     */
    for (f = fdtable; f < fdtable + nfiles; f++)
	if ((f->start >= addpoint) && (f != fd))  f->start += shift;
    return(n);
  }


/* Remove n blocks from the end of the file's space, and compress.  It is
 * the caller's responsibility to be sure that the logical length of the
 * file does not exceed its allocated length.
 */
RemoveBlocksFromFile(fd, n)
  register Filedesc *fd;
  unsigned n;
  {
    register Filedesc *f;
    register mem *oldbase, *newbase;
    unsigned shift = n * BLOCKSIZE;

    if (n > fd->nblocks)  Fatal("Removing too many blocks");
    oldbase = fd->start + fd->nblocks * BLOCKSIZE;
    newbase = oldbase - shift;
    CopyDown(oldbase, memtop, newbase);
    fd->nblocks -= n;

    /* give back released memory to the operating system (wow!) */
    memtop -= shift;
    SetTeamSize(mypid, memtop);

    for (f = fdtable; f < fdtable + nfiles; f++)
	if (f->start >= oldbase) f->start -= shift;

    /* No string table shift, because fdfile never shrinks */
  }


/* Remove all excess allocated blocks from a file */
CloseClearing(fd)
  register Filedesc *fd;
  {
    int excess;

    excess = fd->nblocks - CountBlocks(fd->size);
    if (excess > 0)  RemoveBlocksFromFile(fd, excess);
  }
