/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Internetserver routines for timeout handling.
 */

#include "net.h"
#include "internet.h"
#include "stats.h"

extern int InstanceLockHolder;
extern struct FuncBlock FuncTable[];
extern Boolean StatsEnabled;
extern ProcessId statsGid;
extern int StatsSendCounter;
extern int StatsSendInterval;
extern int StatsEnableCounter;



/*
 * TimeoutTimer:
 * Delays until the next timeout or until awoken; which is at least
 * once a second.  Checks to see if a 
 * timeout has occurred and send a msg to the appropriate process.
 */

TimeoutTimer()
  {
    Message msg;
    IoRequest *rqMsg = (IoRequest *) msg;
    int i, t;
    int secs, clicks;

    Time0 = GetTime(&clicks);
    CurrentTime = clicks;
    NextTimeout = CurrentTime + 100;

    while (True)
      {
        t = NextTimeout - CurrentTime;
	Delay(0, t);
	secs = GetTime(&clicks) - Time0;
	CurrentTime = CTime(secs, clicks);
	NextTimeout = CurrentTime + 100;
				/* Update the current time at least once
				   every second so that Ip reassembly
				   timeouts will have a reasonable time
				   against which to compare. */

	LOCKID(InstanceLock,InstanceLockHolder,15);
	i = NetInstHead;
	while (i != -1)
	  {
	    t = FuncTable[NetInstTable[i].prot].NextTimeout(
		    NetInstTable[i].tcbId);
			    /* t contains the earliest timeout for this
			       connection. */
	    if (t <= CurrentTime)
	      {
		rqMsg->requestcode = NetTimeout;
		msg[1] = ConnTimeout;
		UNLOCK(InstanceLock,InstanceLockHolder);
		Send(msg, NetInstTable[i].pid);
		LOCKID(InstanceLock,InstanceLockHolder,16);
		i = NetInstHead;	/* restart - list may be different */
		continue;
	      }
	    t = FuncTable[NetInstTable[i].prot].NextTimeout(
		    NetInstTable[i].tcbId);
			    /* Have to do it again to get updated timeout
			       values from processing timeouts. */
	    if (t <= CurrentTime)
	      {
		if (InternetDebug)
		  {
		    printf("WARNING - Bad minimum timout value %d!\n",t);
		  }
	      }
	    else if (t < NextTimeout)
	      {
		NextTimeout = t;
	      }
	    i = NetInstTable[i].next;
	  }
	UNLOCK(InstanceLock,InstanceLockHolder);
      }
  }




/*
 * CleanupTimer:
 * At regular intervals do the following:
 *    - check that intstance owners are still alive (if not, reclaim resources)
 *    - return buffers to free memory (if there are many in excess)
 *    - send statistics internet-wide messages (if enabled)
 */

CleanupTimer()
  {
    Message msg;
    IoRequest *rqMsg = (IoRequest *) msg;
    int i;

    while (True)
      {
	Delay(0, CleanupTimerInterval);

	/* Try to give buffers back to free memory */

	if (FreeBufferCount > BufAllocMax) ReturnBuf();

	LOCKID(InstanceLock,InstanceLockHolder,17);
	i = NetInstHead;
	while (i != -1)
	  {
	    if (!ValidPid(NetInstTable[i].ownerPid))
	      {
		rqMsg->requestcode = NetTimeout;
		msg[1] = InvalidConnOwner;
		UNLOCK(InstanceLock,InstanceLockHolder);
		Send(msg, NetInstTable[i].pid);
		LOCKID(InstanceLock,InstanceLockHolder,18);
		i = NetInstHead;	/* restart - list may be different */
		continue;
	      }
	    i = NetInstTable[i].next;
	  }
	UNLOCK(InstanceLock,InstanceLockHolder);

	/* Free up directory instance if reader died or did not release */

	if (NetInstTable[DirectoryInstance].inUse &&
	    !ValidPid(NetInstTable[DirectoryInstance].ownerPid))
	  {
	    if (InternetDebug > 3) printf("directory reader died!\n");
	    rqMsg->requestcode = NetTimeout;
	    rqMsg->fileid = DirectoryFileId;
	    msg[1] = InvalidConnOwner;	
	    Send(msg, NetInstTable[DirectoryInstance].pid);
	  }

        /* Send statistics if enabled and it is time */

	if (StatsEnabled)
	  {
	    if ( ++StatsSendCounter % StatsSendInterval == 0 )
	      {
		SendGlobalStats();
	      }
	    if ( --StatsEnableCounter <= 0) 
	      {
		StatsEnabled = False;
		statsGid = 0;
		if (InternetDebug > 3)
		    printf("Disabling sending of statistics\n");
	      }
	  }
      }
  }




