/************************************************************************/
/*		   (C) COPYRIGHT 1983,1984,1985,1986			*/
/*                         BOARD OF TRUSTEES				*/
/*                 LELAND STANFORD JUNIOR UNIVERSITY			*/
/*                    STANFORD, CA. 94305, U.S.A.			*/
/************************************************************************/

/*
 * Top-level routines for the TCP protocol.
 */

#include "net.h"
#include "internet.h"


/*
 * Constant definitions
 */

#define DefaultLocalPort 100	/* Default local port identifier.  This plus
				   the netinstance index is used as the local
				   port id if none is specified by the
				   client. */

/*
 * Imported routines
 */

extern char *SafeMalloc();
extern PktBuf DeQueueSafe();
extern SystemCode QueryInstance();
extern PktBuf InitTcpConnection();
extern Bit32 SelectIss();
extern SystemCode SendSynSegment();
extern SystemCode QSend();
extern Boolean NoAccess();
extern SystemCode VerifySecurityAndPrecedence();
extern Bit16 CheckSum();
extern TcpTcbPtr AllocateTcb();
extern SystemCode SendIp();
extern PktBuf DeQueue();
extern PktBuf DeQueueMiddle();



/*
 * InitTcp:
 * Global Tcp initialization routine.
 * Dummy routine in this implementation scheme.
 */

InitTcp()
  {
  }




/*
 * CreateTcpConnection:
 * Called from the main InternetServer process to create and initialize a
 * Tcp connection and its associated process. 
 */

SystemCode CreateTcpConnection(msg, eventPid)
    Message msg;
    ProcessId eventPid;
  {
    int TcpProcess();
    ProcessId tcpPid;
    int netId;		/* Index of a network instance. */
    TcpParms1 *ct = (TcpParms1 *)
		((CreateInstanceRequest *)msg)->unspecified;

    netId = FindWhichConnection(True, ct->localPort,
		ct->foreignPort,
		ct->foreignHost);
    if (netId != -1)
        if (ValidPid(NetInstTable[netId].ownerPid))
            return(BUSY);
	else
	    DeallocateTcb(NetInstTable[netId].tcbId, BAD_STATE);
    if (NoAccess())
	return(NO_PERMISSION);
    tcpPid = Create(2, TcpProcess, TcpStackSize);
    if (tcpPid == 0)
      {
	if (InternetDebug)
	  {
	    printf("Couldn't Create TcpProcess.\n");
	  }
        return(NO_MEMORY);
      }
    if (!Ready(tcpPid, 0))
      {
	printf("Couldn't Ready TcpProcess.\n");
	ExitInternetServer();
      }
    Forward(msg, eventPid, tcpPid);
    return(NO_REPLY);
  }



/*
 * CreateTcpInstance:
 * Processes user CreateInstance request.
 */

SystemCode CreateTcpInstance(pTcb, msg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    Message msg;
    ProcessId eventPid;
  {
    SystemCode returnCode;
    int netId = -1;		/* Index of network instance. */
    TcpParms1 *ct = (TcpParms1 *)
		((CreateInstanceRequest *)msg)->unspecified;

    switch (pTcb->state)
      {
	case Closed: 
	      {
		netId = AllocNetInst(TcpProt, eventPid, GetPid(0, LOCAL_PID),
			MaxPacketDataLength, MaxPacketDataLength, pTcb);
		if (netId == -1)
		  {
		    returnCode = NO_SERVER_RESOURCES;
		    break;
		  }

		pTcb->netId = netId;
		if (ct->localPort)
		    pTcb->localPort = ct->localPort;
		else
		    pTcb->localPort = DefaultLocalPort +
		    		(Bit16) (GetTime(0) % 1000);
		pTcb->foreignSocket.port = ct->foreignPort;
		pTcb->foreignSocket.host = ct->foreignHost;
		pTcb->originalForeignSocket = pTcb->foreignSocket;
		pTcb->active = ct->active;
		pTcb->prc = ct->precedence;
		pTcb->security = ct->security;
		pTcb->rcvId = eventPid;
		pTcb->user = User(eventPid);

		if ((returnCode = VerifySecurityAndPrecedence(pTcb)) != OK)
		  {
		    DeallocateTcb(pTcb, returnCode);
		    break;
		  }
		if (!pTcb->active)
		  {
		    pTcb->state = Listen;
		    pTcb->passiveOpen = True;
		    returnCode = OK;
		    break;
		  }
		if (pTcb->foreignSocket.port == Null16)
		  {
		    returnCode = BAD_ARGS;
		    DeallocateTcb(pTcb, BAD_ARGS);
		    break;
		  }
		returnCode = SendSynSegment(pTcb, SelectIss(), False, Null32);
		if (returnCode != OK)
		  {
		    /* Set new state to cause DeallocateTcb to clean up */
		    pTcb->state = SynSent;  
		    DeallocateTcb(pTcb, returnCode);
		  }
		break;
	      }
	case Listen: 
	      {
		if (!ct->active)
		  {
		    returnCode = BAD_ARGS;
		    break;
		  }
		if (ct->foreignPort == Null16)
		  {
		    returnCode = BAD_ARGS;
		    break;
		  }

		pTcb->foreignSocket.port = ct->foreignPort;
		pTcb->foreignSocket.host = ct->foreignHost;
		pTcb->originalForeignSocket = pTcb->foreignSocket;
		pTcb->active = ct->active;
		pTcb->prc = ct->precedence;
		pTcb->security = ct->security;
		pTcb->rcvId = eventPid;
	        /* No options being treated as yet. */

		returnCode = SendSynSegment(pTcb, SelectIss(), False, Null32);
		break;
	      }
	default: 
	      {
		returnCode = BUSY;
		break;
	      }
      }
    if (returnCode == OK)
	returnCode = NO_REPLY;
    return(returnCode);
  }




/*
 * ReleaseTcpInstance:
 * Processes both the regular close and also the abort form of 
 * releasing a TCP connection.
 * Connection is "TCP" closed on Write instance close.  Read instance close
 * is a no-op.
 */

SystemCode ReleaseTcpInstance(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    SystemCode returnCode, CloseCall(), AbortCall();
    int indx = rqMsg->fileid >> 1;

    if (rqMsg->fileid & 1)
	NetInstTable[indx].inUse &= ~ READ_INST;
    else
	NetInstTable[indx].inUse &= ~ WRITE_INST;

    if ((pTcb->instState & TCP_CONN_RELEASED) || ((rqMsg->fileid & 1)))
	return(OK);		/* Do nothing if
				   connection already released or read
				   instance being released. */
    pTcb->instState |= TCP_CONN_RELEASED;

    if (rqMsg->releasemode == REL_STANDARD)
	returnCode = CloseCall(eventPid, pTcb);
    else
	returnCode = AbortCall(pTcb, ABORTED);
    return(returnCode);
  }




/*
 * WriteTcpInstance:
 * Processes user Write request.
 */

SystemCode WriteTcpInstance(pTcb, rqMsg, eventPid, packet, pktSize)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    IoRequest *rqMsg;
    ProcessId eventPid;
    PktBuf *packet;		/* Passed in as a parm. so that a buffer can
				   be preallocated for the next
				   ReceiveWithSegment. */
    int pktSize;		/* Number of bytes already available through
				   the ReceiveWithSegment. */
  {
    SystemCode returnCode;
    register int indx = rqMsg->fileid >> 1;

    if (rqMsg->fileid & 1)
      {
	return(NOT_WRITEABLE);
      }
    if (rqMsg->bytecount > NetInstTable[indx].wblocksize)
      {
	return(BAD_BYTE_COUNT);
      }
    if (rqMsg->blocknumber == (NetInstTable[indx].filelastblock))
				/* This is a duplicate message - ignore it. */
      {
	return(OK);
      }
    if (rqMsg->blocknumber != (NetInstTable[indx].filelastblock+1))
      {
	return(BAD_BLOCK_NO);
      }

    switch (pTcb->state)
      {
	case Listen: 
	      {
		if (pTcb->foreignSocket.port == Null16)
		  {
		    returnCode = BAD_ARGS;
		    break;
		  }
		pTcb->active = True;
		returnCode = SendSynSegment(pTcb, SelectIss(), False, Null32);
		if (returnCode == OK)
		  {
		    if (pTcb->sndUrgBit)
		      {
			pTcb->sndUp = pTcb->sndNxt + rqMsg->bytecount;
		      }
		    returnCode = QSend(pTcb, eventPid, False, rqMsg, 
		    		packet, pktSize);
		  }
		break;
	      }
	case SynSent: 
	case SynRcvd: 
	case Estab: 
	case CloseWait: 
	      {
		if (pTcb->sndUrgBit)
		  {
		    pTcb->sndUp = pTcb->sndNxt + rqMsg->bytecount;
		  }
		returnCode = QSend(pTcb, eventPid, False, rqMsg,
				packet, pktSize);
		break;
	      }
	default: 
	      {
		returnCode = BAD_STATE;
		break;
	      }
      }

    if (returnCode == OK)
      {
	NetInstTable[indx].filelastblock++;
	NetInstTable[indx].filelastbytes = rqMsg->bytecount;
      }
    return(returnCode);
  }




/*
 * ReadTcpInstance:
 * Processes user Read request.
 */

SystemCode ReadTcpInstance(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    register int indx = rqMsg->fileid >> 1;

    if (!(rqMsg->fileid & 1))
      {
	return(NOT_READABLE);
      }
    if (rqMsg->bytecount > NetInstTable[indx].rblocksize)
      {
	return(BAD_BYTE_COUNT);
      }
    if ((rqMsg->blocknumber == (NetInstTable[indx].filenextblock - 1)) &&
    	(pTcb->lastReadBuf != NULL))
      {
	ReplyToRead(OK, eventPid, pTcb->lastReadBuf, rqMsg->bufferptr,
		Min(rqMsg->bytecount, pTcb->lastReadBuf->length));
	return(NO_REPLY);
      }
    if (rqMsg->blocknumber != NetInstTable[indx].filenextblock)
      {
	return(BAD_BLOCK_NO);
      }

    pTcb->rcvId = eventPid;
    pTcb->rcvQ = rqMsg->bufferptr;
    pTcb->rcvByteCnt = rqMsg->bytecount;

    switch (pTcb->state)
      {
	case Listen: 
	case SynSent: 
	case SynRcvd: 
	    break;		/* Read only handled when state is
				   Estab or further; so wait until then. 
				*/
	case Estab: 
	case FinWait1: 
	case FinWait2: 
	      {
		if (pTcb->segQBytesAvail > 0)
		    ServiceRcvRequest(pTcb);
		break;
	      }
	case CloseWait:
	case TimeWait:
	case Closing:
	case LastAck:
	      {
		if (pTcb->segQBytesAvail == 0)
		  {
		    ReplyToRead(END_OF_FILE, pTcb->rcvId, NULL, NULL, 0);
		    pTcb->rcvQ = NULL;
		  }
		else
		    ServiceRcvRequest(pTcb);
		break;
	      }
	case Closed:
	      {
		ReplyToRead(END_OF_FILE, pTcb->rcvId, NULL, NULL, 0);
		pTcb->rcvQ = NULL;
		break;
	      }
	default: 
	      {
		ReplyToRead(BAD_STATE, pTcb->rcvId, NULL, NULL, 0);
		pTcb->rcvQ = NULL;
		break;
	      }
      }
    return(NO_REPLY);
  }



/*
 * QueryTcpInstance:
 * QueryInstance for a Tcp connection.  Does a network query instance.
 */

SystemCode QueryTcpInstance(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;
    QueryInstanceRequest *rqMsg;
    ProcessId eventPid;
  {
    SystemCode replyCode;

    replyCode = QueryInstance(rqMsg);
    return(replyCode);
  }




/*
 * SetTcpInstanceOwner:
 * Sets the owner of a Tcp connection.
 */

SystemCode SetTcpInstanceOwner(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    register int indx = rqMsg->fileid >> 1;

    NetInstTable[indx].ownerPid = rqMsg->instanceowner;
    return(OK);
  }



/*
 * NQueryTcpFile:
 * Not implemented for TCP protocol.
 */

SystemCode NQueryTcpFile( msg, pid )
    Message msg;
    ProcessId pid;
  {
    return( REQUEST_NOT_SUPPORTED );
  }



/*
 * QueryTcpFile:
 * Returns state information about the specified TCP connection.
 */

SystemCode QueryTcpFile(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    QueryFileRequest *rqMsg;
    ProcessId eventPid;
  {
    TcpParms1 *qt1 = (TcpParms1 *) rqMsg->unspecified;
    TcpParms2 *qt2 = (TcpParms2 *) rqMsg->unspecified;

    switch (rqMsg->whichParms)
      {
	case 0:
	    qt1->localPort = pTcb->localPort;
	    qt1->foreignPort = pTcb->foreignSocket.port;
	    qt1->foreignHost = pTcb->foreignSocket.host;
	    qt1->active = pTcb->active;
	    qt1->precedence = pTcb->prc;
	    qt1->security = pTcb->security;
	    return(OK);
	case 1:
	    qt2->rcvUrgFlag = pTcb->rcvUrgentFlag;
	    qt2->sndUrgFlag = pTcb->sndUrgBit;
	    qt2->bytesAvail = pTcb->segQBytesAvail;
	    qt2->state = pTcb->state;
	    return(OK);
	case 2:
	    pTcb->wtSignalId = eventPid;
	    pTcb->waitSignal = True;
	    return(NO_REPLY);
	default:
	    return(BAD_ARGS);
      }
  }




/*
 * ModifyTcpFile:
 * Processes requests to modify or query the state parameters of
 * the designated TCP connection.
 */

SystemCode ModifyTcpFile(pTcb, rqMsg, eventPid)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    ModifyFileRequest *rqMsg;
    ProcessId eventPid;
  {
    switch (rqMsg->whichParm)
      {
	case ModTcpUrgFlag:
	    pTcb->sndUrgBit = ! pTcb->sndUrgBit;
	    break;
	default:
	    return(BAD_ARGS);
      }
    return(OK);
  }




/*
 * RcvTcp:
 * Processes incoming TCP segments handed up from the IP level.
 */

RcvTcp(pTcb, packet)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    PktBuf packet;
  {
    extern Boolean SegmentArrives();
    Bit32 ipsrc, ipdst;
    CurPktRec CurrentPacket;	/* Holds information about current 
				   incoming TCP segment. */

    CurPktPtr curPkt = &CurrentPacket;
    TcpPktPtr p;

    ipsrc = (Bit32) packet->unspecified[0];
    ipdst = (Bit32) packet->unspecified[1];
    p = TcpPkt(packet);

    if (packet->length > MaxPacketDataLength + StandardByteDataOffset)
      {
	if (InternetDebug)
	  {
	    printf("Tossing TCP packet from %s: too long (%d bytes)\n",
	        inet_ntoa(ipsrc), packet->length);
	  }
	FreeBuf(packet);
	return;
      }

    curPkt->foreignPort = p->hdr.sourcePort;
    curPkt->foreignHost = ipsrc;
    curPkt->segAdr = packet;
    curPkt->segLen = packet->length - (int) (TcpPkt(curPkt->segAdr) -> 
						hdr.dataOffset * 4);
    curPkt->segPrc = 0;			/* Currently not set. */
    curPkt->segSecurity = 0;		/* Currently not set. */
    curPkt->segDataOffset = TcpPkt(curPkt->segAdr) -> hdr.dataOffset * 4;

    if (InternetDebug > 7)
      {
	printf("RcvTcp: processing arriving segment\n");
	printf("  fHost : %s\n",inet_ntoa(ipsrc));
	printf("  segLen: %20d     segDataOffset %20d\n",
	    curPkt->segLen, curPkt->segDataOffset);
      }

    pTcb->segmentsRcvd++;
    pTcb->bytesRcvd += DLength(packet);

    /* Note: segment arrival may cause saved segments to be able to
       arrive also - if so, they are returned by SegmentArrives in curPkt. */
    while (SegmentArrives(pTcb, curPkt))
      {
	if (InternetDebug > 5)
	  {
	    printf("RcvTcp: processing saved segment\n");
	    printf("  fHost : %s\n",inet_ntoa(curPkt->foreignHost));
	    printf("  segSeq: %8x  segLen: %14d   segDataOffset %14d\n",
		TcpPkt(curPkt->segAdr)->hdr.sequenceNumber,
		curPkt->segLen, curPkt->segDataOffset);
	    printf("  rcvNxt: %8x  rcvWnd: %8x\n", pTcb->rcvNxt, pTcb->rcvWnd);
	  }
      }

    /* If the arrival(s) allows bytes to be delivered to host, do it */
    if (pTcb->rcvQ != Null && pTcb->segQBytesAvail > 0)
	ServiceRcvRequest(pTcb);
  }




/*
 * NextTcpTimeout:
 * Returns the time of the next timeout for the specified connection.
 */

int NextTcpTimeout(pTcb)
    TcpTcbPtr pTcb;
  {
    int i, min;

    min = pTcb->timers[0];
    for (i = 1; i < NumTcpTimeouts; i++)
	if (min > pTcb->timers[i])
	    min = pTcb->timers[i];
    return(min);
  }




/*
 * TcpTimeout:
 * Handles TCP timeout events.
 */

TcpTimeout(pTcb, msg, eventPid)
    TcpTcbPtr     pTcb;
    Message msg;
    ProcessId eventPid;
  {
    PktBuf packet;
    TcpPktPtr p;
    IoRequest *rqMsg = (IoRequest *) msg;
    int t, clicks;
    register int timeoutsSoFar;
    register int newTimeout;

    if (msg[1] == InvalidConnOwner)
      {
        rqMsg->fileid = pTcb->netId << 1;
	rqMsg->releasemode = REL_ABORT;
	ReleaseTcpInstance(pTcb, rqMsg, eventPid);
	rqMsg->fileid += 1;
	ReleaseTcpInstance(pTcb, rqMsg, eventPid);
	return;
      }
    else if (pTcb->instState & TCP_CONN_CLOSED)
				/* Connection already closed. */
      {
        return;
      }
    else
      {
	if (pTcb->timers[AckTimeout] <= CurrentTime) 
	      {
		SendControlSegment(pTcb, pTcb->localPort,
				pTcb->foreignSocket.port,
				pTcb->foreignSocket.host, pTcb->sndNxt,
				True, pTcb->rcvNxt, False, False);
		StopTimer(pTcb, AckTimeout);
	      }
	if (pTcb->timers[RetransTimeout] <= CurrentTime)
	  {
	    if (Empty(pTcb->q[RetransQueue]))
	      {
		StopTimer(pTcb, RetransTimeout);
	      }
	    else
	      {
	        pTcb->numRetransTimeouts++;
		CompressRetransQueue(pTcb);
		packet = pTcb->q[RetransQueue].head;
		timeoutsSoFar = packet->unspecified[0];
		t = GetTime(&clicks) - Time0;
		t = CTime(t, clicks);

		if (timeoutsSoFar == 0)
		  {
		    /* Reset packet send time on first retransmission */
		    packet->unspecified[1] = t; 
#ifdef TIMERDEBUG
		    if (TimerDebug)
			printf("   new sent time: %8x\n",t);
#endif TIMERDEBUG
		  }
		if ((t - packet->unspecified[1] > MaxRetransTimeoutClicks) ||
		    (TcpPkt(packet)->hdr.syn && 
		        (timeoutsSoFar >= MaxSynRetransTimeouts)))
		  {
		    if (InternetDebug)
		      {
		        printf("TcpTimeout: Max[Syn]RetransTimeout expired.\n");
			printf("t=%u; pkt->unspec[0]=%u; pkt->unspec[1]=%u\n",
			    t,packet->unspecified[0],packet->unspecified[1]);
			Flush(stdout);
		      }
		    DeallocateTcb(pTcb, TIMEOUT);
		    return;	/* Return to avoid processing other timeouts
				   after connection is already closed. */
		  }
		else
		  {
		    TcpPkt(packet)->hdr.acknowledgementNumber = pTcb->rcvNxt;
		    SendIp(pTcb, packet, pTcb->foreignSocket.host);

		    /* Get backoff multipliers from table */
		    newTimeout = (int) (pTcb->lenRetransTimeout * 
			TcpRetransBackoff[Min(timeoutsSoFar, 
					 (MaxRetransTimeoutIndex - 1))]);
		    StartTimer(pTcb, RetransTimeout, newTimeout);
		  }
		packet->unspecified[0] += 1;

		if (InternetDebug>0)
		  {
		    printf("[%d]XXXXX Retransmission # %d (%d) XXXXX\n",
			pTcb->netId, timeoutsSoFar + 1, newTimeout);
		  }
	      }
	  }
	if (pTcb->timers[TmeWaitTimeout] <= CurrentTime) 
	  {
	    if (InternetDebug)
	      {
		printf("[%d]TcpTimeout: TmeWaitTimeout expired!\n",
		    pTcb->netId);
	      }
	    DeallocateTcb(pTcb, TIMEOUT);
	    return;		/* Return immediately so that if any timeout
			       processing is ever done after this one it
			       will not be done on a closed connection. */
	  }
	if (pTcb->timers[SndWndTimeout] <= CurrentTime)
	  {
	    if (InternetDebug)
	      {
		printf("[%d] --- SndWndTimeout occurred ---\n",
			pTcb->netId);
	      }
	    if (!Empty(pTcb->q[SendQueue]))
	      {
		packet = pTcb->q[SendQueue].head;
		p = TcpPkt(packet);
		if (pTcb->sndUp != NULL)
		  {
		    p->hdr.urgentPointer = pTcb->sndUp - 
		    			p->hdr.sequenceNumber;
		    p->hdr.urg = True;
		    if (IN(p->hdr.sequenceNumber, pTcb->sndUp,
		    		p->hdr.sequenceNumber + DLength(packet)))
		      {
		        pTcb->sndUp = NULL;
		      }
		  }
		else
		    p->hdr.urg = False;
		p->hdr.acknowledgementNumber = pTcb->rcvNxt;
		SendIp(pTcb, packet, pTcb->foreignSocket.host);
	      }
	    StartTimer(pTcb, SndWndTimeout, LenSndWndTimeout);
	  }
      }
  }




/*
 * CompressRetransQueue - combine many small packets in the retransmit
 *    queue into a few larger ones.
 */

CompressRetransQueue(pTcb)
    TcpTcbPtr pTcb;
  {
    PktBuf p, q, q2;
    TcpPktPtr tp, tq;
    register char *ptr1;
    register char *ptr2;
    register char *maxptr;

    p = pTcb->q[RetransQueue].head;
    while ((q = p->next) != NULL)
      {
	if (p->length + q->length > 
	    NetInstTable[pTcb->netId].wblocksize + sizeof(struct TcpHeader))
	  {
	    if (InternetDebug>5)
	      {
	        printf("Cannot compress due to blocks too big\n");
		printf("pkt1 length = %d, pkt2 length = %d\n",
			p->length, q->length);
	      }
	    break;
	  }
	tp = TcpPkt(p);
	tq = TcpPkt(q);
	if (tp->hdr.urg || tp->hdr.rst || tp->hdr.syn || tp->hdr.fin ||
	    tq->hdr.urg || tq->hdr.rst || tq->hdr.syn || tq->hdr.fin)
	  {
	    if (InternetDebug>5)
	        printf("Cannot compress due to flags\n");
	    break;
	  }

	if (InternetDebug > 5) printf("compressing retran queue\n");
	tp->hdr.window = tq->hdr.window;
	Copy((char *)(p->dataptr + p->length),
	     (char *)(q->dataptr + sizeof(struct TcpHeader)),
	     q->length - sizeof(struct TcpHeader));
	p->length += q->length - sizeof(struct TcpHeader);
	q2 = DeQueueMiddle(&(pTcb->q[RetransQueue]), p);
	FreeBuf(q2);
      }
  }


/*
 * PrintTcpPktInfo - print info in the tcp header.
 */

PrintTcpPktInfo(p)
    PktBuf p;
  {
    TcpPktPtr tp;
    char *ptr;

    printf("length = %d   ", p->length);
    tp = TcpPkt(p);
    printf("src port %x;  dst port %x\n", 
	tp->hdr.sourcePort, tp->hdr.destinationPort);
    printf("sequence # %d     ack # %d\n",
	tp->hdr.sequenceNumber, tp->hdr.acknowledgementNumber);
    printf("Bit fields: ");
    if (tp->hdr.urg) printf("urg ");
    if (tp->hdr.ack) printf("ack ");
    if (tp->hdr.psh) printf("psh ");
    if (tp->hdr.rst) printf("rst ");
    if (tp->hdr.syn) printf("syn ");
    if (tp->hdr.fin) printf("fin ");
    printf("\nData: ");
    for (ptr = p->dataptr + sizeof(struct TcpHeader);
	 ptr < p->dataptr + p->length;
	 ptr++)
	putchar(*ptr);
    printf("\n");
  }




/*
 * CloseCall:
 * Processes normal closing of a TCP connection.
 */

SystemCode CloseCall(eventPid, pTcb)
    ProcessId eventPid;
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
  {
    SystemCode returnCode;

    switch (pTcb->state)
      {
	case Listen: 
	case SynSent: 
	      {
		DeallocateTcb(pTcb, ABORTED);
		returnCode = OK;
		break;
	      }
	case SynRcvd: 
	      {
		if ((Empty(pTcb->q[SendQueue])) &&
			(Empty(pTcb->q[SegQueue])) &&
			(Empty(pTcb->q[RetransQueue])))
				/* No sends issued and no pending data. */
		    pTcb->state = FinWait1;
		returnCode = QSend(pTcb, eventPid, True, NULL, NULL, 0);
				/* Won't actually get sent until state
				   Estab. */
		break;
	      }
	case Estab: 
	      {
		pTcb->state = FinWait1;
		returnCode = QSend(pTcb, eventPid, True, NULL, NULL, 0);
		break;
	      }
	case FinWait1: 
	case FinWait2: 
	      {
		returnCode = OK;
		break;
	      }
	case CloseWait: 
	      {
		pTcb->state = LastAck;
		returnCode = QSend(pTcb, eventPid, True, NULL, NULL, 0);
		break;
	      }
	default: 
	      {
		returnCode = BAD_STATE;
		break;
	      }
      }
    return(returnCode);
  }




/*
 * AbortCall:
 * Processes abort form of closing a TCP connection.
 */

SystemCode AbortCall(pTcb, status)
    TcpTcbPtr pTcb;		/* Ptr to tcb of the current connection. */
    SystemCode status;
  {
    switch (pTcb->state)
      {
	case SynRcvd: 
	case Estab: 
	case FinWait1: 
	case FinWait2: 
	case CloseWait: 
	      {
		SendControlSegment(pTcb, pTcb->localPort,
			pTcb->foreignSocket.port,
			pTcb->foreignSocket.host,
			pTcb->sndNxt, False, Null32, True, False);
		DeallocateTcb(pTcb, status);
		break;
	      }
	default: 
	      {
		DeallocateTcb(pTcb, status);
		break;
	      }
      }
    return(OK);
  }




/*
 * TcpProcess:
 * Tcp connection process.  Handles all actions for this connection after
 * its initial creation.
 */

TcpProcess()
  {
    struct TcbType tcb;
    TcpTcbPtr pTcb = &tcb;
    Message  msg;
    IoRequest *rqMsg = (IoRequest *)msg;
    IoReply *repMsg = (IoReply *)msg;
    ProcessId eventPid;
    PktBuf packet, writePkt;
    int writePktSize;

    writePkt = InitTcpConnection(pTcb);
				/* Init returns a preallocated packet buffer
				   for use with ReceiveWithSegment. */

    while (True)
      {
        while ((packet = DeQueueSafe(&(pTcb->readerQueue))) != NULL)
	  {
	    if (!(pTcb->instState & TCP_CONN_CLOSED))
	      {
				/* Handle packets only if connection is not
				   yet closed. */
		if (packet->unspecified[2] == TCPprot)
		    RcvTcp(pTcb, packet);
		else if (packet->unspecified[2] == ICMPprot)
		    TcpRcvIcmp(pTcb, packet);
		else if (InternetDebug)
		  {
		    printf("TcpProcess: unknown protocol %d ?\n",
			packet->unspecified[2]);
		    FreeBuf(packet);
		  }
	      }
	    else
		FreeBuf(packet);
	  }

	if ((pTcb->instState & TCP_CONN_CLOSED) &&
		(NetInstTable[pTcb->netId].inUse == 0))
	  {
	    FreeBuf(writePkt);
	    DeallocNetInst(pTcb->netId);
	    Destroy(0);		/* Commit suicide. */
	  }

	writePktSize = MaxPacketDataLength;
	pTcb->receiveBlocked = True;
	eventPid = ReceiveWithSegment(msg, writePkt->dataptr, &writePktSize);
	pTcb->receiveBlocked = False;
	    switch (rqMsg->requestcode)
	      {
	        case CREATE_INSTANCE: 
				/* This is either the initial request
				   forwarded to the newly created connection
				   process or a request to change the 
				   connection state from passive to active. */
		    repMsg->replycode = 
			CreateTcpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case WRITE_INSTANCE:
	        case WRITESHORT_INSTANCE:
		    repMsg->replycode = 
			WriteTcpInstance(pTcb, rqMsg, eventPid, 
				&writePkt, writePktSize);
		    break;
	        case READ_INSTANCE:
		    repMsg->replycode = 
			ReadTcpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case RELEASE_INSTANCE:
		    repMsg->replycode = 
			ReleaseTcpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case QUERY_FILE:
		    repMsg->replycode = 
			QueryTcpFile(pTcb, rqMsg, eventPid);
		    break;
	        case MODIFY_FILE:
		    repMsg->replycode = 
			ModifyTcpFile(pTcb, rqMsg, eventPid);
		    break;
	        case QUERY_INSTANCE:
		    repMsg->replycode = 
		    	QueryTcpInstance(pTcb, rqMsg, eventPid);
		    break;
		case SET_INSTANCE_OWNER:
		    repMsg->replycode = 
		        SetTcpInstanceOwner(pTcb, rqMsg, eventPid);
		    break;
	        case LocalNetPktRcvd:
		    repMsg->replycode = OK;
		    break;
	        case NetTimeout:
		    TcpTimeout(pTcb, rqMsg, eventPid);
		    repMsg->replycode = OK;
		    break;
		default:
		    repMsg->replycode = REQUEST_NOT_SUPPORTED;
		    break;
	      }
	if (repMsg->replycode != NO_REPLY)
	  {
	    eventPid = Reply(msg, eventPid);
	  }
      }
  }




/*
 * PrintTcpConnectionState:
 * Diagnostic routine that prints out information regarding the current
 * state of the specified Tcp connection.
 */

PrintTcpConnectionState(f,pTcb)
    File *f;
    TcpTcbPtr pTcb;
  {
    int i, n;
    PktBuf pkt;

    fprintf(f,"Instance %d: TCP to [%s]  ", pTcb->netId, 
	inet_ntoa(pTcb->foreignSocket.host));
    fprintf(f,"state: %d  release state: %d", pTcb->state, pTcb->instState);
    if (pTcb->receiveBlocked)
        fprintf(f,"  [receiveBlocked]");
    fprintf(f,"\n");
    if (pTcb->readerQueue.head->next->pkt)
        fprintf(f,"Non-empty readerQueue.\n");
    if (pTcb->rcvId)
        fprintf(f,"Process %x is waiting on a Read.  ", pTcb->rcvId);
    fprintf(f,"# bytes avail. for user read: %d\n", 
    	    pTcb->segQBytesAvail);
    if (!Empty(pTcb->q[SendQueue]))
      {
        n = 0;
	for (pkt = pTcb->q[SendQueue].head; pkt != NULL; pkt = pkt->next)
	    n++;
	fprintf(f,"%d packets in the SendQueue.\n", n);
      }
    if (!Empty(pTcb->q[RetransQueue]))
      {
        n = 0;
	for (pkt = pTcb->q[RetransQueue].head; pkt != NULL; pkt = pkt->next)
	    n++;
	fprintf(f,"%d packets in the RetransQueue.\n", n);
      }
    if (!Empty(pTcb->q[SegQueue]))
      {
        n = 0;
	for (pkt = pTcb->q[SegQueue].head; pkt != NULL; pkt = pkt->next)
	    n++;
	fprintf(f,"%d packets in the SegQueue.\n", n);
      }
    if (!Empty(pTcb->q[SaveQueue]))
      {
        n = 0;
	for (pkt = pTcb->q[SaveQueue].head; pkt != NULL; pkt = pkt->next)
	    n++;
	fprintf(f,"%d packets in the SaveQueue.\n", n);
      }
    fprintf(f,"CurrentTime: %d  Timeouts: ", CurrentTime);
    for (i = 0; i < NumTcpTimeouts; i++)
      {
	fprintf(f,"%d  ", pTcb->timers[i]);
      }
    fprintf(f,"\n");
    fprintf(f,"  iss:%10x  sndUna:%10x  sndNxt:%10x  sndWnd   :%10x\n",
    	    pTcb->iss, pTcb->sndUna, pTcb->sndNxt, pTcb->sndWnd);
    fprintf(f,"  irs:%10x  rcvNxt:%10x  rcvWnd:%10x  delRcvWnd:%10x\n",
    	    pTcb->irs, pTcb->rcvNxt, pTcb->rcvWnd, pTcb->delRcvWnd);
    fprintf(f,"# clicks to ack pkt: ");
    for (n = 0, i = pTcb->nextRetransTime; n < NumRetransTimes; i++, n++)
      {
	if (i == NumRetransTimes) i = 0;
        fprintf(f,"%d ", pTcb->retransTimes[i]);
      }
    fprintf(f,"  Avg clicks/retrans: %d\n",pTcb->lenRetransTimeout);
    fprintf(f,"# out of order packets: %d  ", pTcb->numOutOfOrderPkts);
    fprintf(f,"  # retransmissions: %d\n", pTcb->numRetransTimeouts);
  }

