/********************************************************/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/


/* execserver.c - a server to create executives on request
 *
 * Pulled out of the view manager and converted into a server by
 * Kenneth Brooks, November 1983.
 */

#include <Venviron.h>
#include <Vioprotocol.h>
#include <Vteams.h>		/* for Root Message and kill program */
#include <Vprocess.h>
#include <Vexec.h>
#include "sh.h"
#include <Vauthenticate.h>
#include <Vtermagent.h>

#define MAX_EXECS 16
#define MAX_PATH_LENGTH 256
/* for debugging */
/* #define DEBUG -- gets a load of printed debug messages */

extern Exec();			/* imported from exec.c */
extern struct Buffer *CreateBuffer();
static SystemCode CreateExec(), DeleteExec(), KillProgram(), QueryExec();
static SystemCode StartExec(), SetupFile();

static SystemCode AuthExec(), DeAuthExec();
static SystemCode HandleGetRawIO();

static Scavenge(), ExecProcess();

extern AliasRec aliases[];
extern ProcessId execPid;

struct ExecDescriptor *exectable[MAX_EXECS] = {NULL};

Message msg;

extern int SeparateExecProgram;

ExecServer(argc, argv)
  int argc;
  char *argv[];
  {
    register ExecRequest *req = (ExecRequest *) msg;
    register ExecReply *reply = (ExecReply *) msg;
    register ProcessId sender;
    register SystemCode r;

    SeparateExecProgram = 0;
    SetUserNumber(0 , UNKNOWN_USER);
    execPid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
    SetPid(EXEC_SERVER, execPid, LOCAL_PID);
    InitAliases(aliases);

    for (;;)
      {
	sender = Receive(req);
	Scavenge();	/* update our info about execs that have died */
	switch(req->requestcode)
	  {
	  case CREATE_EXEC:
	    r = CreateExec(req);
	    break;
	  case START_EXEC:
	    r = StartExec(req->execid);
	    break;
	  case DELETE_EXEC:
	    r = DeleteExec(req->execid);
	    break;
	  case KILL_PROGRAM:
	    r = KillProgram(req->execid);
	    break;
	  case QUERY_EXEC:
	    r = QueryExec(req);
	    break;
	  case CHECK_EXECS:
	    CheckExecs();
	    r = OK;
	    break;
	  case AUTH_EXEC:
	    r = AuthExec( sender, req );
	    break;
	  case DEAUTH_EXEC:
	    r = DeAuthExec( sender, req );
	    break;
	  case GetRawIO:
	    r = HandleGetRawIO( sender, req );
	    break;
	  default:
	    r = ILLEGAL_REQUEST;
	  }
	
	if (r != NO_REPLY)
	  {
	    reply->replycode = r;
	    Reply(reply, sender);
	  }
      }  /* end for(;;) */
  }


/* Scavenge: called whenever we receive a message.  Checks to see if any
 * of our exec processes have died, and deletes the descriptors of any that
 * have.  This guarantees that (1) we don't waste descriptors, (2) QueryExec
 * returns the truth.
 */
static Scavenge()
  {
    register int i;
    register struct ExecDescriptor *exec;

    for (i=0; i<MAX_EXECS; i++)
      {
	exec = exectable[i];
	if (exec && !ValidPid(exec->execPid))  DeleteExec(i);
      }
  }

static SystemCode AuthExec( pid, req )
ProcessId pid;
MsgStruct *req;
  {
    /* Authenticate the exec server and the requesting exec to the username
     * specified in the segment.
     */
    char buf[100];
    register char *s;
    SystemCode r;

    if (req->segmentSize > 100) return( BAD_BUFFER );
    r = MoveFrom(pid, buf, req->segmentPtr, req->segmentSize);
    if (r != OK) return( r );
    
    s = buf;
    while (*s && (*s != ':')) s++;
    if (*s == 0) return( BAD_ARGS );
    *s++ = 0;
    /* Set our user number, then set the user number of the requesting exec */
    if ( (r = Authenticate(buf, s) ) == OK )
	SetUserNumber(pid, User( 0 ));
    return( r );
  }

static SystemCode DeAuthExec( pid, req )
ProcessId pid;
MsgStruct *req;
  {
    /* Destroy all executives authenticated to the user specified.
     *  If the "all" flag was set in the request, destroy all
     *  execs.  If no user was specified, use the uid of the requestor.
     */
    char buf[100];
    register unsigned int i;
    AuthRec ar;
    SystemCode r;
    UID uid;
    register struct ExecDescriptor *exec;
    register unsigned long all;

    ar.name = NULL;
    ar.fullname = NULL;
    ar.passwd = NULL;
    ar.home = NULL;
    
    all = (req->unspecified[0] == 0);
    if ( !all )
      {
	if (req->segmentSize > 100) return( BAD_BUFFER );
        r = MoveFrom( pid, buf, req->segmentPtr, req->segmentSize );
	if (r != OK) return( r );

	if (strlen(buf) == 0)
	  {
	    uid = User(pid);
	  }
	else
	  {
	    if ( ( r = MapUserName(buf, &ar)) != OK ) return ( r );
	    uid = ar.uid;
	  }
      }

    for (i = 0; i < MAX_EXECS; i++)
      {
	exec = exectable[i];
        if ( (exec != NULL) && ( ( all ) || (User(exec->execPid) == uid) ) )
            DeleteExec( i );
      }
    if ( ( all ) || ( User( 0 ) == uid ) )
      {
	SetUserNumber( 0 , UNKNOWN_USER );
	RemoveAlias(NULL, aliases);	/* remove all aliases */
	clearenv();			/* remove all environment vars */
	ClearLocalNames();		/* remove all local names */
	PrimeCache();			/* reload standard local names */
	UpdateHostStatus("", 1);	/* tell teamserver no one logged in */
      }

    DestroyAuthRec( &ar );

    return( OK );
  }


static SystemCode HandleGetRawIO( pid, req )
ProcessId pid;
RawIOReply *req;
  {
    req->inserver = FileServer(stdin);
    req->infile = FileId(stdin);
    req->outserver = FileServer(stdout);
    req->outfile = FileId(stdout);
    
    return OK;
  }

static SystemCode KillProgram(execid)
  int execid;
  {
    register struct ExecDescriptor *exec = exectable[execid];
#ifdef DEBUG
printf("kill program in %d\n", execid);
#endif

    if (execid > MAX_EXECS || !exec) return(NOT_FOUND);
    else if (exec->status == EXEC_RUNNING)
      {
        return(DestroyProcess(exec->programPid));
      }
    else return(NONEXISTENT_PROCESS);
  }


static SystemCode DeleteExec(execid)
  int execid;
  {
    register struct ExecDescriptor *exec = exectable[execid];
#ifdef DEBUG
printf("delete exec %d\n", execid);
#endif

    if (execid > MAX_EXECS || !exec) return(NOT_FOUND);

    DestroyProcess(exec->execPid);
    if (exec->perProcess.ctxname) free(exec->perProcess.ctxname);
    free(exec);
    exectable[execid] = NULL;
    return(OK);
  }


static SystemCode QueryExec(req)
  register ExecRequest *req;
  {
    register ExecReply *reply = (ExecReply *) req;
    register struct ExecDescriptor *exec = exectable[req->execid];

    if (req->execid > MAX_EXECS || !exec) return(NOT_FOUND);
    reply->pid = exec->execPid;
    reply->program = exec->programPid;
    reply->status = exec->status;
    if (reply->status == EXEC_HOLD1) reply->status = EXEC_HOLD;
#ifdef DEBUG
printf("query exec %d -> pid %x program %x status %d\n", req->execid,
	reply->pid, reply->program, reply->status);
#endif
    return(OK);
  }


static CheckExecs()
  {
    register struct ExecDescriptor *exec;
    register int i;

#ifdef DEBUG
printf("check execs: ");
#endif
    for (i=0; i<MAX_EXECS; i++)
      {
	exec = exectable[i];
	if ( (exec != NULL)
	     && ( (!ValidIOPid(exec->in.fileserver)) 
		|| (!ValidIOPid(exec->out.fileserver)) ) )
	  {
	    DeleteExec(i);
#ifdef DEBUG
	    printf("deleted %d\n");
#endif
	  }
      }
  }

/* ValidIOPid: returns nonzero if it is a valid pid or a kernel device.
 * This stems from a kernel misfeature which does not consider kernel
 * devices to be valid pids.
 */
int ValidIOPid(pid)
  ProcessId pid;
  {
    if (ValidPid(pid)) return(1);
    else if ((pid & 0xFFFF) == 0x80) return(1);
    else return(0);
  }


/* StartExec: If we create an exec which is to own some or all of the
 * instances it does i/o to, we need to wait and let the requestor do
 * a SetInstanceOwner on them before we start the exec up  (lest it should
 * exit before the SetInstanceOwner takes place.)  So in those cases the
 * CreateExec request leaves it in an EXEC_HOLD state until StartExec starts
 * it.  All this is handled in the CreateExec library routine, and the
 * programmer should never have to see it.
 */
static SystemCode StartExec(execid)
  int execid;
  {
    register struct ExecDescriptor *exec = exectable[execid];
#ifdef DEBUG
printf("start exec %d\n", execid);
#endif

    if (execid > MAX_EXECS || !exec) return(NOT_FOUND);
    if (exec->status == EXEC_HOLD)
	Ready(exec->execPid, 4, &exec->in, &exec->out, &exec->err, exec);
    else if (exec->status == EXEC_HOLD1)
	Ready(exec->execPid, 4, &exec->in, &exec->out, &exec->out, exec);
    return(OK);
  }



/* CreateExec: we allocate an exec descriptor, set up the i/o descriptors,
 * and start an exec going.  Here you will see a lot of code which is normally
 * hidden in Create() and OpenFile().
 * The purpose is to pack EVERYTHING which is malloc'ed
 * on behalf of an exec into a single structure, the ExecDescriptor.  This
 * makes it easy to free the whole business when destroying the exec, and
 * avoids losing memory.  Among these items: the PerProcess area, the input
 * and output File descriptors, the stack space, and the i/o buffers.
 */
static SystemCode CreateExec(req)
  register ExecRequest *req;
  {
    register ExecReply *reply = (ExecReply *) req;
    register struct ExecDescriptor *e;
    register File *in, *out, *err;
    Processor_state execState;
    ProcessId pid;
    register int execid;
    int errwithout=0;
    register SystemCode r;

    for (execid=0; execid<MAX_EXECS; execid++) 
	if (exectable[execid] == NULL) break;
#ifdef DEBUG
printf("create exec picks #%d  ", execid);
#endif
    if (execid >= MAX_EXECS) return(NO_SERVER_RESOURCES);

    e = (struct ExecDescriptor *)	malloc(sizeof(struct ExecDescriptor));
    if (e==NULL) return(NO_SERVER_RESOURCES);
    exectable[execid] = e;
#ifdef DEBUG
printf("allocated %x  ", e);
#endif

    in  = &(e->in);
    out = &(e->out);
    err = &(e->err);

    if (req->stdoutserver == req->stderrserver 
	    && req->stdoutfile == req->stderrfile)
        errwithout = 1;

    in->fileserver = req->stdinserver;
    in->fileid = req->stdinfile;
    r = SetupFile(in, FREAD, e->inbuffer);
    if (r != OK) return(r);
    
    out->fileserver = req->stdoutserver;
    out->fileid = req->stdoutfile;
    r = SetupFile(out, FAPPEND, e->outbuffer);
    if (r != OK) return(r);

    if (!errwithout)
      {
	err->fileserver = req->stderrserver;
	err->fileid = req->stderrfile;
	r = SetupFile(err, FAPPEND, e->errbuffer);
	if (r != OK) return(r);
      }

    pid = CreateProcess( 8, ExecProcess, e->stack+MainSize);
    if (pid==0) return(0);
    if (ReadProcessState(pid, &execState) == 0) return (0);
    execState.perProcess = (Unspec *) &e->perProcess;
    execState.perProcessLoc = (Unspec **) &PerProcess;
    if (WriteProcessState(pid, &execState) == 0) return (0);
    e->perProcess = *PerProcess;  /* structure copy */
    /* Request should contain name, not ContextPair!
     *  For now, we inverse-map the supplied context pair
     *  and hope it is still valid, or if none is supplied,
     *  use our current definition of "[home]".   --TPM
     */
    e->perProcess.ctxname = (char *) malloc(MAX_PATH_LENGTH);
    if (req->nameserver)
      {
        e->perProcess.ctx.pid = req->nameserver;
        e->perProcess.ctx.cid = req->context;
	r = GetContextName(e->perProcess.ctxname, MAX_PATH_LENGTH,
		           e->perProcess.ctx);
      }
    else
      {
	strcpy(e->perProcess.ctxname, "[home]");
	r = GetAbsoluteName(e->perProcess.ctxname, MAX_PATH_LENGTH,
			   &e->perProcess.ctx);
      }
    if (r != OK)
      {
	/* Let exec inherit our own current context */
        strcpy(e->perProcess.ctxname, PerProcess->ctxname);
	e->perProcess.ctx = PerProcess->ctx;
      }
    e->perProcess.stackSize = sizeof(e->perProcess) + MainSize;

    e->execPid = pid;
    e->ioflags = req->flags & EXEC_IOFLAGS;

    if (req->flags&EXEC_ROOTFLAGS)/* i.e. some of the files are to be our own */
      {
	if (errwithout)  e->status = EXEC_HOLD1;
	else  e->status = EXEC_HOLD;
      }
    else
      {
	e->status = EXEC_FREE;
	if (errwithout)    Ready(pid, 4, in, out, out, e);
	else  Ready(pid, 4, in, out, err, e);
      }

#ifdef DEBUG
printf("replies %d, pid %x, status %d\n", execid, pid, e->status);
#endif
    reply->execid = execid;
    reply->pid = pid;
    reply->status = e->status;
    return(OK);
}


/* Lobotomized version of _Open that does not malloc its own fad */
static SystemCode SetupFile(fad, mode, buffer)
  register File *fad;
  int mode;
  unsigned char *buffer;
  {
    Message msg;
    register QueryInstanceRequest *req = (QueryInstanceRequest *) msg;
    register QueryInstanceReply *reply = (QueryInstanceReply *) msg;

    req->requestcode = QUERY_INSTANCE;
    req->fileid = fad->fileid;
    req->filemode = mode;
    Send(req, fad->fileserver);
    if( reply->replycode != OK ) return( reply->replycode );

    fad->blocksize = reply->blocksize;
    fad->lastblock = reply->filelastblock;
    fad->lastbytes = reply->filelastbytes;
    fad->type = reply->filetype;
    fad->readLimit = buffer;
    fad->writeLimit = buffer;
    fad->current = buffer;
    fad->buffer = buffer;
    fad->currsize = MAXUNSIGNED;
    fad->state = 0;
    fad->lastexception = 0;
    fad->block = 0;
	
    if ( (mode & FBASIC_MASK) == FAPPEND )
        Seek(fad, 0, FILE_END);

    return(OK);
  }


/* ExecProcess: exists because this process was not created with Create, and
 * its stack is not a malloc block, so it must NOT end up calling Suicide.
 */
static ExecProcess(in, out, err, descriptor)
  File *in, *out, *err;
  struct ExecDescriptor *descriptor;
  {
    Exec(in, out, err, NULL, descriptor);
    DestroyProcess(0);  /* the only safe way to exit this weirdly created
    			   process. */
  }

