/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 * Session handling code.
 * A session is a forked copy of the server, running on behalf of a 
 * particular user.
 */

#include <Venviron.h>
#include <Vioprotocol.h>
#include <pwd.h>
#include <wait.h>
#include "server.h"
#include "config.h"
#include "debug.h"
#include "kernel.h"
#include "swab.h"

#define SigMask(s) (1<<((s)-1))

/* Exports */
extern int ForkNewSession();
extern SystemCode ValidSessionPid();
extern CheckSessions();
SessionDesc *Session = NULL;
int MyUserIsGuest;	/* set if V user is UNKNOWN_USER or has no
			 *  account on this Unix host */
int MyUserId;		/* Effective (Unix) user id of session */
int MyGroupId;    	/* Effective (Unix) group id of session */

/* Imports */
extern ProcessId MyPid, LogicalHostPid, MainServerPid;
extern char *GetUserCorrespondence();
extern struct passwd *GetUnixPwnam();
extern Net;
extern SetStringName();
extern char BlockBuffer[];
extern unsigned MyVUserNumber;
extern int NetworkNum;
extern int (*NetOpen)();

/* Local */
int SessionInitialized, SessionInitFailed;
ProcessId newSessionPid;

/* Environment stuff for Process Instances */
#define MAX_ENV_VAR_LEN		256
extern char MyEnvPath[];		/* set in process.c */
char MyEnvTerm[] = "TERM=sun"; 
char MyEnvUser[ MAX_ENV_VAR_LEN + 5] = "USER=";
char MyEnvShell[ MAX_ENV_VAR_LEN + 6] = "SHELL=";
char MyEnvHome[ MAX_ENV_VAR_LEN + 5] = "HOME=";
char MyEnvPath[] = ENV_SEARCH_PATH;
char *MyEnvironment[] = 
  { 
    MyEnvHome, MyEnvPath, MyEnvShell, MyEnvTerm, MyEnvUser, 0
  };

SessionDesc *SessionList = NULL;

static SessionDesc *AllocSessionDesc()
    /* Creates a new session descriptor, linking it into the list of 
     * descriptors for already existing sessions.
     */
  {
    SessionDesc *desc = (SessionDesc *)calloc(1, sizeof(SessionDesc));
    
    if (desc != NULL)
      {
	desc->link = SessionList;
	SessionList = desc;
      }
    return (desc);
  }

static FreeSessionDesc(desc)
    SessionDesc *desc;
    /* Frees and unlinks the given session descriptor. */
  {
    register SessionDesc *ptr;

    if (SessionList == desc)
	SessionList = SessionList->link;
    else
      {
	for (ptr = SessionList; ptr != NULL; ptr = ptr->link)
	    if (ptr->link == desc)
		break;
	if (ptr == NULL)
	  {
	    if (GDebug) 
		printf("FreeSessionDesc() - descriptor not found!\n");
	    return;
	  }
	else
	    ptr->link = desc->link;
      }
    free(desc);
  }

static FreeSessionDescsExcept(desc)
    SessionDesc *desc;
    /* Frees and unlinks all session descriptors \except/ the one pointed to 
     * by "desc".
     */
  {
    register SessionDesc *ptr = SessionList;

    for (ptr = SessionList; ptr != NULL; ptr = ptr->link)
	if (ptr != desc)
	    free(ptr);
    SessionList = desc;
  }

static SessionDesc *FindSessionFromVPid(Vpid)
    ProcessId Vpid;
    /* Returns a pointer to the descriptor for the session with V process 
     * id "Vpid", or NULL if no such session exists.
     */
  {
    register SessionDesc *ptr;

    for (ptr = SessionList; ptr != NULL; ptr = ptr->link)
	if (ptr->Vpid == Vpid)
	    break;
    return (ptr);    
  }

static SessionDesc *FindSessionFromLocalPid(localpid)
    ProcessId localpid;
    /* Returns a pointer to the descriptor for the session with local process 
     * id "localpid", or NULL if no such session exists.
     */
  {
    register SessionDesc *ptr;

    for (ptr = SessionList; ptr != NULL; ptr = ptr->link)
	if (ptr->localpid == localpid)
	    break;
    return (ptr);    
  }


static SessionReady()
    /* A handler for the signal that a new session sends once it has finished
     * initializing.
     */
  {
    SessionInitialized = 1;
  }


static SessionInitFail()
    /* A handler for the signal that a new session sends if it fails to 
     * initialize (usually, fails to initialize its network interface).
     */
  {
    SessionInitFailed = 1;
  }


int ForkNewSession(vUserNumber, error)
    unsigned vUserNumber;
    SystemCode *error;
    /*
     * Fork off a new session, running as the given V user.  This involves 
     * determining the corresponding Unix uid.  The value returned is like 
     * that of fork() - 0 for the child, Unix pid for the parent, -1 if an 
     * error occurs.  Also, if an error occurs, then an appropriate V error 
     * code is returned in "error".
     */
  {
    register SessionDesc *desc;
    register struct passwd *pwentry;
    char *UserName;
    char *Password;
    SystemCode r;
    long seconds;
    int	i;
    unsigned NameLength;
    char *NamePtr;
    int parent;
    int userIsGuest;

    if (Session != NULL) 
      { 
	/* shouldn't happen! */
	*error = INTERNAL_ERROR;
	return (-1);
      }

    if (SDebug)
        printf("Attempting to create session for V user %d\n", vUserNumber);

    /* Find the local user name corresponding to the V user number.
     * If no such correspondence can be found, then we attempt to map as 
     * UNKNOWN_USER instead.
     */
    userIsGuest = (vUserNumber == UNKNOWN_USER);
    if ( (UserName = GetUserCorrespondence(vUserNumber)) == NULL ||
         (pwentry = GetUnixPwnam(UserName)) == NULL )
      {
	userIsGuest = 1;
        if ( (UserName = GetUserCorrespondence(UNKNOWN_USER)) == NULL ||
             (pwentry = GetUnixPwnam(UserName)) == NULL )
	  {
	    printf("Vserver error: no correspondence for UNKNOWN_USER!\n");
	    *error = INTERNAL_ERROR;
	    return (-1);
	  }
      }

    if (SDebug)
	printf("User validated for new session as \"%s\"\n", UserName);

    seconds = time(0); /* time of login */

    if ((desc = AllocSessionDesc()) == NULL) 
      { 
	*error = NO_SERVER_RESOURCES;
	return (-1);
      }
    newSessionPid = GenerateNewVPid();
    desc->Vpid = newSessionPid;

    /* Fork a session server for this new session */
    SessionInitialized = SessionInitFailed = 0;
    signal(SESSION_INITIALIZED, SessionReady);
    signal(SESSION_INIT_FAILED, SessionInitFail);
    if ((desc->localpid = fork()) == -1)
      {
	FreeSessionDesc(desc);
	*error = NO_SERVER_RESOURCES;
	return (-1);
      }

    if (desc->localpid != 0)
      { 
	/* PARENT - main server */
	long seconds;

	/* Wait for the new session to initialize. */
	seconds = time(0);
	do
	  {
	    pause();
	  }
	while(!SessionInitialized && !SessionInitFailed &&
	      (time(0) - seconds) < MAX_SESSION_INIT_TIME);

	if (SessionInitFailed)
	  {
	    FreeSessionDesc(desc);
	    *error = NO_SERVER_RESOURCES;
	    return (-1);
	  }
	else if (!SessionInitialized)
	  {
	    /* The "pause()" loop timed out, which indicates that the new
	     * session may somehow be wedged.
	     */
	    printf("%d - New session initialization timed out!\n", getpid());
	  }
	FlushNewMessages(vUserNumber);
	return (desc->localpid);
      }

    /* CHILD - new session server */
    sigblock(SigMask(PACKET_RECEIVED));

    close(Net); /* close the parent's network file. */
    
    desc->localpid = getpid();
    parent = getppid();
    /* Put this process in its own Unix process group (i.e. separate from the 
     * parent), just in case the whole process group accidentally gets killed.
     */
    setpgrp(desc->localpid, desc->localpid);

    if (SDebug) printf("New session created, V pid = 0x%08x, Unix pid = 0x%x\n",
		       desc->Vpid, desc->localpid);

    MyPid = newSessionPid;
    MyVUserNumber = vUserNumber;
    MyUserIsGuest = userIsGuest;

    /* Finish setting up the session descriptor */
    FreeSessionDescsExcept(desc);
    desc->uid = pwentry->pw_uid;
    desc->gid = pwentry->pw_gid;
    desc->logintime = seconds;
    Session = desc;

    /* Gross code needed for parent-independent niceness.
     * See the section 3c manual entry for nice().
     */
    nice(-40);
    nice(20);
    nice(0);
    nice(-5); /* This is the nice value that it will run with. */
    /* NOTE:
     * process.c assumes that the sessions are running at -5
     * when it sets the nice value of processes that it forks
     * off. (The sessions aren't privileged, so they can't use
     * the kludgey code above).
     */

    /* Get a new network file for this session */
    if ((Net = (*NetOpen)(MyPid, MyVUserNumber)) == -1) 
      {
	if (SDebug) printf("New session can't open net - aborting.\n");
	kill(parent, SESSION_INIT_FAILED);
	KillSession();
      }

    InitKernel(1);

    /* Now that the new session is able to receive packets (and, in particular,
     * will receive all future retransmissions of the original request), we
     * signal the parent (main server) process that it may continue.
     * (Note that this must be done before changing uids.)
     */
    kill(parent, SESSION_INITIALIZED);

    /* Release our copy of any instances that were maintained by the parent. */
    ReclaimInstances(RI_RELEASE_ALL);
    if (FDebug) CheckFiles();

    /* Change to run as the specified user. */
	initgroups(pwentry->pw_name, pwentry->pw_gid);
    setuid(pwentry->pw_uid);
    MyUserId = geteuid();
    MyGroupId = pwentry->pw_gid; /* Should be getegid(), but this returns 0! */

    /* Set up the well-known contexts. */
    InitContexts();

    /* Set up the Environment for this user */
    MyEnvHome[ 5 ] = NULL;
    if (strlen(pwentry->pw_dir) < MAX_ENV_VAR_LEN)
	strcat(MyEnvHome, pwentry->pw_dir);

    MyEnvUser[ 5 ] = NULL;
    if (strlen(pwentry->pw_name) < MAX_ENV_VAR_LEN)
	strcat(MyEnvUser, pwentry->pw_name);

    MyEnvShell[ 6 ] = NULL;
    if (strlen(pwentry->pw_shell) < MAX_ENV_VAR_LEN)
	strcat(MyEnvShell, pwentry->pw_shell);

    /* Insert the net number and the V user number into the session banner. */
    sprintf(BlockBuffer, SESSION_BANNER, NetworkNum, MyVUserNumber);
    SetStringName(BlockBuffer);

    return (0);

  } /* ForkNewSession */


SystemCode ValidSessionPid(req, pid)
    KernelRequest *req;
    ProcessId pid;
  {
    extern GroupId CommonGroupId;
    extern int ChildrenAreDead;

    ProcessId targetPid;

    /* Make sure we know the state of any sessions that we created. */
     if (ChildrenAreDead)
      {
	ChildrenAreDead = 0;
        CheckSessions();
      }

    targetPid = req->pid;
    if (KDebug) 
        printf("ValidSessionPid: 0x%x\n", targetPid);
    
    if (targetPid == MyPid || targetPid == CommonGroupId ||
        FindSessionFromVPid(targetPid) != NULL)
        return(OK);
    else
      {
	/* No local record of this pid!  Check whether it has our LHN. */
	if ((targetPid&LOGICAL_HOST_PART) != LogicalHostPid)
	  {
	    /* "targetPid" has nothing to do with us! */
	    ProcessId forwarder = Forwarder(targetPid);

	    if (forwarder != CommonGroupId && (forwarder&GROUP_ID_BIT))
		return(DISCARD_REPLY);
	    else
		return(NONEXISTENT_PROCESS);
	  }

	/* As a last resort, try forwarding the request to the pid itself. */
        Forward(req, pid, targetPid);
	return(NOT_AWAITINGREPLY);
      }
  } /* ValidSessionPid */    


CheckSessions()
    /* Checks to see if any sessions have died.  If they have, then their 
     * descriptors are released.
     */
  {
    struct wait status;
    int child;
    SessionDesc *desc;

    while ((child = wait3(&status, WNOHANG|WUNTRACED, 0)) > 0)
      {

	/* Search for all descriptors that match this pid */
        if ((desc = FindSessionFromLocalPid(child)) == NULL)
          {
	    if (SDebug)
	       /* Status on a child we don't have a descriptor for. */
	        printf("Session not found, UNIX pid = %d\n", child);
	    continue;
	  }

        if (SDebug) 
	    printf("Session, UNIX pid = %d\n", child);

	/* Only free the descriptor for a session if it DIEs */
	if (WIFSIGNALED(status) || WIFEXITED(status))
	  {
            FreeSessionDesc(desc);
	    if (SDebug) printf(" freed\n");
	  }
	else if (SDebug) printf(" status is 0x%x\n", status);
      } 

  } /* CheckSessions */
