/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 *
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 * 10 Megabit Ethernet access routines
 *
 */

#include "config.h"
#include <errno.h>
#include <types.h>
#include <enet.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vethernet.h>
#include <Vikc.h>
#include <Vgroupids.h>
#include <net.h>
#include <kernel.h>
#include <debug.h>
#include <swab.h>
#include <pup/filtpri.h>

/* IMPORTS */
extern int 		errno;
extern char		*NetworkDeviceName;

/* EXPORTS */
int Enet10Open(), Enet10Read(), Enet10Write(), Enet10Signal(),
    Enet10ReceiveQLength();

/* LOCALS */
#define AddressTableSize 256	/* size of logical -> physical table */
#define PidToHost(pid) ( (pid>>16) & 0xFF )
static Enet10Address	AddressTable[AddressTableSize];

/* The following offsets (within an Ethernet IK packet) are in shorts. */
#define ETHTYPE		6 /* i.e., the offset from the start of the header. */
#define IKPACKET_START	7
#define dstPid_1stRcvd		IKPACKET_START + 4
#define dstPid_2ndRcvd		IKPACKET_START + 5
#define userNumber_1stRcvd	IKPACKET_START + 8
#define userNumber_2ndRcvd	IKPACKET_START + 9

/* Packet filter for the main server process - this receives V packets 
 * that were not previously received by one of the sessions (the sessions' 
 * filters have higher priority).
 *
 * Note: More stringent checks may be added in the future.
 */
static struct enfilter MainFilter =   
  {
    ENFP_VLISTEN,	/* enf_Priority  */
    3,			/* enf_FilterLen */
    /* must be a V Kernel packet */
	ENF_PUSHWORD + ETHTYPE,
	ENF_PUSHLIT|ENF_EQ,
	BigEndianizeShort(KERNEL_PACKET)
  };

/* Packet filter for each session process. */
static struct enfilter SessionFilter =
  {
    ENFP_VSESSION, /* enf_Priority, MUST be higher than the main server */
    24,   /* enf_FilterLen */
    /* must be a V Kernel packet */
	ENF_PUSHWORD + ETHTYPE,
	ENF_PUSHLIT|ENF_CAND,
	BigEndianizeShort(KERNEL_PACKET), 
    /* and sent either to the specific pid for this process */
	ENF_PUSHWORD + dstPid_2ndRcvd, /* if packet is in big-endian order */
	ENF_PUSHLIT|ENF_COR,
	0,	/* changes with session pids */

	ENF_PUSHWORD + dstPid_1stRcvd, /* if packet is in little-endian order */
	ENF_PUSHLIT|ENF_COR,
	0,	/* changes with session pids */
    /* or to a group id (usually the local process group (containing all local
       processes), or a static group), PROVIDED THAT the user number matches
       the user number of this session. */
	/* Note: for efficiency we only check the low-order 16 bits of the
	 * user number. */
	ENF_PUSHWORD + userNumber_2ndRcvd, /* if in big-endian order */
	ENF_PUSHLIT|ENF_EQ,
	0, /* low-order 16 bits of the user number - filled in later. */

	ENF_PUSHWORD + userNumber_1stRcvd, /* if in little-endian order */
	ENF_PUSHLIT|ENF_EQ,
	0, /* low-order 16 bits of the user number - filled in later. */

	ENF_OR,
	ENF_PUSHZERO|ENF_CNOR, /* rejects packet immediately if user # wrong. */

	/* Check for a group id: */
	ENF_PUSHWORD + dstPid_1stRcvd, /* if packet is in big-endian order */
	ENF_PUSHLIT|ENF_AND,
	BigEndianizeShort(GROUP_ID_BIT >> 16),

	ENF_PUSHWORD + dstPid_2ndRcvd, /* if packet is in little-endian order */
	ENF_PUSHLIT|ENF_AND,
	LittleEndianizeShort(GROUP_ID_BIT >> 16),
	
	ENF_OR
  };

/* Kludge: Packet filter for a session for V user 0.
 * Note that "SessionFilter" above won't work correctly for sessions with a V 
 * user number of 0, because the \high-order/ 16-bits of all V user numbers 
 * happen to be 0.  The `correct' solution would be to properly check the 
 * IKC_LITTLE_ENDIAN bit, and strengthen the user number check according to 
 * the endian-ness.  However, this would require either having two packet 
 * filters instead of one for each session, or making the filters 
 * considerably bigger (and slower).  Instead, we use a special filter for 
 * such sessions:
 */
static struct enfilter User0SessionFilter =
  {
    ENFP_VSESSION, /* enf_Priority, MUST be higher than the main server */
    20,   /* enf_FilterLen */
    /* must be a V Kernel packet */
	ENF_PUSHWORD + ETHTYPE,
	ENF_PUSHLIT|ENF_CAND,
	BigEndianizeShort(KERNEL_PACKET), 
    /* and sent either to the specific pid for this process */
	ENF_PUSHWORD + dstPid_2ndRcvd, /* if packet is in big-endian order */
	ENF_PUSHLIT|ENF_COR,
	0,	/* changes with session pids */

	ENF_PUSHWORD + dstPid_1stRcvd, /* if packet is in little-endian order */
	ENF_PUSHLIT|ENF_COR,
	0,	/* changes with session pids */
    /* or to a group id (usually the local process group (containing all local
       processes), or a static group), PROVIDED THAT the user number matches
       the user number of this session (0 IN THIS CASE!). */
	ENF_PUSHWORD + userNumber_1stRcvd,
	ENF_PUSHZERO|ENF_CAND,
	ENF_PUSHWORD + userNumber_2ndRcvd,
	ENF_PUSHZERO|ENF_CAND,

	/* Check for a group id: */
	ENF_PUSHWORD + dstPid_1stRcvd, /* if packet is in big-endian order */
	ENF_PUSHLIT|ENF_AND,
	BigEndianizeShort(GROUP_ID_BIT >> 16),

	ENF_PUSHWORD + dstPid_2ndRcvd, /* if packet is in little-endian order */
	ENF_PUSHLIT|ENF_AND,
	LittleEndianizeShort(GROUP_ID_BIT >> 16),
	
	ENF_OR
  };

/* Offsets of locations within the filter where session-specific values need 
 * to be filled in.  (The first filter command is at offset 0.)
 */
#define	SF_PIDOFFSET_BIG	5
#define	SF_PIDOFFSET_LITTLE	8
#define SF_USERNUMLOW_BIG	11
#define SF_USERNUMLOW_LITTLE	14

static char *PrintAddress(p)
    unsigned short *p;
    /* Return a string for the given Ethernet address. */
  {
    static char buf[20];

    sprintf(buf,"%04x.%04x.%04x",
	    (unsigned short) BigEndianizeShort(p[0]),
    	    (unsigned short) BigEndianizeShort(p[1]),
	    (unsigned short) BigEndianizeShort(p[2]));

    return buf;
  }


Enet10Open(mypid, myUserNumber)
    ProcessId mypid;
    unsigned myUserNumber;
 {
 /*
  * Open an ethernet connection returning fileid on
  * connection.
  */
    register int i, fid;
    struct eniocb EnetControlBlk;
    u_short bits;
    u_int maxwaiting;
    char fn[16];
    register int failed = 0;
    struct enfilter *filter;

    /* Try to open a network device beginning with the given name.
     *
     * Algorithm: assumption is that minors start at 0
     * and run up as decimal numbers without gaps.  We search
     * until we get an error that is not EBUSY (i.e., probably
     * either ENXIO or ENOENT), or until we sucessfully open one.
     */
    for (i=0; !failed ; i++)
      {
	sprintf (fn, "%s%d", NetworkDeviceName, i);
	if ((fid = open(fn, 2)) >= 0) 	
	  {
	    ioctl(fid, FIOCLEX, NULL);    /* Close the file on a UNIX exec */
	    break;
	  }
	/* if we get past the break, we got an error */
	if (errno != EBUSY) failed++;
      }
    if (failed) return -1;

    if (NDebug) printf("Opened %s\n", fn);

    /* Be sure ENHOLDSIG is set */
    bits = ENHOLDSIG;
    ioctl(fid, EIOCMBIS, &bits);

    /* Get the parameters for this file */
    ioctl(fid, EIOCGETP, &EnetControlBlk);

    /* Ask for maximum incoming packet buffering */
    maxwaiting = EnetControlBlk.en_maxwaiting;
    ioctl(fid, EIOCSETW, &maxwaiting);

    /* Flush packets in the input queue */
    ioctl(fid, EIOCFLUSH, 0);

    if (mypid == LOCAL_KERNEL_PROCESS_PID)
	filter = &MainFilter;
    else /* Session */
      {
	if (myUserNumber != 0)
	    filter = &SessionFilter;
	else
	    filter = &User0SessionFilter;
	
	/* Set the kernel packet filter for this session: */
	filter->enf_Filter[SF_PIDOFFSET_BIG]
	    = BigEndianizeShort(mypid); 
	filter->enf_Filter[SF_PIDOFFSET_LITTLE]
	    = LittleEndianizeShort(mypid); 
	if (myUserNumber != 0)
	  {
	    filter->enf_Filter[SF_USERNUMLOW_BIG]
		= BigEndianizeShort(myUserNumber&0xFFFF);
	    filter->enf_Filter[SF_USERNUMLOW_LITTLE]
		= LittleEndianizeShort(myUserNumber&0xFFFF);
	  }

	if (NDebug)
	    printf("Filter set for %04x, user # (low-order 16 bits) = %04x\n",
		   mypid, myUserNumber);
      }

    ioctl(fid, EIOCSETF, filter);
    return fid;
  }


Enet10Read(net, packetlength, apacket, srcpid, destpid, forwarder, userNumber,
	   length, localaddr, remoteaddr, seqno, msg)
  int	net;			/* File id of ethernet */
  int	packetlength;		/* Actual packet length */
  struct	KernelPacket *apacket;	/* Packet to read */
  int	*srcpid,*destpid,*forwarder,
	*length,*localaddr,*remoteaddr;
  unsigned *userNumber;
  short *seqno;
  long *msg;
  {
  /*
   * Receive a packet from the ethernet
   * specified by net and place it in packet.
   */

    register Enet10Packet *enetpacket 
    		= (Enet10Packet*)&(apacket->header[ENET10_HDR_OFFSET]);
    register kPacketWithSegment *kpacket = &(apacket->kernelpacket);

    packetlength -= ENET10_HDR_OFFSET;
    packetlength = read( net, (char*)enetpacket, packetlength );

    if ( packetlength < sizeof( kPacket ) + ENET10_HEADER_SIZE )
      {
	if ( NDebug ) 
	  {
	    printf("Packet length %d, should be at least %d\n",
		packetlength, ENET10_HEADER_SIZE+sizeof(kPacket) );
			 
	    if ( packetlength < 0 )
	        perror( "packet read" );
	  }
        return( -1 );
      }

    if (DifferentIKCByteOrder(kpacket))
      {
	SwapIKPacket(kpacket);
      }
    kpacket->packetType &= ~IKC_LITTLE_ENDIAN;

    *srcpid = kpacket->srcPid;
    *destpid = kpacket->dstPid;
    *forwarder = kpacket->forwarder;
    *length = kpacket->length;
    *localaddr = (int)(kpacket->localaddress);
    *remoteaddr = (int)(kpacket->remoteaddress);
    *seqno = kpacket->sequenceNo;
    *userNumber = kpacket->userNumber;
	/*
	 * Remember the physical address corresponding to this logical
	 * address so we can send the reply back
	 */
    if ( kpacket->packetType == remoteForward )
      {
	AddressTable[ PidToHost( *forwarder ) ] = enetpacket->SrcHost;
        if ( NDebug ) 
            printf("Packet from %x to %x, forwarder %x host %s\n",
		*srcpid, *destpid, *forwarder,
		PrintAddress( &(AddressTable[ PidToHost(*forwarder) ]) ) );
      }
    else
      {
        AddressTable[ PidToHost( *srcpid ) ] = enetpacket->SrcHost;
        if ( NDebug ) 
            printf("Packet from pid %x, to pid %x hardware host %s\n",
		*srcpid, *destpid, 
		PrintAddress( &(AddressTable[ PidToHost(*srcpid) ]) ) );
      }

    /* Copy the message itself */
    bcopy(&kpacket->msg, msg, sizeof(Message));

    return(kpacket->packetType);
  } /* Enet10Read */



Enet10Write(net, packet, packetlength,
	    packettype, seqno, srcpid, destpid, forwarder, userNumber, 
	    length, localaddr, remoteaddr, msg)
    int	net;				/* file id to write to */
    register struct KernelPacket *packet;	/* Packet to write */
    int packetlength;			/* Actual length of packet */
    short packettype;
    short seqno;
    unsigned srcpid, destpid, forwarder,/* relevent pids*/
	     userNumber,
	     length,		/* length of encapsulated  part of packet */
	     localaddr, remoteaddr;
    register long *msg;		/* message if exists */
   /* Send a Vkernel packet over the ethernet specified by net
    * set up things like srchost, packettype and srcpid.
    */
  {
    int trans; 
    register Enet10Packet *enetpacket 
    		= (Enet10Packet*)&(packet->header[ENET10_HDR_OFFSET]);
    register kPacketWithSegment *kpacket = &(packet->kernelpacket);

/*
 * look up the logical destination host to get the immediate destination
 */
    packetlength -= ENET10_HDR_OFFSET;	/* adjust the packet length
					   to this network */
    enetpacket->DestHost = AddressTable[ PidToHost(destpid) ];

if ( NDebug ) printf("Sending packet to pid 0x%04x, hardware host %s\n",
		destpid, PrintAddress( & (AddressTable[ PidToHost(destpid) ])) );

    enetpacket->EtherType = BigEndianizeShort(KERNEL_PACKET);

    kpacket->packetType = packettype;
    kpacket->sequenceNo = seqno;
    kpacket->srcPid = srcpid;
    kpacket->dstPid = destpid;
    kpacket->forwarder = forwarder;
    kpacket->userNumber = userNumber;
    kpacket->length = length;
    kpacket->localaddress = (Unspec *)localaddr;
    kpacket->remoteaddress = (Unspec *)remoteaddr;

    /* Copy the message itself */
    bcopy(msg, &kpacket->msg, sizeof(Message));

#ifdef LITTLE_ENDIAN
    /* Interkernel packet is sent in little-endian order. */
    kpacket->packetType |= IKC_LITTLE_ENDIAN;
#endif LITTLE_ENDIAN

    for (trans = 0; trans < 4; ++trans)
	if (write(net, (char*)enetpacket, packetlength) != -1) return(OK);

    perror("Net write");
    printf("Network write failed: length = %d\n", packetlength);
    return(DEVICE_ERROR);
  } /* Enet10Write */


Enet10Signal( fid, signum )
   int	fid;		/* file id of an ethernet device */
   int	signum;		/* signal number to set */
  /* Enable or disable a signal on reception of a packet.
   * signum specifies the signal to be invoked if non-zero.
   * Zero signum disable signals.
   */
  {
    ioctl(fid, EIOCENBS, &signum);
  } /* Enet10Signal */


int Enet10ReceiveQLength( net )
   int net;
  /* Return the number of packets waiting in the receive queue */
  {
    int length;

    ioctl( net, FIONREAD, &length );
    return( length );
  }

