/*
* The V UNIX server: a V kernel and V server simulator for VAX/UNIX
* that provides a subset of UNIX system services to SUN workstations
* running the V distributed kernel.
* Copyright (c) 1982 David R. Cheriton, all rights reserved.
*
* Main server code.
*
*/

#include "config.h"
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vtime.h>
#include <Vgroupids.h>
#include <Vsession.h>
#include <pwd.h>
#include <server.h>
#include <debug.h>

/* Exports */
char BlockBuffer[BUFFER_SIZE+1];	/* For MoveFroms, MoveTos, etc. */
char *NetworkDeviceName = NULL;
char *HostName = NULL;
int PublicServer = 0;
char *StringName; /* Pointer to character string name used by ps */

/* Imports */
extern SystemCode CreateInstance();
extern SystemCode ReleaseInstance();
extern SystemCode ReadInstance();
extern SystemCode ReadAndForward();
extern SystemCode WriteInstance();
extern SystemCode QueryInstance();
extern SystemCode SetInstanceOwner();
extern SystemCode QueryFile();
extern SystemCode ModifyFile();
extern SystemCode RemoveFile();
extern SystemCode RenameFile();

extern SystemCode GetContextId();
extern SystemCode GetContextName();
extern SystemCode GetAbsoluteName();
extern SystemCode QueryName();
extern SystemCode GetFileName();
extern SystemCode ReadDescriptor();
extern SystemCode NameReadDescriptor();
extern SystemCode WriteDescriptor();
extern SystemCode NameWriteDescriptor();

extern SystemCode ValidSessionPid();
extern SystemCode QueryKernel();
extern SystemCode ServerJoinGroup();
extern SystemCode ServerQueryGroup();
extern SystemCode ServerGetPid();

extern SystemCode GetUnixTime();

extern SystemCode AddUserCorrespondence();
extern SystemCode DeleteUserCorrespondence();
extern SystemCode GetInstanceInfo();
extern SystemCode LookupServer();

extern SystemCode ProcessTimeout();

extern int AlarmRate;
extern int NetworkNum;
extern SessionDesc *Session;


main(argc, argv, envp)
    int argc;
    char *argv[];
    char *envp[];

    /* The V UNIX server */
  {
    Message msg;
    register IoRequest *req = (IoRequest *) msg;
    register ProcessId pid;
    unsigned segsize;
    SystemCode	reply;
    
    Initialize(argc, argv);
    
    for(;;)
      {
	extern GroupId CommonGroupId;

	unsigned vUserNumber;
	ProcessId forwarder;
	SystemCode error;
	
	/* Receive the next message: */
	segsize = BUFFER_SIZE;
	pid = ReceiveWithSegmentPlus(req, BlockBuffer, &segsize,
				     &forwarder, &vUserNumber);
	/* If:
	 *	1/ This is the main server, and
	 *	2/ We received a real request (i.e. not TIMEOUT), and
	 *	3/ The received message was sent to a group id, and
	 *	4/ The request is not one of the special ones that we 
	 *	   let the main server handle.
	 * then: a new session is forked off to handle the request.
	 * The parent process then continues round the loop.
	 */
	/* Note: The following requests are recognized, and do \not/ cause 
	 * a new session to be forked:
	 * QUERY_NAME, GET_ABSOLUTE_NAME, GET_CONTEXT_ID, any request sent to 
	 * the "V kernel server group":
	 *	These requests frequently fail, and can always be handled by 
	 *	the main server anyway.
	 * ADD_USER_CORRESPONDENCE:
	 *	This allows a new user (for which we do not have a local
	 *	correspondence) to create one for himself, even if this host 
	 *	does not provide `guest' access to such users.
	 */
	if ((Session == NULL) && (req->requestcode != TIMEOUT) &&
	    ((forwarder&GROUP_ID_BIT) != 0) &&
	    (req->requestcode != QUERY_NAME) &&
	    (req->requestcode != GET_ABSOLUTE_NAME) &&
	    (req->requestcode != GET_CONTEXT_ID) &&
	    (forwarder != VKERNEL_SERVER_GROUP) &&
	    (req->requestcode != ADD_USER_CORRESPONDENCE))
	  {
	    int result = ForkNewSession(vUserNumber, &error);
	    
	    if (result != 0)
	      { /* PARENT */
		if (error)
		  { /* no session was created, so we reply with error code */
		    req->requestcode = error;
		    if (GDebug)
			printf("Replycode was 0x%x\n", error);
		    Reply(req, pid);
		  }
		else
		  { /* Deal with this req, without actually sending a reply. */
		    req->requestcode = DISCARD_REPLY;
		    Reply(req, pid);
		  }
	        continue;
	      }
	    /* The CHILD now handles the request as usual: */
	  }
	
	/* Reset the alarm to fast rate when have an operation pending */
	if (AlarmRate == SLOW_ALARM_RATE && req->requestcode != TIMEOUT)
	    alarm(AlarmRate = FAST_ALARM_RATE);
	    
	if (KDebug ||
	    GDebug && req->requestcode != TIMEOUT)
	  {
	    int i;

	    printf("requestcode is 0x%x\n", req->requestcode);
	    for (i = 0; i < 8; i++)
		printf("%x ", ((long*)req)[i]);
	    printf("\n");
	  }
	
	/* Handle the request: */
	switch (req->requestcode)
	  {
	    /* I/O protocol requests: */

	    case CREATE_INSTANCE:
	    case CREATE_INSTANCE_RETRY:
		reply = CreateInstance(req, pid, segsize);
		break;
	    case RELEASE_INSTANCE:
		reply = ReleaseInstance(req, pid);
		break;
	    case READ_INSTANCE:
		reply = ReadInstance(req, pid);
		break;
	    case READ_AND_FORWARD:
		reply = ReadAndForward(req, pid);
		break;
	    case WRITE_INSTANCE:
	    case WRITESHORT_INSTANCE:
		reply = WriteInstance(req, pid, segsize);
		break;
	    case QUERY_INSTANCE:
		reply = QueryInstance(req, pid);
		break;
	    case SET_INSTANCE_OWNER:
		reply = SetInstanceOwner(req, pid);
		break;
	    case QUERY_FILE:
		reply = QueryFile(req, pid);
		break;
	    case MODIFY_FILE:
		reply = ModifyFile(req, pid);
		break;
	    case REMOVE_FILE:
		reply = RemoveFile(req, pid, segsize);
		break;
	    case RENAME_FILE:
		reply = RenameFile(req, pid, segsize);
		break;

	    /* Naming protocol requests: */

	    case GET_CONTEXT_ID:
		reply = GetContextId(req, pid, segsize);
		break;
	    case GET_CONTEXT_NAME:
		reply = GetContextName(req, pid);
		break;
	    case GET_ABSOLUTE_NAME:
		reply = GetAbsoluteName(req, pid, segsize);
	        break;
	    case QUERY_NAME:
		reply = QueryName(req, pid, segsize);
	        break;
	    case GET_FILE_NAME:
		reply = GetFileName(req, pid);
		break;
	    case READ_DESCRIPTOR:
		reply = ReadDescriptor(req, pid);
		break;
	    case NREAD_DESCRIPTOR:
		reply = NameReadDescriptor(req, pid, segsize);
		break;
	    case WRITE_DESCRIPTOR:
		reply = WriteDescriptor(req, pid, segsize);
		break;
	    case NWRITE_DESCRIPTOR:
		reply = NameWriteDescriptor(req, pid, segsize);
		break;

	    /* `Kernel process' requests: */

	    case QUERY_PROCESS:
		reply = ValidSessionPid(req, pid);
		break;
	    case QUERY_KERNEL:
		reply = QueryKernel(req, pid);
		break;
	    case JOIN_GROUP:
		reply = ServerJoinGroup(req, pid);
		break;
	    case QUERY_GROUP:
		reply = ServerQueryGroup(req, pid);
		break; 
	    /* GET_PID should soon be obsolete! */
	    case GET_PID:
		reply = ServerGetPid(req, pid);
		break;

	    /* Time server requests: */

	    case GET_TIME:
		reply = GetUnixTime(req, pid);
		break;

	    /* V server requests: */

	    case ADD_USER_CORRESPONDENCE:
		reply = AddUserCorrespondence(req, pid, segsize);
		break; 
	    case DELETE_USER_CORRESPONDENCE:
		reply = DeleteUserCorrespondence(req, pid);
		break; 
	    case GET_INSTANCE_INFO:
		reply = GetInstanceInfo(req, pid);
		break;
	    /* LOOKUP_SERVER will go away soon - it's needed only for Vload! */
	    case LOOKUP_SERVER:
		reply = LookupServer(req, pid, segsize);
		break;

	    /* A "timeout" pseudo-message generated by the kernel simulator: */

	    case TIMEOUT:
		reply = ProcessTimeout(req, pid);
		break;

	    /* Anything else: */

	    default:
		reply = ILLEGAL_REQUEST;
		break;
	  }
	if (reply != OK && GDebug)
	    printf("Replycode was 0x%x\n", reply);

	/* Reply to the sender, unless we have already done so as part of 
	 * handling the request.
	 */
	if (reply == NO_REPLY || reply == NOT_AWAITINGREPLY)
	    continue;

	/* If the request was sent to a group, then don't reply to failed 
	 * requests.  Instead, we simply discard the reply.
	 */
	if ( forwarder != CommonGroupId && (forwarder&GROUP_ID_BIT) &&
	     (reply == NOT_FOUND || reply == NOT_HERE || 
	      reply == INVALID_CONTEXT || reply == ILLEGAL_REQUEST ||
	      reply == TIMEOUT) )
	  {
	    reply = DISCARD_REPLY;
	  }

	req->requestcode = reply;
	Reply(req, pid);
      }
  }


usage(name)
    char *name;
  {
    printf("usage: %s [pAGKNIFPSDCLM] [n<hostname>] device-name\n", StringName);
  }


Initialize(argc, argv)
    int argc;
    register char *argv[];
    /* Initialize the server, setting up the file instance descriptors.
     * Also, cause the pseudo-kernel to initialize.
     */
  {
    extern long Debug;	/* just for setting */

    SystemCode r;
    char banner[MAX_BANNER_LEN + 1];
    
    Debug = 0;
    /* Handle arguments if any */
    StringName = *argv;

    if (argc < 2)
      {
	usage(StringName);
	exit(1);
      }

    while (--argc > 1)
      {
	++argv;
	while (**argv != '\0')
            switch ( *((*argv)++) )
	      {
		case '-':
			break;

		/* real arguments */
		case 'p':	
			PublicServer = 1; /* answer GetPid requests */
			break;

		case 'n':
			if (*(HostName = *argv) == NULL)
			  {
			    printf("%s: hostname must follow 'n'\n",
				   StringName);
			    usage(StringName);
			    exit(1);
			  }
			while (*++*argv != 0); /* clear to end of arg */
			break;

		/* Following are debugging options */
		case 'A':	Debug = -1;	/* turn on all debugging bits */
				break;
	      	case 'G':	Debug |= GENERIC_DEBUG;	
				break;
		case 'K':	Debug |= KERNEL_DEBUG;
				break;
		case 'N':	Debug |= NET_DEBUG;
				break;
		case 'I':	Debug |= INSTANCE_DEBUG;
				break;
		case 'F':	Debug |= FILE_DEBUG;
				break;
		case 'P':	Debug |= PROCESS_DEBUG;
				break;
		case 'S':	Debug |= SESSION_DEBUG;
				break;
		case 'D':	Debug |= DIRECTORY_DEBUG;
				break;
		case 'C':	Debug |= CONTEXT_DEBUG;
				break;
		case 'L':	Debug |= LOOKUP_DEBUG;
				break;
		case 'M':	Debug |= MULTICAST_DEBUG;
				break;
		default:
				printf("%s: unknown option '%c'\n",
				       StringName, (*argv)[-1]);
				usage(StringName);
				exit(1);
				break;
	      }
      }

    NetworkDeviceName = *++argv;

    if (GDebug)
    	printf("Device name is \"%s\"\n", NetworkDeviceName);
    
    /* For some reason we cannot close stdio, this would be nice to
       supply some extra file descriptors to the client.  When stdio
       is closed funny things happen, such as, stderr output from 
       remote execution no longer works.
    */

    InitKernel(0); /* Cause the kernel to initialize */
    InitInstances();
    InitIOQueue();
    sprintf(banner, SERVER_BANNER, NetworkNum);

    /* Set up the well-known contexts. */
    if ((r = InitContexts()) != OK)
      {
	printf("%s: Error in InitContexts(): V error code = 0x%x\n",
	       banner, r);
	exit(1);
      }

    SetStringName(banner);

    /* Set the logical pid map */
    SetPid(STORAGE_SERVER, GetPid(0)); 
    SetPid(UNIX_SERVER, GetPid(0));
    SetPid(TIME_SERVER, GetPid(0)); 
  } /* Initialize */

SetStringName(str)
  register char *str;
  /* Set the command name read by the ps command */
  {
    register char *p = StringName;

    if (SDebug) printf("A new %s%s started\n"
    		, PublicServer ? "public " : "\0", str);
    while (*str && *p) *p++ = *str++;
    while (*p) *p++ = ' ';
  } /* SetStringName */


SystemCode ProcessTimeout(req, pid)
  IoRequest *req;
  ProcessId pid;
  /*
   * Performs periodic housekeeping.
   * For example, the LRU clocks on each file instance are updated.
   * Also, the main server checks for the death of any of the sessions, and
   * each session checks whether or not its continued existence is justified.
   */
  {
    extern int ChildrenAreDead;
    extern KillSession();
    extern SessionDesc *Session;
    register int inactive;
    static int LRUtimer = 0;

    if ((LRUtimer += AlarmRate) > LRU_CLOCK_TICK)
      {
        LRUtimer = 0;
	IncInstanceClocks();
      }

    if (Session == NULL) 
      { /* Main server */
	if (ChildrenAreDead)
	  {
	    ChildrenAreDead = 0;
	    CheckSessions();
	  }
	/* Reset the alarm, since we have been idle for some time */
	/* we really should have an inactivity count for the main server */
	AlarmRate = SLOW_ALARM_RATE;
        return(OK);
      }

    /* Session */
    inactive = Session->inactivity;
    if (ChildrenAreDead) 
      {
	ChildrenAreDead = 0;
	CheckProcesses();
      }
     
    CheckIOQueue();

    Session->inactivity = inactive;
    Session->inactivity += AlarmRate;

    if (Session->inactivity > MAX_SESSION_INACTIVITY)
      {
	/* If we're no longer maintaining any valid file instances, then we 
	 * commit suicide.
	 */
	extern FileInstance *FileInstances;

	ReclaimInstances(RI_CHECK_ONLY);
	if (FileInstances->link == NULL)
	    KillSession(); /* we don't return from this */

	Session->inactivity = 0;
      }
    AlarmRate = SLOW_ALARM_RATE;	/* slow up, we're not so busy */
    return(OK);
  } /* ProcessTimeout */
