/* groups.c: Vserver group operations.
 */
#include "Venviron.h"
#include "debug.h"

/* exports */

extern InitGroupIpc();
extern int RegisteredGroup();
extern SystemCode JoinGroup();
extern SystemCode ServerQueryGroup();
extern SystemCode ServerJoinGroup();


#define MAX_GROUP_DESC		16 /* Total group member descriptors. */
#define GROUP_MASK		0xF /* For Group_bundle hash table. */

struct _GroupMember
  {
    struct _GroupMember	*link;
    Process_id		groupId;
    Process_id		pid;
  };
typedef struct _GroupMember GroupMember;

#define GROUP_FREE_LIST (sizeof(GroupMember)*MAX_GROUP_DESC)

/* Memory for group descriptors */
char Group_free_list[GROUP_FREE_LIST];
GroupMember *Free_group_descs;
GroupMember *Group_bundle[GROUP_MASK]; /* Edge vector for group desc. */

InitGroupIpc()
    /* Allocate group member descriptors and initialize */
  {
    register GroupMember *grpmem;
    register unsigned i;

    /* Group_bundle is assumed to be zero. */

    grpmem = ( GroupMember * ) &Group_free_list[0];
    for( i = 0; i < MAX_GROUP_DESC; ++i )
      {
	grpmem->link = Free_group_descs;
	Free_group_descs = grpmem;
	++grpmem;
      }
    for (i=0; i < GROUP_MASK ; i++) Group_bundle[i] = NULL;
  }

int GroupToPid(gid)
register Process_id gid;
  {
    extern ProcessId MyPid;
    extern GroupId CommonGroupId;

    register GroupMember *grpmem;

    /* First check for the special local group to which everyone belongs: */
    if (gid == CommonGroupId)
        return(MyPid);

    if (MDebug) printf("**GroupToPid: 0x%x.\n", gid);
    grpmem = ( GroupMember * ) Group_bundle[gid&GROUP_MASK];
    while (grpmem != NULL)
      {
	if (grpmem->groupId == gid)
	  {
	    if (grpmem->pid == CommonGroupId)
		return(MyPid);
	    else
	        return(grpmem->pid);
	  }
        grpmem = grpmem->link;
      }
    if (MDebug) printf("Notfound.\n");
    return(0);
  }

SystemCode ServerQueryGroup( req, pid )
register KernelRequest *req;
ProcessId pid;
/* Determine whether req->pid would be allowed to join the 
 * group specified by req->unspecified[0].
 */
  {
    register GroupId groupId;
    register GroupMember *grpmem, *prev;
    register unsigned members, umembers;
    SystemCode r;

    groupId = (GroupId) req->unspecified[0];
    if( (groupId & GROUP_ID_BIT) == 0 ) return( BAD_ARGS );
    members = 0; /* Count of members in group. */
    umembers = 0; /* Count of member in group with same userNumber. */
    if( req->pid != 0 ) pid = req->pid;
    if (MDebug) printf("Query of group %x: %x => %x\n", groupId, pid, req->pid);
    prev = (GroupMember *) &Group_bundle[groupId&GROUP_MASK];
    while( (grpmem=prev->link) != NULL )
      {
	if (grpmem->groupId == groupId)
	  { /* Found a valid group member. */
	    ++members;
	    if( grpmem->pid == pid ) r = DUPLICATE_NAME;
	  }
	prev = grpmem;
      }
    req->unspecified[1] = members;
    if (members == 0)
        if (Forwarder( pid ) & GROUP_ID_BIT)
	    r = DISCARD_REPLY;
	else
	    r = NOT_FOUND;
    else
	if ( UserNumber( pid ) != SYSTEM_USER )
	    r = NO_PERMISSION;
        else
	    if (r != DUPLICATE_NAME) r = OK;
    if (MDebug) printf("membership: %d, %d -> %x\n", members, umembers, r );
    return( r );
  }

SystemCode ServerJoinGroup( req, pid )
KernelRequest *req;
ProcessId pid;
  {
    if ( ServerQueryGroup( req, pid ) == NO_PERMISSION )
	return ( NO_PERMISSION );
    if ( Forwarder( pid ) & GROUP_ID_BIT )
	return( DISCARD_REPLY );
    else
	return( NO_PERMISSION );
  }

SystemCode JoinGroup( groupId, pid )
register Process_id groupId, pid;
  {
    GroupMember *grpmem;
    register int index;
    
    if( (grpmem=Free_group_descs) == NULL )
      {
	printf("Vserver panic: No group descriptors to add 0x%x to 0x%x\n",
	    pid, groupId);
	return( NO_GROUP_DESC );
      }
    Free_group_descs = grpmem->link;

    /* Initialize descriptor. */
    grpmem->groupId = groupId;
    grpmem->pid = pid;
    /* Link into hash table. */
    index = groupId & GROUP_MASK;
    grpmem->link = Group_bundle[index];
    Group_bundle[index] = grpmem;
    return( OK );
  }
