/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 *
 * Implements the directory information requests: READ & WRITE _DESCRIPTOR
 * Also handles read instance on DIRECTORY_INSTANCEs.
 */

#include "config.h"
#include <types.h>
#include <stat.h>
#include <pwd.h>
#include <grp.h>
#include <dir.h>
#include <errno.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vnaming.h>
#include <Vdirectory.h>
#include <Vgroupids.h>
#include <server.h>
#include <debug.h>
#include <swab.h>

extern int errno;
extern SystemCode GetName();
extern char *ParseName();
extern FileInstance *GetInstance();
extern struct passwd *getpwuid();
extern struct group *getgrgid();
extern SystemCode ReadDirectory();
extern SystemCode ChangeContext();
extern SystemCode ModifyDescriptor();
extern char *rindex();
extern char *GetUnixUserName(), *GetUnixGroupName();
extern char *FindContext();

extern char BlockBuffer[];
extern GroupId CommonGroupId;
extern char *HostName;


static ByteSwapDescriptorIfNecessary(pid, desc, type)
    ProcessId pid;
    ArbitraryDescriptor *desc;
    unsigned type;
    /* Byte-swap appropriate fields in the descriptor "desc", 
     * if "pid"s machine has a different byte order to ours.
     */
  {
    if (!DifferentByteOrder(pid)) return;

    switch (type)
      {
        case UNIXFILE_DESCRIPTOR:
	  ByteSwapUnixDescriptor(&desc->u);
	  break;
	case CONTEXT_DESCRIPTOR:
	  ByteSwapContextDescriptor(&desc->c);
	  break;
	case EMPTY_DESCRIPTOR:
	  ByteSwapEmptyDescriptor(&desc->e);
	  break;
      }
  }


UnixFileDescriptor *ConstructDescriptor( st, name )
    register struct stat *st;
    register char *name;
    
/*
 * Take the stat structure and name and create a descriptor
 * entry and return a pointer to a UnixFileDescriptor.  Name 
 * has to be in the directory entry form (ie. no '/'s)
 */
  {
    register UnixFileDescriptor *Udesc = (UnixFileDescriptor *)BlockBuffer;

    Udesc->descriptortype = UNIXFILE_DESCRIPTOR;

    strncpy( Udesc->name, name, MAX_NAME_LENGTH );

    Udesc->st_dev = st->st_dev;
    Udesc->st_ino = st->st_ino;
    Udesc->st_mode = st->st_mode;
    Udesc->st_nlink = st->st_nlink;
    Udesc->st_uid = st->st_uid;
    Udesc->st_gid = st->st_gid;
    Udesc->st_rdev = st->st_rdev;
    Udesc->st_size = st->st_size;
    Udesc->st_atime = st->st_atime;
    Udesc->st_mtime = st->st_mtime;
    Udesc->st_ctime = st->st_ctime;
    
    if ( (name = GetUnixUserName( st->st_uid )) != NULL )
	strncpy( Udesc->ownername, name, UnixUserNameLen );
    else
        *((short *)(Udesc->ownername)) = 0;	/* guarrantee right byte */
   
    if ( (name = GetUnixGroupName( st->st_gid )) != NULL )
	strncpy( Udesc->groupname, name, UnixUserNameLen );
    else
        *((short*)(Udesc->groupname)) = 0;

    if ( DDebug )
      {
        printf( "owner %8s, group %8s, name %s\n", Udesc->ownername
		, Udesc->groupname, Udesc->name );
	printf( "st_mode = 0%o, Udesc->st_mode 0%o\n", st->st_mode, Udesc->st_mode );
      }
    
    return( Udesc );
    
  } /* ConstructDescriptor */
    
SystemCode ModifyDescriptor( Name, Descriptor )
    char *Name;
    UnixFileDescriptor *Descriptor;
/*
 * Update the directory entry for Name to look like Descriptor.
 * This only means to modify those fields which can be changed
 * in UNIX. 
 */
  {
    if ( DDebug )
        printf( "ModifyDescriptor: mode 0%o\n", Descriptor->st_mode&~S_IFMT );

    if ( chmod( Name, Descriptor->st_mode&~S_IFMT ) != 0 )
        switch ( errno )
	  {
	    case ENOENT:
	    	return( NOT_FOUND );
		break;
	    default:		/* should be more specific */
	        return( NO_PERMISSION );
		break;
	  }
    else
        return( OK );
  } /* ModifyDescriptor */


SystemCode GetDescriptor( req, pid, segsize, desc )
    DescriptorRequest *req;
    ProcessId pid;
    unsigned segsize;
    register UnixFileDescriptor *desc;
/*
 * Retrieve the descriptor from the process.
 */
  {
    SystemCode r;
    
    /* Was the descriptor sent with the packet? */
    if ( segsize < sizeof(UnixFileDescriptor) + req->dataindex )
      { /* no, have to retrieve from remote host */
	r = MoveFrom( pid, req->segmentptr + req->dataindex, desc,
			    sizeof(UnixFileDescriptor) );
	if (r != OK) return r;
      }
    else
      { /* yes, copy from BlockBuffer */
	bcopy( &BlockBuffer[req->dataindex], desc,
	       sizeof(UnixFileDescriptor) );
      }
   
    ByteSwapDescriptorIfNecessary(pid, desc, UNIXFILE_DESCRIPTOR);

    if (DDebug)
        printf("GetDescriptor: desctype = 0x%x\n", desc->descriptortype);
    if (desc->descriptortype != UNIXFILE_DESCRIPTOR)
	return(BAD_ARGS);

    return(OK);
  } /* GetDescriptor */

SystemCode ReadDirectory( req, pid, desc, block, bytes )
    register IoRequest *req;
    ProcessId pid;
    register DirectoryInstance *desc;
    register unsigned block;
    unsigned bytes;
/*
 * Read a directory entry and format it as a UnixFileDescriptor.
 */
  {
    struct direct *dentry;
    struct stat entrystat;
    UnixFileDescriptor *Udesc;
    char *ptr;
#define reply	((IoReply*)req)

    if ( block < 0 )
        return( BAD_ARGS );

    if ( block < desc->nextblock )
      {
        rewinddir( desc->unixfile );
	desc->nextblock = 0;
      }

    do
      {
        if ( (dentry = readdir( desc->unixfile))==NULL )
            return( END_OF_FILE );
      } while ( block > desc->nextblock++ );
    
    /* change to the context so that we may interpret the name
       of the directory entry */
    if ( (block = (unsigned)ChangeContext( desc->ctx.cid )) != OK )
        return( (SystemCode) block );
    
    if ( stat( dentry->d_name, &entrystat ) != 0 )
        switch( errno )
	  {
		/* only known reasons for failure: */
	    case ENOENT: /* inode == 0 */
	    case EACCES: /* permission denied */
	    case ELOOP: /* symbolic link loop */
		Udesc = (UnixFileDescriptor*)BlockBuffer;
		Udesc->descriptortype = EMPTY_DESCRIPTOR;
		break;
	    default:
		if (DDebug)
		    printf("error on stat(%s, ...) - %d\n",
		           dentry->d_name, errno);
	    	return( INTERNAL_ERROR );
		break;
	  }
    else
        Udesc = ConstructDescriptor( &entrystat, dentry->d_name );

    /* where to put it */
    ptr = req->bufferptr;

    /* Everything is okay */
    reply->replycode = OK;
    reply->bytecount = bytes;

    ByteSwapDescriptorIfNecessary(pid, Udesc, Udesc->descriptortype);

    /* Using ReplyWithSegment could change if descriptor changes */
    ReplyWithSegment( reply, pid, (char *)Udesc, ptr, bytes );
    return( NOT_AWAITINGREPLY );

#undef reply
  } /* ReadDirectory */


ContextDescriptor *ConstructContextDescriptor(cid)
  ContextId cid;
  {
    static ContextDescriptor Cdesc;

    Cdesc.descriptortype = CONTEXT_DESCRIPTOR;

    switch (cid)
      {
	case GLOBAL_ROOT_CONTEXT:
	  Cdesc.ctxpair.pid = VCSNH_SERVER_GROUP;
	  Cdesc.ctxpair.cid = GLOBAL_ROOT_CONTEXT;
	  strcpy(Cdesc.name, "[");
	  break;

	case STORAGE_SERVER_CONTEXT:
	  Cdesc.ctxpair.pid = VSTORAGE_SERVER_GROUP;
	  Cdesc.ctxpair.cid = STORAGE_SERVER_CONTEXT;
	  strcpy(Cdesc.name, "storage");
	  break;

	case DEFAULT_CONTEXT:
	  Cdesc.ctxpair.pid = CommonGroupId;
	  Cdesc.ctxpair.cid = DEFAULT_CONTEXT;
	  strcpy(Cdesc.name, HostName);
	  break;
      }

    return &Cdesc;
  }

SystemCode ReadMMContext( req, pid, desc, block, bytes )
    register IoRequest *req;
    ProcessId pid;
    register DirectoryInstance *desc;
    register unsigned block;
    unsigned bytes;
/*
 * Read from a multi-manager context directory
 */
  {
    ContextDescriptor *Cdesc;
#define reply	((IoReply*)req)
    ContextPair ctx;
    char *ptr;

    if ( block > 0 ) return END_OF_FILE;
    if ( bytes < sizeof(ContextDescriptor) ) return BAD_BYTE_COUNT;

    switch (desc->ctx.cid)
      {
	case GLOBAL_ROOT_CONTEXT:
    	  Cdesc = ConstructContextDescriptor(STORAGE_SERVER_CONTEXT);
	  break;

	case STORAGE_SERVER_CONTEXT:
	  Cdesc = ConstructContextDescriptor(DEFAULT_CONTEXT);
	  break;
      }

    /* where to put it */
    ptr = req->bufferptr;

    /* Everything is okay */
    reply->replycode = OK;
    reply->bytecount = bytes;

    ByteSwapDescriptorIfNecessary(pid, Cdesc, Cdesc->descriptortype);

    /* Using ReplyWithSegment could change if descriptor changes */
    ReplyWithSegment( reply, pid, (char *)Cdesc, ptr, bytes );
    return( NOT_AWAITINGREPLY );

#undef reply
  } /* ReadDirectory */



/* DESCRIPTOR REQUEST HANDLERS */
SystemCode ReadDescriptor( req, pid )
    DescriptorRequest *req;
    ProcessId pid;
/*
 * Read the descriptor for the file specified by the fileid
 * in the request.
 */
  {
    register FileInstance *desc;
    char *ptr, *ptr2;
    struct stat FileStat, NameStat;
    UnixFileDescriptor *Udesc;
    unsigned bytes;
#define reply	((DescriptorReply*)req)

    if( Session == NULL ) 
        return( ILLEGAL_REQUEST );

    if( (desc = GetInstance(req->fileid)) == NULL ) 
        return( INVALID_FILE_ID );

    Session->inactivity = 0;

    switch (desc->type)
      {
	default:
          return( REQUEST_NOT_SUPPORTED );

	case MMCONTEXT_INSTANCE:
	  bytes = sizeof(ContextDescriptor);
          Udesc = (UnixFileDescriptor *) ConstructContextDescriptor(
			((DirectoryInstance *)desc)->ctx.cid );
	  break;

	case FILE_INSTANCE:
	case DIRECTORY_INSTANCE:
	  bytes = sizeof(UnixFileDescriptor);
	  /* Should always be able to obtain status on open files */
	  if (desc->type == DIRECTORY_INSTANCE)
	    {
	      if ( fstat( ((DIR *)desc->unixfile)->dd_fd, &FileStat ) != 0 )
	          return( INTERNAL_ERROR );
	    }
	  else
	    {
	      if ( fstat( desc->unixfile, &FileStat ) != 0 )
	          return( INTERNAL_ERROR );
	    }
	  /* Do a consistency check on the validity of the name and
	     the instance */
	  if ( stat( desc->name, &NameStat ) != 0 
	       || NameStat.st_ino != FileStat.st_ino 
	       || NameStat.st_dev != FileStat.st_dev )
	      return( NOT_FOUND );
      
	  /* have to make a directory entry equivalent name ie.
	     "xyzzy" not "/zz/yy/xyzzy" */
	  if ( *(ptr = rindex( desc->name, '/' ) + 1) == '\0' )
	    {
	      /* Directory instance name */
	      ptr--;		/* skip '/' */
	      while( ptr > desc->name && *--ptr != '/' );
	      ptr++;
	    }
	      
	  /* get equivalent of directory entry for the instance name */
	  if ( DDebug ) printf( "Constructing descriptor for %s\n", ptr );
	 
	  Udesc = ConstructDescriptor( &NameStat, ptr );
          break;
      }

    /* got enough room in the segment? */
    if ( req->dataindex + bytes > req->segmentlen )
        return( BAD_BUFFER );

    /* where to put it */
    ptr = req->segmentptr + req->dataindex;
    
    reply->replycode = OK;
    
    ByteSwapDescriptorIfNecessary(pid, Udesc, Udesc->descriptortype);

    /* Using ReplyWithSegment could change if descriptor changes */
    ReplyWithSegment( reply, pid, (char *)Udesc, ptr, bytes );
    return( NOT_AWAITINGREPLY );
#undef reply

  } /* ReadDescriptor */



SystemCode NameReadDescriptor( req, pid, segsize )
    DescriptorRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Return the descriptor for the name in the request.
 */
  {
    register SystemCode r;
    char *NamePtr;
    char *ptr,*ptr2;
    struct stat NameStat;
    UnixFileDescriptor *Udesc;
    ContextPair ctx;
    unsigned bytes;
#define reply	((DescriptorReply*)req)

    /* we do this check so that we don't have to explicitly check the
       permissions on all the directories down the path */
    if( Session == NULL ) 
        return( ILLEGAL_REQUEST );
 
    Session->inactivity = 0;

    r = GetName( req, pid, &segsize, &NamePtr, &ctx );

    if( DDebug ) printf( "NameReadDescriptor: \"%s\"\n", NamePtr );

    switch (r)
      {
	default:
          return r;

	case MULTI_MANAGER:
	  bytes = sizeof(ContextDescriptor);
          Udesc = (UnixFileDescriptor *)
	  		ConstructContextDescriptor(ctx.cid);
	  break;

	case OK:
	  bytes = sizeof(UnixFileDescriptor);
	  if ( stat( NamePtr, &NameStat ) != 0 )
	      return( NOT_FOUND );
      
	  /* make the name so that it will look like a directory entry */
	  ptr = ParseName( (char*)malloc( strlen(NamePtr) + 2 ), NamePtr, 0 );
      
	  /* check for the special instance of the name being '/' */
	  if ( *(ptr2 = rindex( ptr, '/' ) + 1) == '\0' )
	      ptr2--;
	  
	  Udesc = ConstructDescriptor( &NameStat, ptr2 );
	  
	  free( ptr );	/* free the name that we allocated */
	  break;
      }

    /* got enough room in the segment? */
    
    if ( req->dataindex + bytes > req->segmentlen)
        return( BAD_BUFFER );

    /* where to put it */
    ptr2 =  req->segmentptr;
    ptr2 += req->dataindex;
    
    reply->replycode = OK;
    
    ByteSwapDescriptorIfNecessary(pid, Udesc, Udesc->descriptortype);

    /* Using ReplyWithSegment could change if descriptor changes */
    ReplyWithSegment( reply, pid, (char *)Udesc, ptr2, bytes );

    return( NOT_AWAITINGREPLY );
#undef reply
  } /* NameReadDescriptor */

SystemCode WriteDescriptor( req, pid, segsize )
    DescriptorRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Write the descriptor for the file specified by the fileid
 * in the request.
 */
  {
    register FileInstance *desc;
    struct stat FileStat, NameStat;
    UnixFileDescriptor Descriptor;
    register SystemCode r;

    if ( Session == NULL ) 
        return( ILLEGAL_REQUEST );

    if( (desc = GetInstance(req->fileid)) == NULL ) 
        return( INVALID_FILE_ID );

    if( desc->type != FILE_INSTANCE && desc->type != DIRECTORY_INSTANCE ) 
        return( REQUEST_NOT_SUPPORTED );

    Session->inactivity = 0;

    /* Should always be able to obtain status on open files */
    if (desc->type == DIRECTORY_INSTANCE)
      {
        if ( fstat( ((DIR *)desc->unixfile)->dd_fd, &FileStat ) != 0 )
	    return( INTERNAL_ERROR );
      }
    else
      {
        if ( fstat( desc->unixfile, &FileStat ) != 0 )
	    return( INTERNAL_ERROR );
      }

    /* Do a consistency check on the validity of the name and
       the instance */
    if ( stat( desc->name, &NameStat ) != 0 
	 || NameStat.st_ino != FileStat.st_ino 
         || NameStat.st_dev != FileStat.st_dev )
        return( NOT_FOUND );

    if ( (r = GetDescriptor( req, pid, segsize, &Descriptor )) != OK )
        return( r );

    return( ModifyDescriptor( desc->name, &Descriptor ) );

  } /* WriteDescriptor */

SystemCode NameWriteDescriptor( req, pid, segsize )
    DescriptorRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Get the descriptor specified in the request and modify
 * the entries of the named file.
 */
  {
    register SystemCode r;
    char *NamePtr;
    UnixFileDescriptor Descriptor;
    ContextPair dummy;

    /* we do this check so that we don't have to explicitly check the
       permissions on all the directories down the path */
    if( Session == NULL ) return( NO_PERMISSION );
 
    Session->inactivity = 0;

    if ( (r = GetName( req, pid, &segsize, &NamePtr, &dummy )) != OK )
	  return( r );

    if( DDebug ) printf( "NameWriteDescriptor: \"%s\"\n", NamePtr );

    if ( (r = GetDescriptor( req, pid, segsize, &Descriptor )) != OK )
        return( r );

    return( ModifyDescriptor( NamePtr, &Descriptor ) );

  } /* NameWriteDescriptor */


SystemCode SetupDirectoryInstance(dp, desc, nameptr, pid)
    DIR *dp;
    register FileInstance *desc;
    char *nameptr;
    ProcessId pid;
  {
    extern SessionDesc *Session;
    
    struct stat buf;
    ContextId cid;
    register SystemCode r;
    int file = dp->dd_fd;
    unsigned short mode;

    if (fstat( file, &buf )<0) printf("Error doing fstat\n");
    mode = buf.st_mode;

    if( Session == NULL )
      {
	if( !(mode&(S_IREAD >> 6)))
	  return( NO_PERMISSION);  
      }

	/* NOTE: have to be able to cd to it, inorder to contruct Descriptors
	 	 so it is best to try and put it in the context cache.
         */
	if ( (r = InsertContextNode( nameptr, strlen(nameptr), &cid )) != OK )
	    return( r );
	
        desc->name = FindContext( cid );	/* should be there */
	((DirectoryInstance *)desc)->ctx.cid = cid;
	desc->blocksize = sizeof( UnixFileDescriptor );
        desc->lastblock = MAXBLOCKS;	/* too much trouble to compute */
        desc->lastbytes = MAXBYTES;
        desc->nextblock = 0;
        desc->unixfile = (int)dp; /* Remember the directory descriptor */
	desc->owner = pid;
        return(OK);
  } 


SystemCode SetupMMContextInstance(ctx, desc, pid)
    ContextPair ctx;
    register DirectoryInstance *desc;
    ProcessId pid;
  {
    switch (ctx.cid)
      {
	case GLOBAL_ROOT_CONTEXT:
	  desc->name = "[";
	  break;
	case STORAGE_SERVER_CONTEXT:
	  desc->name = "[storage";
	  break;
      }

    desc->ctx = ctx;
    desc->blocksize = sizeof( UnixFileDescriptor );
    desc->lastblock = MAXBLOCKS;	/* too much trouble to compute */
    desc->lastbytes = MAXBYTES;
    desc->nextblock = 0;
    desc->unixfile = -2;
    desc->owner = pid;
    return(OK);
  } 
