/* Unix open/close/read/write routines compatibility routines for V */

#include <Venviron.h>
#include <Vio.h>
#include "/usr/include/errno.h"

char _sobuf[1];         /* Some programs call setbuf(fout, _sobuf) */

#define NFDS 30 /* Number of Unix file descriptors */
static File* _unix_fds[NFDS];
static int _unix_stdinit;

/*
 * stdin, stdout, and stderr can't be initialized statically so each
 * compatibility routine has to check if they've been set
 */
#define stdinit() \
    if (!_unix_stdinit) \
    { \
	_unix_fds[0] = stdin; _unix_fds[1] = stdout; _unix_fds[2] = stderr; \
	_unix_stdinit = 1; \
    }

static _unix_umask = 0002;	/* Used for umask() */

/* Translate V to Unix error codes. */
/* This table is by no means complete, and some of the entries can
   probably be improved on. */
struct errtrans
{
    SystemCode v_error;
    int     unix_errno;
} _V_to_unix_errors[] =
{
/*     V error code      Unix error code      */
    {ILLEGAL_REQUEST,       EINVAL},
    {NO_REPLY,              EIO},
    {BAD_ARGS,              EINVAL},
    {BAD_BUFFER,            EFAULT},
    {NO_SERVER_RESOURCES,   ENFILE},
    {NO_MEMORY,             ENOMEM},
    {NO_PERMISSION,         EACCES},
    {NOT_FOUND,             ENOENT},
    {BUSY,                  ETXTBSY},
    {TIMEOUT,               EIO},
    {0, 0}          /* End of list */
};

int errno;              /* Unix error code is put here */

/* Note: this handles only old-style open calls */

/*
 * We can't #include "/usr/include/sys/file.h" because of conflicting names
 * (e.g. FREAD) in V include files.
 */

open(filename, type, protmode)
    char *filename;
    int type;
    int protmode;	/* Protection mode for new files */
{
    unsigned short mode;
    SystemCode error;
    register int fd;
    File *f;

    stdinit();

    /* Check first if we have a file descriptor free */
    for (fd = 0; fd < NFDS; fd++)
    {
	if (_unix_fds[fd] == 0)
	    goto gotit;
    }
    return EMFILE;  /* Too many files open */

gotit:
    if (type & (01000 | 02000)) /* Check for O_CREAT or O_TRUNC bits */
	mode = FCREATE;	/* Create the file if it doesn't exist */
    else
	switch(type & 0x3)		/* Check read/write bits */
	  {
	    case 0:
		mode = FREAD;
		break;
	    case 1:
		mode = FAPPEND;
		break;
	    case 2:
		mode = FMODIFY;
		break;
	    default:
		errno = EINVAL;
		return -1;
          }
    f = Open(filename, mode, &error);
    if (error == OK)
    {
	_unix_fds[fd] = f;
	if (mode == FCREATE)	/* If we're creating the file, */
	    chmod(filename, protmode & ~_unix_umask); /* set the protection */
	return fd;
    }
    unix_errno(error);		/* Translate to Unix error number */
    return -1;
}

static unix_errno(error)
    register SystemCode error;
{
    /* Translate a V error to a Unix error, and set the global errno */

    register struct errtrans *p;
    
    /* Figure out which Unix error code to return */
    errno = EIO;    /* Catch-all error code as default */
    for (p = _V_to_unix_errors; p->v_error != 0; p++)
    {
	if (p->v_error == error)
	{
	    errno = p->unix_errno;
	    break;
	}
    }
}


close(fd)
    int fd;
{
    File *f;

    stdinit();
    if (fd < 0 || fd >= NFDS || (f = _unix_fds[fd]) == 0)
	return EBADF;
    Close(f);           /* Close the file */
    _unix_fds[fd] = 0;  /* Mark it as being closed */
}


read(fd, buf, len)
    int fd;
    char buf[];
    int len;
{
    int c;
    register char *cp;      /* Pointer to next char in buffer */
    register char *end;     /* Pointer to end of buffer */
    register File *f;

    stdinit();
    if (fd < 0 || fd >= NFDS || (f = _unix_fds[fd]) == 0)
	return EBADF;

    for (cp = buf, end = &buf[len]; cp < end;)
    {
	if ((c = getc(f)) == EOF)
	    break;
	*cp++ = c;
    }
    return cp - buf;    /* Return number of bytes read */
}


write(fd, buf, len)
    int fd;
    char buf[];
    int len;
{
    int c;
    register char *cp;      /* Pointer to next char in buffer */
    register char *end;     /* Pointer to end of buffer */
    register File *f;

    stdinit();
    if (fd < 0 || fd >= NFDS || (f = _unix_fds[fd]) == 0)
	return EBADF;

    for (cp = buf, end = &buf[len]; cp < end; cp++)
    {
	if (putc(*cp, f) == EOF)
	    break;
    }
    Flush(f);
    return cp - buf;    /* Return number of bytes written */
}



long
lseek(fd, offset, origin)
    int fd;
    long offset;
    int origin;
{
    File *f;

    stdinit();
    if (fd < 0 || fd >= NFDS || (f = _unix_fds[fd]) == 0)
	return EBADF;
    if (!Seek(f, offset, origin))
    {
	errno = EINVAL;
	return -1;
    }
    return BytePosition(f);
}

creat(filename, protmode)
    char *filename;
    int protmode;
{
    return open(filename, 01000, protmode);
}


umask(numask)
    int numask;
{
    int omask;

    omask = _unix_umask;
    _unix_umask = numask;
    return omask;
}





#include <Vioprotocol.h>
#include <Vnaming.h>
#include "Vdirectory.h"
#include "/usr/include/sys/types.h"
#include "/usr/include/sys/stat.h"

#define MAX_NAME_LEN 255

fstat(fd, statbuf)
    int fd;
    struct stat *statbuf;
/*
 * Do a READ_DESCRIPTOR on the file instance and copy the data
 * into the Unix structure;
 */
{
    Message msg;
    register DescriptorRequest *req = (DescriptorRequest*)msg;
#define reply	((DescriptorReply*)req)

    ArbitraryDescriptor	desc;
    File *f;

    stdinit();
    if (fd < 0 || fd >= NFDS || (f = _unix_fds[fd]) == 0)
    {
	errno = EBADF;
	return -1;
    }

    /* format the request */
    req->requestcode = READ_DESCRIPTOR;
    req->fileid = FileId(f);
    req->nameindex = 0;
    req->dataindex = 0;
    req->segmentptr = (char *)&desc;
    req->segmentlen = sizeof desc;

    /* route the request to the appropriate name server */
    NameSend( req );

    if ( reply->replycode != OK )
    {
	unix_errno(reply->replycode);  /* Translate to Unix error number */
	return -1;
    }

    /* Check that this is a Unix file... otherwise results won't
       make much sense. */
    if (desc.e.descriptortype != UNIXFILE_DESCRIPTOR)
    {
	errno = EISDIR;
	return -1;
    }

    vdir_to_stat(&desc.u, statbuf); /* Copy data to statbuf */
    return 0;
#undef reply
}


stat(filename, statbuf)
    char *filename;
    struct stat *statbuf;

/*
 * Do an NREAD_DESCRIPTOR on the file name and copy the data
 * into the Unix structure;
 */
{
    Message msg;
    register DescriptorRequest *req = (DescriptorRequest*)msg;
#define reply	((DescriptorReply*)req)

    /* save the name and descriptor in a box */
    struct 
      { 
        ArbitraryDescriptor	desc;
				/* note this has to be longest name possible */
	char			name[MAX_NAME_LEN+1];
      } box;

    File *f;
    int namelen;

    stdinit();
    namelen = strlen(filename);
    if (namelen > MAX_NAME_LEN)
    {
	errno = ENAMETOOLONG;
	return -1;
    }

    strcpy(box.name, filename);

    /* format the request */
    req->requestcode = NREAD_DESCRIPTOR;
    req->nameindex = (char *)(box.name) - (char *)&box;
    req->dataindex = 0;
    req->fileid = 0;	/* none */
    req->segmentptr = (char *)&box;
    req->segmentlen = sizeof box.desc + namelen + 1;

    /* route the request to the appropriate name server */
    NameSend( req );

    if (reply->replycode != OK)
    {
	unix_errno(reply->replycode); /* Translate to Unix error number */
	return -1;
    }

    /* Check that this is a Unix file... otherwise results won't
       make much sense. */
    if (box.desc.e.descriptortype != UNIXFILE_DESCRIPTOR)
    {
	errno = EISDIR;
	return -1;
    }

    vdir_to_stat(&box.desc.u, statbuf);	/* Copy data to statbuf */
    return 0;
#undef reply
}


vdir_to_stat(desc, statbuf)
    register UnixFileDescriptor	*desc;
    register struct stat *statbuf;
/*
 * Copy the data from the V unix-descriptor to the stat buf.  These
 * two are very similar but not enough to just do a bcopy().  In
 * particular, st_ino is not the same length.
 */

{
    clear((char *)statbuf, sizeof *statbuf);  /* Zero any unused fields */
    statbuf->st_dev = desc->st_dev;
    statbuf->st_ino = desc->st_ino;
    statbuf->st_mode = desc->st_mode;
    statbuf->st_nlink = desc->st_nlink;
    statbuf->st_uid = desc->st_uid;
    statbuf->st_gid = desc->st_gid;
    statbuf->st_rdev = desc->st_rdev;
    statbuf->st_size = desc->st_size;
    statbuf->st_atime = desc->st_atime;
    statbuf->st_mtime = desc->st_mtime;
    statbuf->st_ctime = desc->st_atime;
}


chmod(filename, protmode)
    char *filename;
    short protmode;
/*
 * Do an NWRITE_DESCRIPTOR on the file name.
 */
{
    Message msg;
    register DescriptorRequest *req = (DescriptorRequest*)msg;
#define reply	((DescriptorReply*)req)

    /* save the name and descriptor in a box */
    struct 
      { 
        ArbitraryDescriptor	desc;
				/* note this has to be longest name possible */
	char			name[MAX_NAME_LEN+1];
      } box;

    File *f;
    int namelen;

    stdinit();
    namelen = strlen(filename);
    if (namelen > MAX_NAME_LEN)
    {
	errno = ENAMETOOLONG;
	return -1;
    }

    strcpy(box.name, filename);

    clear((char *)&box.desc, sizeof box.desc);
    box.desc.u.descriptortype = UNIXFILE_DESCRIPTOR;

    /*
     * Strictly speaking, we should read the descriptor first so all the
     * other fields will remain unchanged.  However, the Unix V server
     * currently sets only the mode field.  Here all the other fields are
     * made 0 just in case.
     */
    box.desc.u.st_mode = protmode;

    /* format the request */
    req->requestcode = NWRITE_DESCRIPTOR;
    req->nameindex = (char *)(box.name) - (char *)&box;
    req->dataindex = 0;
    req->fileid = 0;	/* none */
    req->segmentptr = (char *)&box;
    req->segmentlen = sizeof box.desc + namelen + 1;

    /* route the request to the appropriate name server */
    NameSend( req );

    if (reply->replycode != OK)
    {
	unix_errno(reply->replycode); /* Translate to Unix error number */
	return -1;
    }

    return 0;
#undef reply
}


rename(oldname, newname)
    char *oldname, *newname;
{

    Message msg;
    register NameRequest *req = (NameRequest*)msg;
#define reply	((DescriptorReply*)req)

    /* Leave room for both names plus terminating nulls */
    char	names[MAX_NAME_LEN + MAX_NAME_LEN + 2];
    int		oldnamelen, newnamelen;

    /*
     * The request expects to get the old name followed by a null
     * followed by the new name.
     */

    oldnamelen = strlen(oldname);
    newnamelen = strlen(newname);
    if (oldnamelen > MAX_NAME_LEN ||
        newnamelen > MAX_NAME_LEN)
    {
	errno = ENAMETOOLONG;
	return -1;
    }
    strcpy(names, oldname);	/* strcpy will add a null at the end */
    strcpy(&names[oldnamelen+1], newname);

    req->requestcode = RENAME_FILE;
    req->nameptr = names;
    req->namelength = oldnamelen + newnamelen + 2; /* Include 2 nulls */
    req->nameindex = 0;
    req->unspecified[0] = oldnamelen;         /* Length of old name */

    NameSend(req);

    if (reply->replycode != OK)
    {
	unix_errno(reply->replycode); /* Translate to Unix error number */
	return -1;
    }

    return 0;
#undef reply
}



/* The Unix V server doesn't have any way to make links. */
/* KLUDGE */

link(filename, newname)
    char *filename, *newname;
{
    char cmdstring[200];

    sprintf(cmdstring, "ln %s %s", filename, newname);
    system(cmdstring); 		/* Blech */
    return 0;  /* ugh */
}
