/*
 * _start.s
 *
 * Assembly code to get a new team started off on the right foot.
 *
 * The initial team space layout is:
 *   | code | data | bss | istack | TEB |
 * where	code=	program code space
 *      data=   initialized data
 *	bss=	uninitialized data (actually initialized to 0, in this code)
 *      istack= initial stack space; used for the root stack
 *		until the TEB has been unpacked.
 *	TEB=	team environment block; contains the command line args,
 *		environment variables, etc.
 *
 * This assembly code points the stack into the istack space, calls
 *   TeamRoot to initialize the standard I/O and unpack the TEB,
 *   then resets the stack to the size called for by RootStackSize,
 *   and finally calls main().  If main() returns, we call exit(0).
 *
 * The team layout when it starts execution is:
 *   | code | data | bss | root stack | heap |
 * The root process's PerProcess area is at the start of its stack, as
 *   with all PerProcess areas.
 */

	.set    _INIT_STACK_SIZE, 1024

	.data
BssCleared:
	.word	0  /* flag indicating whether bss has already been cleared */

	.text
	.globl	__start
__start:

/*
 * First, set Frame and Argument Pointers to 0, so stackdump routines know
 *   when to terminate.  They seemed to survive without this, but it's nice
 *   to be certain.
 */
	clrq	ap		/* fp is next register */

/* Fetch the root message. */

	moval   __RootMsg, r2
	movl    r3, (r2)+       /* Stored in r3-r10 */
	movl    r4, (r2)+
	movl    r5, (r2)+
	movl    r6, (r2)+
	movl    r7, (r2)+
	movl    r8, (r2)+
	movl    r9, (r2)+
	movl    r10, (r2)+

/*
 * Temporarily position the stack pointer at _end + _INIT_STACK_SIZE so we
 *   are using only the small istack area, to be sure the stack doesn't
 *   smash the TEB or go past the end of our team space.
 */
	moval   _end + _INIT_STACK_SIZE, sp

/* Zero the bss segment */

	tstw	BssCleared	/* Check if program was dumped */
	bneq	1f
	subl3   $_edata, $_end, r0  /* Length of bss */
	pushl	r0
	pushal	_edata
	calls	$2, _clear	/* clear(&_edata, &_end - &_edata) */
	incw	BssCleared	/* Don't do it next time (if there is one) */

/* Call TeamRoot() to do the rest of the setup */

1:
	calls   $0, _TeamRoot

/* Move the end of the stack where it belongs */

	moval   _end, sp
	addl2   _RootStackSize, sp

/* Call main(argc, argv) */

	pushl   _Argv
	pushl   _Argc
	calls   $2, _main

/* Call exit() with the value returned by main(). */

	pushl   r0
	calls   $1, _exit

/* Die with a privilege violation if exit() returns -- it isn't supposed to! */

	bugl    $0x911
