/*
 * Team root function, called when program starts execution.
 *   This code is independent of the differences between VAX
 *   and 68000, but may need to be scrutinized when porting to
 *   other systems.
 */

#include "Venviron.h"
#include "Vgroupids.h"
#include "Vio.h"
#include "Vnaming.h"
#include "Vteams.h"
#include "Vprocess.h"

#ifdef VAX
#define _end end
#endif VAX

extern File *OpenFile();
extern char *malloc(), *strsave();
extern int RootStackSize;
extern char _end;

/* Align pointer p to the next b-byte boundary, assuming b is a power of 2. */
#define align(p, b) (  (p) + ( -(int)(p) & ((b)-1) )  )

PerProcessArea *PerProcess;
Message _RootMsg = { 0 };		/* Force to data segment */
RootMessage *RootMsg = (RootMessage *) _RootMsg;
ProcessId TeamCreator;
ProcessId Kernel_Process_Pid;
int Argc;
char **Argv;
EnvironmentVariable *EnvVars;
NameCacheEntry *NameCache;

TeamRoot()
    /*
     * TeamRoot() is an initialization function executed by the root
     *   process of a V team.
     * It is called by assembly code that zeroes the bss (uninitialized data)
     *   area, copies the root message (initial reply from the team's creator)
     *   from registers to global storage, and initializes the stack.
     * The C code below sets up the root PerProcess area, including the 
     *   standard input and output, initializes the environment variables 
     *   and name cache, etc.
     */
  {
    Processor_state state;
    register char *teb, **av, *p, *q;
    register unsigned long i;

    /***** First find the local kernel server *****/
    Kernel_Process_Pid = GetPid(KERNEL_PROCESS, LOCAL_PID);
    TeamCreator = Creator(0);

    /***** Tell the kernel what our PerProcess pointer is *****/
    PerProcess = (PerProcessArea *) align(&_end, 4);
    ReadProcessState(0, &state);
    state.perProcessLoc = (Unspec **) &PerProcess;
    state.perProcess = (Unspec *) PerProcess;
    WriteProcessState(0, &state);

    PerProcess->env = &EnvVars;
    PerProcess->namecache = &NameCache;

    /***** Unpack team environment block, if any *****/
    if (RootMsg->envblock)
      {
	/* Point to start of TEB */
	teb = RootMsg->envblock;
	i = *(unsigned long *)teb;	/* Get TEB size */
	teb += sizeof(unsigned long);
    
        /* Be sure stack size is big enough to clear TEB, then
         *  set team size to end of stack area so that malloc() can
	 *  start the heap from there.  (We are still using the
	 *  initial small stack for now.)
	 */
	if ((char *)PerProcess + RootStackSize < teb + i)
	    RootStackSize = teb + i - ((char *)PerProcess);
	SetTeamSize(0, (char *) PerProcess + RootStackSize);
        PerProcess->stackSize = RootStackSize;

	/* Get argc and argv */
	Argc = i = *(unsigned long *)teb;
	teb += sizeof(unsigned long);
	Argv = av = (char **) malloc((Argc+1)<<2);
	while (i--)
	  {
	    p = teb;
	    while (*teb++) ;
	    *av = (char *) malloc(teb - p);
	    strcpy(*av, p);
	    av++;
	  }
	 *av = NULL;

        /* Get environment */
	teb = align(teb, 4);
	i = *(unsigned long *) teb;
	teb += sizeof(unsigned long);
	while (i--)
	  {
	    p = teb;
	    while (*teb++) ;
	    setenv(p, teb);
	    while (*teb++) ;
	  }

	/* Get name cache */
	teb = align(teb, 4);
	i = *(unsigned long *) teb;
	teb += sizeof(unsigned long);
	while (i--)
	  {
	    p = teb + 2*sizeof(ContextPair) + 
			sizeof(unsigned short); /* prefix */
	    q = p;
	    while (*q++) ;		        /* truename */

	    /* NameCacheAdd(prefix, length, from, to, truename, flags) */
	    NameCacheAdd( p, q-p-1, *(ContextPair *)teb,
			   *(ContextPair *)(teb + sizeof(ContextPair)), q,
			   *(unsigned short *)(teb + 2*sizeof(ContextPair)) );
	    teb = q;
	    while (*teb++);
	    teb = align(teb, 4);
	  }

        /* Get current context */
	PerProcess->ctx = *(ContextPair *)teb;
	teb += sizeof(ContextPair);
	p = teb;
	while (*teb++) ;
	PerProcess->ctxname = q = (char *) malloc(teb-p);
	strcpy(q, p);
      }
    else
      {
        /* Set team size to end of stack area so that malloc() can
	 *  start the heap from there.  (We are still using the
	 *  initial small stack for now.)
	 */
	SetTeamSize(0, (char *) PerProcess + RootStackSize);
        PerProcess->stackSize = RootStackSize;

	/* Set current context to default */
	PerProcess->ctx.pid = VSTORAGE_SERVER_GROUP;
	PerProcess->ctx.cid = PUBLIC_CONTEXT;
	PerProcess->ctxname = strsave("[sys]");
      }

    /***** Finish setting up PerProcess area *****/

    /* Setup open files */
    stdin = OpenFile( RootMsg->stdinserver, RootMsg->stdinfile, FREAD 
	+ (RootMsg->rootflags&RELEASE_INPUT ? FRELEASE_ON_CLOSE : 0), NULL );
    stdout = OpenFile( RootMsg->stdoutserver, RootMsg->stdoutfile,
	(RootMsg->rootflags&STDOUT_APPEND ? FAPPEND : FCREATE)
	+ (RootMsg->rootflags&RELEASE_OUTPUT ? FRELEASE_ON_CLOSE : 0), NULL );

    /* Set stderr to same file as stdout if same instance is specified */
    if( RootMsg->stdoutserver == RootMsg->stderrserver &&
	    RootMsg->stdoutfile == RootMsg->stderrfile )
      {
	stderr = stdout;
      }
    else
      {
	stderr = OpenFile( RootMsg->stderrserver, RootMsg->stderrfile,
	    (RootMsg->rootflags&STDERR_APPEND ? FAPPEND : FCREATE) +
	    (RootMsg->rootflags&RELEASE_ERR ? FRELEASE_ON_CLOSE : 0), NULL );
      }

  }
