/* #define NAMING_STATS */

/*
 * NameSend() and GetContextId()
 *
 * NameSend() is used to send off a naming protocol request using
 *  the naming cache in the standard way.  It accepts the message to
 *  be sent as its argument.  As with Send, NameSend returns
 *  the pid of the replier, and modifies its argument to contain the first
 *  reply message received.  The caller can invoke GetReply() upon
 *  return if additional replies are anticipated.
 *
 * Algorithm:  First checks whether the name is relative or absolute,
 *  and fills in the proper (serverpid, contextid) accordingly.
 *  Then checks if the name is to be mapped in a single-server context.
 *  If so, it is sent off with no further ado.  If not, the name
 *  is looked up in the team name prefix cache.  If it misses in the cache, 
 *  a QueryName request is sent off to get more cache information, and the 
 *  resulting information is used to (hopefully) unicast the request.
 *
 * Warning: the string passed in the request message must be null-terminated;
 *  req->namelength is neither examined nor set by this routine.
 */

#include "Vnaming.h"
#include "Vgroupids.h"
#include "Vnamecache.h"

#ifdef NAMING_STATS 
NamingStats NS;
#endif NAMING_STATS

extern NameCacheEntry *NameCacheLookup();
extern NameCacheEntry *NameCacheAdd();
extern SystemCode NameCacheDelete();
/*forward*/ SystemCode GetContextId();

ProcessId NameSend(req)
    register NameRequest *req;
  {
#   define reply ((ContextReply *) req)
    register NameCacheEntry *ce = NULL;
    register char *p;
    ContextPair context;
    register startindex, relname = 0;
    register ProcessId replier;
    Message msg;
    register ContextRequest *qreq = (ContextRequest *) msg;
#   define qreply ((ContextReply *) msg)	    
    SystemCode err;
    NameRequest savereq;
#ifdef NAMING_STATS 
    register cacheHitThisTry;

    if (NS.time == 0)
      {
	unsigned long clicks;

	NS.time = GetTime(&clicks);
	NS.time = NS.time * CLICKS_PER_SEC + clicks;
      }
    NS.calls++;
#endif NAMING_STATS

    /* Save original message in case of retries */
    savereq = *req;

retry:
#ifdef NAMING_STATS 
    NS.tries++;
    cacheHitThisTry = 0;
#endif NAMING_STATS
    p = &req->nameptr[req->nameindex];
    if (p[0] == ROOT_ESCAPE_CHAR)
      {
        /* An absolute name.  Strip root escape character and use
	 * the root context. */
	p++;
	context.pid = VCSNH_SERVER_GROUP;
	context.cid = GLOBAL_ROOT_CONTEXT;
      }
    else
      {
	/* A relative name.  Just use the current context. */
#ifdef NAMING_STATS 
	if (!relname) NS.relative++;
#endif NAMING_STATS
	relname = 1;
	context = PerProcess->ctx;
      }

    /* If we are in a multi-manager context (context.pid is a non-local
     *  group id) and the remaining name suffix is non-null, do a cache 
     *  lookup.  Might need to do more than one depending on just what is
     *  in the cache.
     */
    while (*p && (context.pid&(GROUP_ID_BIT|LOCAL_GROUP_BIT)) == GROUP_ID_BIT)
      {
	/* Do cache lookup */
#ifdef NAMING_STATS 
	NS.lookups++;
#endif NAMING_STATS
        ce = NameCacheLookup(p, context);
        if (ce != NULL)
	  {
   	    /* Cache hit, on entry "ce". */
#ifdef NAMING_STATS 
	    NS.lookupSuccesses++;
	    cacheHitThisTry++;
#endif NAMING_STATS

	    /* If the prefix length is zero and the new context
	     *  is the same as the old one, we can't avoid
	     *  doing a group Send for this name.
	     */
	    if (ce->namelength == 0 && ContextPairEqual(context, ce->to))
	    	break;

	    /* Skip past the match and move to the new context. */
	    p += ce->namelength;
	    if (*p) p++;	/* skip delimiter if not end of string  */
	    context = ce->to;	

	    /* Temporary feature to allow access to old GetPid based
	     *  CSNH servers:
	     */
	    if (ce->flags & LOGICAL_PID)
	      {
#ifdef NAMING_STATS 
		NS.getPids++;
#endif NAMING_STATS
	        context.pid = GetPid(context.pid, ANY_PID);
	      }
	    continue;
	  }

	/* Prevent special name "all" in multi-manger contexts 
         *  from being queried or cached.  Doing this test here 
         *  seems pretty kludgey.
	 */
	if (strncmp(p, "all", 3) == 0 && IsDelimiter(p[3])) break;

	/* Cache miss.  Invoke the QUERY_NAME operation to find out 
	 *  which server(s) implement the named object (if any).  The 
	 *  name does not have to represent a valid object, as long as 
	 *  a name prefix is recognized by at least one server.
	 */
#ifdef NAMING_STATS 
	NS.queries++;
	if (cacheHitThisTry != 0) NS.partialHits++;
#endif NAMING_STATS
	qreq->requestcode = QUERY_NAME;
	qreq->namecontextid = context.cid;
	qreq->nameptr = p;
	qreq->nameindex = 0;
	qreq->namelength = strlen(p);
	
	replier = Send(qreq, context.pid);
    
	if (qreply->replycode != OK)
	  {
	    /* We assume that if the query fails (replycode != OK), that
	     *  a Send with the same name specified will also fail.
	     */
#ifdef NAMING_STATS 
	    NS.queryFailures++;
	    if (qreply->replycode == KERNEL_TIMEOUT) NS.queryTimeouts++;
#endif NAMING_STATS
	    reply->replycode = qreply->replycode;
	    goto did_send;
	  }
      
	NameCacheAdd(p, qreply->nameindex, context, 
		     qreply->context, NULL, 0);
     
	/* Skip past the prefix mapped by the query operation,
	 *  and move to the new context.
	 */
#ifdef NAMING_STATS 
	if (qreply->nameindex == 0 && 
	    ContextPairEqual(context, qreply->context)) NS.queryNoHelp++;
#endif NAMING_STATS
	p += qreply->nameindex;
	context = qreply->context;

	/* Do at most one query operation */
	break;
      }

    /* Cache plus query op have done all they can; time to Send */
    req->nameindex = p - req->nameptr;
    req->namecontextid = context.cid;
    replier = Send(req, context.pid);

did_send:
    switch (reply->replycode)
      {
        case OK:
#ifdef NAMING_STATS 
	  NS.successes++;
	  /* fall through */
#endif NAMING_STATS 
	default:
	  return replier;

        case KERNEL_TIMEOUT:
#ifdef NAMING_STATS 
	  NS.timeouts++;
	  /* fall through */
#endif NAMING_STATS 
	case NONEXISTENT_PROCESS:
	case INVALID_CONTEXT_ID:
	case NOT_HERE:
	  if (ce == NULL)
	    {
	      /* Didn't use cache */
	      if (relname)
	        {
		  /* Try remapping the current context
		   *  If we get a different result this time, maybe that
		   *  will fix the problem.
		   */
#ifdef NAMING_STATS 
		  NS.remapCurrCtx++;
#endif NAMING_STATS 
		  if ( PerProcess->ctxname != NULL &&
		       PerProcess->ctxname[0] == ROOT_ESCAPE_CHAR )
		    {
		      err = GetContextId(PerProcess->ctxname, &context);
		    }
		  if (  err == OK && 
		       !ContextPairEqual(context, PerProcess->ctx) )
		    {
		      PerProcess->ctx = context;
		      *req = savereq;
		      goto retry;
		    }
#ifdef NAMING_STATS 
		  NS.remapCurrCtxFail++;
#endif NAMING_STATS 
		}	

	      /* Not a relative name, or GetContextId didn't help */
	      return replier;
	    }

          if (!(ce->flags&DONT_FLUSH))
	    {
	      /* The cache entry we used may well have been stale.
	       * Flush it.
	       */
#ifdef NAMING_STATS 
	      NS.stale++;
#endif NAMING_STATS 
	      NameCacheDelete(ce);
	      req->nameindex = startindex;

	      /* Now retry.  We should never get into an infinite retry
	       *  loop because we will eventually either succeed, or
	       *  flush all the flushable cache entries we used, and then
	       *  either fail using a non-flushable entry or fail without
	       *  having hit at all.
	       */
	      *req = savereq;
	      goto retry;  
	    }

	  if (ce->flags&ALIAS)
	    {
	      /* Try remapping the true name for this alias.
	       *  If we get a different result this time, maybe that
	       *  will fix the problem.
	       */
#ifdef NAMING_STATS 
	      NS.remapAlias++;
#endif NAMING_STATS 
	      err = GetContextId(ce->truename, &context);
	      if (err == OK && !ContextPairEqual(context, ce->to))
	        {
		  ce->to = context;
	          *req = savereq;
		  goto retry;
		}
#ifdef NAMING_STATS 
	      NS.remapAliasFail++;
#endif NAMING_STATS 
	    }

	  /* Can't retry; just return what we have */
	  return replier;
      }

#undef reply
#undef qreply
  }


/*
 * getcontextid.c
 *
 * Function-call interface to GET_CONTEXT_ID message.  Interprets the 
 *   given name in the current context and returns a corresponding
 *   serverpid/contextid pair.
 */
#include "Vnaming.h"

SystemCode GetContextId(name, context)
    char *name;
    ContextPair *context;
  {
    Message msg;
    register ContextRequest *req = (ContextRequest *) msg;
#   define reply ((ContextReply *)req)
    ProcessId pid;

    req->requestcode = GET_CONTEXT_ID;
    req->nameindex = 0;
    req->nameptr = name;
    req->namelength = strlen( name );

    /* Interpret the given name in the current context */
    NameSend(req);

    if (reply->replycode != OK) return (reply->replycode);

    /* Return context identifier */
    *context = reply->context;

    return (OK);

#   undef reply;
  }
