/*
 * V-System I/O library
 * Copyright (c) 1985, Board of Trustees, Leland Stanford Junior University
 *
 * Seek
 */

#include "Vioprotocol.h"

#define BufferValid(fad) \
	( fad->currsize != MAXBYTES || \
	  fad->writeLimit > fad->buffer && fad->current > fad->buffer )

#define BufferModified(fad) \
	( (fad->state & MODIFIED_BUFFER) || \
	  fad->writeLimit > fad->buffer && fad->current > fad->buffer )


int Seek( fad, offset, origin ) 
  register File *fad; 
  int offset; 
  unsigned origin;
  /*
   * Seek's primary function is to reset the current byte position
   *   of the specified FAD.  It also tries to get the new current
   *   block into the buffer, and normalizes some things.  To be specific:
   *
   * Seek attempts to...
   *   Make specified position current.
   *   Clear EOF_BYTE.
   *   Set MODIFIED_BUFFER iff current block is modified.
   *   Set (lastblock, lastbytes) to end of file.
   *   Force (new) current position to be normalized.
   *   Kludge: for string files, view buffer as always being valid for
   *     all of block 0 of the file and never needing to be read or written.
   *   If moving to a different block or seeking in FLUSH mode,
   *	and current block is modified, write it.
   *   Force current block to be valid if file is READABLE and block is in
   *    file (i.e., 0 <= block < lastblock + (lastbytes != 0) ).
   *	May require reading a block.
   *   Set currsize to size of block.
   *   Optimization:  Set writeLimit != buffer if current == buffer and
   *    WRITEABLE.  (Without this optimization we would always set 
   *    writeLimit = buffer.)
   *   Return non-zero on success.
   * ...else fails:
   *   If called when fad is in block mode, call abort().
   *   Should perhaps abort if user tries to seek on a STREAM, but
   *    currently doesn't because other I/O routines use Seek on
   *    streams in various (legitimate) ways.
   *   On other failures (e.g., I/O errors), set EOF_BYTE and lastexception,
   *	and set readLimit = writeLimit = buffer.  If a Read fails, set
   *	currsize to the amount read anyway; if a Write fails, reset 
   *	MODIFIED_BUFFER anyway.  Return 0.
   */
  {
    register unsigned newblock;
    register int newindex;
    register int blocksize = (int) fad->blocksize;
        /* must be signed to cause signed division below */
    register unsigned bytesdone, m;

    /* Detect a common programmer error:
     *   use of byte-mode routines on a fad created in block mode */
    if ( fad->type & FBLOCK_MODE )
	abort("Seek called in block mode");

    /* Clear error flag in fad.
     *  Set again later if another error occurs. */
    fad->state &= ~EOF_BYTE;

    /*
     * If BufferModified(), set MODIFIED_BUFFER and update
     *  (lastblock, lastbytes) and currsize.
     * Must be done here so that seeking relative to FILE_END 
     *  will work correctly.
     */
    if (BufferModified(fad))
      {
	fad->state |= MODIFIED_BUFFER;

	/* Update currentEOF if needed */
	if (  ( fad->block > fad->lastblock ) ||
	      ( fad->block == fad->lastblock && 
	        (fad->current - fad->buffer) > fad->lastbytes )  )
	  {
	    fad->lastblock = fad->block;
	    fad->lastbytes = fad->current - fad->buffer;
	  }

	/* Compute new currsize.  Note: since buffer is modified,
	 *  we can assume BufferValid is true in this computation.
	 */
	m = ( (fad->writeLimit < fad->current) ? 
		fad->writeLimit : fad->current ) - fad->buffer;
	if (fad->currsize == MAXBYTES || fad->currsize < m)
		fad->currsize = m;
      }

    /* Calculate the new position in the file in terms of the
     * block number and position within that block.
     */
    switch( origin )
      {
	case ABS_BLK:
	  newblock = offset;
	  newindex = 0;
	  break;

	case REL_BYTE:
	  newblock = fad->block;
	  newindex = (fad->current - fad->buffer) + offset;
	  break;

	case ABS_BYTE:
	  newblock = 0;
	  newindex = offset;
	  break;

	case FILE_END:
	  newblock = fad->lastblock;
	  newindex = fad->lastbytes + offset;
	  break;

	case FLUSH:
	  /* No change in position */
	  newblock = fad->block;
	  newindex = fad->current - fad->buffer;
	  break;

	default:
	  abort("Seek: invalid origin");
      }

    /* Adjust newindex such that 0 <= newindex < blocksize */
    newblock += newindex/blocksize;
    newindex %= blocksize;
    if ( newindex < 0 )
      {
	--newblock;
	newindex += blocksize;
      }

    /* Kludge for string files: buffer always contains all of
     *  block 0, with no need to ever call Read or Write.  Can
     *  never seek out of block 0.
     */
    if (fad->fileserver == 0)
      {
	fad->current = fad->buffer + newindex;
	return 1;
      }	  

    /* Write the block if needed */
    if ( (fad->state & MODIFIED_BUFFER) &&
         (origin == FLUSH || newblock != fad->block) )
      {
	fad->currsize = Write(fad, fad->buffer, fad->currsize);
	fad->state &= ~MODIFIED_BUFFER;
	if (fad->lastexception != OK)
	  {
	    /* I/O error; fail, considering write to have been done. */
	    /* Note that we haven't yet moved to the new position,
	     *  and that we have updated size(block) to be the amount
	     *  actually written. */
	    fad->state |= EOF_BYTE;
	    fad->readLimit = fad->writeLimit = fad->buffer;
	    return 0;
	  }
      }
    
    /*
     * Invalidate buffer if moving to a new block.
     */
    if ( newblock != fad->block )
      {
	/* Invalidate buffer.  Must do this here in case
	 *  calling process is destroyed during Read.
	 * Note: MODIFIED_BUFFER flag cannot be set at this point,
	 *  so we don't need to turn it off.
	 */
	fad->currsize = MAXBYTES;
	fad->readLimit = fad->writeLimit = fad->buffer;
      }

    /* Set current position to new position
     */
    fad->block = newblock;
    fad->current = fad->buffer + newindex;

    /* Try to read the new current block if file is readable,
     *  it is not already in the buffer, and it is not beyond
     *  the known end of file.  Else leave the buffer in its
     *  current state.
     */
    if ( (fad->type & READABLE) && !BufferValid(fad) &&
	 ( newblock < fad->lastblock ||
	    (newblock == fad->lastblock && fad->lastbytes != 0) ) )
      {
	/* Block is within file */
	fad->currsize = Read(fad, fad->buffer, blocksize);
    
	/* If the last block grew (or is no longer the last block),
	 * pretend that nothing has changed, to keep from getting
	 * confused.  A ClearEof will be required to read further,
	 * which will update lastblock & lastbytes
	 */
	if (fad->block == fad->lastblock && fad->currsize > fad->lastbytes)
	    fad->currsize = fad->lastbytes;
	
	if (fad->lastexception == END_OF_FILE)
	  {
	    /* Hit end, so set lastblock and lastbytes accordingly
	     *  (May be different from old values).  This allows
	     *  files to grow shorter, including the case where
	     *  the length was initially unknown.
	     */
	    fad->lastblock = fad->block;
	    fad->lastbytes = fad->currsize;
	  }
	else if (fad->lastexception != OK)
	  {
	    /* I/O error; fail.  Note that readLimit and writeLimit
	     *  are already set to fad->buffer at this point.
	     */
	    fad->state |= EOF_BYTE;
	    return 0;
	  }
      }

    /*
     * Update readLimit
     *  Note: currsize is up to date at this point, because (1) if
     *  we have moved to a new block without reading, the buffer
     *  was invalidated above; (2) if we have read a block, currsize
     *  was set to the amount read; (3) if we are in the same block,
     *  and buffer was modified when Seek was entered, currsize was 
     *  updated above; (4) if we are in the same block and buffer
     *  was not modified, currsize was (and remains) correct.
     */
    if ((fad->type & READABLE) && (fad->currsize != MAXBYTES))
        fad->readLimit = fad->buffer + fad->currsize;
    else
        fad->readLimit = fad->buffer;  /* (should be correct already) */
  
    /* Optimization:  Enable writing if WRITEABLE and current == buffer
     * If optimization not possible, set writeLimit = buffer
     */
    if ( (fad->type & WRITEABLE) && fad->current == fad->buffer )
      {
	/* Enable writing */
        if ( fad->block == fad->lastblock && (fad->type&FIXED_LENGTH) )
	    fad->writeLimit = fad->buffer + fad->lastbytes;
	else
	    fad->writeLimit = fad->buffer + fad->blocksize;
      }
    else
	fad->writeLimit = fad->buffer;
    
    return 1;
  }

