/*
 * V-System I/O library
 * Copyright (c) 1985, Board of Trustees, Leland Stanford Junior University
 *
 * Resynch
 * Clear any exception recorded in the given fad and resynchronize our
 *  local view of the associated file instance with that of the 
 *  server, including (lastblock, lastbytes) and nextblock to read.
 * If fad is not of type STREAM, the contents of the local buffer are
 *  discarded even if the buffer was modified and not yet rewritten.
 * If fad is of type STREAM, and our current file position would cause 
 *  us to violate the stream condition (always read nextblock or write 
 *  lastblock+1), reposition.  The contents of the local buffer
 *  are discarded only if repositioning is necessary.
 */
#include "Vioprotocol.h"

#define BufferValid(fad) \
	( (fad)->currsize != MAXBYTES || \
	  ( (fad)->writeLimit > (fad)->buffer && \
	    (fad)->current > (fad)->buffer ) )

Resynch( fad )
	File *fad;
  {
    Message msg;
    register QueryInstanceRequest *req = (QueryInstanceRequest *) msg;
    register QueryInstanceReply *reply = (QueryInstanceReply *) msg;

    if (fad->fileserver == 0) return;	/* for string files */

    req->requestcode = QUERY_INSTANCE;
    req->fileid = fad->fileid;

    /* Horrible hack to support the VGTS overloaded instance-id's --
     *  SHOULD GO AWAY AS SOON AS POSSIBLE!!!
     */
    req->filemode = (fad->type & WRITEABLE) ? FAPPEND : FREAD;

    Send( req, fad->fileserver );
    if( reply->replycode != OK )
      {
	fad->lastexception = reply->replycode;
	fad->state |= EOF_BYTE;
        abort("Resynch got %z from server", reply->replycode);
      }

    /* Adjust local view of end of file.
     */
    fad->lastblock = reply->filelastblock;
    fad->lastbytes = reply->filelastbytes;

    /* Clear exception record for this fad */
    fad->lastexception = OK;
    fad->state &= ~EOF_BYTE;

    /* If not a stream... */
    if ( !(fad->type & STREAM) )
      {
	/* Discard currently buffered block in case it
	 *  has changed without our knowledge.
	 */
	fad->state &= ~MODIFIED_BUFFER;
	fad->currsize = MAXBYTES;    
	fad->readLimit = fad->buffer;
	if (fad->current != fad->buffer) fad->writeLimit = fad->buffer;

	/* Done */
	return;
      }

    /* Check whether READABLE or WRITEABLE -- a STREAM fad cannot be both.
     */
    if (fad->type & READABLE)
      {
        /* If there's anything in the buffer, make it visible */
        if ( (fad->type & READABLE) && fad->currsize != MAXBYTES )
            fad->readLimit = fad->buffer + fad->currsize;

	/* Are we synchronized already? */
	if ( BufferValid(fad) ? fad->block != reply->filenextblock - 1
			      : fad->block != reply->filenextblock )
	  {
	    /* No; mark buffer as invalid and position
	     *  to start of next block to read.
	     */
	    fad->readLimit = fad->buffer;
	    fad->currsize = MAXBYTES;
	    fad->current = fad->buffer;
	    fad->block = reply->filenextblock;
	  }
      }
    else
      {
	/* Are we synchronized already? */
	if ( fad->block != reply->filelastblock + 1 )
	  {	
	    /* No; mark buffer as not modified and position
	     *  to start of next block to write.
	     */
	    fad->state &= ~MODIFIED_BUFFER;
	    fad->writeLimit = fad->buffer;
	    Seek(fad, reply->filelastblock + 1, ABS_BLK);
	  }
      }
  }
