/* Macros for semi-portably initializing bitmaps.
 * Example:
 * short X[16] = { B_16(_F,_C,_0,_F), B_16(...), ...};
 * - generates a 16*16 pattern whose top line is ******......****
 * The programmer defines the pattern in terms of big-endian nibbles,
 * and the assumption that '1's are black.
 * The macros will then translate into the correct pattern, depending on:
 * BIG_ENDIAN == 1 iff a big-endian machine, else 0.
 * FIXUP(x) can be used to invert the bits if needed.
 * I.e for a framebuffer where black==0, one could say:
 * #define FIXUP(x) (~(x))
 */

#ifnef BIG_ENDIAN
#ifdef LITTLE_ENDIAN
#define BIG_ENDIAN 0
#else
#define BIG_ENDIAN 1
#endif
#endif

#ifnef FIXUP
#define FIXUP(x) (x)
#endif

#define _0 /* 0000 */ (BIG_ENDIAN ?  0: 0)
#define _1 /* 0001 */ (BIG_ENDIAN ?  1: 8)
#define _2 /* 0010 */ (BIG_ENDIAN ?  2: 4)
#define _3 /* 0011 */ (BIG_ENDIAN ?  3:12)
#define _4 /* 0100 */ (BIG_ENDIAN ?  4: 2)
#define _5 /* 0101 */ (BIG_ENDIAN ?  5:10)
#define _6 /* 0110 */ (BIG_ENDIAN ?  6: 6)
#define _7 /* 0111 */ (BIG_ENDIAN ?  7:14)
#define _8 /* 1000 */ (BIG_ENDIAN ?  8: 1)
#define _9 /* 1001 */ (BIG_ENDIAN ?  9: 9)
#define _A /* 1010 */ (BIG_ENDIAN ? 10: 5)
#define _B /* 1011 */ (BIG_ENDIAN ? 11:13)
#define _C /* 1100 */ (BIG_ENDIAN ? 12: 3)
#define _D /* 1101 */ (BIG_ENDIAN ? 13:11)
#define _E /* 1110 */ (BIG_ENDIAN ? 14: 7)
#define _F /* 1111 */ (BIG_ENDIAN ? 15:15)

#define B_8(nibble0, nibble1) FIXUP( \
	BIG_ENDIAN ? (nibble0 << 4) + nibble1 : (nibble1 << 4) + nibble0)

#define B_16(nibble0, nibble1, nibble2, nibble3) FIXUP( \
	BIG_ENDIAN ? (nibble0<<12) + (nibble1<<8) + (nibble2<<4) + nibble3
	: (nibble3<<12) + (nibble2<<8) + (nibble1<<4) + nibble0))
