/* Internal support routines only - subject to change without notice! */

/* NOTE: "OpenProgFile()" is, however, used by the team server, when loading 
 * a postmortem debugger.  Thus, if the interface to "OpenProgFile()" is 
 * changed, then the team server must be changed accordingly.
 */

#include <Vio.h>
#include <Vquerykernel.h>
#include <Vnaming.h>

#define FALSE 0
#define TRUE 1

#define LineLength 128		/* Max size of a char. string buffers */

char DefaultProgram[] = "[bin]fexecute";

extern char *ProcSuffix[];

File *OpenProgFile(nameBuf, procFamily, error)
    char nameBuf[];
    unsigned char procFamily;
    SystemCode *error;
    /* Attempt to open a program file, given the candidate file name
     * (unsuffixed) in "nameBuf".  The file should be a version that can
     * execute on processors of type "procFamily"
     * (as defined in <Vquerykernel.h>).
     * If an appropriate program file can be opened, then the file pointer 
     * is returned, otherwise 0 is returned and an error code is returned in
     * "error".
     */
  {
    int rawNameSize;
    char *procSuffix;
    File *progFile;

    rawNameSize = strlen(nameBuf);
    procSuffix = ProcSuffix[procFamily >> PROC_FAMILY_SHIFT];

    /* First attempt to open a file with an appropriate suffix. */
    strncpy(&nameBuf[rawNameSize], procSuffix, LineLength - rawNameSize);
    progFile = Open(nameBuf, FREAD | FBLOCK_MODE, error);
    nameBuf[rawNameSize] = '\0';
    if (progFile != NULL)
	return(progFile);

    /* Now attempt to open the unsuffixed file. */
    progFile = Open(nameBuf, FREAD | FBLOCK_MODE, error);
    if (progFile != NULL)
      {
	/* Find out if this file runs on the type of machine that we want. */
	long magicNum;

	if ( Read(progFile, &magicNum, sizeof(long)) == sizeof(long) &&
	     ValidMagicNum(magicNum, procFamily) )
	    return(progFile);
	else
	  {
	    *error = NOT_FOUND;
	    Close(progFile);
	  }
      }

    return(NULL); /* no luck */
  }


int SearchPathMatch(argv, path, nameBuf, specificHost,
		    procFamily, progFile, /* meaningful only if specificHost */
		    foundProgFiles, /* meaningful only if !specificHost */
		    error)
    char *argv[]; char *path; char nameBuf[]; int specificHost;
    unsigned char procFamily; File **progFile;
    File *foundProgFiles[];
    SystemCode *error;
    /* Searches "path" to try to match a program given by "argv[0]".
     * The meaning of `match' varies slightly, depending on 
     * whether "specificHost" is set - i.e., on whether the execution 
     * host is known in advance.
     * If "specificHost" IS set, then "procFamily" is used, and a single 
     *     open program file is returned in "progFile".
     * If "specificHost" is NOT set, then an array of open program files is 
     *     returned in "foundProgFiles".
     * Apart from this, the path searching code is the same for both cases.
     * 0 is returned, and "error" is set, if no match can be found.
     */
  {
    int success = 0;
    int absProgName = 0;

    /* Check if an absolute command name was given: */
    if (*argv[0] == ROOT_ESCAPE_CHAR)
      {
	path = "./"; /* semi-kludge */
	absProgName = 1;
      }

    while (*path != '\0')
      {
	register char *name;
	register char lastchar;

	name = nameBuf;
    
	/* Copy the string name of the next directory to try into the 
	   nameBuf. */
	while ((*path != ' ') && (*path != '\0'))
	  {
	    *name++ = *path++;
	  }
	if ((lastchar = name[-1]) != '/' && lastchar != ']')
	    *name++ = '/'; /* ensures that path component ends correctly */
	*name = '\0';
	/* Check for "./", indicating the current context. */
	if (Equal(nameBuf, "./"))
	    nameBuf[0] = '\0';

	/* Append the cmd name to the directory name. */
	strncat(nameBuf, argv[0], LineLength - strlen(nameBuf) - 1);
    
	/* Check for a match: */
	if (specificHost)
	  {
	    *progFile = OpenProgFile(nameBuf, procFamily, error);
	    success = (*progFile != NULL);
	  }
	else /* arbitrary host */
	    success = FindMatch(foundProgFiles, nameBuf, error);
	if (success)
	    break; /* we found a file! */

	/* Continue the search... */
	while (*path == ' ')
	    path++;		/* Skip over the blank separator(s) between
			           directory entries in the search path. */
      }

    /* If we couldn't find a program file, then look for the default program. */
    if (!success && !absProgName)
      {
	strcpy(nameBuf, DefaultProgram);
	if (specificHost)
	  {
	    *progFile = OpenProgFile(nameBuf, procFamily, error);
	    success = (*progFile != NULL);
	  }
	else /* arbitrary host */
	    success = FindMatch(foundProgFiles, nameBuf, error);
      }

    return (success);
  }


FindMatchingProgs(foundProgFiles, progName, argv, path, error)
    File *foundProgFiles[];
    char **progName;
    char *argv[];
    char *path;		/* Search path to use for finding the program
			   file.  NULL indicates that the default should
			   be used. */
    SystemCode *error;
    /* Uses the search path "path" to find all versions of the program given
     * by "argv".  On exit, the array "foundProgFiles" will contain
     * pointers to open files for the versions that were found.
     * The full (but unsuffixed) program name is returned in "progName".
     *
     * Note: Eventually we will probably want to change this to return an 
     * array of file \names/, rather than pointers to open file descriptors.
     */
  {
    char nameBuf[LineLength];

    /* Check the search path, to find a set of matching program files: */
    SearchPathMatch(argv, path, nameBuf, 0 /* => arbitrary host. */,
		    0, NULL, /* These parameters are irrelevent for us. */
		    foundProgFiles,
		    error);

    *progName = nameBuf;
  }


static int FindMatch(foundProgFiles, nameBuf, error)
    File *foundProgFiles[];
    char nameBuf[];
    SystemCode *error;
    /* Tries to find matching program files, given the candidate file name 
     * in "nameBuf".  TRUE is returned iff matches are found.
     */
  {
    int rawNameSize;
    int success;
    File *unsuffixedProgFile;
    register int i;

    rawNameSize = strlen(nameBuf);

    for (i = 0; i < NUM_PROC_FAMILIES; ++i)
	foundProgFiles[i] = NULL;

    /* Determine which versions (if any) of the program exist. */
/*TEMPORARY!*/
    /* NOTE: This would ideally be done using pattern matching to find all 
     * suffixes of the file name given in "nameBuf".  However, we currently 
     * cannot perform such pattern matching efficiently.  Thus, as 
     * a \temporary/ measure only, we explicitly look for each possible 
     * version in turn.  This is still reasonably efficient as long as only 2 
     * processor families are supported.
     */

    /* First attempt to open the unsuffixed file. */
    unsuffixedProgFile = Open(nameBuf, FREAD | FBLOCK_MODE, error);
    if (unsuffixedProgFile != NULL)
      {
	/* Find out what type of machine this file runs on. */
	long magicNum;

	if (Read(unsuffixedProgFile, &magicNum, sizeof(long)) == sizeof(long))
	  {
	    for (i = 0; i < NUM_PROC_FAMILIES; ++i)
		if (ValidMagicNum(magicNum, i<<PROC_FAMILY_SHIFT))
		  {
		    foundProgFiles[i] = unsuffixedProgFile;
		    break;
		  }
	    if (i == NUM_PROC_FAMILIES) /* unrecognized magic number */
		Close(unsuffixedProgFile);
	  }
	else
	    Close(unsuffixedProgFile);
      }
  
    /* Attempt to open suffixed files, for versions we haven't yet found. */
    for (i = 0; i < NUM_PROC_FAMILIES; ++i)
        if (foundProgFiles[i] == NULL)
	  {
	    strncpy(&nameBuf[rawNameSize], ProcSuffix[i],
	    	    LineLength - rawNameSize);
	    foundProgFiles[i] = Open(nameBuf, FREAD | FBLOCK_MODE, error);
	  }

    nameBuf[rawNameSize] = '\0';
    for (i = 0; i < NUM_PROC_FAMILIES; ++i)
	if (foundProgFiles[i] != NULL)
	  {
	    *error = OK;
	    return (TRUE);
	  }
    return (FALSE);
  }
