/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1985				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Routines to create (and start running) a program.
 *
 * Marvin Theimer
 */


#include <Venviron.h>
#include <Vteams.h>

extern char *getenv();

#define FALSE 0
#define TRUE 1

#define DEFAULT_PATH "./ [bin]";


/*
 * LoadProgram:
 * Tries to load the program specified by the first argument in 
 * argv.  The program is placed in a separate team and is set ready to
 * run.  Its root pid is returned as the funtion return value.
 *
 * The program is found by searching for it in the directories 
 * specified by path.  If it is not found (and no other significant 
 * errors occurred) then the default command program is executed.  
 * (This currently tries to run the specified command as a remote 
 * command on the file server host.)
 */

ProcessId LoadProgram(argv, hostSpec, rtMsg, path, concurrent, error)
    char *argv[];	/* Program arguments (including name). */
    SelectionRec *hostSpec;
    			/* Specifies the host to execute on.
			   (NULL => default, i.e. local host) */
    RootMessage *rtMsg;
			/* Root message to use.  NULL => default settings. */
    char *path;		/* Search path to use for finding the program
			   file.  NULL indicates that the default should
			   be used. */
    int concurrent;	/* Specifies whether the program should be 
			   owned by the system (concurrent = 1) or 
			   by the user (concurrent = 0). */
    SystemCode *error;	/* Return code. */
  {
    Message msg;
    SelectionRec hostSpec1; /* used only if the hostSpec parameter was NULL */

    *error = OK;

    /* If any of the parameters were given as NULL, to indicate default values,
     * then set up these defaults.
     */
    if (rtMsg == NULL)
      {
        rtMsg = (RootMessage *) msg;
	DefaultRootMessage(rtMsg);
      }

    if (path == NULL)
      {
	path = getenv("PATH");
        if (path == NULL)
	    path = DEFAULT_PATH;
      }

    if (hostSpec == NULL)
      {
	hostSpec = &hostSpec1;
	DefaultSelectionRec(hostSpec);
	hostSpec->teamServerPid = GetPid(TEAM_SERVER, LOCAL_PID);
      }

    /* Determine whether a specific host has been specified, or if 
     * an `arbitrary' host is to be chosen (according to "hostSpec").
     */
    if (hostSpec->teamServerPid != 0)
	return SpecLoadProgram(argv, hostSpec->teamServerPid, rtMsg, path,
			       concurrent, error);
    else
	return ArbLoadProgram(argv, hostSpec, rtMsg, path, concurrent, error);
  }


/*
 * ExecProgram:
 * Tries to execute the program specified by the first argument in 
 * argv.  The program is placed in a separate team and is set 
 * running.  Its root pid is returned as the funtion return value.
 *
 * The program is found by searching for it in the directories 
 * specified by path.  If it is not found (and no other significant 
 * errors occurred) then the default command program is executed.  
 * (This currently tries to run the specified command as a remote 
 * command on the file server host.)
 *
 * If the status parameter is NULL then the program is run 
 * concurrently, otherwise the function waits until the program has
 * terminated and returns its exit status in status.
 */

ProcessId ExecProgram(argv, hostSpec, rtMsg, path, status, error)
    char *argv[];	/* Program arguments (including name). */
    SelectionRec *hostSpec;
    			/* Specifies the host to execute on.
			   (NULL => default, i.e. local host) */
    RootMessage *rtMsg;	/* Root message to use.  NULL => default settings. */
    char *path;		/* Search path to use for finding the program
			   file. */
    int *status;	/* Return code from program executed or NULL
			   if the program is to be run concurrently. */
    SystemCode *error;	/* Return code. */
  {
    ProcessId newpid;
    int concurrent;
    Message msg;

    if (status == NULL)
	concurrent = TRUE;
    else
	concurrent = FALSE;
    if (rtMsg == NULL)
      {
        rtMsg = (RootMessage *) msg;
	DefaultRootMessage(rtMsg);
      }

    newpid = LoadProgram(argv, hostSpec, rtMsg, path, concurrent, error);
    if ((newpid == 0) || (*error != OK))
        return(newpid);

    Reply( rtMsg, newpid ); 	/* pass stdio on to new process */

    if (!concurrent)
	Wait(newpid, status);
    return(newpid);
  }


extern ProcessId NameServerPid;

DefaultRootMessage(rtMsg)
    register RootMessage *rtMsg;
  {
    rtMsg->replycode = OK;
    rtMsg->rootflags = STDOUT_APPEND|STDERR_APPEND;
    rtMsg->stdinserver = stdin->fileserver;
    rtMsg->stdinfile = stdin->fileid;
    rtMsg->stdoutserver = stdout->fileserver;
    rtMsg->stdoutfile = stdout->fileid;
    rtMsg->stderrserver = stderr->fileserver;
    rtMsg->stderrfile = stderr->fileid;
    rtMsg->envblock = NULL;	/* set later */
  }
