/*
 * Routine to print out the stack of the current process (in hex) for the VAX
 */
#include <Vprocess.h>
#include <Vio.h>

/*
 * The VAX stack looks like this after a calls instruction:
 *
 *           |--------------------------------|
 *           |           args                 |
 *           |        (longwords)             |
 *           |            .                   |
 *           |--------------------------------|
 *     AP -> |        number of args          |
 *           |--------------------------------|
 *           |       longword padding         |        |
 *           |--------------------------------|        |
 *           |             .                  |        |
 *           |             .                  |        |
 *           |       saved registers          |        |
 *           |     (according to mask)        |        V
 *           |             .                  |      lower
 *           |             .                  |    addresses
 *           |--------------------------------|
 *           |          saved PC              |
 *           |--------------------------------|
 *           |          saved FP              |
 *           |--------------------------------|
 *           |          saved AP              |
 *           |--------------------------------|
 *           | pad length, entry mask, PSW    |
 *           |--------------------------------|
 *     FP -> |             0                  |
 *           |--------------------------------|
 *
 */

PrintStackDump(fout, dumppid)
    File *fout;
    ProcessId dumppid;
{
    ProcessId pid, mypid;
    unsigned *currentFp;
    unsigned *currentAp;
    unsigned char *returnPc;
    unsigned teamSize;
    unsigned i, nparams;
    int      offset;

    mypid = GetPid(ACTIVE_PROCESS, LOCAL_PID);
    pid = dumppid ? dumppid : mypid;  /* dumppid = 0 ==> stackdump myself */
    if (!SameTeam(pid, mypid))
      {
	fprintf(fout, "Not on same team: 0x%x\n", pid);
	return;
      }
    fprintf(fout, "StackDump for %x:\n", pid);

    if (pid == mypid)
      {	
	/* 
	 * Can't call QueryProcessState on myself, because the info it returns
	 *   describes my state during the call to QueryProcessState etc.
	 */
	get_Ap_Fp(&currentAp, &currentFp);
      }
    else
      {
	ProcessBlock pb;
	if (QueryProcessState( pid, &pb ) != pid)
	  {
	    fprintf(fout, "Invalid Process Identifier 0x%x\n", pid );
	    return;
	  }
	currentAp = (unsigned *) pb.ps.proc_state.regs[12];
	currentFp = (unsigned *) pb.ps.proc_state.regs[13];
      }

    teamSize = (unsigned) GetTeamSize(pid);
    if (((unsigned)currentFp < 0x200) ||
				/* Which magic hat was 200 pulled out of? */
	((unsigned)currentFp > teamSize-0x14))
				/* 0x14 = size (bytes) of basic stack frame */
      {
	fprintf(fout, "Invalid frame pointer: %x\n", currentFp);
	return;
      }
    while ((unsigned)currentFp >= 0x200 && (unsigned)currentFp <= teamSize-0x14)
    {
	returnPc = (unsigned char *) currentFp[4];
	if ((unsigned)returnPc < teamSize)
	  {
	    offset = print_pc(fout, returnPc);
	  }
	else
	  {
	    fprintf(fout, "(bogus return PC: 0x%x)", returnPc);
	    offset = 0;
	  }
	fprintf(fout, "("); /* Start of argument list */
	if ((unsigned) currentAp <= teamSize-4 && /* Really do need && here */
            (unsigned)(currentAp+ (nparams=currentAp[0]) ) <= teamSize-4 )
	  {
	    if (nparams < 20) {
		for (i=1; i <= nparams; i++)
		  {
		    fprintf(fout, "%x%s", currentAp[i], (i<nparams)?", ":"");
		  }
	      }
	    else
	      {
		fprintf(fout, "%x args", nparams);
	      }
	  }
	else
	  {
	    fprintf(fout, " bogus Argument Pointer: 0x%x ", currentAp);
	  }
	fprintf(fout, ") called from %x", returnPc+offset);
	if (offset >= 0)
	    fprintf(fout, "-?");
	fprintf(fout, "\n");
	if ((unsigned)returnPc >= teamSize)
	    return;
	
	currentAp = (unsigned *)currentFp[2];
	currentFp = (unsigned *)currentFp[3];
      }
  }


static
print_pc(fout, Pc)
    unsigned char *Pc;
    File *fout;
{
    int offset;     /* Offset to subroutine call instruction from pc */

    /*
     * This assumes the standard calling sequence generated by
     * the C compiler: a calls instruction with a literal argument
     * count and longword relative addressing.  That means the opcode
     * is at pc-7, and argument count at pc-6, the address mode at pc-5,
     * and the displacement at pc-4 through pc-1.
     *
     * If pc-7 does not contain the opcode for calls or if pc-6
     * does not specify literal mode or if pc-5  does not
     * specify longword displacement off the pc, then we
     * don't really know what address was called.  This routine
     * could be modified to look for other modes if necessary.
     *
     * We assume that Pc is a valid address in the team space.
     */

    offset = 0;     /* In case we don't know adr of calling instruction */
    if ((unsigned)Pc >= 7 /* bomb-proofing */ &&
	    (Pc[-7] & 0xFF) == 0xFB /* calls */ &&
	    (Pc[-6] & 0xC0) == 0    /* Literal mode */ &&
	    (Pc[-5] & 0xFF) == 0xEF /* Longword relative mode */)
    {
	fprintf(fout, "%x", Pc + *(long *)(&Pc[-4]));
	offset = -7;    /* calls instruction is at pc-7 */
    }
    else
    {
	fprintf(fout,"?");    /* Don't know what address was called */
    }
    return offset;
}

/*
 * get_Ap_Fp returns the values of the Argument Pointer and Frame Pointer
 *   (in the calling procedure, not in get_Ap_Fp).  This is easy to do here,
 *   since they're stored in the stack frame.
 */
static
get_Ap_Fp(return_Ap, return_Fp)
    unsigned *return_Ap;
    unsigned *return_Fp;
  {
    asm("	movl	0x8(fp), *4(ap)");  /* Argument Pointer */
    asm("	movl	0xC(fp), *8(ap)");  /* Frame Pointer    */
  }
