/*
 * Distributed V Kernel 
 * Copyright (c) 1982 by David Cheriton, Tim Mann, Willy Zwaenepoel
 *
 * Kernel Ethernet driver, hacked into a busy-wait version for standalone
 * use, and then hacked a bit more.
 *
 * Reference:
 *   [UG] EK-DEQNA-UG-001, DEQNA Ethernet User's Guide, 1st ed., August 1984.
 *
 * Byte-swapping:
 *   We now use DEQNAs (and this driver) in MicroVAXen and in the 680x0-based
 *   Fireflies.  The former case is easy: both the DEQNA and the MicroVAX are
 *   little-endian, and we only have to worry about the fact that Ethernet
 *   packet headers (source address, destination address, ether type) are sent
 *   over the network in big-endian order, whereas we like to treat them as
 *   little-endian numbers [Flame: if we treated them as bytes instead of
 *   words, the problem would go away].
 *
 *   For the Firefly, however, we must byte-swap all data to and from the
 *   DEQNA.  This applies both to the DEQNA registers and to the DEQNA data
 *   structures (buffer descriptor lists) in memory [Funny - I thought the
 *   680x0 Firefly had a hardware hack that byte-swapped the register case for
 *   us].
 *
 *   V inter-kernel packets are sent out with a flag which specifies whether
 *   the sending processor is big- or little-endian.  V process-IDs also have
 *   a flag which says whether the process lives on a little-endian host.
 *
 *   To deal with all these, we use:
 *     #ifdef LITTLE_ENDIAN for swapping packet headers and for setting the 
 *         flag in V inter-kernel packets and process-IDs.
 *     #ifdef LITTLE_ENDIAN also for putting ethernet addresses in big-endian
 *	   order before building setup packets fro the DEQNA.
 *     SWAPREG(x) to swap, if necessary, data going to/from the DEQNA's
 *	   registers
 *     SWAPBDL(x) to swap, if necessary, data going to/from the buffer
 *	   descriptor lists
 *   LITTLE_ENDIAN is (had better be) defined in the standard V buildfiles
 *   for anything that runs on a MicroVAX.  SWAPREG() and SWAPBDL() will
 *   normally be defined in "deqnalocal.h"; there will be a separate version
 *   of this for the MicroVAX II, 680x0 Firefly and anything else we dream up.
 *
 * Addressing - where the DEQNA registers are
 *   On the MicroVAX II, this is easy: the #1 DEQNA lives at 20001920-2000192f
 *   On the Firefly, Q-bus addresses are mangled so badly that the DEQNA
 *     registers are in eight non-contiguous locations.
 *   In deqna.h we define a pointer to each register.  The definitions of the
 *     pointers depend on the macros DEQNA_BASE_ADDRESS and MANGLE_ADDRESS,
 *     which will normally be defined in deqnalocal.h.
 *
 * Addressing - where the DEQNA data lives
 *   The addresses which we write into the DEQNA's Tx/Rx BDL registers, and
 *   the buffer/chain addresses which we put in the buffer descriptor lists,
 *   are really Q-bus addresses, not processor (physical or virtual) addresses.
 *   The functions or macros virt_to_QbusLO() and virt_to_QbusHI should perform
 *   any necessary mapping from a processor address to the low 16 and high 6
 *   bits, respectively, of the Q-bus address.
 */

#include "deqnalocal.h"
#include "deqna.h"

#include "process.h"	/* Must come before Vikc.h, because of the horrible */
#include "Vikc.h"	/*   "#define forwarder last_sender" in process.h   */
#include "Vethernet.h"
#include "Vmachine.h"
#define MAXLOOPCOUNT 200000

#define printx printf

#ifdef DEBUG
#define debug(c) K_putchar(c)
#else
#define debug(c)
#endif DEBUG

#define CopyMsg(to, from) Copy(to, from, 32)

/* Exports */
extern SystemCode	EnetPowerup();
extern SystemCode	AwaitKernelPacket();
extern int		WriteKernelPacket();
int KernelPacketType =  KERNEL_PACKET;
char	HexEnetAddress[sizeof(Enet10Address)*2 + 1];
kPacketWithSegment	*kPacketSave;	/* Save area for kernel packets */

Enet10Address	EnetHostNumber;		/* physical ethernet address */
Enet10Packet	*EnetPacket;
Enet10Address	ServerHostAddress;	/* A one-address "cache" */
int		ServerLogicalHost = -1;

Enet10Address	BroadcastAddress = { 0xFFFF, 0xFFFF, 0xFFFF };
char		Broadcast[] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};
Enet10Address	VMulticastAddr = V_MULTICAST_ADDRESS;
char		VMultiCast[6];

Qna_t	Buffers;	/* The Enet buffer structure */
int	RcvIndex;	/* The receive buffer we are currently using */
int	LastValidBuffer;/* The last valid buffer */
kPacketWithSegment	PacketSave;



SystemCode EnetPowerup()
  /* Powerup and initialize the Ethernet Interface Board.
   *  Done just once, from kernel Powerup() routine. 
   */
  {
    extern Process	*Active;
    register int  j;
    register char *q;
    
    /* get my host address */

#define qna_adr(x) SWAPREG(*(Deqna_enet_addr(x)))

    EnetHostNumber.addrhigh = ((qna_adr(0) & 0xFF)<<8) | (qna_adr(1) & 0xFF);
    EnetHostNumber.addrmid  = ((qna_adr(2) & 0xFF)<<8) | (qna_adr(3) & 0xFF);
    EnetHostNumber.addrlow  = ((qna_adr(4) & 0xFF)<<8) | (qna_adr(5) & 0xFF);
    
    /* Convert to hex for use elsewhere */
    q = HexEnetAddress;
    for (j=0; j < ENETADDRLEN; j++)
      {
	*q++ = MakeHexDigit( ( qna_adr(j) >> 4 ) & 0xF );
	*q++ = MakeHexDigit(   qna_adr(j)        & 0xF );
      }
    *q = '\0';
    
#ifdef LITTLE_ENDIAN
    swab(&VMulticastAddr, VMultiCast, sizeof(VMulticastAddr));
#else
    Copy(VMultiCast, &VMulticastAddr, sizeof(VMulticastAddr));
#endif
    
    Active->pid = ( 0x4000 | (EnetHostNumber.addrlow<<16) ) & ~GROUP_ID_BIT;

#ifdef LITTLE_ENDIAN
    Active->pid |=  LITTLE_ENDIAN_HOST;
#else
    Active->pid &= ~LITTLE_ENDIAN_HOST;
#endif

    kPacketSave = &PacketSave;

    /* Ignore deqna's interrupt vector */

    EnetReset( ENET_DEFAULT_MASK );

    return( OK );
 }


EnetReset( receiveMask )
    int receiveMask;
  {
    register int i, j, time;
    register QNABD_t *bdl;
    short	rcvMode;
    unsigned short	csr;
    int		numTimesRetry;
    static int numGoodResets = 0;

    /* We need to convert receiveMask into a rcvMode the deqna understands. */
    rcvMode = (receiveMask & ENET_BROADCAST ? QNA_AllMultiCast : 0) |
    		(receiveMask & ENET_PROMISCUOUS ? QNA_Promiscuous : 0);
    rcvMode += sizeof(Buffers.setupbuff);

#ifdef DEBUG
    printx("EnetReset(%d); rcvMode = 0x%x...\n", receiveMask, rcvMode);
#endif DEBUG

    numTimesRetry = 10;
  while (1)
   {    
    /*
     * Reset the DEQNA.  At least, I hope that's what it does.  The DEC DEQNA
     *   driver for the MicroVAX I and the DEC-SRC DEQNA driver for the 680x0
     *   version of the Firefly both do roughly this, although each has only
     *   two delay loops.  In the true spirit of compromise, I've put a delay
     *   loop where either of them had one.  Questions:
     *     - Why do they pulse the Boot/Diagnostic ROM bit?  It turns on the
     *       DEQNA's LEDs (which the first driver uses) and could be used to
     *       load stuff out of the DEQNA ROM (which neither DEQNA uses), but 
     *       does it do anything useful for resetting?
     *     - Why do all the drivers use "|=" and "&=" rather than just 
     *       clobbering the csr?  Since we're doing a reset, and particularly
     *	     since QNARESET sets all the bits to fixed values, it seems odd.
     *   Quite likely there really isn't a reason for all of this; maybe they
     *   were as much in the dark as I am, and just followed the MicroVAX I
     *   version or some earlier (PDP-11?) version
     */
    *Deqna_csr |= SWAPREG( QNARESET);
    coffee_break();
    *Deqna_csr &= SWAPREG(~QNARESET);
    coffee_break();
    *Deqna_csr |= SWAPREG( QNABOOTROM);
    coffee_break();
    *Deqna_csr &= SWAPREG(~QNABOOTROM);

    /* initialize the setup packet */
    for (i = 0; i < 16; i++)
        for (j = 0; j < 8; j++)
	    Buffers.setupbuff[i][j] = 0;
    /*
     * The DEQNA manual (such as it is) keeps telling us to initialize all
     *   unused addresses to the self-address, presumably on the grounds that
     *   any other value (even all 0) may bring in unwanted packets.  Still,
     *   this seems to work OK.
     */
    for (i = 0; i < 6; i++)
      {
        if (receiveMask & ENET_SELF)
	  {
	    Buffers.setupbuff[i][1] = SWAPREG(*Deqna_enet_addr(i)) & 0xFF;
	  }
        /* is this even needed? */
        if (receiveMask & ENET_BROADCAST)
	  {
	    Buffers.setupbuff[i][2] = Broadcast[i];
	    Buffers.setupbuff[i][3] = VMultiCast[i];
	  }
      }

    /* reset and flush the buffers */
    RcvIndex = 0;
    LastValidBuffer = RXLISTLEN-1;

    for (bdl = &Buffers.txlist[0]; bdl <= &Buffers.txlist[TXLISTLEN]; bdl++)
      {
	bdl->bd1	= SWAPBDL(QNA_BDLUnused);
	bdl->bd2	= SWAPBDL(0); /* sets VALID bit = 0 */
	bdl->addrlo	= SWAPBDL(0);
	bdl->size	= SWAPBDL(0);
	bdl->status1	= SWAPBDL(QNA_BDLUnused);
	bdl->status2	= SWAPBDL(0);
      }

    /* Initialize descriptors 0..RXLISTLEN-1 */
    for (i = 0, bdl = &Buffers.rxlist[0]; i < RXLISTLEN; i++, bdl++)
      {
        bdl->bd1	= SWAPBDL(QNA_BDLUnused);
	bdl->bd2	= SWAPBDL(QNA_BD2_VALID | 
			   ( virt_to_QbusHI(&Buffers.rxbuff[i]) 
			     << QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask ) );
        bdl->addrlo	= SWAPBDL( virt_to_QbusLO(&Buffers.rxbuff[i]) );
        bdl->size	= SWAPBDL(-((sizeof(Buffers.rxbuff[i]) + 1) /
				     sizeof(short)) );
        bdl->status1	= SWAPBDL(QNA_BDLUnused);
        bdl->status2	= SWAPBDL(0);
      }
      
    /* set up last BD */
    bdl->bd1	= SWAPBDL(QNA_BDLUnused);
    bdl->bd2	= SWAPBDL(QNA_BD2_VALID | QNA_BD2_CHAIN |
			   ( virt_to_QbusHI(Buffers.rxlist) 
			      << QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask ) );
    bdl->addrlo = SWAPBDL(virt_to_QbusLO(Buffers.rxlist));
    bdl->size   = 0;/* Size and status fields aren't used if chain is true, */
		    /*   but we initialize them to make debugging easier    */
    bdl->status1= SWAPBDL(QNA_BDLUnused);
    bdl->status2= SWAPBDL(0);
    
    i = SWAPREG(*Deqna_csr);
    i |= QNARCVINT | QNAILOOP | QNAXMTINT;
    i &= ~(QNARCVENABLE | QNAINTENABLE | QNATIMERENABLE | QNAELOOP);
    *Deqna_csr = SWAPREG(i);
    if ( (SWAPREG(*Deqna_csr) & QNARXBDLBAD) == 0 )
      {
	Debug_Deqna(SWAPREG(*Deqna_csr));
	puts("EnetReset: Receiver BDL should be invalid, isn't\n");
	RETURN_TO_MONITOR;
      }
    *Deqna_rxBDLlo = SWAPREG(virt_to_QbusLO(Buffers.rxlist));
    *Deqna_rxBDLhi = SWAPREG(virt_to_QbusHI(Buffers.rxlist));

    /* send a setup packet
    */
    LoadXmitDescriptor((unsigned)rcvMode, (int)Buffers.setupbuff, 1);
    if ( (SWAPREG(*Deqna_csr) & QNATXBDLBAD) == 0 )
      {
	puts("EnetReset: Transmitter BDL should be invalid, isn't\n");
	RETURN_TO_MONITOR;
      }
    *Deqna_txBDLlo = SWAPREG(virt_to_QbusLO(Buffers.txlist));
    *Deqna_txBDLhi = SWAPREG(virt_to_QbusHI(Buffers.txlist));
    /* wait for it to be looped back */
    time = 0;
    /*
     * We wait on the Tx, not Rx, buffer because Tx status is written after
     *   Rx status for a setup packet. [UG] section 1.3.4 (LoopBack), p. 1-9
     */
    while ( (SWAPBDL(Buffers.txlist[0].status1) & QNA_RX1_USEmask) == 
		QNA_BDLUnused &&
	     time <= QNAWAITTIME)
        time++;
    if (time > QNAWAITTIME)
      {
        /* we timed out waiting for a looped back packet, abort */
	Debug_Deqna(SWAPREG(*Deqna_csr));
	puts("Deqna wouldn't loop back a setup packet...\n");
	RETURN_TO_MONITOR;
      }
    if ( (SWAPBDL(Buffers.txlist[0].status1) & QNA_TX1_USEmask) == 
		QNA_BDLLastNoErr )
	break;
    else
      {
	Debug_Deqna(SWAPREG(*Deqna_csr));
	printx(
	    "\nPreviously did %d (decimal) consecutive successful EnetResets",
	    numGoodResets);
	numGoodResets = 0;
	if (numTimesRetry-- > 0)
	    printx(
	"\nDeqna errored on a setup packet .. retrying at most %d more time%s",
		   numTimesRetry+1, (numTimesRetry==0)?"":"s");
	else
	  {
	    puts("Deqna erred on a setup packet...stopping");
	    RETURN_TO_MONITOR;
	  }
      }
   }

    numGoodResets++;
    Buffers.rxlist[RcvIndex].bd1	= SWAPBDL(QNA_BDLUnused);
    Buffers.rxlist[RcvIndex].status1	= SWAPBDL(QNA_BDLUnused);
    RcvIndex = (RcvIndex+1) % (RXLISTLEN+1);
    /* Clear Rx, Tx interrupts from setup packet, enable receiver */
    *Deqna_csr |= SWAPREG(QNARCVINT | QNAXMTINT | QNARCVENABLE);
  }


LoadXmitDescriptor( msize, buff, setup )
  /* Loads the Transmit descriptor according to the given information */
    unsigned msize;	/* the length of the message */
    int buff;		/* the buffer containing the message */
    int setup;		/* 1 if this is a setup packet */
  {
    if (msize < ENET_MIN_DATA + ENET10_HEADER_SIZE ||
        msize > ENET_MAX_DATA + ENET10_HEADER_SIZE)
      {
	puts("LoadXmitDescriptor: bogus packet size\n");
	RETURN_TO_MONITOR;
      }
    /*
     * We write more than is really necessary.  Slows things down a bit, but it
     *   it helps when debugging if we set everything possible to 0, so we can
     *   see what the deqna has changed since we wrote it.
     */
    Buffers.txlist[0].bd1	= SWAPBDL(QNA_BDLUnused);
    Buffers.txlist[0].bd2	= SWAPBDL(QNA_BD2_VALID | QNA_BD2_ENDOFMSG |
		(setup ? QNA_BD2_SETUP : 0) |
		((msize & 1) ? QNA_BD2_LBOEND : 0) |
		(virt_to_QbusHI(buff) 
		   << QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask) );
	/*   chain and HBOStart are zero, as required.  */
    Buffers.txlist[0].addrlo	= SWAPBDL(virt_to_QbusLO(buff));
    Buffers.txlist[0].size	= SWAPBDL(-((msize + 1) / sizeof(short)) );
    Buffers.txlist[0].status1	= SWAPBDL(QNA_BDLUnused);
    Buffers.txlist[0].status2	= 0; /* Unnecessary but nice for debugging */
#ifdef LITTLE_ENDIAN
    if (!setup)
        swab((char *)buff, (char *)buff, ENET10_HEADER_SIZE);
#endif LITTLE_ENDIAN
  }


SystemCode AwaitKernelPacket()

   /* Wait until a valid kernel packet is received.  Return
    *   TIMEOUT if we time out before getting one.
    */

   /*
    * I think this works as follows:  In EnetReset we initialized the Buffer
    *   Descriptor List to be a ring, with "valid" set in all the buffer
    *   descriptors and in the chain descriptor.  Therefore the DEQNA will
    *   (???) read incoming packets indefinitely, wrapping around the ring
    *   when necessary.  If it ever overtakes us, it's not clear what will
    *   happen: it depends on what the DEQNA does when it reads a descriptor
    *   which is already marked as used (probably doesn't notice).
    * At no stage do we reset the Receive BDL.
    * Should we be writing "unused" into the chain descriptor, or doesn't
    *   it matter?
    * LastValidBuffer isn't used at all.  Strange.
    */
  {
    SystemCode		ret;
    register int	loopcount = 0;
    register int	done, i;
    register QNABD_t	*bdl;

    ret = TIMEOUT;
    done = 0;
    bdl = &Buffers.rxlist[RcvIndex];

debug('a');
    /* Find a valid error-free packet */
    while (loopcount++ < MAXLOOPCOUNT && !done)
      {
        if ( (SWAPBDL(bdl->status1) & QNA_RX1_USEmask) == QNA_BDLUnused &&
	    !(SWAPBDL(bdl->bd2    ) & QNA_BD2_CHAIN) )
	    continue;	/* no packet yet */
debug('b');
	if ( (SWAPBDL(bdl->status1) & QNA_RX1_USEmask) == QNA_BDLLastNoErr &&
	    !(SWAPBDL(bdl->bd2    ) & QNA_BD2_CHAIN) )
	  {
	    /* We have a valid error-free packet */
	    /* Get the first two words of the packet and figure out
	    *   if it is a kernel packet
	    */
debug('c');
	    EnetPacket = &Buffers.rxbuff[RcvIndex];
#ifdef LITTLE_ENDIAN
	    swab((char *)EnetPacket, (char *)EnetPacket, ENET10_HEADER_SIZE);
#endif LITTLE_ENDIAN
	    if (EnetPacket->EtherType == KernelPacketType)
	      {
debug('d');
		Copy(&PacketSave, EnetPacket->data, sizeof(kPacketWithSegment));
		if (DifferentIKCByteOrder(&PacketSave))
		  {
		    SwapIKPacket(&PacketSave);
		  }
		PacketSave.packetType &= ~IKC_LITTLE_ENDIAN;
		ServerHostAddress = EnetPacket->SrcHost;    
		ServerLogicalHost = PacketSave.srcPid >> 16;
		done = 1;
		ret = OK;
	      }
	  }
	bdl->bd1	= SWAPBDL(QNA_BDLUnused);
	/* Leave address descriptor, address, length alone */
	bdl->status1	= SWAPBDL(QNA_BDLUnused);
	RcvIndex = (RcvIndex+1) % (RXLISTLEN+1);
	bdl = &Buffers.rxlist[RcvIndex];
      }
debug('f');
    i = SWAPREG(*Deqna_csr);
    i |= QNARCVINT | QNAILOOP | QNARCVENABLE;
    i &= ~(QNAINTENABLE | QNATIMERENABLE | QNAELOOP | QNAXMTINT);
    *Deqna_csr = SWAPREG(i);
    return( ret );
  }


/* Routines specific to the distributed version */


extern int WriteKernelPacket(type, srcPid, dstPid, forwarder, length, localaddress,
				remoteaddress,sequenceNo, msg, data )	

/* Writes kernel packet to the Ethernet */

  short		type;
  ProcessId	srcPid, dstPid, forwarder;
  unsigned	length;
  Unspec	*localaddress;
  Unspec	*remoteaddress;
  unsigned short	sequenceNo;
  long		*msg;
  unsigned char *data;
  {
    Enet10Packet	*enetpacket;
    register kPacket	*packet;
    register int	dstLogicalHost;
    int		i, j;

    enetpacket = (Enet10Packet *)(&Buffers.txbuff);

    dstLogicalHost = dstPid>>16;
    if (dstLogicalHost == ServerLogicalHost)
        enetpacket->DestHost = ServerHostAddress;
    else
        enetpacket->DestHost = BroadcastAddress;

    enetpacket->SrcHost = EnetHostNumber;
    enetpacket->EtherType = KernelPacketType;
    packet = (kPacket *) enetpacket->data;
#ifdef LITTLE_ENDIAN
    packet->packetType = type |  IKC_LITTLE_ENDIAN;
#else
    packet->packetType = type & ~IKC_LITTLE_ENDIAN;
#endif
    packet->sequenceNo = sequenceNo;
    packet->srcPid = srcPid;
    packet->dstPid = dstPid;
    packet->forwarder = forwarder;
    packet->userNumber = 0;		/* user number */
    packet->length = length;
    packet->localaddress = localaddress;
    packet->remoteaddress = remoteaddress;
    if( msg != NULL )
      {
	CopyMsg(&packet->msg, msg);
      }
    if( data != NULL )
      {
	Copy( ((kPacketWithSegment *)packet)->data, data, length );
      }
    
    LoadXmitDescriptor(ENET10_HEADER_SIZE + sizeof(kPacket) + (data?length:0),
		       (int)enetpacket, 0);
    if ( (SWAPREG(*Deqna_csr) & QNATXBDLBAD) == 0)
      {
	puts("WriteKernelPacket: Transmitter BDL should be invalid, isn't\n");
	RETURN_TO_MONITOR;
      }
    *Deqna_txBDLlo = SWAPREG(virt_to_QbusLO(Buffers.txlist));
    *Deqna_txBDLhi = SWAPREG(virt_to_QbusHI(Buffers.txlist));

debug('y');
    for (j=0; j<QNAWAITTIME && 0 == (QNAXMTINT&(i=SWAPREG(*Deqna_csr))); j++)
	;
    if ((i & QNAXMTINT) == 0)
	puts(
"WriteKernelPacket: Timed out waiting for Transmit \"interrupt\"; Continuing..\n");
debug('x');
    i |= QNAXMTINT | QNAILOOP | QNARCVENABLE;
    i &= ~(QNAINTENABLE | QNATIMERENABLE | QNAELOOP | QNARCVINT);
    *Deqna_csr = SWAPREG(i);
    
    return (OK);
  }

int ReadDataPacket ( bytes, dst )
    unsigned bytes; 
    unsigned char *dst;
  {

debug('z');
    Copy( dst, PacketSave.data, bytes );
    return( OK );
  }

int DiscardDataPacket ( bytes ) unsigned bytes;
  {
    return( OK );
  }

coffee_break()
  {
    int i;
    /*
     * Haven't a clue how long the delay has to be.  MicroVAX I version (in DEC
     *   Pascal) uses a count of 500; so does 680x0 Firefly version (in C).
     *   Being very ignorant, hence very conservative (there's a political
     *   lesson somewhere here), and not having much feel for the speed of 
     *   MicroVAX II vs MicroVAX I or Firefly, and VAX C vs. DEC Pascal or 
     *   68000 C, I'll shoot for 2000.
     */
    for (i=0; i<2000; i++)
	one_sip();
  }

one_sip()
  { }

/*
 * Print debugging info for this DEQNA driver.  Must (1) not occupy too many
 *   lines on the display, and (2) tell as near as we can get to the whole 
 *   truth.  Might also be nice if it (3) made some attempt to pretty-print
 *   things, but this may conflict with (1) and (2) and might even require
 *   some work (gasp).  Prints stuff backwards so that the most important stuff
 *   appears last, and so doesn't vanish from sight.
 * If the DEQNA does anything creative while we're printing this, then what we
 *   print won't be very helpful.  Partly for this reason, we print whatever
 *   csr value the calling routine decides to hand us, and also print the csr's
 *   value when we're almost done.
 */
Debug_Deqna(csr)
    unsigned short csr; /* Assumes SWAPREG() has already been applied */
  {
    int i;
    printx("\nRx:");
    for (i = RXLISTLEN; i >= 0; i--)
	Debug_BDL(i, &Buffers.rxlist[i], (i&1)?"  " : "\r\n");
    printx("Tx:");
    for (i = TXLISTLEN; i >= 0; i--)
	Debug_BDL(i, &Buffers.txlist[i], (i&1)?"  " : "\r\n");
    printx(
       "RcvIndex = 0x%x, LastValidBuffer = 0x%x, Deqna CSR was 0x%x, is 0x%x",
	    RcvIndex, LastValidBuffer, csr, SWAPREG(*Deqna_csr));
  }

Debug_BDL(index, address, string)
    int index;
    unsigned short *address;
    char *string;
  {
    printx("%x: %x %x %x %x %x %x%s", index,
	   SWAPBDL(address[0]), SWAPBDL(address[1]), SWAPBDL(address[2]),
	   SWAPBDL(address[3]), SWAPBDL(address[4]), SWAPBDL(address[5]),
	   string);
  }
