/*
 * V Kernel - Copyright (c) 1985 by Stanford University
 *
 * Kernel Framebuffer support routines
 *
 * The "qvss" provides access to the framebuffer and to the framebuffer's
 * machine registers.  Opening this device causes the framebuffer to be
 * mapped into your address space, and reading from or writing to this
 * device causes a read/write to the machine register that controls it.
 */

#include "Venviron.h"
#include "Vteams.h"
#include "Vnaming.h"
#include "Vquerykernel.h"
#include "dm.h"
#include "interrupt.h"
#include "qvss.h"


/* Imports */
extern SystemCode NotSupported();
extern Page	*FreePageList;	/* Free list of pages */
extern int	FreePages;
extern PageTableEntry	SysPageTable[];
extern QvssKeyboardInit();

/* Exports */
SystemCode QvssCreate();
SystemCode QvssUnmap();	/* so can be called when destroying a team */

/* Private */

/* Forward */
SystemCode QvssWrite();
SystemCode QvssRead();
SystemCode QvssQuery();
SystemCode QvssRelease();
SystemCode QvssUnmap();
SystemCode QvssMap();

/* type of script with which to initialize the entire qvss */
typedef	struct {
    int		regnum;	/* register number to store value into */
    short	value;	/* value to be stored */
} QvssScript;

QvssScript  QvssInitScript[] = {
	    /* ignore interrupts and turn video on */
		{QV_CSR, QV_ENABLEVID},
	    /* initialize the CRT controller */
		{QV_CRTCA, 0}, {QV_CRTCD, 0x27},
		{QV_CRTCA, 1}, {QV_CRTCD, 0x20},
		{QV_CRTCA, 2}, {QV_CRTCD, 0x21},
		{QV_CRTCA, 3}, {QV_CRTCD, 0xb4},
		{QV_CRTCA, 4}, {QV_CRTCD, 0x38},
		{QV_CRTCA, 5}, {QV_CRTCD, 0x5},
		{QV_CRTCA, 6}, {QV_CRTCD, 0x36},
		{QV_CRTCA, 7}, {QV_CRTCD, 0x36},
		{QV_CRTCA, 8}, {QV_CRTCD, 0x4},
		{QV_CRTCA, 9}, {QV_CRTCD, 0xf},
		{QV_CRTCA, 10}, {QV_CRTCD, 0x20},
		{QV_CRTCA, 11}, {QV_CRTCD, 0},
		{QV_CRTCA, 12}, {QV_CRTCD, 0},
		{QV_CRTCA, 13}, {QV_CRTCD, 0},
		{QV_CRTCA, 14}, {QV_CRTCD, 0},
		{QV_CRTCA, 15}, {QV_CRTCD, 0},
	    /* Initialize the two UARTs */
		{QV_UCMD+UARTAOFFS, 0x15},
		{QV_UMODE+UARTAOFFS, 0x17}, /* doc uses 0x13, Ultrix 0x17 */
		{QV_UMODE+UARTAOFFS, 0x07},
		{QV_USCSR+UARTAOFFS, 0x99}, /* supposed to be 4800 rx, tx */
		{QV_UCMD+UARTBOFFS, 0x15},
		{QV_UMODE+UARTBOFFS, 0x17},
		{QV_UMODE+UARTBOFFS, 0x07},
		{QV_USCSR+UARTBOFFS, 0x99},
		{QV_UISM, 0x2},		/* rxints only */
	    /* Initialize the Interrupt controller */
	    /*
	     * The index into the SCB that the QVSS interrupts at is
	     * equal to SCBOffset = (0x80 | (ValueStored>>2)).
	     */
		{QV_ICSR, 0},		/* reset the controller */
		{QV_ICSR, 0x40},	/* Clear the IRR */
		{QV_ICSR, 0x80},	/* Set Interrupt, MultiVec, RotPrior */
		{QV_ICSR, 0xc0},	/* Select the ACR */
		{QV_ICDR, 0xff},	/* All are auto clear */
		{QV_ICSR, 0xe0},	/* Select UART int Vec */
		{QV_ICDR, 0xff&(VecQvssUart<<2)},
		{QV_ICSR, 0x28},	/* enable ints for that device */
		{QV_ICSR, 0xe1},	/* Select Vsync int Vec */
		{QV_ICDR, 0xff&(VecQvssMousePoll<<2)},
		{QV_ICSR, 0x29},	/* enable ints for that device */
		{QV_ICSR, 0xe4},	/* Select MouseA int Vec */
		{QV_ICDR, 0xff&(VecQvssMousePoll<<2)},
		{QV_ICSR, 0x2c},	/* enable ints for that device */
		{QV_ICSR, 0xe5},	/* Select MouseB int Vec */
		{QV_ICDR, 0xff&(VecQvssMousePoll<<2)},
		{QV_ICSR, 0x2d},	/* enable ints for that device */
		{QV_ICSR, 0xe6},	/* Select MouseC int Vec */
		{QV_ICDR, 0xff&(VecQvssMousePoll<<2)},
		{QV_ICSR, 0x2e},	/* enable ints for that device */
		
		{QV_ICSR, 0xb0},	/* Select IMR */
		{QV_ICDR, 0x0c},	/* mask irrelevant ints */
		{QV_ICSR, 0xa1},	/* Enable ints and ISR */
		{QV_CSR, QV_INTENABLE|QV_ENABLEVID}
	};



QvssPowerup()
  /*
   * This powers up the entire physical Qvss device -- which contains
   * four logical devices: the mouse, framebuffer, and two serial ports,
   * of which the first is the keyboard and the second is currently unused.
   *
   * The reason it is all done here is that interrupts need to be
   * coordinated among the four logical devices.  So even though there
   * is a qvssmouse.c and a qvssserial.c, their respective initializations
   * appear here.
   */
  {
    register int i;
    extern int Asm_IntQvssUart(), Asm_IntQvssMousePoll();

    /* plug the interrupt vectors */
    setexvec(Asm_IntQvssUart, VecQvssUart);
    setexvec(Asm_IntQvssMousePoll, VecQvssMousePoll);

    for (i = 0; i < sizeof(QvssInitScript)/sizeof(QvssScript); i++)
	Qvss->regs[QvssInitScript[i].regnum] = QvssInitScript[i].value;
    
    QvssKeyboardInit();
  }



/* Device Routines */

SystemCode QvssCreate( pd, inst )  
    Process *pd;
    DeviceInstance *inst;
  {
  /*
   * Create an instance for the Qvss.
   */
    CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    /*
     * Initialize the device instance descriptor, can assume all fields
     * are zero except for id and first instance's owner.
     */


    inst->readfunc = QvssRead;
    inst->writefunc = QvssWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = QvssQuery;
    inst->releasefunc = QvssRelease;

    inst->type = (READABLE+WRITEABLE);
    inst->blocksize = sizeof(short);
    inst->lastbytes = inst->blocksize;
    inst->lastblock = NUMUSERQVSSREGS - 1;

    return ( QvssMap( pd->team ) );

  }

SystemCode QvssRead( pd, inst )
    Process		*pd;
    DeviceInstance	*inst;

   /* Handle a read instance request for the console
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    register IoReply *reply = (IoReply *) &(pd->msg);
    register unsigned short *buf = (unsigned short *)&(reply->shortbuffer[0]);

    if( req->bytecount != inst->blocksize ) return( BAD_BYTE_COUNT );
    if( req->blocknumber >= NUMUSERQVSSREGS ) return( END_OF_FILE );
    
    *buf = Qvss->regs[req->blocknumber];
    
    return (OK);
    
  }

static SystemCode QvssWrite( pd, inst )
    register Process *pd;
    register DeviceInstance	*inst;

   /* Handle a write instance request to a serial line
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    register IoReply *reply = (IoReply *) &(pd->msg);
    register unsigned short *buf = (unsigned short *)&(reply->shortbuffer[0]);

    if( req->bytecount != inst->blocksize ) return( BAD_BYTE_COUNT );
    if( req->blocknumber >= NUMUSERQVSSREGS ) return ( END_OF_FILE );

    Qvss->regs[req->blocknumber] = *buf;
    
    return (OK);
    
  } /* QvssWrite */

SystemCode QvssQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register MsgStruct *reply = (MsgStruct *) &(pd->msg);

    reply->unspecified[0] = QVSSVBASEADDR;

    return( OK );
  }

static SystemCode QvssRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release VAX screen (console) instance
   */
  {
    inst->owner = 0;
    
    QvssUnmap( pd->team );

    return( OK );
  }

SystemCode QvssMap( td )
    Team *td;
  /*
   * Map the Qvss framebuffer into the specified process's (actually team's)
   * address space.  It will be at a constant location, which is returned
   * by a query operation on the instance.
   */
  {
    register int teamno = td->team_space.teamno;
    register int i, pageend;
    register PageTableEntry *pte, *pteend;
    register Page *newpg;
    Process *pd;
    
    if (td->team_space.qvss) /* already mapped */
	return( OK );
    
    /* make sure there are enough free pages to do this */
    i = 0;
    newpg = FreePageList;
    while (i < NUMSYSQVPAGES && newpg != NOPAGE)
      {
	newpg = newpg->nextpage;
	i++;
      }
    if (i != NUMSYSQVPAGES)
	return( NO_MEMORY );

    /* allocate the pages in the process's page table */
    newpg = FreePageList;
    pageend = TEAMSYSQVEND(teamno);
    for (i = TEAMSYSQVSTART(teamno); i < pageend; i++)
      {
	SysPageTable[i].word
		 = (SegOffs(newpg)>>PAGESIZEBITS) | VM_VALID | VM_RWRW;
	/* I think this works, but we'll just invalidate all of them */
	/*
	r11 = i - TEAMSYSPTSTART(teamno);
	r11 *= (PAGESIZE/sizeof(PageTableEntry))<<PAGESIZEBITS;
	r11 |= SYSSEGSTART;
	asm("	mtpr	r11, $tbis");
	*/

	/* Remove it from the free list */
	FreePageList = newpg->nextpage;
	FreePages--;
	newpg = FreePageList;
      }
    /* create the process's page table map */
    i = 0;
    pteend = (PageTableEntry *)TEAMPTQVEND(teamno);
    pte = (PageTableEntry *)TEAMPTQVBASE(teamno);
    while (pte < pteend)
      {
	pte++->word = ((QVSSPBASEADDR>>PAGESIZEBITS)+i++)|VM_VALID|VM_RWRW;
	/* would need toinvalidate the cache here */
      }
  
    asm("	mtpr	$0, $tbia"); /* invalidate the cache */

    td->team_space.qvss = 1;
    pd = MapPid(td->team_root);
    if (pd != NULL)
	KAssignSize(pd, td, MAXPAGEFRAMES);

    return(OK);

  }

SystemCode QvssUnmap( td )
    Team *td;
  /*
   * Unmaps the Qvss framebuffer from the specified process's (actually team's)
   * address space.  It was at a constant location.  Note that this should
   * be the last process (with an open qvss open) in the team, otherwise
   * the other processes will get addressing exceptions when referencing
   * the framebuffer.  This routine is also is called from machine.c when
   * we are destroying a team with the Qvss mapped.
   */
  {
    register int teamno = td->team_space.teamno;
    register int i, pageend;
    register PageTableEntry *pte;
    register Page *newpg;
    Process *pd;
    
    if (!td->team_space.qvss)
	/* already unmapped */
	return( OK );

    /* deallocate the pages used by the process's page table */
    pageend = TEAMSYSQVEND(teamno);
    for (i = TEAMSYSQVSTART(teamno); i < pageend; i++)
      {
	pte = &SysPageTable[i];
	newpg = (Page *)((pte->fields.pfn << PAGESIZEBITS) | SYSSEGSTART);
	pte->word = 0;
	newpg->nextpage = FreePageList;
	FreePageList = newpg;
	FreePages++;
      }
  
    asm("	mtpr	$0, $tbia"); /* invalidate the cache */

    td->team_space.qvss = 0;
    pd = MapPid(td->team_root);
    if (pd != NULL)
	KAssignSize(pd, td, td->team_space.p0len);
    return(OK);
  } 

/*
 * QvssEnableVideo: turn on "video enable" bit in the Qvss' CSR.  Called from
 *   Kabort() to ensure that error messages are seen even if software outside
 *   the kernel asked for video to be disabled (screen saver).  Just in case
 *   we get a Kabort while running without memory mapping, we check and do the
 *   right thing.  If memory mapping is enabled, we half-heartedly check
 *   whether we can access the QVSS.
 */
void QvssEnableVideo()
  {
    register int r11;

    ;asm("	mfpr	$mapen,	r11");
    if ( (r11 & 1) == 0 )
	/* Physical addressing */
	((qvss_reg_t *)PhysQvss)->regs[QV_CSR] |= QV_ENABLEVID;
    else if (KernelCanRead( &(Qvss->regs[QV_CSR]),
		       sizeof(Qvss->regs[QV_CSR]) ))
	/* Should really test for write access, but... */
	Qvss->regs[QV_CSR] |= QV_ENABLEVID;
    /* else do nothing */
  }
