/*
 * Distributed V Kernel
 * Copyright (c) 1986 by Stanford University, All rightrs reserved.
 *
 * Kernel MSCP disk driver, William Lees
 */

/* This source file encapuslates the subroutines and data structures
 * required to speak the MSCP protocol.  There are three entry points:
 *
 *	o MscpStartInit		to start the port init sequence
 *	o MscpUnitBlockIO	to initiate a transfer
 *	o MscpInterrupt		when a controller interrupt occurs
 *
 * The hardware port init sequence used by the RQDX is the same as that
 * used by the UDA devices of larger VAXen.  The aspects of MSCP used will
 * be a superset of those employed here.  The major enhancement required
 * to this driver is to implementent separate data structures for each
 * controller, i.e. separate communications area, separate packets, etc
 *
 * In order for the controller to DMA to host memory it must be mapped by
 * the Qbus registers.  As of this writing only kernel memory is mapped.
 * This requires a double buffering scheme.  Note that VAX memory to memory
 * transfers are pretty fast.  Because large buffers are required, only one
 * transfer is allowed to use the single buffer at a time.  Unfortunately,
 * this prevents the controller from performing seek optimization. In the
 * future, Qbus registers should be allocated to map the user buffers.
 *
 * Speaking of addressing, Qbus addresses are only 22 bits.  Only kernel memory
 * is mapped in.  To convert a kernel address to a Qbus address suitable for
 * passing to the controller, just use the low order 22 bits.
 *
 * When concurrent transfers are implemented, note that each outstanding
 * command must have a unique command reference number.  Using the requestor's
 * pid or a pointer to an additional information data block works well.
 * The latter is a mechanism for passing software context to the response
 * handling routine.
 */

/* Common definitions */

#include "rqdx.h"		/* RQDX controller */
#include "mscp.h"		/* MSCP protocol */
#include "Venviron.h"		/* Defs for SystemCode, ... */
#include "interrupt.h"		/* Defs for Call_inthandler, ... */
#include "Vioprotocol.h"	/* Defs for IoReply, ... */
#include "process.h"		/* Defs for Process, ... */

/* Routine Exports */

extern SystemCode MscpStartInit();	/* Start port init sequence */
extern SystemCode MscpUnitBlockIO();	/* Block transfer */

/* Variable Exports */

short RqdxDriverState;			/* Drive state (software) */
struct rqdx_unit_info RqdxUnitInfo[3];	/* Info on each unit */

/* Forward */

void MscpCommunicationInit();	/* Initialize data structures */
void MscpControllerInit();	/* Set controller characteristics */

void MscpResponsePacket();	/* Handle a response packet */
void MscpErrorDatagram();	/* Handle a datagram */

void MscpUnitOnline();		/* Send packet to bring unit on-line */
void MscpGetUnitStatus();	/* Send packet to get unit status */

void MscpConvertMediaId();	/* Decode media id field */
void MscpDecodeStatusEvent();	/* Decode status / event code */

struct mscp * MscpGetCommandPacket();	/* Find next free command packet */

/* Private */

short		Credits;	/* Transfer credits */
short		LastCommand;	/* Pointer into command ring */
short		LastResponse;	/* Pointer into response ring */
struct mscp	Response[NRSP];	/* Response packets */
struct mscp	Command[NCMD];	/* Command packets */

struct rqdx_ca Rqdx_ca;		/* RQDX Communications area */

Process		*DiskRequestorPD = NULL;		/* Requestor */
unsigned	DiskRequestorBuffer;			/* User buffer ptr */
char		RqdxDiskBuffer[RQDX_DISK_BUFFER_SIZE];	/* Kernel buffer */

/****************************************************************************
 * Routine name:	Asm_MscpInterrupt
 *
 * Description:
 *
 *	This routine is a jacket which perserves registers before
 *	calling the C interrupt routine.  It is produced from a macro
 *	expansion.
 *
 * Inputs:	None.
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

Call_inthandler(MscpInterrupt)

/****************************************************************************
 * Routine Name:	MscpStartInit
 *
 * Description:
 *
 *	This routine starts the port init sequence for the controller.
 *
 * Inputs:	None.
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxDriverState
 */

SystemCode MscpStartInit()
{
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;

	RqdxDriverState = S_IDLE;

	rqdx->ip = 0;	/* Writing to IP dev reg means initialize */

	while ( (rqdx->sa & UDA_STEP1) == 0) ;
	if (rqdx->sa & UDA_ERR) return( DEVICE_ERROR );

/* Tell it number of command packets, number of response packets, to
 * interrupt during this sequence, and our interrupt vector.
 */
	RqdxDriverState = S_STEP1;
	rqdx->sa = UDA_ERR | (NCMDL2<<11) | (NRSPL2<<8) |
			UDA_IE | RQDX_INTERRUPT;

	return( OK );	/* Sequence continues in interrupt routine */
}

/****************************************************************************
 * Routine name:	MscpInterrupt
 *
 * Description:
 *
 *	This routine is called by the assembly language jacket when a
 *	controller interrupt occurs.  This routine handles the remainder
 *	of the port initialization sequence as well as passing response
 *	packets off for handling.
 *
 * Inputs:	None.
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:
 *
 *	RqdxDriverState
 *	Rqdx_ca
 */

void MscpInterrupt()
{
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;
	unsigned temp;
	int i;

	switch (RqdxDriverState) {

	case S_IDLE:
		printx("[Random RQDX interrupt ignored]\n");
		return;

/* We've received an interrupt and we were in Step 1.  If the Step 2 flag
 * isn't set then error.  Load the first half of the communication area
 * into SA.  Go to state step 2.  Note the two step assignment to the
 * address register.  It is required because of a compiler bug.
 */
	case S_STEP1:
		if ( (rqdx->sa & UDA_STEP2) == 0 ) {
			RqdxDriverState = S_IDLE;
			return;
		};
		temp = (unsigned) &Rqdx_ca.ResponseDesc[0];	/* Kludge */
		rqdx->sa = (short) temp;			/* Kludge */
		RqdxDriverState = S_STEP2;
		return;

/* We've received an interrupt and we were in Step 2.  If the Step 3 flag
 * isn't set then error.  Load the upper half of the communication area
 * into SA.  Go to state step 3.
 */
	case S_STEP2:
		if ( (rqdx->sa & UDA_STEP3) == 0 ) {
			RqdxDriverState = S_IDLE;
			return;
		};
		temp = ( ( (unsigned) &Rqdx_ca.ResponseDesc[0] ) >> 16) & 0x3f;
		rqdx->sa = (short) temp;			/* Kludge */
		RqdxDriverState = S_STEP3;
		return;

/* We've received an interrupt and we were in Step 3.  If the Step 4 flag
 * isn't set then error.  We made it!  Find out the version and model, then
 * init the data structures and send the controller its first packet!
 */
	case S_STEP3:
		if ( (rqdx->sa & UDA_STEP4) == 0 ) {
			RqdxDriverState = S_IDLE;
			return;
		};
		printx("RQDX disk controller: ");
		printx("port init seq OK, version %d model %d.\n",
			(rqdx->sa & 0xf), ( (rqdx->sa>>4) & 0xf ) );

		rqdx->sa = UDA_GO;
		RqdxDriverState = S_SCHAR;

		MscpCommunicationInit();
		MscpControllerInit();
		return;    

	case S_SCHAR:
	case S_RUN:
		break;
	default:
		printx("[Random RQDX interrupt ignored]\n");
		return;
	}

/* Check for controller error */

	if (rqdx->sa & UDA_ERR)
		printx("Rqdx fatal error (sa = %x)\n", rqdx->sa);
	else

/* Check for buffer purge request */

	if (Rqdx_ca.ca_bdp) {
		printx("[UDA buffer purge]\n");
		Rqdx_ca.ca_bdp = 0;
		rqdx->sa = 0; /* Signal buffer purge complete */
	}
	else

/* Check for response ring transtion */

	if (Rqdx_ca.ResponseInterrupt) {
/*		printx("[ResponseInterrupt]\n"); */
		Rqdx_ca.ResponseInterrupt = 0;
		for (i = LastResponse;; i++) {
			i %= NRSP;
			if ( Rqdx_ca.ResponseDesc[i] & UDA_OWN ) break;
			MscpResponsePacket(i);
			Rqdx_ca.ResponseDesc[i] |= UDA_OWN;
		}
		LastResponse = i;
	}
	else

/* Check for command ring transition */

	if (Rqdx_ca.CommandInterrupt) {
		printx("[CommandInterrupt]\n");
		Rqdx_ca.CommandInterrupt = 0;
	}
	else
		printx("[Random RQDX interrupt ignored]\n");

};

/****************************************************************************
 * Routine Name:	MscpCommunicationInit
 *
 * Description:
 *
 *	This routine initializes the packet descriptors.
 *
 * Inputs:	None
 *
 * Outputs:	None
 *
 * Implicit Inputs/Outputs:
 *
 *	LastCommand
 *	LastResponse
 *	Credits
 *	Rqdx_ca
 */

void MscpCommunicationInit()
{
	int i;

	LastCommand = 1;
	LastResponse = 0;
	Credits = 0;

#define qbus(object) ( ( (unsigned) &(object) ) & 0x3fffff )

/* Mark the response descriptors as owned by the controller.  Not having
 * the INT flag set means its a new one for us to read.
 */
	for (i = 0; i < NRSP; i++) {
		Rqdx_ca.ResponseDesc[i] =
			(UDA_OWN | UDA_INT | qbus(Response[i].mscp_cmd_ref) );
		Response[i].mscp_header.uda_msglen = mscp_msglen;
	};

/* Mark the command descriptor as interrupt enable on completion. */

	for (i = 0; i < NCMD; i++) {
		Rqdx_ca.CommandDesc[i] =
			(UDA_INT | qbus( Command[i].mscp_cmd_ref ) );
		Command[i].mscp_header.uda_msglen = mscp_msglen;
	};
};


/****************************************************************************
 * Routine Name:	ControllerInit
 *
 * Description:
 *
 *	This routine sends the first packet to the controller.  It is not
 *	built using the GetCommandPacket subroutine because at this point
 *	no credits have been assigned.  This packet does a "set controller
 *	characteristics" to tell it what kind of datagrams we want.
 *
 * Inputs:	None.
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:
 *
 *	Command
 *	Rqdx_ca
 *
 */

void MscpControllerInit()
{
	short i;
	struct mscp *mp;
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;

	mp = &Command[0];

	mp->mscp_opcode = M_OP_STCON;
	mp->mscp_cmd_ref = mp->mscp_unit = 0;
	mp->mscp_flags = mp->mscp_modifier = 0;

	mp->mscp_version = 0;
	mp->mscp_cnt_flgs = M_CF_ATTN | M_CF_MISC | M_CF_THIS;
	mp->mscp_hst_tmo = mp->mscp_use_frac = 0;
	mp->mscp_time_low = mp->mscp_time_high = 0;
	mp->mscp_cnt_dep = 0;

/* Setting the OWN flag tells the controller it can have it */

	Rqdx_ca.CommandDesc[0] |= UDA_OWN | UDA_INT;

/* Reading from IP means start looking for commands */

	i = rqdx->ip;	/* Initiate polling */
};

/****************************************************************************
 * Routine Name:	ResponsePacket
 *
 * Description:
 *
 *	This routine processes a response packet found by the interrupt
 *	routine.  A response packet that indicates the completion of a
 *	command has the M_OP_END bit set in the opcode field.
 *
 * Inputs:
 *
 *	Index	Index of response packet to process
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:
 *
 *	Response
 *	RqdxDriverState
 *
 */

void MscpResponsePacket(index)
	int index;
{
	unsigned short s;
	struct mscp *mp;
	struct rqdx_unit_info *ui;
	SystemCode r;
	IoReply *reply;
	Team *oldteam;

	mp = &Response[index];
	ui = &RqdxUnitInfo[mp->mscp_unit];
	s = mp->mscp_status & M_ST_MASK;

/* Give us some credit */

	mp->mscp_header.uda_msglen = mscp_msglen;
	Credits += mp->mscp_header.uda_credits & 0xf;
	if ( (mp->mscp_header.uda_credits & 0xf0) > 0x10 )
		printx("Unknown RQDX message type\n");

/* High 4 bits are msg type: send error datagrams off for further processing */

	if ( (mp->mscp_header.uda_credits & 0xf0) == 0x10) {
		MscpErrorDatagram( (struct mslg *)mp );
		return;
	}

	if ( ( ( mp->mscp_opcode & M_OP_END ) == M_OP_END ) &&
		( mp->mscp_flags != 0 ) ) {
		printx("[RQDX report of bad block on unit %d ignored]\n",
			mp->mscp_unit);
		return;
	}

	switch (mp->mscp_opcode) {

/* After completion of the port init sequence the interrupt routine issues
 * a set controller characteristics request to the controller.  When this
 * completes here, send packets to check the status of each of the units.
 */
	case M_OP_STCON | M_OP_END:	
		if (s != M_ST_SUCC) {
			printx("RQDX MSCP STCON failure\n");
			MscpDecodeStatusEvent( mp->mscp_status );
			RqdxDriverState = S_IDLE;
			return;
		}
		printx("RQDX disk controller: ");
		printx("version %d, hardware rev %d, class %d, model %d.\n",
			mp->mscp_cnt_rsn & 0xff,
			(mp->mscp_cnt_rsn>>8) & 0xff,
			mp->mscp_cnt_class,
			mp->mscp_cnt_model);

		RqdxDriverState = S_RUN;

		MscpGetUnitStatus(0);
		MscpGetUnitStatus(1);
		MscpGetUnitStatus(2);
		break;

/* Handle completion of a get unit status request.  If the unit is available,
 * i.e. has media mounted, then send a packet to bring the unit on-line.
 */
	case M_OP_GTUNT | M_OP_END:
		MscpConvertMediaId(mp->mscp_media_id,
			ui->device_type, ui->media_name);

		if (s == M_ST_AVLBL)
			MscpUnitOnline(mp->mscp_unit);
		else {
			printx("Device %s%d: %s is now OFFLINE\n",
				ui->device_type, mp->mscp_unit, ui->media_name);
			ui->unit_online = 0;
			MscpDecodeStatusEvent( mp->mscp_status );
		}
		break;

/* Handle completion of unit on-line command.  If it succeeds print a message
 * and mark the unit as on-line.
 */
	case M_OP_ONLIN | M_OP_END:
		if (s != M_ST_SUCC) {
			printx("Device %s%d: %s has failed to come ONLINE\n",
				ui->device_type,
				mp->mscp_unit,
				ui->media_name);
			MscpDecodeStatusEvent( mp->mscp_status );
			return;
		}

		printx("Device %s%d: %s is ONLINE with %d 512-byte blocks\n",
			ui->device_type, mp->mscp_unit,
			ui->media_name,	mp->mscp_unit_size);
		ui->unit_online = 1;
		ui->unit_last_block = mp->mscp_unit_size - 1;
		break;

/* An unit attention response means a unit MAY be able to come on-line now.
 * Closing a floppy door, for example, will produce these.  Send a get unit
 * status packet to check the status of the unit now.
 */
	case M_OP_AVATN:
		MscpGetUnitStatus(mp->mscp_unit);
		break;

/* Handle completion of a read or write transfer.  If it doesn't succeed
 * mark the reply as a device error.  If it succeeds and the request was
 * a read then copy the user's data from the kernel buffer to the user's
 * buffer.  Wake up the requestor now that the transfer is complete.
 */
	case M_OP_READ | M_OP_END:
	case M_OP_WRITE | M_OP_END:
		reply = (IoReply *) &(DiskRequestorPD->msg);
		if (s != M_ST_SUCC) {
			printx("Device %s%d: %s had a disk transfer error.\n",
				ui->device_type, mp->mscp_unit,
				ui->media_name);
			MscpDecodeStatusEvent( mp->mscp_status );
			reply->replycode = DEVICE_ERROR;
			reply->bytecount = 0;
		}
		else {
			reply->replycode = OK;
			if ( mp->mscp_opcode == (M_OP_READ | M_OP_END) ) {

				oldteam = GetAddressableTeam();
				SetAddressableTeam( DiskRequestorPD->team );
				Copy( DiskRequestorBuffer,
					RqdxDiskBuffer, mp->mscp_byte_cnt);
				SetAddressableTeam( oldteam );
			}
		}
		Addready( DiskRequestorPD );
		DiskRequestorPD = NULL;
		break;

	default:
		printx("[RQDX unexpected 0%o response packet ignored]\n",
			mp->mscp_opcode);
	}
}

/****************************************************************************
 * Routine Name:	MscpConvertMediaId
 *
 * Description:
 *
 *	This routine converts the MSCP media id field into two Digital
 *	standard names: the generic device type name (e.g. DU), and the
 *	media name (e.g. RD52).
 *
 * Inputs:
 *
 *	media_id	Longword containing encoded media information
 *
 * Outputs:
 *
 *	device_type	Pointer to string to receive device type
 *	media_name	Pointer to string to receive media name
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpConvertMediaId(media_id, device_type, media_name)
	unsigned media_id;
	char device_type[];
	char media_name[];
{
	char c;
	int p = 0;

/* Decode device type.  Its always two characters */

	c = (char) ( (media_id >> 27) & 0x1f );
	device_type[p++] = c + 64;
	c = (char) ( (media_id >> 22) & 0x1f );
	device_type[p++] = c + 64;
	device_type[p]='\0';

/* Decode media name.  Its 1-3 characters.  Its followed by a two-digit
 * number.
 */
	p = 0;
	c = (char) ( (media_id >> 17) & 0x1f );
	media_name[p++] = c + 64;
	c = (char) ( (media_id >> 12) & 0x1f );
	if (c != 0) media_name[p++] = c + 64;
	c = (char) ( (media_id >> 7) & 0x1f );
	if (c != 0) media_name[p++] = c + 64;

	c = (char) (media_id & 0x3f);
	media_name[p++] = (c / 10) + 48;
	media_name[p++] = (c % 10) + 48;

	media_name[p] = '\0';
}

/****************************************************************************
 * Routine Name:	MscpGetUnitStatus
 *
 * Description:
 *
 *	This routine builds and sends a "get unit status" packet.
 *
 * Inputs:
 *
 *	unit	Unit whose status is to be obtained
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpGetUnitStatus(unit)
	short unit;
{
	short i;
	struct mscp *mp;
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;

	mp = MscpGetCommandPacket();
	mp->mscp_unit = unit;
	mp->mscp_opcode = M_OP_GTUNT;
	
	i = rqdx->ip;	/* Initiate polling */
};

/****************************************************************************
 * Routine Name:	MscpUnitOnline
 *
 * Description:
 *
 *	This routine builds and sends a "unit on-line" command
 *
 * Inputs:
 *
 *	unit	Unit to be brought on-line.
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpUnitOnline(unit)
	short unit;
{
	short i;
	struct mscp *mp;
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;

	mp = MscpGetCommandPacket();
	mp->mscp_unit = unit;
	mp->mscp_opcode = M_OP_ONLIN;
	
	i = rqdx->ip;	/* Initiate polling */
};

/****************************************************************************
 * Routine Name:	MscpUnitBlockIO
 *
 * Description:
 *
 *	This routine builds a transfer command and sends it.
 *
 * Inputs:
 *
 *	pd		Desciptor of requesting process
 *	read		1 means read, 0 means write
 *	unit		Unit to transfer to/from
 *	byte_count	How many bytes to transfer.
 *	user_buffer	Address of user's buffer
 *	block		Block to transfer to/from
 *
 * Outputs:
 *
 *	SystemCode	V system status code
 *
 * Implicit Inputs/Outputs:
 *
 *	DiskRequestorPD
 *	DiskRequestorBuffer
 */

SystemCode MscpUnitBlockIO(pd, read, unit, byte_count, user_buffer, block)
	Process *pd;
	char read;
	short unit;
	unsigned byte_count;
	unsigned user_buffer;
	unsigned block;
{
	short i;
	struct mscp *mp;
	struct rqdx *rqdx = (struct rqdx *) RQDX_BASE_ADDRESS;
	Team *oldteam;

/* Punt if transfer in progress and return RETRY.  Really should implement
 * some sort of a queue here.  Won't be required when Qbus mapping is
 * implemented.
 */
	if (DiskRequestorPD != NULL) {
		printx("Transfer in progress; returning RETRY\n");
		return( RETRY );
	}

	DiskRequestorPD = pd;

/* On a write request copy the user's data into the kernel buffer */

	if (read == 0) {
		oldteam = GetAddressableTeam();
		SetAddressableTeam( pd->team );
		Copy( RqdxDiskBuffer, user_buffer, byte_count );
		SetAddressableTeam( oldteam );
	}
	else
		DiskRequestorBuffer = user_buffer;

	mp = MscpGetCommandPacket();
	mp->mscp_cmd_ref = pd->pid;	/* Using PID as unique cmd ref id */
	mp->mscp_unit = unit;
	mp->mscp_opcode = read ? M_OP_READ : M_OP_WRITE;
	mp->mscp_byte_cnt = byte_count;
	mp->mscp_buffer = ( (unsigned) RqdxDiskBuffer ) & 0x3fffff;
	mp->mscp_lbn = block;
	
	i = rqdx->ip;	/* Initiate polling */

	return( OK );
};

/****************************************************************************
 * Routine Name:	MscpGetCommandPacket
 *
 * Description:
 *
 *	This routine finds a free command packet and initializes it.
 *
 * Inputs:	None.
 *
 * Outputs:
 *
 *	struct mscp *	Pointer to free mscp command packet
 *
 * Implicit Inputs/Outputs:
 *
 *	LastCommand
 *	Rqdx_ca
 *	Credits
 *
 */

struct mscp * MscpGetCommandPacket()
{
	struct mscp *mp;
	int i;

/* Busy wait until command packet found.  Must have atleast two credits and
 * packet must not be owned by controller. */

	while (1) {
		i = LastCommand;
		if ( ( (Rqdx_ca.CommandDesc[i] &
			(UDA_OWN|UDA_INT)) == UDA_INT ) && (Credits >= 2) ) {
			Credits--;
			mp = &Command[i];
			mp->mscp_cmd_ref = mp->mscp_unit = 0;
			mp->mscp_xxx1 = mp->mscp_opcode = 0;
			mp->mscp_flags = mp->mscp_modifier = 0;
			mp->mscp_byte_cnt = mp->mscp_buffer = 0;
			mp->mscp_mapbase = mp->mscp_xxx2 = 0;
			mp->mscp_lbn = mp->mscp_xxx3 = 0;

			Rqdx_ca.CommandDesc[i] |= UDA_OWN | UDA_INT;
			LastCommand = (i + 1) % NCMD;
			break;
		}
	};
	return(mp);
}

/****************************************************************************
 * Routine Name:	MscpErrorDatagram
 *
 * Description:
 *
 *	Handle a MSCP error log datagram.
 *
 * Inputs:
 *
 *	mp	Pointer to datagram
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpErrorDatagram(mp)
	struct mslg *mp;
{
	struct rqdx_unit_info *ui = &RqdxUnitInfo[mp->mslg_unit];

	switch (mp->mslg_format) {
	case M_FM_CNTERR:
		printx("RQDX disk controller error\n");
		MscpDecodeStatusEvent( mp->mslg_event );
		break;
	case M_FM_BUSADDR:
		printx("RQDX error accessing host memory");
		printx("at addrress 0x%x\n", mp->mslg_bus_addr);
		MscpDecodeStatusEvent( mp->mslg_event );
		break;
	case M_FM_DISKTRN:
		printx("Device %s%d: %s had a disk transfer error.\n",
			ui->device_type, mp->mslg_unit,
			ui->media_name);
		MscpDecodeStatusEvent( mp->mslg_event );
		break;
	case M_FM_SDI:
		printx("Device %s%d: %s had a SDI error.\n",
			ui->device_type, mp->mslg_unit,
			ui->media_name);
		MscpDecodeStatusEvent( mp->mslg_event );
		break;
	case M_FM_SMLDSK:
		printx("Device %s%d: %s had a small disk error.\n",
			ui->device_type, mp->mslg_unit,
			ui->media_name);
		MscpDecodeStatusEvent( mp->mslg_event );
		break;
	default:
		printx("RQDX unknown error ( event 0%o, unit %d, format %d )\n",
			mp->mslg_event, mp->mslg_unit, mp->mslg_format);
		MscpDecodeStatusEvent( mp->mslg_event );
		break;
	};
}

/****************************************************************************
 * Routine Name:	MscpDecodeStatusEvent
 *
 * Description:
 *
 *	This routine decodes an MSCP status or event code and prints
 *	the corresponding English error message.  MSCP codes have a 5 bit
 *	major error code and an 11 bit subcode.
 *
 * Inputs:
 *
 *	code	Word status code to be decoded
 *
 * Outputs:	None.
 *
 * Implicit Inputs/Outputs:	None.
 *
 */

void MscpDecodeStatusEvent( code )
unsigned short code;
{
	unsigned short major_code = (code & 0x1f);
	unsigned short sub_code = (code >> 5);
	char *m, *s;

	switch (major_code) {
	case M_ST_SUCC:
		m = "Success";
		if (sub_code == 0) s = "Normal";
		else
		if (sub_code == 1) s = "Spin-down ignored";
		else
		if (sub_code == 2) s = "Still connected";
		else
		if (sub_code == 4) s = "Duplicate unit number";
		else
		if (sub_code == 8) s = "Already online";
		else
		if (sub_code == 16) s = "Still online";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_ICMD:
		m = "Invalid command";
		if (sub_code == 0) s = "Invalid message length";
		else
			s = "Field in error";
		break;

	case M_ST_ABRTD:
		m = "Command aborted";
		s = "";
		break;

	case M_ST_OFFLN:
		m = "Unit offline";
		if (sub_code == 0) s = "Unit unknown or online to another";
		else
		if (sub_code == 1) s = "No volume mounted or drive disabled";
		else
		if (sub_code == 2) s = "Unit is inoperative";
		else
		if (sub_code == 4) s = "Duplicate unit number";
		else
		if (sub_code == 8) s = "Unit disabled by field service";
		else
			s = "Unknown erorr sub-code";
		break;

	case M_ST_AVLBL:
		m = "Unit available";
		s = "";
		break;

	case M_ST_MFMTE:
		m = "Media format error";
		if (sub_code == 2) s = "FCT unreadable - invalid sector head";
		else
		if (sub_code == 3) s = "FCT unreadable - data sync timeout";
		else
		if (sub_code == 5)
			s = "Disk isn't formatted with 512 byte sectors";
		else
		if (sub_code == 6) s = "Disk isn't formatted or FCT corrupted";
		else
		if (sub_code == 7)
			s = "FCT unreadable - Uncorrectable ECC error";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_WRTPR:
		m = "Write protected";
		if (sub_code == 128) s = "Unit is software write protected";
		else
		if (sub_code == 256) s = "Unit is hardware write protected";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_COMP:
		m = "Compare error";
		s = "";
		break;

	case M_ST_DATA:
		m = "Data error";
		if (sub_code == 2) s = "Header compare error";
		else
		if (sub_code == 3)
			s = "Data sync not found (data sync timeout)";
		else
		if (sub_code == 7) s = "Uncorrectable ECC error";
		else
		if (sub_code == 8) s = "One symbol ECC error";
		else
		if (sub_code == 9) s = "Two symbol ECC error";
		else
		if (sub_code == 10) s = "Three symbol ECC error";
		else
		if (sub_code == 11) s = "Four symbol ECC error";
		else
		if (sub_code == 12) s = "Five symbol ECC error";
		else
		if (sub_code == 13) s = "Six symbol ECC error";
		else
		if (sub_code == 14) s = "Seven symbol ECC error";
		else
		if (sub_code == 15) s = "Eight symbol ECC error";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_HSTBF:
		m = "Host buffer access error";
		if (sub_code == 1) s = "Odd transfer address";
		else
		if (sub_code == 2) s = "Odd byte count";
		else
		if (sub_code == 3) s = "Non-existent memory error";
		else
		if (sub_code == 4) s = "Host memory parity error";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_CNTLR:
		m = "Controller error";
		if (sub_code == 1) s = "SERDES overrun error";
		else
		if (sub_code == 2) s = "EDC error";
		else
		if (sub_code == 3) s = "Inconsistent internal data structure";
		else
			s = "Unknown error sub-code";
		break;

	case M_ST_DRIVE:
		m = "Drive error";
		if (sub_code == 1) s = "SDI command timed out (no response)";
		else
		if (sub_code == 2) s = "Controller detected transmission error";
		else
		if (sub_code == 3) s = "Positioner error (mis-seek)";
		else
		if (sub_code == 4) s = "Lost read/write ready";
		else
		if (sub_code == 5) s = "Driver clock dropout";
		else
		if (sub_code == 6) s = "Lost receiver ready between sectors";
		else
		if (sub_code == 7) s = "Drive detected error";
		else
		if (sub_code == 8) s = "Controller detected pulse";
		else
			s = "Unknown error sub-code";
		break;

	default:
		m = "Unknown error code";
		s = "Unknown error sub-code";
		break;
	}

	printx("      \"%s - %s\"\n", m, s);
}

