/*
 * V Kernel - Copyright (c) 1982 by David Cheriton, Tim Mann
 * (Transliterated from Zed and Verex Kernel)
 *
 * Header file describing Vax memory mapping
 */

#ifndef MEMORY
#define MEMORY

#define PAGESIZE	512		/* in bytes */
#define PAGESIZEBITS	9		/* LOG2 (512) */
#define	SEGMENTSIZE	0x40000000	/* in bytes */
#define	IOSPACESIZE	0x2000		/* in bytes */
#define	PHYSMEMSIZE	0x400000	/* in bytes */
#define	MAXPAGEFRAMES	(PHYSMEMSIZE>>PAGESIZEBITS)
#define	NUMIOPAGES	(IOSPACESIZE>>PAGESIZEBITS)
#define	IOMEMBASE	0x20000000	/* base address of physical IO space */

#define	P0SEGSTART	0x00000000
#define	P1SEGSTART	0x40000000
#define	SYSSEGSTART	0x80000000

#define KERNEL_START	0x0	/* Kernel is loaded at this physaddress */
#define PHYS_TEAM_START	0x40000	/* First team is at this physaddress */
#define TEAM_START	0x0	/* Where the teams start virtually */
#define TEAM_LIMIT	PHYSMEMSIZE	/* Largest Team Size */

#define MEMLIMIT	PHYSMEMSIZE	/* End of mappable address space */
#define STACK_SIZE	0x2000   /* should be enough */

/* A virtual address */
typedef struct
  {
    unsigned	offset:9,
		page:21,
		segment:2;
  } vaddr_t;

/* a physical page */
typedef union pageu
  {
    union pageu	*nextpage;
    char	byte[PAGESIZE];
    short	word[PAGESIZE/sizeof(short)];
    long	longword[PAGESIZE/sizeof(long)];
  } Page;

/* a Page table Entry */
typedef union
  {
    unsigned	word;
    struct
      {
        unsigned pfn:21,	/* the physical page frame number */
		 unused:4,
		 mbz:1,
		 modified:1,	/* page modified ? */
		 protection:4,	/* protection code */
		 valid:1;	/* valid entry ? */
      } fields;
  } PageTableEntry;

/* A Qbus page map entry */
typedef union
  {
    unsigned	word;
    struct
      {
	unsigned pfn:15,	/* local pf to be used for this qbus addr */
		 unused:16,
		 valid:1;	/* valid entry ? */
      } fields;
  } QBusMapEntry;

#define	PageOffs(addr)	((unsigned)(addr) & (PAGESIZE-1))
#define	SegOffs(addr)	((unsigned)(addr) & (SEGMENTSIZE-1))


/* round addr up to the nearest page boundary */
#define	PageAlign(addr) \
	((((long)(addr)-1) & ~(PAGESIZE-1)) + PAGESIZE)

/* return the page number of the next unused page after addr */
#define	NumPages(addr) \
	(PageAlign(addr)>>PAGESIZEBITS)

/* return the number of bytes required to create a page table for pages. */
#define	PageTblLen(pages) \
	(sizeof(PageTableEntry)*(pages))

/* length of a team's page table in bytes */
#define	TEAMPTLEN \
	PageAlign(PageTblLen(MAXPAGEFRAMES))

/* length of the system page table in PageTableEntry's */
#define	SYSPTLEN \
	(MAXPAGEFRAMES+NUMIOPAGES+MAX_TEAMS*NumPages(TEAMPTLEN))

/* for a given team number, the index into the system page table containing
*  the map for the team's page table */
#define	TEAMSYSPTSTART(teamno) \
	(MAXPAGEFRAMES+NUMIOPAGES+(teamno)*NumPages(TEAMPTLEN))

/* for a given team number, gives the base address of team's page table in
*  system space */
#define	TEAMPTBASE(teamno) \
	(SYSSEGSTART+PHYSMEMSIZE+IOSPACESIZE+(teamno)*TEAMPTLEN)

/* Usable Protection Codes */
#define	VM_____		(0<<27)
#define	VM_RW__		(2<<27)
#define	VM_R___		(3<<27)
#define	VM_RWRW		(4<<27)
#define	VM_RWR_		(12<<27)
#define	VM_R_R_		(15<<27)
#define	VM_VALID	(1<<31)

/* Probe values, for probing memory */
#define	PROBE_INIT	0
#define	PROBE_READ	1
#define	PROBE_PRESERVE	2

/* a nil pointer */
#define	NOPAGE	((Page *)(~0))

/*
 * Qbus memory map.  This maps Q-bus memory addresses, generated by DMA
 * devices on the Q-bus, to addresses in the CPU's local memory space.
 * The MicroVAX I doesn't have any local memory (it's all on the Q-bus)
 * so this map is unnecessary.
 */
#define QBMAP_Valid		(1<<31)
#define QBMAP_NumEntries	(0x2000)
#define	QBMAP_Base		(0x20088000)

/*
 * Mapping the other way - from CPU physical addresses into Q-bus memory
 * addresses.  On the MicroVAX I they are identical, whereas on the 
 * MicroVAX II Q-bus memory appears at 30000000-303FFFFF in the CPU's
 * physical address space.  Memory.c sets this offset appropriately.
 */
extern unsigned QbusMemOffset;/* It's an address, but using Unspec is messy */
#define UVAX1_QBUSMEMOFFSET 0
#define UVAX2_QBUSMEMOFFSET 0x30000000

#endif MEMORY
