/*
 * V Kernel - Copyright (c) 1982 by David Cheriton, Tim Mann
 *
 * Memory mapping routines--machine dependent
 */

#include "process.h"
#include "memory.h"
#include "interrupt.h"
#include "Vexceptions.h"
#include "Vquerykernel.h"
#include "b.out.h"
#include "bootparam.h"

/* Imports */
extern MachineConfigurationReply MachineConfig;
extern	(*SysControlBlock[])();
extern RestartParamBlock RPB;

/* Exports */
struct bhdr	KernelBheader = {0},
		TeamBheader = {0};	/* Force to data segment */
Team		*AddressableTeam;	/* the currently addressable team */
unsigned	AddressableTemp;	/* temporary for SetAddressableTeam */
unsigned	QbusMemOffset;
SystemCode      ClearModifiedPages();

/* Pseudo Locals (qvss might use them) */
Page	*FreePageList = NOPAGE;	/* Free list of pages */
int	FreePages = 0;
PageTableEntry	SysPageTable[SYSPTLEN];

/* Forward */
unsigned	GetMemorySize();

Init_memory_mapping()
  /*
   * Setup the memory mapping as required.  This routines starts off in
   *   Physical memory and ends up in Virtual memory.  It prints a message
   *   if it thinks it succeeds.
   *
   * We assume the loader program has placed the kernel at KERNEL_START and
   *   the first team at PHYS_TEAM_START, with the team's b.out header just
   *   below PHYS_TEAM_START, and the kernel's b.out header just below that.
   *
   * We assume that all the physical memory is allocated sequentially
   *  starting from 0.  If there are holes, things get more complicated.
   * This routine maps memory as follows:
   * The kernel is assigned addresses between 0 and PHYS_TEAM_START.  Any
   *   unused memory in this area is added to the free list.
   * Addresses from PHYS_TEAM_START to TEAM_LIMIT are available to teams.
   *   The first team is initially given just enough mapped memory in this area
   *   to hold its code and data.  All other memory is added to the free
   *   list.
   */
  {
    register int	r11;
    register int	pages;
    register char	*cptr;
    register int	i;
    register unsigned	stopat, page;
    extern char etext, edata, end;

    /* Copy the b.out headers into kernel data space */
    TeamBheader = *((struct bhdr *)(PHYS_TEAM_START - sizeof(struct bhdr)));
    KernelBheader = *((struct bhdr *)(PHYS_TEAM_START-2*sizeof(struct bhdr)));

    /* If the first team is NMAGIC (410), then the loader should have moved */
    /*   the data section to the first 1K boundary after the text.  We      */
    /*   round tsize up to the next 1K - a bit of a crock, but then we can  */
    /*   forget about thbe problem.					    */
    if ( TeamBheader.fmagic == NMAGIC )
	TeamBheader.tsize = (TeamBheader.tsize + SEGSIZE - 1) & ~(SEGSIZE-1);

    /* Zero the bss areas in case the loader didn't do it */
    clear(PHYS_TEAM_START + TeamBheader.tsize + TeamBheader.dsize,
	  TeamBheader.bsize);
    clear(&edata, &end-&edata);

    FindConfiguration();

    pages = GetMemorySize();

    switch (MachineConfig.processor)
      {
	case PROC_UVAX1:
	  QbusMemOffset = UVAX1_QBUSMEMOFFSET;
	  break;

	case PROC_UVAX2:
	  QbusMemOffset = UVAX2_QBUSMEMOFFSET;
	  break;

	default:
	  printx("Don't know where Q-Bus memory should be mapped\n");
	  break;
      }

    MachineConfig.fastMemory = pages * PAGESIZE;
    MachineConfig.slowMemory = 0;
    MachineConfig.memory = MachineConfig.fastMemory + 
	MachineConfig.slowMemory;

    printx("Memory: 0x%x bytes.\r\n", MachineConfig.fastMemory);

    /* map all of physical memory into system space with proper protections */
    stopat = NumPages(KERNEL_START);
    for (page = 0; page < stopat; page++) {
	SysPageTable[page].word = page | VM_VALID | VM_RW__;
	FreePageframe(page);
    }
    stopat += NumPages(KernelBheader.tsize);
    for (; page < stopat; page++)
	SysPageTable[page].word = page | VM_VALID | VM_R___;
    stopat += NumPages(KernelBheader.dsize+KernelBheader.bsize);
    for (; page < stopat; page++)
	SysPageTable[page].word = page | VM_VALID | VM_RW__;

    stopat = NumPages(PHYS_TEAM_START);
    for (; page < stopat; page++) {
	SysPageTable[page].word = page | VM_VALID | VM_RW__;
	FreePageframe(page);
    }
    
    stopat += NumPages(TeamBheader.tsize);
    for (; page < stopat; page++)
	SysPageTable[page].word = page | VM_VALID | VM_RW__;
    stopat += NumPages(INIT_STACK+TeamBheader.dsize+TeamBheader.bsize);
    for (; page < stopat; page++)
	SysPageTable[page].word = page | VM_VALID | VM_RW__;

    for (; page < pages; page++) {
	SysPageTable[page].word = page | VM_VALID | VM_RW__;
	FreePageframe(page);
    }

    for (; page < MAXPAGEFRAMES; page++)
	SysPageTable[page].word = page | VM_____;

    /* map all of IO space into system space with proper protections */
    for (; page < MAXPAGEFRAMES + NUMIOPAGES; page++)
	SysPageTable[page].word = (IOMEMBASE/PAGESIZE) | (page-MAXPAGEFRAMES) | VM_VALID | VM_RW__;

     /* make all of the team page tables nonexistent */
    for (; page < SYSPTLEN; page++)
	SysPageTable[page].word = 0;

    /* initialize the internal registers so we can turn on memory management */
    r11 = (unsigned)SysPageTable;
    asm("	mtpr	r11, $sbr");
    r11 = SYSPTLEN;
    asm("	mtpr	r11, $slr");
    asm("	mtpr	$0, $p0br");
    asm("	mtpr	$0, $p0lr");
    asm("	mtpr	$-1, $p1br");
    asm("	mtpr	$0x200000, $p1lr");

    /* turn on memory management */
    asm("	mtpr	$0, $tbia");
    asm("	mtpr	$1, $mapen");
  }


LockSCB()
  {
    /*
     * Write-protect the system control block to keep it from being
     * stomped on.  This assumes that the system control block be
     * page-aligned, which the hardware requires anyway, and that
     * it be a multiple of the page length (since any remaining
     * data in the last page would also be non-writable).
     */
    long page;

    for (page = ((long)&SysControlBlock[0] & ~0x80000000) >> PAGESIZEBITS;
	 page < ((long)&SysControlBlock[NumExceptions] & ~0x80000000) >>
								PAGESIZEBITS;
	 page++)
	SysPageTable[page].word = page | VM_VALID | VM_R___;
  }

FreePageframe(pageframe)
    register unsigned pageframe;
  /*
   * This function adds the specified page frame to the free list.
   * The addresses it uses for links are addresses that map to
   * the System segment.
   */
  {
    register Page *page = (Page *)(SYSSEGSTART | (pageframe<<PAGESIZEBITS));
    
    page->nextpage = FreePageList;
    FreePageList = page;
    FreePages++;
  }


SetUpRootPT(pd, size)
    Process *pd;
    int size;
  /*
   * This function maps the first team's address space into the p0 region,
   * and allocates space for its page table.  It does not need to
   * invalidate the translation buffers because the first team's address
   * space will not be referenced before we do a ldpctx, which does that
   * for us.
   */
  {
    register int page, i, stopat;
    register PageTableEntry *pte;
    register int numPages = NumPages(size);
    register Page *newpg;
    Team *td = pd->team;

    /* give the root team enough space for its page table */
    stopat = NumPages(PageTblLen(numPages));
    pte = &SysPageTable[td->team_space.p0index];
    for (i = 0; i < stopat; i++)
      {
	newpg = FreePageList;
	FreePageList = newpg->nextpage;
	FreePages--;
	pte->word = (SegOffs(newpg)>>PAGESIZEBITS) | VM_VALID | VM_RW__;
	pte++;
      }
    
    /* Initialize the root team's page table */
    pte = (PageTableEntry *)TEAMPTBASE(td->team_space.teamno);
    page = PHYS_TEAM_START>>PAGESIZEBITS;
    for (i = 0; i < numPages; i++)
      {
	(pte++)->word = page++ | VM_VALID | VM_RWRW;
      }

    pd->proc_state.ProgLenReg = numPages;
    td->team_space.size = size;
    td->team_space.p0len = numPages;
    td->team_space.qvss = 0;

  }


SystemCode SetTeamSize( active )
    Process *active;
  /*
   * Set the memory size for a team, mapping or unmapping pages as needed.
   */
  {
    register KernelRequest *req = (KernelRequest *) &(active->msg);
    register Process *pd;

    if( !(pd = MapPid(req->pid)) ) return( NONEXISTENT_PROCESS );

    req->unspecified[0] = KSetTeamSize(pd->team, req->unspecified[0]);
    return( OK );
  }


int KSetTeamSize(td, size)
    register Team *td;
    register unsigned size;
  /*
   * Called by SetTeamSize to do the actual work.  Note that AllocatePage
   * and FreePage change the value of td->team_space.p0len .
   */
  {
    register int numPages;
    register int fail = 0;
    register Process *pd;

    /* Range check on size */
    if (size >= TEAM_LIMIT) size = TEAM_LIMIT;

    numPages = NumPages(size) - td->team_space.p0len;

    if (numPages >= 0) { /* More memory needed */
	for (; numPages; numPages--) {
	    if (fail = AllocatePage(td)) break;
	}
    }
    else {  /* Less memory needed */
	for (; numPages; numPages++) {
	    FreePage(td);
	}
    }

    size = fail ? td->team_space.p0len<<PAGESIZEBITS : size;

    /* if MapPid doesn't work things are so FUBAR that it doesn't matter */
    if (!td->team_space.qvss)
      {
	pd = MapPid(td->team_root);
	if (pd != NULL)
	    KAssignSize(pd, td, NumPages(size));
      }

    return(td->team_space.size = size);
  }

KAssignSize(pd, td, size)
    register Process	*pd;
    register Team	*td;
    register unsigned	size;
  /*
   * Assign size to the Process contexts of all the processes in this team.
   *  this is a bad implementation, because there is no single list of all
   *  the processes on this team, so we search the process tree, looking
   *  for processes with this team descriptor, and setting their ProgLenReg
   *  fields to be size.
   */
  {
    if (pd->team == td)
      {
        /* set this pd and search its kids */
	pd->proc_state.ProgLenReg = size;
	for (pd = pd->son; pd != NULL; pd = pd->brother)
	    KAssignSize(pd, td, size);
      }
  }


SystemCode ClearModifiedPages( req, segPtr, segSize )
KernelRequest *req;
register long **segPtr;
unsigned long segSize;
  {
    return MODE_NOT_SUPPORTED;  /* Some day... */
  }


FreePage(td)
    Team *td;
  /*
   * Add a page to the free list, unmapping it from the team's context
   */
  {
    register int	r11;
    register Page	*newpg;
    register int	pages = td->team_space.p0len;
    register int	ptbllen = PageTblLen(pages - 1);
    register PageTableEntry	*pte;

    if (td->team_space.size == 0)
	return;
    pte = (PageTableEntry *)td->team_space.p0base + pages - 1;
    newpg = (Page *)((pte->fields.pfn << PAGESIZEBITS) | SYSSEGSTART);
    pte->word = 0;
    r11 = (pages<<PAGESIZEBITS) - 1;
    asm("	mtpr	r11, $tbis"); /* invalidate the cache */

    /* Add newpg to the free list */
    newpg->nextpage = FreePageList;
    FreePageList = newpg;
    FreePages++;

    if (0 == PageOffs(ptbllen)) { /* free system page frame */
	pte = &SysPageTable[td->team_space.p0index + NumPages(ptbllen)];
	newpg = (Page *)((pte->fields.pfn << PAGESIZEBITS) | SYSSEGSTART);
	pte->word = 0;
	r11 = (td->team_space.p0index + NumPages(ptbllen))<<PAGESIZEBITS;
	r11 |= SYSSEGSTART;
	asm("	mtpr	r11, $tbis"); /* invalidate the cache */

	/* Add newpg to the free list */
	newpg->nextpage = FreePageList;
	FreePageList = newpg;
	FreePages++;
    }

    td->team_space.p0len--;
  }


AllocatePage(td)
    Team *td;
  /*
   * Map a page into the specified team's context at the next available
   *  address, if a free page can be found.  Runs in Virtual memory.
   *  Return 0 for success, 1 for failure.
   */

  {
    register int	r11;
    register Page	*newpg;
    PageTableEntry	*pte;
    register int	pages = td->team_space.p0len;
    register int	ptbllen = PageTblLen(pages);

    /* Find a page, if available */
    if ( (newpg = FreePageList) == NOPAGE ) {
	return(1);
    }
    if (0 == PageOffs(ptbllen)) {
        /* Map a new page into the team's page table */
        if ( FreePageList->nextpage == NOPAGE ) {
	    return(1);
        }
	SysPageTable[td->team_space.p0index+NumPages(ptbllen)].word
		 = (SegOffs(newpg)>>PAGESIZEBITS) | VM_VALID | VM_RWRW;
	r11 = td->team_space.p0base;
	r11 += ptbllen<<PAGESIZEBITS;
	asm("	mtpr	r11, $tbis"); /* invalidate the cache */

        /* Remove it from the free list */
        FreePageList = newpg->nextpage;
        FreePages--;
        newpg = FreePageList;
    }

    pte = (PageTableEntry *)td->team_space.p0base + pages;
    pte->word = (SegOffs(newpg)>>PAGESIZEBITS) | VM_VALID | VM_RWRW;
    r11 = td->team_space.size;
    asm("	mtpr	r11, $tbis"); /* invalidate the cache */

    /* Remove it from the free list */
    FreePageList = newpg->nextpage;
    FreePages--;

    td->team_space.p0len++;

    return (0);
  }


ReclaimMemory(td)
    Team *td;
  /*
   * Reclaim the memory allocated to a defunct team.
   */
  {
    KSetTeamSize(td, 0);
    QvssUnmap(td);
  }


InterTeamCopy(dest, dteam, src, steam, bytes)
    Team *dteam, *steam;
    char *dest, *src;
    unsigned long bytes;
  /*
   * This routine is used by MoveFrom and MoveTo to copy bytes between
   *   different teams.  We do have to worry about the case where one
   *   of the `teams' is in fact the kernel process.
   */

  {
    register int r11;
    Team *oldteam;

    /* save current context */
    oldteam = GetAddressableTeam();

    /* map source team into P0 */
    if (((int)src & SYSSEGSTART) == 0) {
	r11 = steam->team_space.p0len;
	asm("	mtpr	r11, $p0lr");
	r11 = steam->team_space.p0base;
	asm("	mtpr	r11, $p0br");
    }
    
    /* do the copy */
    if (((int)dest & SYSSEGSTART) == 0) {
	/* map destination team into P1 */
	asm("	mtpr	$0, $p1lr");
	r11 = dteam->team_space.p0base;
	asm("	mtpr	r11, $p1br");

	Copy((char *)(P1SEGSTART | SegOffs(dest)), src, (unsigned)bytes);

	asm("	mtpr	$0x200000, $p1lr");
	asm("	mtpr	$-1, $p1br");
    }
    else
	Copy(dest, src, (unsigned)bytes);

    /* restore the old contexts */
    SetAddressableTeam(oldteam);
  }


unsigned GetMemorySize()
  /*
   * Probe to determine size of memory on a VAX.  Runs in physical memory.
   *  Returns the amount of memory, in pages.
   */
  {
    register unsigned pages;

    /* Test pages until one doesn't work */
    pages = PHYS_TEAM_START/PAGESIZE;  /* Don't test kernel memory */
    while (pages < MAXPAGEFRAMES
		&& Probe((short *)(pages<<PAGESIZEBITS), PROBE_PRESERVE))
	pages++;	/* more memory here */
#ifdef MICROVAX
    /*
     * The MicroVAX primary bootstrap probably passed us various magic numbers,
     *   including some that tell us what pages the MicroVAX monitor wants to
     *   itself (way up the top of memory).  We try to use these here, but are
     *   a bit suspicious: if it looks as though the Monitor took more than 64K
     *   (arbitrary figure, pulled out of the air) we ignore it.  In practice,
     *   it should only take two pages plus enough pages for the bitmap (one
     *   page for each two megabytes of memory).  We could check the bitmap for
     *   consistency, or even use the bitmap to initialize our free list, but
     *   that might be just too thorough.
     */
    if ((unsigned)RPB.pagecount              < pages &&
	(unsigned)RPB.pagecount + 65536/512 >= pages )
      {
	printx("Left 0x%x bytes for MicroVAX Monitor\r\n", 
		512 * (pages - RPB.pagecount) );
	pages = RPB.pagecount;
      }
#endif MICROVAX
    return(pages);
  }

Probe(address, how)
    short *address;
    int how;
  /*
   * Look for memory at address.
   * If how == PROBE_INIT,
   *   write before reading to initialize, then look for writeable memory.
   * If how == PROBE_PRESERVE,
   *   look for writeable memory, but preserve contents of location.
   * If how == PROBE_READ,
   *   just try reading and see whether it causes a bus error.
   */
  {
    register short x;
    extern int NoMemOnRead(), NoMemOnWrite();
    int (*oldBusVector)(), (*oldCheckVector)();
    int i;

    /* Set bus error vector */
    oldBusVector = SysControlBlock[VecBusTimeout];
    oldCheckVector = SysControlBlock[VecMachineChk];
    setexvec(NoMemOnWrite, VecBusTimeout);
    setexvec(NoMemOnRead, VecMachineChk);

    if (how == PROBE_INIT) 
	*address = 0x5c5c;	/* Initialize the test location */

    x = ~*address;		/* Read from it */
    
    if (how == PROBE_READ) {
	/* Success if we get here with no bus error */
	setexvec(oldBusVector, VecBusTimeout);
	setexvec(oldCheckVector, VecMachineChk);
	return (1);
    }
    else {
	*address = x;		/* Write back the complement */
	if (*address == x) {
	    /* Success */
	    *address = ~x;
	    setexvec(oldBusVector, VecBusTimeout);
	    setexvec(oldCheckVector, VecMachineChk);
	    return (1);
        }
    	else {
	    /* Failure */
	    asm("ProbeFailed:");
	    setexvec(oldBusVector, VecBusTimeout);
	    setexvec(oldCheckVector, VecMachineChk);
            return (0);
        }
    }

    asm("	.align	2");
    asm("_NoMemOnWrite:");
    asm("	addl2	$8, sp");
    asm("	mtpr	$0x1c, $ipl");
    asm("	jbr	ProbeFailed");

    asm("	.align	2");
    asm("_NoMemOnRead:");
    asm("	addl2	$48, sp");
      {
	extern MachineConfigurationReply MachineConfig;
	int i;

	if (MachineConfig.processor == PROC_UVAX2)
	  {
	    i = *(int *)(0x20080004);
	    *(int *)(0x20080004) = i;
	  }
      }
    ;asm("	mtpr	$0x1c, $ipl");
    asm("	mtpr	$1, $mcesr");
    asm("	jbr	ProbeFailed");

  }

/*
 * More routines that tell you whether you can read a given location in
 *   memory.  These ones, however, are intended specifically for virtual
 *   addresses.  They don't actually try to read from the location, just
 *   look at the memory mapping registers and tables, which had better be set
 *   up properly.
 *
 * VIRTUAL MEMORY IMPLEMENTORS NOTE: If we ever decide to page the per-process
 *   (well, per-team) page tables, then the P0 and P1 cases will need some
 *   work.
 */

KernelCanRead(address, length)
    unsigned	address, length;
  {
    unsigned	testadr;
    /* Test one byte in each page from address to address+length-1 */
    /*   If we can read one byte in the page, we can read anything */
    for (testadr = address; (unsigned)(testadr-address) < length;
			     /* ^ Does this work for all cases? */
	 testadr = (testadr+PAGESIZE) & (-PAGESIZE) /*gag*/)
	if (! KernelCanReadByte((char *)testadr))
	    return 0;
    return 1;
  }

KernelCanReadByte(address)
    char	*address;
  {
    register unsigned r11 = 
		((unsigned)address &(SEGMENTSIZE-1)) >> PAGESIZEBITS;
    register PageTableEntry *r10;

    switch((unsigned)address >> 30)
      {
	case 3:
	    ;asm("naughty:");
	    return 0;
	case 2:
	    asm("	mfpr	$slr,	r0");
	    asm("	cmpl	r11,	r0");
	    asm("	bgequ	naughty	  ");
	    r10 = &(SysPageTable[r11]);
	    break;
	case 1:	   /* We don't use P1 at present, but let's get this right */
	    asm("	mfpr	$p1lr,	r0");
	    asm("	cmpl	r11,	r0");	/* Bounds-check */
	    asm("	blssu	naughty	  ");
	    asm("	ashl	$2, r11,r10");	/* Form virtual address of */
	    asm("	mfpr	$p1br,	r0");   /*   P1 page table entry   */
	    asm("	addl2	r0,	r10");
		/* Check that this gives an address in System Space.  Should */
		/*   be, unless our software is really screwy or the hardware*/
		/*   is less clever than we thought.  Still, check it anyway;*/
		/*   if things went wrong we might otherwise recurse.	     */
	    asm("	bgeq	naughty   ");
	    if (! KernelCanReadByte(r10))     /* We'll actually want to read*/
		return 0;		      /* a PTE (4 bytes) but this   */
	    break;			      /* gives the right answer     */
	case 0:
	    asm("	mfpr	$p0lr,	r0");
	    asm("	cmpl	r11,	r0");
	    asm("	bgequ	naughty   ");
	    asm("	ashl	$2, r11,r10");
	    asm("	mfpr	$p0br,	r0");
	    asm("	bgeq	naughty   "); /* Checks P0BR->System space  */
	    asm("	addl2	r0,	r10");
	    if (! KernelCanReadByte(r10))
		return 0;
	    break;
      }
    if (!r10->fields.valid || r10->fields.protection < 2)
	return 0;
    return 1;
  }
  

/*
 * Get memory statistics for QueryKernel operation
 */
GetMemoryStats(reply)
    register MemoryStatisticsReply *reply;
  {
    reply->unusedFastMemory = FreePages * PAGESIZE;
    reply->unusedSlowMemory = 0;
  }


/*
 * Routines to set/get the currently addressable team space.
 * Very simple on the VAX.
 */
/*
Team *GetAddressableTeam()
  {
    return(AddressableTeam);
  }

SetAddressableTeam(newteam)
    register Team *newteam;
  {
    register int r10;
    
    AddressableTeam = newteam;
    r10 = newteam->team_space.p0base;
    asm("	mtpr	r10, $p0br");
    r10 = newteam->team_space.qvss ? MAXPAGEFRAMES : newteam->team_space.p0len;
    asm("	mtpr	r10, $p0lr");
    asm("	mtpr	$0, $tbia");
  }
*/


/*
 * Map kernel local memory into the Qbus address space so that
 * it can be DMA'd with.  Since we only map kernel local memory, all our
 * buffers must be statically allocated.
 *
 * Assumes that we are running in physical memory.
 *
 * We never map the page mapping registers into kernel space, so once this
 * is done, it can't be changed easily.
 */
MapLocalIntoQBus()
  {
    register QBusMapEntry *map;
    register int i;

    map = (QBusMapEntry *)QBMAP_Base;
    for (i = 0; i < NumPages(PHYS_TEAM_START-KERNEL_START); i++, map++)
	map->word = QBMAP_Valid | i;
    for (; i < QBMAP_NumEntries; i++, map++)
	map->word = 0;
    printx("Mapped Local -> QBus\n");
  }

