/*
 * Distributed V Kernel
 * Copyright (c) 1982 by David Cheriton, Tim Mann, Willy Zwaenepoel
 * (Transliterated from Zed and Verex Kernel)
 *
 * Machine-dependent kernel routines
 */

#include "process.h"
#include "asmdefs.h"
#include "asmprocess.h"
#include "memory.h"
#include "Vquerykernel.h"
#include "b.out.h"

/* Imports */
extern Process	*Active;
extern unsigned long MaxProcesses;
extern char	KStack[], IStack[];
extern int	Found_Qvss;

/* Saved configration information */
MachineConfigurationReply	MachineConfig;
PeripheralConfigurationReply	PeripheralConfig;
KernelConfigurationReply	KernelConfig;

unsigned char *LastPeripheral = &PeripheralConfig.peripheral[0];

short DiskType = PRF_NONE;
short FramebufferType = PRF_NONE;
short MouseType = PRF_NONE;
short ConsoleInputDevice = PRF_CONSOLE_TERMINAL;
short ConsoleOutputDevice = PRF_CONSOLE_TERMINAL; 

/* Temporoary location for storing ipl in the Lockq and Unlockq macros */
long GlobalLockKludge;

/* These would break the optimizer if they appeared inside the function */
asm("	.globl	Switch");
asm("	.globl	ActivateReadyqHead");

/* Basic Process Implementation */
dummySwitch()

    /*
     * Save the state of the active process and activate the process
     * specified by Readyq.head, loading its state from its process
     * descriptor.  It is assumed to be called with interrupts
     * disabled.  Switch and ActivateReadyqHead are never called as
     * functions; they are jumped to.  The code is written inside this
     * dummy function body to allow C statements as well as asms to be
     * used.
     *
     * Warning: Do not use any automatic variables in this code, and
     * remember not to declare more register variables than there are
     * registers!
     */

  {
    extern SyncQueue Readyq;
    register KProcessor_state *r11;
    register Team *r10;
    register Process *pd;
    register Unspec (*finish_up)();

    /*
     * Switch is jumped to when we know that we are going to do a
     * context switch.  Currently, however, the only difference between
     * it and ActivateReadyqHead is that Switch saves the process context.
     *
     * Here we save our state on the stack.  It is assumed that the
     * PC and PSL are still there from whatever got us here (an interrupt
     * or a fault), so we can do a svpctx right off.
     */
    asm("Switch:");
    asm("	svpctx");

    asm("ActivateReadyqHead:");

newprocess:
    Active = pd = Readyq.head;    /* Remove new active process readyq */
    Readyq.head = pd->link;
    pd->queuePtr = NULL;
    r10 = pd->team;
    SetAddressableTeam(r10);
    /* An optimized version of the above is:
	r10 = AddressableTeam = pd->team;
	if (pd->team->team_space.qvss)
	  {
	    asm("	mtpr	$0x2000, $p0lr");
	  }
	else
	  {
	    asm("	mtpr	0x20(r10), $p0lr");
	  }
	;
    */

    /* Call a function to finish up if necessary */
    if (finish_up = pd->finish_up) {

        disable;		/* Ensure running at disable level */
        pd->finish_up = NULL;	/* This has to be in front of the call to
				   finish_up because finish_up may set it to
				   a new value. */
        (*finish_up)(pd);

        /* Check in case we are no longer ready */
        disable;
        if( pd != Readyq.head ) goto newprocess;
	Readyq.head = pd->link;
	pd->queuePtr = NULL;
    }


    r11 = &pd->proc_state;
    
    /* Set the per-process data area pointer */
    if (r11->perProcessLoc)
	*r11->perProcessLoc = r11->perProcess;

    /* HACK: an interrupt while in the kernel may have changed this */
    r11->KernelSP = &KStack[STACK_SIZE];
    
    /* load the new context */
    asm("	mtpr	r11, $pcbb");
    asm("	ldpctx");
    

    /*
     * Message-register interface goodies.  Check whether we're meant to
     *   return a message in registers r3-r10, and do it if necessary.
     *   Always clears, as well as testing, the MSG_FLAG in pd_flags, so
     *   that next time we don't accidentally think we're meant to return
     *   a message in registers.
     *   We have to be careful about preserving registers now, since the
     *   process may originally have been interrupted and we've done the
     *   ldpctx.
     *   Note difference from 680X0 version: there, we only restore all
     *   registers (the equivalent of a ldpctx) if we're not returning a
     *   message in registers.  Here, we do the ldpctx in either case 
     *   (seems like too much trouble to try faking the bits of a ldpctx
     *   that we actually want) so we do it first and then check for
     *   message-register stuff.
     */
    asm("	pushl	r11");
    asm("	movl	_Active, r11");
    asm("	bbcc	$1, _PD_FLAGS(r11), nomessage");
    asm("	movq	_TRAP_PARAMS(r11), r0");
    asm("	moval	_MSG(r11), r11");
    asm("	movq	(r11)+, r3");
    asm("	movq	(r11)+, r5");
    asm("	movq	(r11)+, r7");
    asm("	movq	(r11) , r9");
    asm("	addl2	$4, sp");	/* throw away saved r11 */
    asm("	rei");
    asm("nomessage:");
    asm("	movl	(sp)+, r11");
    asm("	rei");			/* Switch to user process */
  }

#define	MAX_PROCESSES	64
extern Process	*ProcessDescriptors, *EndPds;
extern unsigned long MaxProcesses;
Process		ProcessDescriptorArray[ MAX_PROCESSES ];
Team TeamDescriptorArray[ MAX_TEAMS ];

extern unsigned long MaxTeams;
extern Team *TeamDescriptors;

AllocatePds()
  {
    TeamDescriptors = TeamDescriptorArray;
    KernelConfig.maxTeams = MaxTeams = MAX_TEAMS;
printx("Allocating %d team descriptors.\n", MaxTeams);

    ProcessDescriptors = ProcessDescriptorArray;
    KernelConfig.maxProcesses = MaxProcesses = MAX_PROCESSES;
printx("Allocating %d process descriptors.\n", MaxProcesses);
    EndPds = (Process *) &ProcessDescriptorArray[ MaxProcesses ];
  }


/* Process Management */

Ms_CreateProcess( pd, entry, stack )
    register Process *pd;
    Unspec entry;
    Unspec stack;

  /* Perform the machine-specific portion of process creation,
   * namely initializing the state and such like.
   */
  {
    register MsgStruct *msg;
    register KProcessor_state *state;
    register Team_space *space;
    register int i;

    state = &pd->proc_state;
    space = &pd->team->team_space;
    for (i = 0; i < 14; state->regs[i++] = 0);
    state->UserSP = stack;	/* set the fp = sp = stack */
    state->regs[13] = stack;
    state->KernelSP = &KStack[STACK_SIZE];
    state->ExecSP = state->SuperSP = (Unspec)~0;
    state->PSL = (Unspec)0x03c00000;
    state->PC = entry;	/* Set entry point */
    state->ProgBaseReg = space->p0base;
    state->ProgLenReg = space->qvss ? MAXPAGEFRAMES : space->p0len;
    state->MBZfield1 = 0;
    state->MBZfield2 = 0;
    state->MBZfield3 = 0;
    state->AstLevel = NO_AST_PENDING;
    state->CtrlBaseReg = ~0;
    state->CtrlLenReg = 0x200000;

    /* Fill up message buffer */
    msg = &(pd->msg);
    msg->sysCode = PUT_ON_GREEN_HAT;
    msg->segmentPtr = (char *) 0;
    msg->segmentSize = TEAM_LIMIT;

    /* Grant creator full access to team space */
    pd->dataSegmentPro = READ_ACCESS + WRITE_ACCESS;
    pd->dataSegmentPtr = (Unspec *) 0;
    pd->dataSegmentSize = TEAM_LIMIT;
  }

Ms_CreateTeam( td, pd )
    register Process *pd;
    register Team *td;

  /* Perform the machine-specific portion of team creation.
   */
  {
    register KProcessor_state *state;
    register Team_space *space;

    state = &pd->proc_state;
    space = &td->team_space;
    space->size = 0;
    space->p0len = 0;
    state->ProgBaseReg = space->p0base;
    state->ProgLenReg = 0;
    ReclaimMemory(td);	/* Be sure team space is null */
    space->qvss = 0;
  }


/*** Kernel Initialization ***/


Init_root_team()

  /* Create the first or root team and first process
   */
  {
    extern Team		*TdFreeList, *FirstTeam;
    extern LhnRec	*LhnFreeList;
    extern Process	*IdleProcessPtr;
    extern SystemCode	CreateTeam();
    extern SyncQueue	*RemoveQueue();
    register Process_id pid;
    extern struct bhdr	TeamBheader;
    register KernelRequest *req;
    register unsigned	teamsize;
    register Process	*pd, *active;
    extern Team *AddressableTeam;

    active = IdleProcessPtr;
    AddressableTeam = active->team;
    /* Assume first team is already loaded at TEAM_START */
    teamsize = TeamBheader.tsize + TeamBheader.dsize
		+ TeamBheader.bsize + INIT_STACK;

    TdFreeList->lhn = LhnFreeList;
				/* We need this so that the first process has
				   an lhost available. */
    LhnFreeList->lhn = 0;

    req = (KernelRequest *) &(active->msg);
    req->unspecified[0] = ROOT_PRIORITY;
    req->unspecified[1] = TeamBheader.entry;
    req->unspecified[2] = teamsize-16;
    req->unspecified[3] = 0;
    if( (req->opcode=CreateTeam(active)) != OK )
      {
	printx( "Failed to create first team: %x\n", req->opcode );
        Kabort( "first team" );
      }
    pid = req->pid;
    pd = MapPid( pid );
    RemoveQueue( pd );
    SetUpRootPT(pd, teamsize);
    pd->team->team_priority = 0x0100;
    pd->priority = PROCESS_PRIORITY_MASK | pd->team->team_priority;
    pd->userNumber = SYSTEM_USER;
    FirstTeam = pd->team; /* First team is used for permission checking. */

    /* We'll use the V_BOOT_LHN while checking for lhn collisions, then
     * change our lhn when we find a unique value.
     */
#ifdef LITTLE_ENDIAN
    pd->localPid = pd->pid |= ((V_BOOT_LHN << 16) | LITTLE_ENDIAN_HOST);
#else
    pd->localPid = pd->pid |= (V_BOOT_LHN << 16);
#endif
    NewLhn(pd, pid);
  }

Ms_td_init(td, i)
    register Team *td;
    register unsigned i;

  /*
   * Perform machine-specific initialization of team descriptors
   */
  {
    td->team_space.teamno = i;	/* the team id number */
    td->team_space.size = 0;	/* the initial size */
    td->team_space.p0len = 0;	/* the initial number of pages used */
    td->team_space.qvss = 0;
    td->team_space.p0base = TEAMPTBASE(i);	/* base of the page table */
    td->team_space.p0index = TEAMSYSPTSTART(i);	/* index of page table */
  }

Ms_pd_init(pd, i)
    register Process *pd;
    register unsigned i;
    
  /*
   * Perform machine specific initialization of process descriptors
   */
  {
    register KProcessor_state *state;
    register int j;
    
    state = &pd->proc_state;
    state->KernelSP = &KStack[STACK_SIZE];
    state->ExecSP = state->SuperSP = (Unspec)~0;
    for (j = 0; j < 14; state->regs[j++] = 0);
    state->PSL = (Unspec)0x03c00000;
    state->MBZfield1 = 0;
    state->MBZfield2 = 0;
    state->MBZfield3 = 0;
    state->AstLevel = NO_AST_PENDING;
    state->CtrlBaseReg = ~0;
    state->CtrlLenReg = 0x200000;
  }


Idle()

  /* Idle process function. Wait for an interrupt and
   * loop as soon as redispatched.
   */
  {
    extern Process	Idle_process;
    KProcessor_state *state = &Idle_process.proc_state;
    register int r11;

    /* Initialize the Idle Process */
    state->KernelSP = &KStack[STACK_SIZE];
    state->ExecSP = state->SuperSP = (Unspec)~0;
    for (r11 = 0; r11 < 14; state->regs[r11++] = 0);
    state->PSL = (Unspec)0x00000000;
    state->MBZfield1 = 0;
    state->MBZfield2 = 0;
    state->MBZfield3 = 0;
    state->AstLevel = NO_AST_PENDING;
    state->CtrlBaseReg = ~0;
    state->CtrlLenReg = 0x200000;
    
    /* Add to ready queue. */
    Readyq.head = Active;
    Active->queuePtr = &Readyq;
    Active->state = READY;

    r11 = (int)&KStack[STACK_SIZE];
    asm("    movl	r11, sp");
    asm("    movl	r11, fp");
    asm("    movl	r11, ap");
    asm("    pushl	$0");
    asm("    pushal	loop");		/* Set up for rei to loop */
    asm("    jmp	Switch");	/* Switch to head of readyq */
    asm("loop:");
    while (1);
  }


/*
 * getchar:
 * Kernel version of getchar.
 */
GetChar()
  {
	register int r11;
	int i;

	do
	  {
	    ;asm("    mfpr    $rxcs, r11");
	  }
	while (!(r11 & 0x80));  /* Wait for character */
	;asm("    mfpr    $rxdb, r11"); /* Read it in */
	if ((r11 & 0xff00) != 0)/* Check for break or errors */
	{
	    return -1;
	}
	r11 &= 0x7f;    /* Turn off parity */
	if ((char)r11 == '\r')  /* CRMOD translation */
	    r11 = '\n';
	return r11;
}


/* These are needed for the physical addresses of the device registers */
#include "qvss.h"
#include "rqdx.h"

FindConfiguration()
    /*
     * Determine make of workstation.  Information used by device routines
     *   and QueryKernel operation.  We make a few assumptions here instead
     *   of actually looking for everything.
     *
     */
  {
    register int r11;	/* used to get the config information */
    extern unsigned versionnum; /* from vers.c */

    /* Pseudo MI configuration information */
#ifdef ENET3MEG
    KernelConfig.ikcType = IKC_3MBIT;
#else
#ifdef ENET10MEG
    KernelConfig.ikcType = IKC_10MBIT;
#else
    KernelConfig.ikcType = IKC_NONE;
#endif
#endif
    KernelConfig.maxProcesses = MaxProcesses;
    KernelConfig.maxTeams = MaxTeams;
    KernelConfig.maxLogicalId = MAX_LOGICAL_ID;
    KernelConfig.rootPriority = ROOT_PRIORITY;
    KernelConfig.initStack = INIT_STACK;
    KernelConfig.vmConfig = VM_NONE;
    KernelConfig.versionNumber = versionnum;
    
    /* We KNOW we are a VAX of some sort and this is true of all Vaxen */
    MachineConfig.machine = MACH_VAX;
    *LastPeripheral++ = PRF_CONFREG;

    /* find type of processor */
    asm("	mfpr	$sid, r11");
    MachineConfig.confreg = r11;
    switch (r11 >> 24)
      {
        case 1:
	  MachineConfig.processor = PROC_VAX11_780;
	  printx("VAX-11/780 processor\n");
	  break;

	case 2:
	  MachineConfig.processor = PROC_VAX11_750;
	  printx("VAX-11/750 processor\n");
	  break;

	case 3:
	  MachineConfig.processor = PROC_VAX11_730;
	  printx("VAX-11/730 processor\n");
	  break;

	case 7:
	  MachineConfig.processor = PROC_UVAX1;
	  printx("MicroVax I processor");
	  printx(" (KD32-A%c, %c_floating)\n",
	    r11 & 0x10000 ? 'B' : 'A',
	    r11 & 0x10000 ? 'D' : 'G');
	  break;

	case 8:
	  MachineConfig.processor = PROC_UVAX2;
	  printx("MicroVax II processor\n");
	  MapLocalIntoQBus();
	  break;

	default:
	  Kabort("Unknown processor type\n");
	  break;

      }

    if ( Probe(PhysQvss, PROBE_READ) )
      {
	printx("Found a QVSS framebuffer, mouse, and keyboard\n");
	MouseType = *LastPeripheral++ = PRF_MOUSE_QVSS;
	FramebufferType = *LastPeripheral++ = PRF_FRAMEBUFFER_QVSS;
	*LastPeripheral++ = PRF_SERIAL_QVSS;
	ConsoleInputDevice = PRF_SERIAL_QVSS;
	Found_Qvss = 1;
      }
    if ( Probe(PhysRqdx, PROBE_READ) )
      {
	printx("Found an RQDX disk drive\n");
	DiskType = *LastPeripheral++ = PRF_DISK_RQDX;
      }
    /* Assume we have a terminal as our console */
    printx("Assuming terminal on console port\n");
    *LastPeripheral++ = PRF_CONSOLE_TERMINAL;

  }


/* Copy between interal and external processor state representations.
 *   Also protects the system from attempts to assign invalid states
 *   to processes.
 */
SystemCode CopyInProcessorState(internal, external, modify_self)
    register KProcessor_state *internal;
    register Processor_state *external;
  {
    register short i;

    /* Integrity checks */
    if (external->perProcessLoc != NULL && 
    	BadUserPtr(external->perProcessLoc))
      {
	return (BAD_ADDRESS);
      }
    if (((unsigned)external->PSL & PSL_CURMOD) == PSL_KERNEL)
      {
	return (NO_PERMISSION);
      }

    /* PerProcess variable location and value */
    internal->perProcessLoc = external->perProcessLoc;
    internal->perProcess = external->perProcess;

    /* Don't let a process modify its own registers */
    if (modify_self) return( OK );

    /* Registers */
    for (i=0; i<14; i++)
        internal->regs[i] = external->regs[i];
    internal->UserSP = external->regs[14];
    internal->PC = external->regs[15];
    internal->PSL = (Unspec)(((int)internal->PSL & 0xFFFF0000) |
			((int)external->PSL & 0x0000FFFF));
    return (OK);
  }


CopyOutProcessorState(external, internal)
    register Processor_state *external;
    register KProcessor_state *internal;
  {
    register short i;

    /* PerProcess variable location and value */
    external->perProcessLoc = internal->perProcessLoc;
    external->perProcess = internal->perProcess;

    /* Registers */
    for (i=0; i<14; i++)
        external->regs[i] = internal->regs[i];
    external->regs[14] = (Unspec) internal->UserSP;
    external->regs[15] = (Unspec) internal->PC;
    external->PSL = internal->PSL;
  }


Reboot()
  {
    asm("	halt");
#ifdef undef
/*
 *   The magic below only works for uVAX 1.  Should have cases for uVAX II,
 *	 and a "please reboot me" message otherwise.  (Or do it anyway, in case
 *	 Halt/reboot is set to halt).
 */
    asm("	mtpr	$0xf04, $txdb");
    asm("	mtpr	$0xf02, $txdb");
#endif
  }

