/*
 * V Kernel - Copyright (c) 1981 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 * Copyright (c) 1982 Stanford University, all rights reserved.
 *
 * Exception handling for Vaxen
 */

#include "Venviron.h"
#include "Vexceptions.h"
#include "Vquerykernel.h"
#include "process.h"
#include "naming.h"
#include "interrupt.h"
#include "asmdefs.h"

/* Imports */
extern Logical_id_entry	Logical_id_map[];
extern short		KernelInterrupted, ProcessInterrupted;
extern char		version[];

/* Exports */
int			Found_Qvss = 0;	/* Set in machine.c, used here */

/* Forward declarations */
static short	 		ExceptionType;
static ExceptionStackFrame	Frame;
char *FmtHex();

/*
*  Definitions of routines to vector to on exceptions and interrupts.
*  exvecn means pop n parameters off the stack before doing a svpctx.
*/
#define exvec0(vecname,vecnum) 					\
asm("	.align	2");						\
asm("vecname:");						\
	ExceptionType = vecnum;					\
asm("	jmp	callException");

#define exvec1(vecname,vecnum) 					\
asm("	.align	2");						\
asm("vecname:");						\
	ExceptionType = vecnum;					\
asm("	movl	(sp)+, _Frame");				\
asm("	jmp	callException");

#define exvec2(vecname,vecnum) 					\
asm("	.align	2");						\
asm("vecname:");						\
	ExceptionType = vecnum;					\
asm("	movq	(sp)+, _Frame");				\
asm("	jmp	callException");

#define exvec4(vecname,vecnum)					\
asm("	.align	2");						\
asm("vecname:");						\
	ExceptionType = vecnum;					\
asm("	movq	(sp)+, _Frame");				\
asm("	movq	(sp)+, _Frame+8");				\
asm("	jmp	callException");

#define exvec10(vecname,vecnum)					\
asm("	.align	2");						\
asm("vecname:");						\
	ExceptionType = vecnum;					\
asm("	movq	(sp)+, _Frame");				\
asm("	movq	(sp)+, _Frame+8");				\
asm("	movq	(sp)+, _Frame+16");				\
asm("	movq	(sp)+, _Frame+24");				\
asm("	movq	(sp)+, _Frame+32");				\
asm("	jmp	callException");


DummyExceptionHandler()
  {
    /* Message register traps */
    extern	Asm_SendReg_trap(),
		Asm_ForwardReg_trap(),
		Asm_ReceiveReg_trap(),
		Asm_ReplyWithSeg_trap(),
		Asm_GetReplyReg_trap();

   /*
    * Establish the routines for each exception to vector to.
    */
    exvec0(_UnknownTrap, 0);
    exvec4(_MachineChk,VecMachineChk);
    exvec0(_KernStackInval,VecKernStackInval);
    exvec0(_PwrFail,VecPwrFail);
    exvec0(_ResPrivInstrn,VecResPrivInstrn);
    exvec0(_XFC,VecXFC);
    exvec0(_OperandReserved,VecOperandReserved);
    exvec0(_AddrModeReserved,VecAddrModeReserved);
    exvec2(_AccessCntlViol,VecAccessCntlViol);
    exvec2(_XlationInval,VecXlationInval);
    exvec0(_TracePending,VecTracePending);
    exvec0(_BreakPoint,VecBreakPoint);
    exvec1(_Compatibility,VecCompatibility);
    exvec1(_Arithmetic,VecArithmetic);
    /*  we have our own handler for CHMK, see below */ 
    exvec1(_CHME,VecCHME);	/* Since these stacks are invalid, these */
    exvec1(_CHMS,VecCHMS);	/* traps will generate Access Violations */
    exvec1(_CHMU,VecCHMU);
    exvec0(_SBI_SILO,VecSBI_SILO);
    exvec0(_CorrMemRead,VecCorrMemRead);
    exvec0(_SBI_Alert,VecSBI_Alert);
    exvec0(_SBI_Fault,VecSBI_Fault);
    exvec0(_BusTimeout,VecBusTimeout);
    exvec0(_Trap1,VecTrap1);
    exvec0(_Trap2,VecTrap2);
    exvec0(_Trap3,VecTrap3);
    exvec0(_Trap4,VecTrap4);
    exvec0(_Trap5,VecTrap5);
    exvec0(_Trap6,VecTrap6);
    exvec0(_Trap7,VecTrap7);
    exvec0(_Trap8,VecTrap8);
    exvec0(_Trap9,VecTrap9);
    exvec0(_TrapA,VecTrapA);
    exvec0(_TrapB,VecTrapB);
    exvec0(_TrapC,VecTrapC);
    exvec0(_TrapD,VecTrapD);
    exvec0(_TrapE,VecTrapE);
    exvec0(_TrapF,VecTrapF);
    exvec0(_IntervalTimer,VecIntervalTimer);
    exvec10(_UndoneEMT,VecUndoneEMT);
    exvec0(_DoneEMT,VecDoneEMT);
    exvec0(_RxConsStor,VecRxConsStor);
    exvec0(_TxConsStor,VecTxConsStor);
    exvec0(_RxConsole,VecRxConsole);
    exvec0(_TxConsole,VecTxConsole);
    exvec0(_UnknownInterrupt,VecInterrupt);

    /* the CHMK handler */
    asm("	.align	2");
    asm("_HandleCHMK:");
    disable;
		ExceptionType = VecCHMK;
    asm("	movl	(sp), _Frame");
    asm("	casel	(sp)+, $0, $12");
    asm("SwitchBase:");
    asm("	.word	Trap0-SwitchBase");
    asm("	.word	Trap1-SwitchBase");
    asm("	.word	Trap2-SwitchBase");
    asm("	.word	Trap3-SwitchBase");
    asm("	.word	callException-SwitchBase");
    asm("	.word	callException-SwitchBase");
    asm("	.word	callException-SwitchBase");
    asm("	.word	callException-SwitchBase");
    asm("	.word	callException-SwitchBase");
    asm("	.word	callException-SwitchBase");
    asm("	.word	callException-SwitchBase");
    asm("	.word	callException-SwitchBase");
    asm("	.word	TrapC-SwitchBase");
    asm("	jmp	callException");
  
  /* jump to the message register traps */
    asm("Trap0:	jmp	Asm_SendReg_trap");
    asm("Trap1:	jmp	Asm_ForwardReg_trap");
    asm("Trap2:	jmp	Asm_ReceiveReg_trap");
    asm("Trap3:	jmp	Asm_ReplyWithSeg_trap");
    asm("TrapC:	jmp	Asm_GetReplyReg_trap");

  }


/*
 * Asm routine to call the HandleException() function
 *  Registers are saved and the stack is cleaned up
 *  so that the process state record will
 *  be accurate as of the time of the exception.
 *
 * We assume that the Process Control Block Base register
 *  is pointing to the right place so that we can do a 
 *  svpctx right off and have all be well.
 * Note that if the trap was caused by a Machine check, Memory Management,
 *  CHMK, CHMU, CHMS, CHME, CorrMemRead (not mVAX), BusTimeout (not mVAX),
 *  or EMT, we need to pop the parameters off of the stack
 *  and into the variable Frame before the jump to callException.
 *
 * Also note that the svpctx will switch us over to the interrupt stack.
 */
asm("	.text");
asm("callException:");
disable;
asm("	svpctx");
asm("	pushl	_Active");
asm("	calls	$1, _HandleException");
asm("	jmp	ActivateReadyqHead");


InitException()

/*
 * Function to initialize exception vectors and kernel traps
 *
 * The VAX doesn't put the vector number on the stack, so we have a
 *  separate routine for each exception to vector to, which loads the
 *  vector number into a global variable and jumps to the common routine
 * CHMx will halt if we try to put them on the interrupt stack, so we
 *  let the svpctx in the vector routine do that for us.
 */


  {
    register int r11;

    extern  MachineChk(), KernStackInval(), PwrFail(), ResPrivInstrn(),
	    XFC(), OperandReserved(), AddrModeReserved(), AccessCntlViol(),
	    XlationInval(), TracePending(), BreakPoint(), Compatibility(),
	    Arithmetic(), HandleCHMK(), CHME(), CHMS(), CHMU(), SBI_SILO(),
	    CorrMemRead(), SBI_Alert(), SBI_Fault(), BusTimeout(),
	    Trap1(), Trap2(), Trap3(), Trap4(), Trap5(), Trap6(), Trap7(),
	    Trap8(), Trap9(), TrapA(), TrapB(), TrapC(), TrapD(), TrapE(),
	    TrapF(),
	    IntervalTimer(), UndoneEMT(), DoneEMT(), RxConsStor(),
	    TxConsStor(), RxConsole(), TxConsole(),
	    UnknownTrap(), UnknownInterrupt();

    for (r11 = 0; r11 < NumExceptions; r11++)
        setexvec(UnknownTrap, r11);

    for (r11 = VecIntStart; r11 < VecInterrupt; r11++)
        setexvec(UnknownInterrupt, r11);

    setexvec(MachineChk, VecMachineChk);
    setexvec(KernStackInval, VecKernStackInval);
    setexvec(PwrFail, VecPwrFail);
    setexvec(ResPrivInstrn, VecResPrivInstrn);
    setexvec(XFC, VecXFC);
    setexvec(OperandReserved, VecOperandReserved);
    setexvec(AddrModeReserved, VecAddrModeReserved);
    setexvec(AccessCntlViol, VecAccessCntlViol);
    setexvec(XlationInval, VecXlationInval);
    setexvec(TracePending, VecTracePending);
    setexvec(BreakPoint, VecBreakPoint);
    setexvec(Compatibility, VecCompatibility);
    setexvec(Arithmetic, VecArithmetic);
    setexvec(HandleCHMK, VecCHMK);
    setexvec(CHME, VecCHME);
    setexvec(CHMS, VecCHMS);
    setexvec(CHMU, VecCHMU);
    setexvec(SBI_SILO, VecSBI_SILO);
    setexvec(CorrMemRead, VecCorrMemRead);
    setexvec(SBI_Alert, VecSBI_Alert);
    setexvec(SBI_Fault, VecSBI_Fault);
    setexvec(BusTimeout, VecBusTimeout);
    setexvec(Trap1, VecTrap1);
    setexvec(Trap2, VecTrap2);
    setexvec(Trap3, VecTrap3);
    setexvec(Trap4, VecTrap4);
    setexvec(Trap5, VecTrap5);
    setexvec(Trap6, VecTrap6);
    setexvec(Trap7, VecTrap7);
    setexvec(Trap8, VecTrap8);
    setexvec(Trap9, VecTrap9);
    setexvec(TrapA, VecTrapA);
    setexvec(TrapB, VecTrapB);
    setexvec(TrapC, VecTrapC);
    setexvec(TrapD, VecTrapD);
    setexvec(TrapE, VecTrapE);
    setexvec(TrapF, VecTrapF);
    setexvec(IntervalTimer, VecIntervalTimer);
    setexvec(UndoneEMT, VecUndoneEMT);
    setexvec(DoneEMT, VecDoneEMT);
    setexvec(RxConsStor, VecRxConsStor);
    setexvec(TxConsStor, VecTxConsStor);
    setexvec(RxConsole, VecRxConsole);
    setexvec(TxConsole, VecTxConsole);

    r11 = (unsigned)&Active->proc_state;
    asm("	mtpr	r11, $pcbb");
  }


HandleException(active)
    register Process *active;
/*
 * Kernel exception handler.  Packages up the information
 *  about the exception and causes the offending process to send
 *  a message off to the exception handler process, if one is
 *  registered.  Otherwise, the process sends the message to itself.
 *
 * This function sees the exception information automatically pushed
 *   on the stack by the processor as its arguments.
 */

  {
    extern Process Idle_process;
    extern ProcessId KernelServerPid;
    ExceptionRequest *req;

    req = (ExceptionRequest *) &(active->msg);

    /* Build msg to send to exception handler */
    req->requestcode = EXCEPTION_REQUEST;
    req->type = ExceptionType;
    req->errpc = (unsigned)active->proc_state.PC;
    req->status = (unsigned)active->proc_state.PSL;
    req->instruction = req->accaddr = req->code = req->subtype = 0;

    switch (ExceptionType)
      {
	case VecMachineChk:
	  {
	    extern MachineConfigurationReply MachineConfig;
	    int i;

	    if (MachineConfig.processor == PROC_UVAX2)
	      {
#ifdef undef
goto skipbug; /* skip this piece of code for now, since it's wrong */
	      /* It'll be unreachable, so the compiler should complain */
	      /* until I do something about it */
		i = *(unsigned short *)(0x20080004);
		/* Writing 1's to any bits that were set clears them, so */
		/*   just write the same thing back			 */
		*(unsigned short *)(0x20080004) = i;
skipbug:	;
#endif
		req->code = i;
	      }
	  }
	  req->subtype = Frame.MCheck.type;
	  req->accaddr = Frame.MCheck.par1;
	  if (MachineConfig.processor == PROC_UVAX1)
	    {
	      asm("	mtpr	$0, $mcesr");	/* say we handled the Check */
	    }
	  break;

	case VecAccessCntlViol:
	case VecXlationInval:
	  req->accaddr = Frame.MM.virtaddr;
	  req->code = Frame.MM.acctype;
	  break;

	case VecCompatibility:
	case VecArithmetic:
	case VecCHMK:
	case VecCHME:
	case VecCHMS:
	case VecCHMU:
	  req->subtype = Frame.one_param.type;
	  break;

	case VecUndoneEMT:
	  req->instruction = Frame.EMT.ir;
	  req->errpc = Frame.EMT.oldpc;
	  break;
      }

    req->segment = (char *) TEAM_START; /* Include access to the team space */
    req->segmentsize = ((unsigned) (active->team->team_space.size))
	 - TEAM_START;

    if( KernelInterrupted || ProcessInterrupted ) /* Fatal kernel error */
      {
	KPrintException( active, req, "Exception in interrupt routine" );
      }
    if( active == &Idle_process )
      {
	KPrintException( active, req, "Exception in idle process" );
      }

#ifdef VM
    /* See if exception was a page fault */
    if (PageFault(active, req) != BAD_ADDRESS) return;
#endif VM

    /* Send to the exception server and send to self if that fails */
    active->blocked_on = Logical_id_map[EXCEPTION_SERVER].pid;
    active->returnMessage = NULL;
    active->forwarder = KernelServerPid;
    active->finish_up = NULL; /* We don't want to get the reply */
    /* Setup data segment fields in PD - not set on forward. */
    active->dataSegmentPro = (READ_ACCESS+WRITE_ACCESS);
    active->dataSegmentPtr = (Unspec *) req->segment;
    active->dataExpected = NULL;
    active->dataSegmentSize = req->segmentsize;

    if( !MapPid(active->blocked_on) )
      {
	printx("No exception server\r\n" );
        KPrintException( active, req, NULL );
	active->blocked_on = active->pid;
	KSend( active );
      }
    else if ( active->blocked_on == active->pid )
      {
	printx("Exception in exception server\r\n" );
        KPrintException( active, req, NULL );
	active->blocked_on = active->pid;
	KSend( active );
      }
    else if ( active->pid == KernelServerPid )
      {
        KPrintException( active, req, "Exception in kernel server process" );
      }
    else
      {
	/* Non-fatal exception */
/*** DumpProcState(&(active->proc_state)); ***/
	KSend( active );
      }

  }


static char	*Uvax1MachChk[] = {
			"Memory Controller died accessing 0x%x",
			"Read, Parity, or ECC error at 0x%x",
			"Nonexistent Memory at 0x%x",
			"Unaligned I/O Access at 0x%x",
			"Can't read page table at 0x%x",
			"Can't write page table at 0x%x",
			"GAK! Control Store parity error",
			"Tora Tora Tora, Invalid Micromachine state",
			"Can't read interrupt vector from Q22 bus",
			"Unexpected stack write error at Vaddr 0x%x"
			};
#define 	 MinUvax1MachChk 0
#define		 MaxUvax1MachChk (sizeof(Uvax1MachChk) / sizeof(char *))

static char	*Uvax2MachChk[] = {
			"Impossible microcode state (FSD)",
			"Impossible microcode state (SSD)",
			"Undefined FPU error code 0",
			"Undefined FPU error code 7",
			"Undefined memory management status (TB miss)",
			"Undefined memory management status (M = 0)",
			"Process PTE address in P0 space",
			"Process PTE address in P1 space",
			"Undefined interrupt ID code"
			};
#define		 MinUvax2MachChk 1
#define		 MaxUvax2MachChk (sizeof(Uvax2MachChk) / sizeof(char *))

static char	*BogusMachChk =
			" code = 0x%x (!?), first param = 0x%x";

static char	*ArithError[] = {
			"Integer Overflow (trap)",
			"Integer Divide by Zero (trap)",
			"Floating Overflow (trap)", 
			"Floating or Decimal String Divide by Zero (trap)", 
			"Floating Underflow (trap)", 
			"Decimal String Overflow (trap)", 
			"Subscript Range (trap)",
			"Floating Overflow (fault)",
			"Floating Divide by Zero (fault)",
			"Floating Underflow (fault)"
			};
#define		 MinArithError 1
#define		 MaxArithError (sizeof(ArithError) / sizeof(char *))


static int recursionCount = 0;	/* Used by KPrintException to trap the case */
				/* where something that KPrintException does*/
				/* causes a further exception.  This assumes*/
				/* that we won't be interrupted (and maybe  */
				/* generate another exception) while we're  */
				/* in KPrintException.  Also requires that  */
				/* any successful exits from KPrintException*/
				/* decrement the counter.		    */

KPrintException( pd, req, abortstring )
    register Process *pd;
    register ExceptionRequest *req;
    register char *abortstring;

    /* Kernel exception handling routine.  Prints out some information
     * about the process incurring the exception and returns the pc at
     * which the exception occurred.  If abortstring is nonNULL, it prints
     * the message, followed by a stackdump and then halts.  I do this
     * instead of calling Kabort because two stackdumps would take up too much
     * space on the screen and overwrite the DumpProcState stuff.
     */
  {
    int noStackDumpYet = 1;

    switch (recursionCount++)
      {
	case 0:
	    break;
	case 1:
	    printx(
	    "** Detected recursion in exception-printing routine - Quit **\n");
	    Halt();
	case 2:
	    Halt();
	case 3:
	    asm("halt");
	default:
	    while (1) ;	/* Can't even do a VAX HALT properly; just hang. */
      }
	    
    DumpProcState(&(pd->proc_state));

    /* Print information about the state of the process */
    switch( req->type )
      {

        case VecMachineChk:
	  printx(" Machine Check: ");
	  {
	    extern MachineConfigurationReply MachineConfig;
	    switch (MachineConfig.processor)
	      {
		case PROC_UVAX1:
		  if ( req->subtype >= MinUvax1MachChk &&
		       req->subtype <= MaxUvax1MachChk )
		    {
		      printx(Uvax1MachChk[ req->subtype - MinUvax1MachChk ],
			     req->accaddr);
		    }
		  else
		    {
		      printx(BogusMachChk, req->subtype, req->accaddr);
		    }
		  break;

		case PROC_UVAX2:
		  if ( req->subtype >= MinUvax2MachChk &&
		       req->subtype <= MaxUvax2MachChk )
		    {
		      printx("%s, VAP = 0x%x", 
				Uvax2MachChk[ req->subtype - MinUvax2MachChk ],
				req->accaddr);
		    }
		  else if ( req->subtype >= 0x80 && req->subtype <= 0x83 )
		    {
		      printx("%s bus error, %s address = 0x%x",
			       (req->subtype & 2) ? "write" : "read",
			       (req->subtype & 1) ? "physical" : "virtual",
				req->accaddr );
		    }
		  else
		    {
		      printx(BogusMachChk, req->subtype, req->accaddr);
		    }
		  break;

	        default:
		  printx("code = 0x%x, first param = 0x%x",
			    req->subtype, req->accaddr);
		  break;
	      }
	  }
	  break;

        case VecKernStackInval:
	  printx(" Kernel Stack Invalid");
	  break;

        case VecPwrFail:
	  printx(" Power Fail");
	  break;

        case VecResPrivInstrn:
	  printx(" Reserved/Privileged Instruction");
	  break;

        case VecXFC:
	  printx(" Extended Function Call");
	  break;

        case VecOperandReserved:
	  printx(" Reserved Operand");
	  break;

        case VecAddrModeReserved:
	  printx(" Reserved Addressing Mode");
	  break;

        case VecAccessCntlViol:
        case VecXlationInval:
          if (req->type == VecAccessCntlViol)
	    printx(" Access Control Violation");
	  else
	    printx(" Translation Not Valid");
	  printx(" at Vaddr 0x%x, mode %s", req->accaddr,
		(req->code & MM_Write) ? "write" : "read");
	  if (req->code & MM_Length)
		printx(", length violation");
	  if (req->code & MM_ProcessPT)
		printx(", while accessing Process Page Table");
	  break;

        case VecTracePending:
	  printx(" Trace Pending");
	  break;
        case VecBreakPoint:
	  printx(" Breakpoint Instruction");
	  break;

        case VecCompatibility:
	  printx(" Compatibility mode fault number 0x%x", req->subtype);
	  break;

        case VecArithmetic:
	  if ( req->subtype >= MinArithError &&
	       req->subtype <= MaxArithError )
	    {
	      printx("%s", ArithError[ req->subtype - MinArithError ]);
	    }
	  else
	    {
	      printx("Arithmetic error, code = 0x%x (!?)", req->subtype);
	    }
	  break;

        case VecCHMK:
	  printx(" CHMK 0x%x Fault", req->subtype);
	  break;

        case VecCHME:
	  printx(" CHME 0x%x Fault", req->subtype);
	  break;

        case VecCHMS:
	  printx(" CHMS 0x%x Fault", req->subtype);
	  break;

        case VecCHMU:
	  printx(" CHMU 0x%x Fault", req->subtype);
	  break;

        case VecSBI_SILO:
	  printx(" SBI SILO Fault");
	  break;

        case VecCorrMemRead:
	  printx(" Corrected Memory Read Fault");
	  break;

        case VecSBI_Alert:
	  printx(" SBI Alert");
	  break;

        case VecSBI_Fault:
	  printx(" SBI Fault");
	  break;

        case VecBusTimeout:
	  printx(" Memory Write Timeout");
	  break;

        case VecTrap1: case VecTrap2: case VecTrap3: case VecTrap4:
        case VecTrap5: case VecTrap6: case VecTrap7: case VecTrap8:
        case VecTrap9: case VecTrapA: case VecTrapB: case VecTrapC:
        case VecTrapD: case VecTrapE: case VecTrapF:
	  printx(" Unexpected level %d AST", req->type - VecTrap1 + 1);
	  break;

        case VecIntervalTimer:
	  printx(" Timer Interrupt");
	  break;

        case VecUndoneEMT:
	  printx(" Undone EMT");
	  break;

        case VecDoneEMT:
	  printx(" Done EMT");
	  break;

        case VecRxConsStor:
	  printx(" Console Storage Rx");
	  break;

        case VecTxConsStor:
	  printx(" Console Storage Tx");
	  break;

        case VecRxConsole:
	  printx(" Console Rx Interrupt");
	  break;

        case VecTxConsole:
	  printx(" Console Tx Interrupt");
	  break;

        case VecInterrupt: 
	  printx( " Unknown interrupt" );
	  break;

        default: 
	  printx( " Unknown exception(0x%x)", req->type);
	  break;
      }

    printx( " in process %x\r\n",pd->pid );
    if( req->instruction != 0 ) 
        printx( "Instruction " );
    printx( "Program counter    Status register\r\n" );
    if( req->instruction != 0 )
        printx( " %s   ", FmtHex(req->instruction) );
    printx( "   %s", FmtHex(req->errpc));
    printx( "           %s\r\n", FmtHex(req->status) );


    printx("%s\n", version);
    if ((req->status & PSL_CURMOD) == PSL_KERNEL) {
	extern	Process *Active;

	StackDump();
	noStackDumpYet = 0;
	printx( " Last kernel trap invoked from address %x\r\n", 
		Active->proc_state.PC);
    }
    
    if (abortstring != NULL)
      {
	printx("Kernel abort: %s\n", abortstring);
	if (noStackDumpYet)
	    StackDump();
	Halt();
      }
    

#ifdef NOTDEF
    {
      KProcessor_state	*state;
      unsigned		stackSize;

      state = &(pd->proc_state);
      if (state->perProcess != 0)
	{
	  /* Check for user stack overflow */
	  stackSize = ((PerProcessArea *) state->perProcess)->stackSize;

	  if ((unsigned)state->USER_STACK_POINTER <=
	  			(unsigned)state->perProcess ||
	      (unsigned) state->USER_STACK_POINTER >
	      			(unsigned) state->perProcess + stackSize)
	      printx( " Probable stack overflow\r\n");
	}
    }
#endif NOTDEF
    recursionCount--;
  }


/* Now that we know how to drop back gracefully to the MicroVAX Monitor, we */
/*   could probably abolish this and do a real HALT.  However, that would   */
/*   waste a couple of extra lines on the display (error msg from Monitor). */
Halt()
{
    register unsigned long  *r11;
    unsigned long GetHexNum();
    
    for(;;)
      {
	r11 = (unsigned long *)GetHexNum();
	if (KernelCanRead(r11, 4))
	  {
	    printx("/%s\n", FmtHex(*r11));
	    K_putchar('\n');	/* Superfluous?? */
	  }
	else
	    printx("/can't access\n");
      }
}


/* Read a number in hex from the console, ending with space or slash */
unsigned long
GetHexNum()
{
    register int c;
    register unsigned long n;
    int count;
    static unsigned long oldn;

    n = 0;          /* Number read in */
    count = 0;      /* Character count */
    for(;;)
    {
	c = GetChar();
	if (c >= '0' && c <= '9')
	{
	    n = (n << 4) + (c - '0');
	    K_putchar(c);
	    count++;
	    continue;
	}
	if (c >= 'a' && c <= 'f')
	{
	    n = (n << 4) + (c - 'a') + 10;
	    K_putchar(c);
	    count++;
	    continue;
	}
	if (c >= 'A' && c <= 'F')
	{
	    n = (n << 4) + (c - 'A') + 10;
	    K_putchar(c);
	    count++;
	    continue;
	}
	if (c == '\b')  /* Backspace */
	{
	    if (count > 0)
	    {
		K_puts("\b \b");
		count--;
		n >>= 4;
	    }
	    continue;
	}
	if (c == ' ' || c == '/' || c == '\n')
	{
	    if (count == 0)
	    {
		n = oldn + 4;   /* Use last value + 4 */
		printx("%s", FmtHex(n));
	    }
	    oldn = n;   /* Save for next time */
	    return n;
	}
	K_putchar('\007');
    }
}


/*
 * Primitive stack dump package, will trace back into user space also.
 */
StackDump()
  {
    unsigned	*Fp;
    unsigned	*Ap;
    unsigned	Pc;
    unsigned	numArgs;
    int		offset;
    register int r11;
    Team	*saveTeam;
    extern char		KStack[], IStack[];
    extern Process	*Active;

#define	ReadablePointer(p) \
    (( ( (char *)(p) > IStack && (char *)(p) < &KStack[STACK_SIZE] ) ||	\
	      (unsigned)(p) < (unsigned)SYSSEGSTART			\
	    ) && KernelCanRead((p), 0x14))
    /*
     * If we trace down into user (P0/P1) space, then we have to ensure that
     *   the addresses we see on the stack are in the currently addressable
     *   team.  If something in the kernel did a SetAddressableTeam(), this
     *   is probably false, so we SetAddressableTeam(Active->team).  Will
     *   this always get us the right address space?
     * We must restore the addressable team when we exit.
     */
    saveTeam = GetAddressableTeam();
    SetAddressableTeam(Active->team);

    printx("StackDump:\n");
    asm("	movl	ap, r11");
    Ap = (unsigned *)r11;
    asm("	movl	fp, r11");
    Fp = (unsigned *)r11;

    /*
     * We make the asssumption here that the two stacks are allocated
     * sequentially.
     */
    if (! ReadablePointer(Fp))
      {
	printx("Invalid frame pointer (0x%x).\n", Fp);
	goto cleanup;
      }

    while ( ReadablePointer(Fp) )
      {
	Pc = Fp[4];
	offset = print_pc((unsigned char *)Pc);
	printx("(");
	if (KernelCanRead(Ap,0x4) && KernelCanRead(&Ap[1],4*Ap[0]))
	  {
	    for (numArgs = 1; numArgs <= Ap[0] && numArgs <= 10; numArgs++)
		printx("0x%x%s", Ap[numArgs], 
			(numArgs < Ap[0] && numArgs < 10) ? ", " : "" );
	    if (Ap[0] > 10)
		printx("..(0x%x args altogether)..", Ap[0]);
	  }
	else
	    printx(" Invalid argument pointer (0x%x) ", Ap);

	printx(") Called from 0x%x%s\n", Pc+offset, 
			(offset >= 0) ? "-?" : "" );
	if ((Pc < (unsigned)SYSSEGSTART) && (Pc > TEAM_LIMIT+P0SEGSTART))
	  {
	    printx("That last program counter looks bogus\n");
	    goto cleanup;
	  }
	Ap = (unsigned *)Fp[2];
	if ((unsigned)Fp == Fp[3])
	  {
	    printx("Next stack frame at same address (0x%x) - pretty bogus\n",
			Fp);
	    goto cleanup;
	  }
	Fp = (unsigned *)Fp[3];
      }
  cleanup:
    SetAddressableTeam(saveTeam);
  }

print_pc(Pc)
    unsigned char *Pc;
{
    int offset;     /* Offset to subroutine call instruction from pc */

    /*
     * This assumes the standard calling sequence generated by
     * the C compiler: a calls instruction with a literal argument
     * count and longword relative addressing.  That means the opcode
     * is at pc-7, and argument count at pc-6, the address mode at pc-5,
     * and the displacement at pc-4 through pc-1.
     *
     * If pc-7 does not contain the opcode for calls or if pc-6
     * does not specify literal mode or if pc-5  does not
     * specify longword displacement off the pc, then we
     * don't really know what address was called.  This routine
     * could be modified to look for other modes if necessary.
     *
     * We assume that Pc is a valid address in the team space.
     */

    offset = 0;     /* In case we don't know adr of calling instruction */
    if (KernelCanRead(Pc-7,7) /* bomb-proofing */ &&
	    (Pc[-7] & 0xFF) == 0xFB /* calls */ &&
	    (Pc[-6] & 0xC0) == 0    /* Literal mode */ &&
	    (Pc[-5] & 0xFF) == 0xEF /* Longword relative mode */)
    {
	printx("%x", Pc + *(long *)(&Pc[-4]));
	offset = -7;    /* calls instruction is at pc-7 */
    }
    else
    {
	printx("?");    /* Don't know what address was called */
    }
    return offset;
}


Kabort( str ) register char *str;

  /* Kernel abort: print message on console and stop with
   * only refresh interrupt enabled.
   */
  {
    disable;			/* Ensure that Kabort can't be recursively
				   called from an interrupt routine. */
    if (Found_Qvss)
	QvssEnableVideo();
    printx("Kernel abort: %s\n", str);
    printx("%s\n", version);
    StackDump();
    Halt();
  }

/*
 * DumpProcState:
 */

DumpProcState(ps)
    register KProcessor_state *ps;
  {
    printx(
	"Process state (from process descriptor, not current registers):\n");
    printx("perProcess: %s    ", FmtHex(ps->perProcess));
    printx("perProcessLoc: %s\n", FmtHex(ps->perProcessLoc));
    printx(" ksp: %s    ", FmtHex(ps->KernelSP));
    printx(" esp: %s    ", FmtHex(ps->ExecSP));
    printx(" ssp: %s    ", FmtHex(ps->SuperSP));
    printx(" usp: %s\n",   FmtHex(ps->UserSP));
    printx("  r0: %s    ", FmtHex(ps->regs[0]));
    printx("  r1: %s    ", FmtHex(ps->regs[1]));
    printx("  r2: %s    ", FmtHex(ps->regs[2]));
    printx("  r3: %s\n",   FmtHex(ps->regs[3]));
    printx("  r4: %s    ", FmtHex(ps->regs[4]));
    printx("  r5: %s    ", FmtHex(ps->regs[5]));
    printx("  r6: %s    ", FmtHex(ps->regs[6]));
    printx("  r7: %s\n",   FmtHex(ps->regs[7]));
    printx("  r8: %s    ", FmtHex(ps->regs[8]));
    printx("  r9: %s    ", FmtHex(ps->regs[9]));
    printx(" r10: %s    ", FmtHex(ps->regs[10]));
    printx(" r11: %s\n",   FmtHex(ps->regs[11]));
    printx("  ap: %s    ", FmtHex(ps->regs[12]));
    printx("  fp: %s    ", FmtHex(ps->regs[13]));
    printx("  pc: %s    ", FmtHex(ps->PC));
    printx(" psl: %s\n",   FmtHex(ps->PSL));
    printx("p0br: %s    ", FmtHex(ps->ProgBaseReg));
    printx("p0lr: %s    ", FmtHex(ps->ProgLenReg));
    printx("p1br: %s    ", FmtHex(ps->CtrlBaseReg));
    printx("p1lr: %s\n",   FmtHex(ps->CtrlLenReg));
  }

char *FmtHex(i)
    register unsigned int i;
  {
    register int j;
    static char s[9];
    static char hex[] = "0123456789abcdef";

    for (j = 7; j >= 0; j--, i >>= 4)
        s[j] = hex[i&0x0000000F];
    return(s);
  }
