/*
 * Distributed V Kernel 
 * Copyright (c) 1982 by Stanford University.
 *
 * Kernel Ethernet driver
 * Deqna Ethernet Controller ("qna")
 *
 * Reference:
 *   [UG] EK-DEQNA-UG-001, DEQNA Ethernet User's Guide, 1st ed., August 1984.
 *
 * Byte-swapping:
 *   We now use DEQNAs (and this driver) in MicroVAXen and in the 680x0-based
 *   Fireflies.  The former case is easy: both the DEQNA and the MicroVAX are
 *   little-endian, and we only have to worry about the fact that Ethernet
 *   packet headers (source address, destination address, ether type) are sent
 *   over the network in big-endian order, whereas we like to treat them as
 *   little-endian numbers [Flame: if we treated them as bytes instead of
 *   words, the problem would go away].
 *
 *   For the Firefly, however, we must byte-swap all data to and from the
 *   DEQNA.  This applies both to the DEQNA registers and to the DEQNA data
 *   structures (buffer descriptor lists) in memory [Funny - I thought the
 *   680x0 Firefly had a hardware hack that byte-swapped the register case for
 *   us].
 *
 *   V inter-kernel packets are sent out with a flag which specifies whether
 *   the sending processor is big- or little-endian.  V process-IDs also have
 *   a flag which says whether the process lives on a little-endian host.
 *
 *   To deal with all these, we use:
 *     #ifdef LITTLE_ENDIAN for swapping packet headers and for setting the 
 *         flag in V inter-kernel packets and process-IDs.
 *     #ifdef LITTLE_ENDIAN also for putting ethernet addresses in big-endian
 *	   order before building setup packets fro the DEQNA.
 *     SWAPREG(x) to swap, if necessary, data going to/from the DEQNA's
 *	   registers
 *     SWAPBDL(x) to swap, if necessary, data going to/from the buffer
 *	   descriptor lists
 *   LITTLE_ENDIAN is (had better be) defined in the standard V buildfiles
 *   for anything that runs on a MicroVAX.  SWAPREG() and SWAPBDL() will
 *   normally be defined in "deqnalocal.h"; there will be a separate version
 *   of this for the MicroVAX II, 680x0 Firefly and anything else we dream up.
 *
 * Addressing - where the DEQNA registers are
 *   On the MicroVAX II, this is easy: the #1 DEQNA lives at 20001920-2000192f
 *   On the Firefly, Q-bus addresses are mangled so badly that the DEQNA
 *     registers are in eight non-contiguous locations.
 *   In deqna.h we define a pointer to each register.  The definitions of the
 *     pointers depend on the macros DEQNA_BASE_ADDRESS and MANGLE_ADDRESS,
 *     which will normally be defined in deqnalocal.h.
 *
 * Addressing - where the DEQNA data lives
 *   The addresses which we write into the DEQNA's Tx/Rx BDL registers, and
 *   the buffer/chain addresses which we put in the buffer descriptor lists,
 *   are really Q-bus addresses, not processor (physical or virtual) addresses.
 *   The functions or macros virt_to_QbusLO() and virt_to_QbusHI should perform
 *   any necessary mapping from a processor address to the low 16 and high 6
 *   bits, respectively, of the Q-bus address.
 */

#include "Venviron.h"
#include "Vethernet.h"
#include "Vikc.h"
#include "Vquerykernel.h"
#include "deqnalocal.h"
#include "deqna.h"
#include "interrupt.h"
#include "dm.h"
#include "ikc.h"

#ifdef DEBUG
#define debug(c) K_putchar(c)
#else
#define debug(c)
#endif

typedef Process_id (*PFPID)();

typedef struct { char *ptr; unsigned long bytes; } BufferList;

/* Imports */
extern unsigned char	*LastPeripheral;

/* Variables imported from enet.c */
extern DeviceInstance	*EthernetInstance;
extern ProcessId	EnetWriter;	/* Pid of writer; NULL if free */
extern NetAddress	MulticastDefault;
extern char		UserPacketBuffer[];
extern int		UserPacketLength;
extern SyncQueue	IkpOutq;

/* Exports */
extern SystemCode EnetModify();
extern SystemCode EnetQuery();
extern EnetPowerup();
extern NetworkWrite();

/* Used internally */
extern SystemCode EnetReadPacket();

Enet10Address	EnetHostNumber;		/* physical ethernet address */
int		EnetReceiveMask = ENET_DEFAULT_MASK;
unsigned short	EnetCollisions = 0;	/* Number of collision errors */
unsigned short	EnetOverflows = 0;	/* Queue overflow errors */
unsigned short	EnetCRCErrors = 0;	/* Packets with bad CRC's */
unsigned short	EnetSyncErrors = 0;	/* Receiver out of sync */
unsigned short	EnetTimeouts = 0;	/* Transmitter timeouts */
int		EnetValidPackets = 0;
int		NetAlarm = 0;		/* set to nonzero when we receive an
					* unclaimed packet.  If it times out,
					* the kernel flushes the packet using
					* NetCheck */
unsigned short  EnetPromiscuous = 0;	/* Are we in promiscuous mode. */
Enet10Packet 	*EnetPacket;	/* Points to current incoming ethernet pkt. */

Qna_t		Buffers;	/* The Enet buffer structure */
int		RcvIndex;	/* The receive buffer we are currently using */
int		LastValidBuffer;/* The last valid buffer in the list */


Enet10Address	VMulticastAddr = V_MULTICAST_ADDRESS;
char		VMultiCast[6];
char		Broadcast[] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff};


/* Macro expansion to interrupt-invoked C call to Ethernetinterrupt */
Call_inthandler(EnetInterrupt);


EnetPowerup()
  /* Powerup and initialize the Ethernet Interface Board.
   *  Done just once, from kernel Powerup() routine. 
   */
  {
    extern unsigned long GenerateRandomNumber();
    extern int Asm_EnetInterrupt();
    
    /* Indicate we have a deqna board */
    *LastPeripheral++ = PRF_ENET_DEQNA;
    IkpOutq.tail = (Process *) &(IkpOutq.head);
    IkpOutq.type = IKP_OUT_QUEUE;

    /* get my host address */

#define qna_adr(x) SWAPREG(*Deqna_enet_addr(x))

    EnetHostNumber.addrhigh = ((qna_adr(0) & 0xFF)<<8) | (qna_adr(1) & 0xFF);
    EnetHostNumber.addrmid  = ((qna_adr(2) & 0xFF)<<8) | (qna_adr(3) & 0xFF);
    EnetHostNumber.addrlow  = ((qna_adr(4) & 0xFF)<<8) | (qna_adr(5) & 0xFF);

    printx("Ethernet Address is %x%x%x%x.%x%x%x%x.%x%x%x%x\n",
    	0xf&(qna_adr(0)>>4), qna_adr(0)&0xf,
    	0xf&(qna_adr(1)>>4), qna_adr(1)&0xf,
    	0xf&(qna_adr(2)>>4), qna_adr(2)&0xf,
    	0xf&(qna_adr(3)>>4), qna_adr(3)&0xf,
    	0xf&(qna_adr(4)>>4), qna_adr(4)&0xf,
    	0xf&(qna_adr(5)>>4), qna_adr(5)&0xf);

    /* initialize VMulticast */
#ifdef LITTLE_ENDIAN
    ByteSwapShortCopy(&MulticastDefault, VMultiCast, sizeof(VMulticastAddr));
#else
    Copy(VMultiCast, &MulticastDefault, sizeof(VMulticastAddr));
#endif
    srandom((EnetHostNumber.addrmid<<16)+EnetHostNumber.addrlow);
    
    /* Plug interrupt location and deqna's interrupt vector */
    *Deqna_vector = VecDeqna << 2;
    setexvec(Asm_EnetInterrupt, VecDeqna);

    EnetReset( ENET_DEFAULT_MASK );

  }


EnetReset( receiveMask )
    int receiveMask;
  {
    register int i, j, time;
    register QNABD_t *bdl;
    short	rcvMode;
    int numTimesRetry;
    char *bug, *lastbug;
    int save_csr;
    static int numGoodResets = 0;

    EnetReceiveMask = receiveMask;
    /* We need to convert receiveMask into a rcvMode the deqna understands. */
    rcvMode = (receiveMask & ENET_BROADCAST ? QNA_AllMultiCast : 0) |
    		(receiveMask & ENET_PROMISCUOUS ? QNA_Promiscuous : 0);
    rcvMode += sizeof(Buffers.setupbuff);

#ifdef DEBUG
    printx("EnetReset(%d); rcvMode = 0x%x...\n", receiveMask, rcvMode);
#endif DEBUG

    /* Main retry loop.  If all goes well we'll only execute this once. */
    numTimesRetry = 0;
    lastbug = NULL;
    while (1)
      {    
	/*
	 * Assert and hold the RESET bit in the DEQNA; we clear it when we're
	 *   ready, after we've initialized lots of data.  Dunno whether we
	 *   really need to hold the DEQNA reset all this time, but it seems
	 *   like a reasonable idea.
	 */
	*Deqna_csr |= SWAPREG( QNARESET);

	/* initialize the setup packet */
	for (i = 0; i < 16; i++)
            for (j = 0; j < 8; j++)
		Buffers.setupbuff[i][j] = 0;
	for (i = 0; i < 6; i++)
	  {
            if (receiveMask & ENET_SELF)
	      {
		Buffers.setupbuff[i][1] = SWAPREG(*Deqna_enet_addr(i)) & 0xFF;
	      }
	    /* is this even needed? */
	    if (receiveMask & ENET_BROADCAST)
	      {
		Buffers.setupbuff[i][2] = Broadcast[i];
		Buffers.setupbuff[i][3] = VMultiCast[i];
	      }
	  }

	/* reset and flush the buffers */
	UserPacketLength = 0;
	RcvIndex = 0;
	LastValidBuffer = RXLISTLEN-2; /* RXLISTLEN-1 is marked invalid,    */
				       /* RXLISTLEN is the chain descriptor */
    
	/* Initialize descriptors 0..TXLISTLEN */
	for (bdl= &Buffers.txlist[0]; bdl<= &Buffers.txlist[TXLISTLEN]; bdl++)
	  {
	    bdl->bd1	= SWAPBDL(QNA_BDLUnused);
	    bdl->bd2	= SWAPBDL(0); /* sets VALID bit = 0 */
	    bdl->addrlo	= SWAPBDL(0);
	    bdl->size	= SWAPBDL(0);
	    bdl->status1	= SWAPBDL(QNA_BDLUnused);
	    bdl->status2	= SWAPBDL(0);
	  }
    
	   /* Initialize descriptors 0..RXLISTLEN-1 */
	for (i = 0, bdl = &Buffers.rxlist[0]; i < RXLISTLEN; i++, bdl++)
	  {
            bdl->bd1	= SWAPBDL(QNA_BDLUnused);
	    bdl->bd2	= SWAPBDL( (i==RXLISTLEN-1 ? 0 : QNA_BD2_VALID) | 
			   ( virt_to_QbusHI(&Buffers.rxbuff[i]) 
			     << QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask ) );
	    bdl->addrlo	= SWAPBDL( virt_to_QbusLO(&Buffers.rxbuff[i]) );
	    bdl->size	= SWAPBDL(-((sizeof(Buffers.rxbuff[i]) + 1) /
				     sizeof(short)) );
	    bdl->status1	= SWAPBDL(QNA_BDLUnused);
	    bdl->status2	= SWAPBDL(0);
	  }
      
	/* set up last BD */
	bdl->bd1    = SWAPBDL(QNA_BDLUnused);
	bdl->bd2    = SWAPBDL(QNA_BD2_VALID | QNA_BD2_CHAIN |
			   ( virt_to_QbusHI(Buffers.rxlist) 
			      << QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask ) );
	bdl->addrlo = SWAPBDL(virt_to_QbusLO(Buffers.rxlist));
	bdl->size   = 0;/* Size and status fields aren't used if chain is   */
			/* true; we initialize them to make debugging easier*/
	bdl->status1= SWAPBDL(QNA_BDLUnused);
	bdl->status2= SWAPBDL(0);
    
	/*
	 * Take the DEQNA out of the RESET state.  Most other DEQNA
	 *   drivers delay for quite a while before clearing RESET, then read
	 *   from the DEQNA's boot ROM (and ignore what they read).  Wish I
	 *   knew whether we really have to do those things.  In the meantime,
	 *   we don't bother and just hope things work.
	 */
	*Deqna_csr &= SWAPREG(~QNARESET);
	i = *Deqna_csr;
	i |= SWAPREG( QNARCVINT | QNAILOOP | QNAXMTINT );
	i &= SWAPREG(
		 ~(QNARCVENABLE | QNAINTENABLE | QNATIMERENABLE | QNAELOOP) );
	*Deqna_csr = i;
	if ( (SWAPREG(*Deqna_csr) & QNARXBDLBAD) == 0 )
	  {
	    save_csr = SWAPREG(*Deqna_csr);
	    bug = "Receiver BDL should be invalid, isn't";
	    goto reset_bug;
	  }
	*Deqna_rxBDLlo = SWAPREG(virt_to_QbusLO(Buffers.rxlist));
	*Deqna_rxBDLhi = SWAPREG(virt_to_QbusHI(Buffers.rxlist));

	/* send a setup packet (receiver is disabled, so we know that the
	*  packet we are receiving is the looped back setup packet.)
	*/
	XmitBuffer((unsigned)rcvMode, (int)Buffers.setupbuff, 1);
    
	/* wait for it to be looped back */
	time = 0;
	/*
	 * Wait on the Tx, not Rx, buffer because Tx status is written after
	 * Rx status for a setup packet. [UG] section 1.3.4 (LoopBack), p. 1-9
	 */
	while ( (SWAPBDL(Buffers.txlist[0].status1) & QNA_TX1_USEmask) == 
		    QNA_BDLUnused &&
		 time <= QNAWAITTIME)
	    time++;
	if (time > QNAWAITTIME)
	  {
	    /* we timed out waiting for a looped back packet, Kabort */
	    save_csr = SWAPREG(*Deqna_csr);
	    bug = "Timed out waiting for setup packet to loop back";
	    goto reset_bug;
	  }
	if ( (SWAPBDL(Buffers.txlist[0].status1) & QNA_TX1_USEmask) != 
		    QNA_BDLLastNoErr )
	  {
	    save_csr = SWAPREG(*Deqna_csr);
	    bug = "Got bad transmit status when looping back a setup packet";
	    goto reset_bug;
	  }

	/* Did all the setup stuff successfully; get out of the retry loop */
	break;

reset_bug:

	/* numTimesRetry, bug, save_csr and lastbug should all be defined */
#ifndef VERBOSE
	/*
	 * If we just get one or two DEQNA problems and then everything
	 *   works again, don't bother telling anybody
	 */
    if (numTimesRetry >= 2)
      {
#endif  VERBOSE
	if (lastbug == NULL)
	  {
	    printx(
	    "Previously did %d (decimal) consecutive successful EnetResets\n",
	            numGoodResets);
	    numGoodResets = 0;
	  }
	if (bug != lastbug ||
	    (numTimesRetry & (-numTimesRetry)) == numTimesRetry )
	    /* ^ i.e. it's a power or 2 (or zero), so print something just */
	    /*   to let the user know what's going on.  We use powers of 2 */
	    /*   so the messages become less frequent if the bug persists  */
	  {
	    printx("DEQNA EnetReset: %s\n  (have already done %d retries)\n",
			bug, numTimesRetry);
	  }
	if (numTimesRetry > 1000000)
			    /* ^ Very arbitrary limit */
	  {
	    Debug_Deqna(save_csr);
	    printx("DEQNA EnetReset: %s\n  (have done %d retries)\n",
			bug, numTimesRetry);
	    Kabort("DEQNA EnetReset: Couldn't reset DEQNA despite retrying");
	  }
	lastbug = bug;
#ifndef VERBOSE
      }	/* Finishes the conditionally-compiled if-statement above */
#endif  VERBOSE
	numTimesRetry++;
      } /* End of the retry loop */

    if (numTimesRetry >
#ifdef VERBOSE
		0
#else  VERBOSE
		2
#endif VERBOSE
		 )
      {
	printx("DEQNA EnetReset - finally succeeded after %d retries\n",
		numTimesRetry);
      }

    numGoodResets++;
    Buffers.rxlist[RcvIndex].bd1	= SWAPBDL(QNA_BDLUnused);
    Buffers.rxlist[RcvIndex].status1	= SWAPBDL(QNA_BDLUnused);
    GoToNextBD();
    i = *Deqna_csr;
    i |= SWAPREG(QNAINTENABLE | QNARCVINT | QNAILOOP | QNAXMTINT | QNARCVENABLE);
    i &= SWAPREG(~(QNATIMERENABLE | QNAELOOP));
    *Deqna_csr = i;
    
    EnetWriter = 0;
    NetAlarm = ENET_READER_TIMEOUT;
  }


XmitBuffer( msize, buff, setup )
  /*
   * Loads the Transmit and sends the packet
   * descriptor according to the given information
   */
    unsigned msize;	/* the length of the message */
    int buff;		/* the buffer containing the message */
    int setup;		/* 1 if this is a setup packet */
  {
    /*
     * We write more than is really necessary.  Slows things down a bit, but it
     *   it helps when debugging if we set everything possible to 0, so we can
     *   see what the deqna has changed since we wrote it.
     */
    Buffers.txlist[0].bd1	= SWAPBDL(QNA_BDLUnused);
    Buffers.txlist[0].bd2	= SWAPBDL(QNA_BD2_VALID | QNA_BD2_ENDOFMSG |
		(setup ? QNA_BD2_SETUP : 0) |
		((msize & 1) ? QNA_BD2_LBOEND : 0) |
		(virt_to_QbusHI(buff) 
		   << QNA_BD2_ADDRHIpos & QNA_BD2_ADDRHImask) );
	/*   chain and HBOStart are zero, as required.  */
    Buffers.txlist[0].addrlo	= SWAPBDL(virt_to_QbusLO(buff));
    Buffers.txlist[0].size	= SWAPBDL(-((msize + 1) / sizeof(short)) );
    Buffers.txlist[0].status1	= SWAPBDL(QNA_BDLUnused);
    Buffers.txlist[0].status2	= 0; /* Unnecessary but nice for debugging */
#ifdef LITTLE_ENDIAN
    if (!setup)
        ByteSwapShortCopy((char *)buff, (char *)buff, ENET10_HEADER_SIZE);
#endif LITTLE_ENDIAN
    if ( EnetWriter == 0 && !setup )
      {
	/*
	 * Debugging code only:
	 *
	 * If EnetWriter == 0 and this is a real packet that we're trying to
	 *   send, then someone screwed up the management of EnetWriter.
	 *   May provoke real problems (like the BDL Kabort below)
	 */
#ifdef VERBOSE
	printx("XmitBuffer: EnetWriter == 0 (continuing)\n");
#endif VERBOSE
	EnetWriter = 1; /* probably too late, but... */
      }
    if ( (SWAPREG(*Deqna_csr) & QNATXBDLBAD) == 0 )
      {
	int i;

	for (i=0; i<1000000 && (SWAPREG(*Deqna_csr) & QNATXBDLBAD) == 0; i++)
	    ;    /* ^ arbitrary.  Worth retrying for a long time */
	if ( (SWAPREG(*Deqna_csr) & QNATXBDLBAD) == 0)
	  {
	    Debug_Deqna(SWAPREG(*Deqna_csr));
	    printx("XmitBuffer: EnetWriter is %d (%s) just before Kabort\n",
		EnetWriter, (EnetWriter ? "OK" : "bad") );
	    Kabort("XmitBuffer: Transmitter BDL should be invalid, isn't\n");
	  }
#ifdef VERBOSE
	else
	    printx("XmitBuffer: Tx BDL got it right after %d tries\n", i);
#endif VERBOSE
      }
    *Deqna_txBDLlo = SWAPREG(virt_to_QbusLO(Buffers.txlist));
    *Deqna_txBDLhi = SWAPREG(virt_to_QbusHI(Buffers.txlist));
  }
    

EnetInterrupt()
   /* Handle an interrupt from the ethernet interface.
    */
  {
    extern ProcessId EnetWriter;
    register Process *pd;
    register QNABD_t *bdl;
    register int csr;
    register int i;
 
    /* First, bring our IPL down to a reasonable value */
    asm("	mtpr	$0x15, $ipl");
    
    /*
     * Determine type of interrupt.
     * We acknowledge the interrupts now, to prevent problems if a packet
     * comes in while we are processing.
     */
    csr = SWAPREG(*Deqna_csr);
    *Deqna_csr = SWAPREG(csr | QNARCVINT | QNAXMTINT);
    if ( csr & QNANXMINT )
      {
	Debug_Deqna(csr);
        Kabort("Deqna accessed nonexistent memory...");
      }
    if ((SWAPBDL(Buffers.txlist[0].status1) & QNA_TX1_USEmask) != QNA_BDLUnused
         || (csr & QNAXMTINT) )
      {
debug('x');
        /* check for errors on last transmit */
	bdl = &Buffers.txlist[0];
	EnetCollisions += (SWAPBDL(bdl->status1) & QNA_TX1_COUNTmask)
				>> QNA_TX1_COUNTpos;
	if ((SWAPBDL(bdl->status1) & QNA_TX1_USEmask) != QNA_BDLLastNoErr)
	  {
	    /* we had a transmit error */
	    if (SWAPBDL(bdl->status1) & QNA_TX1_ABORT)
	        EnetCollisions += 16;
	    if (SWAPBDL(bdl->status1) & (QNA_TX1_LOSS | QNA_TX1_NOCAR))
	      {
	        printx("Bad packet transmit due to malfunctioning hardware\n");
	      }
	  }

	/* If IkpOutq is non-empty, transmit next interkernel packet. */
	Lockq( &IkpOutq );
	if( (pd=IkpOutq.head) != NULL )
	  {
	    if( (IkpOutq.head = pd->link) == NULL )
		IkpOutq.tail = (Process *) &(IkpOutq.head);
	    pd->queuePtr = NULL;
	    Unlockq( &IkpOutq );
	    WriteNextKPacket( pd );
	  }
	else
	  {
	    Unlockq( &IkpOutq );
	    EnetWriter = 0; /* No transmission in progress now */
	  }
        
	/* Mark this buffer as unused */
/*
 * Doesn't this stomp on the buffer descriptor that WriteNextKPacket has
 *   just set up, and the DEQNA is probably using?
 */
	bdl->status1 = SWAPBDL(QNA_BDLUnused);
	bdl->status2 = SWAPBDL(      0      ); /* Unnecessary, nice for debug*/
	bdl->bd1     = SWAPBDL(QNA_BDLUnused);
      }
    if ( (csr & QNARCVINT) || 
		(SWAPBDL(Buffers.rxlist[RcvIndex].status1) & QNA_RX1_USEmask)
			 != QNA_BDLUnused )
      {
debug('r');
	EnetReadPacket();
      }
    if (EnetWriter == 0) NetAlarm = ENET_READER_TIMEOUT;
  }


SystemCode EnetReadPacket()
   /*
    *  Physically transfer a packet from the Ethernet
    *  Interface Board into the reading process's buffer.
    */
  {
    extern PFPID	RemoteTrapTable[];
    Process		*pd;
    register DeviceInstance *inst;
    register kPacket	*kp;
    register unsigned	bytes;
    QNABD_t		*bdl;
    unsigned short	lhn;
    extern int		DeliverToEnetReader;

    /* Find a valid error-free packet */
    bdl = &Buffers.rxlist[RcvIndex];
    while ((SWAPBDL(bdl->status1) & QNA_RX1_USEmask) != QNA_BDLUnused)
      {
	if ((SWAPBDL(bdl->status1) & QNA_RX1_USEmask) == QNA_BDLLastNoErr)
	  {
	    /* A valid packet */
	    EnetPacket = &Buffers.rxbuff[RcvIndex];
#ifdef LITTLE_ENDIAN
	    ByteSwapShortCopy((char *)EnetPacket, (char *)EnetPacket,
			      ENET10_HEADER_SIZE);
#endif LITTLE_ENDIAN
	    kp = (kPacket *) EnetPacket->data;
	    DeliverToEnetReader = 0;
	    if (EnetPacket->EtherType == KERNEL_PACKET)
	      {
		/* Exactly WHY is all the following stuff in the ethernet */
		/*   driver when it belongs in the ikc routines?          */

		if ( DifferentIKCByteOrder(kp) )
		  {
		    SwapIKPacket(kp);
		  }
		kp->packetType &= ~IKC_LITTLE_ENDIAN; /* Just to be tidy */

		if ( !(kp->dstPid & REMOTE_ALIAS_PROCESS) )
		  {
		    /* a kernel packet */
		    lhn = ((kp->packetType == remoteForward) ?
					kp->forwarder : kp->srcPid) >> 16;
		/* HACK */
		    if (kp->packetType != remoteForward)
			HostCacheEnter( lhn, EnetPacket->SrcHost );

#ifndef DEBUG2
		    if (kp->packetType < 1 || kp->packetType > 14)
			/* Bad: shouldn't hard-code packet-Type numbering, */
			/*      particularly not here.			   */
#endif  DEBUG2
/* Do this if packet looks bogus, or always if DEBUG2 */
		      {
			register int *i = (int *)&kp->msg;
			printx(
 "Type  seq  srcPid   dstPid forwardr userNumb  length  localadd remotead\n");
/* should be:		printx("%4x %4x %8x %8x %8x %8x %8x %8x %8x\n", */
			printx("%x %x %x %x %x %x %x %x %x\n",
				kp->packetType, kp->sequenceNo, kp->srcPid,
				kp->dstPid, kp->forwarder, kp->userNumber,
				kp->length, kp->localaddress, 
				kp->remoteaddress);
/* should be:		printx("*%8x %8x %8x %8x %8x %8x %8x %8x\n", */
			printx("*%x %x %x %x %x %x %x %x\n",
				i[0],i[1],i[2],i[3],i[4],i[5],i[6],i[7]);
		      }

		    /* Ensure in correct packetType range. */
		    kp->packetType = kp->packetType & 0xF;

		    (*RemoteTrapTable[kp->packetType])(kp);
		  }
	      }
	    if( (EnetPacket->EtherType != KERNEL_PACKET)
		|| (kp->dstPid & REMOTE_ALIAS_PROCESS)
		|| DeliverToEnetReader
		|| EnetPromiscuous )
	      {
		/* it is not a kernel packet */
		/* and not to a remote alias process */
		/* If in promiscuous mode, let the enet reader get too. */

		bytes  = (SWAPBDL(bdl->status2) & QNA_RX2_RBLLOmask)
				>> QNA_RX2_RBLLOpos;
		bytes |= (SWAPBDL(bdl->status1) & QNA_RX1_RBLHImask)
				>> QNA_RX1_RBLHIpos << 8;
			/* But QNA_RX1_RBLHIpos = 8; wonder whether the */
			/* compiler is smart enough to optimize this?   */
		bytes += 60;	/* The deqna gives you the length - 60 */

		if( (inst = EthernetInstance) == NULL )
		  {
		    /* Ethernet device not open */
		  }
		else if( !MAP_TO_RPD(pd, inst->reader) )
		  {
		    /* No reader process */
		    if( !MAP_TO_RPD(pd, inst->owner) )
		      {
			/* Ethernet device not open */
			inst->owner = 0; /* Free the instance */
			EthernetInstance = NULL;
			inst = NULL;
			EnetReset( ENET_DEFAULT_MASK );
		      }
		    else if (!EnetPromiscuous ||
				EnetPacket->EtherType != KERNEL_PACKET ||
				!(kp->dstPid & REMOTE_ALIAS_PROCESS) )
		      /* If in promiscuous mode, don't save packet if
		       * it is a kernel packet - too risky.*/
		      {
		        /* No one is waiting for this packet, so copy it into
		         * the UserPacketBuffer */
		        UserPacketLength = bytes;
		        Copy(UserPacketBuffer, (char *)EnetPacket, bytes);
		      }
		  }
		else
		  {
		    /* give the reader this packet */
		    ReadIntoProcess(pd, inst, (int)bytes, (char *)EnetPacket);
		  }
	      }

	    /* Return packet to receiver */
	    ++EnetValidPackets;
	  }
	else
	  {
	    /* Otherwise we discard this packet */
debug('#');
	    if ( SWAPBDL(bdl->status1) & QNA_RX1_OVF    ) EnetOverflows++;
	    if ( SWAPBDL(bdl->status1) & QNA_RX1_CRCERR ) EnetCRCErrors++;
	    if ( SWAPBDL(bdl->status1) & QNA_RX1_FRAME  ) EnetSyncErrors++;
	  }

	bdl->bd1	= SWAPBDL(QNA_BDLUnused);
	bdl->status1	= SWAPBDL(QNA_BDLUnused);
	bdl->status2	= SWAPBDL(      0      ); /* Again, unnecessary */
	GoToNextBD();
	bdl = &Buffers.rxlist[RcvIndex];
      }

  }


NetworkWrite( bufferptr )
register BufferList *bufferptr;
  {
    register char *bp;

    if (bufferptr->bytes < sizeof(Enet10Header)) Kabort("Bad network header");
  
    /* Transfer a packet to the Ethernet Interface Board. */
    bp = (char *)&Buffers.txbuff;
    while (bufferptr->ptr)
      {
	if (bufferptr->bytes)
	  {
	    Copy(bp, bufferptr->ptr, bufferptr->bytes);
	    bp += bufferptr->bytes;
	  }
	bufferptr++;
      }
    ((Enet10Header *) &Buffers.txbuff)->SrcHost = EnetHostNumber;

    XmitBuffer(bp - (char *)&Buffers.txbuff, (int)&Buffers.txbuff, 0);
    NetAlarm = ENET_WRITER_TIMEOUT;
  }

SystemCode EnetModify( pd, inst, dirIndex )
    register Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &(pd->msg);

    if( reply->NumCollisions != -1 )
	EnetCollisions = reply->NumCollisions;
    if( reply->NumOverflows != -1 )
	EnetOverflows = reply->NumOverflows;
    if( reply->NumCRCErrors != -1 )
	EnetCRCErrors = reply->NumCRCErrors;
    if( reply->NumSyncErrors != -1 )
	EnetSyncErrors = reply->NumSyncErrors;
    if( reply->NumTimeOuts != -1 )
	EnetTimeouts = reply->NumTimeOuts;
    if( reply->NumValidPackets != -1 )
	EnetValidPackets = reply->NumValidPackets;
    if( reply->ReceiveMask != EnetReceiveMask &&
        reply->ReceiveMask != -1 ) EnetReset( reply->ReceiveMask );

   return( OK );
 }    

SystemCode EnetQuery( pd, inst, dirIndex ) 
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QueryEnetReply *reply = (QueryEnetReply *) &(pd->msg);

    reply->NetworkType = ENET_TYPE_10MBIT;
    reply->NumCollisions = EnetCollisions;
    reply->NumOverflows = EnetOverflows;
    reply->NumCRCErrors = EnetCRCErrors;
    reply->NumSyncErrors = EnetSyncErrors;
    reply->NumTimeOuts = EnetTimeouts;
    reply->ReceiveMask = EnetReceiveMask;
    reply->NumValidPackets = EnetValidPackets;
    reply->HostAddress.e10 = EnetHostNumber;

    return( OK );
  } 


int NetCheck()
  {
    register Process *pd;

    EnetReset( ENET_DEFAULT_MASK );
    /* If IkpOutq is non-empty, transmit next interkernel packet. */
    Lockq( &IkpOutq );
    if( (pd = IkpOutq.head) != NULL )
      {
        if( (IkpOutq.head = pd->link) == NULL )
	    IkpOutq.tail = (Process *) &(IkpOutq.head);
	pd->queuePtr = NULL;
	if (EnetWriter == 0)	/* Not really sure this is right, but it */
	    EnetWriter = 1;	/*   seems more right than nothing at all*/
	Unlockq( &IkpOutq );
	WriteNextKPacket( pd );
      }
    else
      {
        EnetWriter = 0; /* No transmission in progress now */
        Unlockq( &IkpOutq );
	NetAlarm = ENET_READER_TIMEOUT;
      }
  }	  

ReadIntoProcess(pd, inst, size, buf)
register Process *pd;
register int size;
register char *buf;
register DeviceInstance *inst;
  {
    register IoRequest *req = (IoRequest *) &pd->msg;
    register char *p;
    Team *oldteam = GetAddressableTeam();


    /* First, fill in the bytecount in the reply */
    if ( size < req->bytecount )
      req->bytecount = size;
    /* Now decide where to put the data */
    if( req->bytecount <= IO_MSG_BUFFER ) 
      p = (char *) req->shortbuffer;
    else
      p = (char *) req->bufferptr;

    /* Copy data to correct location */
    SetAddressableTeam( pd->team );
    Copy( p, buf, req->bytecount );
    SetAddressableTeam( oldteam );
    EthernetInstance->reader = 0;

    req->requestcode = OK;
    Addready( pd );
  }


GoToNextBD()
/*
 * This function handles going to the next Buffer descriptor, reinitializing
 * the list if it is invalid.
 */
  {
    register QNABD_t *bdl;
    
    if ( RcvIndex == LastValidBuffer )
      {
	Buffers.rxlist[LastValidBuffer].bd2 &= SWAPBDL(~QNA_BD2_VALID);
        RcvIndex = (RcvIndex+1) % RXLISTLEN;
	bdl = &Buffers.rxlist[RcvIndex];
	bdl->bd2 |= SWAPBDL(QNA_BD2_VALID);
	if (SWAPREG(*Deqna_csr) & QNARXBDLBAD)
	  {
	    *Deqna_rxBDLlo = SWAPREG(virt_to_QbusLO(bdl));
	    *Deqna_rxBDLhi = SWAPREG(virt_to_QbusHI(bdl));
	  }
	LastValidBuffer = (LastValidBuffer+RXLISTLEN-1) % RXLISTLEN;
      }
    else
      {
        RcvIndex = (RcvIndex+1) % RXLISTLEN;
      }
  }

/*
 * Print debugging info for this DEQNA driver.  Must (1) not occupy too many
 *   lines on the display, and (2) tell as near as we can get to the whole 
 *   truth.  Might also be nice if it (3) made some attempt to pretty-print
 *   things, but this may conflict with (1) and (2) and might even require
 *   some work (gasp).  Prints stuff backwards so that the most important stuff
 *   appears last, and so doesn't vanish from sight.
 * If the DEQNA does anything creative while we're printing this, then what we
 *   print won't be very helpful.  Partly for this reason, we print whatever
 *   csr value the calling routine decides to hand us, and also print the csr's
 *   value when we're almost done.
 */
Debug_Deqna(csr)
    unsigned short csr; /* Assumes SWAPREG() has already been applied */
  {
    int i;
    printx("\nRx:");
    for (i = RXLISTLEN; i >= 0; i--)
	Debug_BDL(i, &Buffers.rxlist[i], (i&1)?"  " : "\r\n");
    printx("Tx:");
    for (i = TXLISTLEN; i >= 0; i--)
	Debug_BDL(i, &Buffers.txlist[i], (i&1)?"  " : "\r\n");
    printx(
       "RcvIndex = 0x%x, LastValidBuffer = 0x%x, Deqna CSR was 0x%x, is 0x%x",
	    RcvIndex, LastValidBuffer, csr, SWAPREG(*Deqna_csr));
  }

Debug_BDL(index, address, string)
    int index;
    unsigned short *address;
    char *string;
  {
    printx("%x: %x %x %x %x %x %x%s", index,
	   SWAPBDL(address[0]), SWAPBDL(address[1]), SWAPBDL(address[2]),
	   SWAPBDL(address[3]), SWAPBDL(address[4]), SWAPBDL(address[5]),
	   string);
  }

SystemCode AddLogicalHostGroup() {return( OK );}
SystemCode DeleteLogicalHostGroup() {return( OK );}

