/*
 * V Kernel - Copyright (c) 1982 by Stanford University
 *
 * Kernel Console support routines for the device server
 *
 * The "console" provides access to the default keyboard and display
 *  for standalone programs.  The association between this logical
 *  device and physical device(s) depends on the workstation's
 *  type and configuration.
 */

#include "Venviron.h"
#include "Vteams.h"
#include "Vnaming.h"
#include "Vquerykernel.h"
#include "dm.h"
#include "process.h"
#include "machine.h"
#include "interrupt.h"
#include "cbuff.h"


/* Imports */
extern SystemCode NotSupported();
extern short ConsoleInputDevice, ConsoleOutputDevice;

/* Exports */
extern SystemCode ConsoleCreate();
int	(*ConsoleBellRoutine)();

/* Private (ConsoleInInstance and ConsoleInBuff get used in qvssserial.c) */
DeviceInstance *ConsoleInInstance, *ConsoleOutInstance;
CircularBuffer ConsoleInBuff, ConsoleOutBuff;
unsigned ConsoleXmitReady;

/* Forward */
SystemCode	ConsoleWrite();
SystemCode	ScreenRelease();
SystemCode	ConsoleRead();
SystemCode	KeyboardRelease();

/* Routines */
Call_inthandler(ConsoleRcv);	/* Macro expansion to interrupt-invoked
				 * C call to ConsoleRcv */

Call_inthandler(ConsoleXmit);	/* Macro expansion to interrupt-invoked
				 * C call to ConsoleXmit */
InitConsole()
    /* Initialize the Transmit and Receive registers of the console,
    *  and plug in the interrupt handlers for them */
  {
    extern int Asm_ConsoleRcv(), Asm_ConsoleXmit();

    setexvec(Asm_ConsoleRcv, VecRxConsole);
    setexvec(Asm_ConsoleXmit, VecTxConsole);
    asm("	mtpr	$0x40, $rxcs");
    asm("	mtpr	$0x40, $txcs");
  }


/* Routines that interact directly with the console device */

unsigned long ConsoleGetchar()
  {
  /*
   * Read the character in from the VAX console.
   */
    asm("	mfpr	$rxdb, r0");	/* read the character */
    asm("	movzbl	r0, r0"	);	/* Zero 3 high bytes of r0 */
  }

ConsolePutchar(ch)
    int ch;
  {
  /*
   * Write the character out to the VAX console.
   */
    asm( "	movzbl	4(ap), r0" ); /* Guarantee that 3 high bytes are 0 */
    asm("	mtpr	r0, $txdb");  /* write the character */

    if (ch == 7 && ConsoleBellRoutine)
        (*ConsoleBellRoutine)();
  } 

/* INTERRUPT HANDLERS */
ConsoleRcv() 
  /*
   * Interrupt handler for receiving characters for the console.
   */
  {
    register unsigned char c;
    register Process *pd;
    register DeviceInstance *inst;

    /* Machine specific read of character */
    c = ConsoleGetchar();

    /* figure out where it goes */
    if ( (inst = ConsoleInInstance ) != NULL ) {
	if (MAP_TO_RPD( pd, inst->reader) != NULL ) {
	    /* Someone waiting for this character, so give it to
	       him directly */
	    
	    ((IoReply *)&pd->msg)->replycode = OK;
	    ((IoReply *)&pd->msg)->bytecount = 1;
	    ((IoReply *)&pd->msg)->shortbuffer[0] = c;
	    inst->nextblock++;
	    inst->reader = 0;

	    Addready( pd );
	}
	else { 
	    /* no one is waiting so queue the character. */
	    CbuffPut( &ConsoleInBuff, c );
	}
    }
  } /* ConsoleRcv */

ConsoleXmit()
  /* 
   * Console transmit buffer empty interrupt handler
   */
  {
    register DeviceInstance *inst = ConsoleOutInstance;
    register Process *pd;
    register int c;

    if ( (c = CbuffGet(&ConsoleOutBuff)) != CBUFF_EMPTY )
      {
        /* print the buffer */
        ConsolePutchar( c );
        if ( inst != NULL && MAP_TO_RPD(pd, inst->writer) != NULL )
	  {
	    /* There is someone waiting to transmit a character */
	    inst->lastblock++;
	    CbuffPut(&ConsoleOutBuff, ((IoRequest *)&pd->msg)->shortbuffer[0]);
	    ((IoReply *)&pd->msg)->replycode = OK;
	    ((IoReply *)&pd->msg)->bytecount = 1;
	    inst->writer = NULL;
	    Addready( pd );
	  }
      }
    else if ( inst != NULL && MAP_TO_RPD(pd, inst->writer) != NULL )
      {
	/* There is someone waiting to transmit a character */
	inst->lastblock++;
	ConsolePutchar( ((IoRequest *)&pd->msg)->shortbuffer[0] );
	((IoReply *)&pd->msg)->replycode = OK;
	((IoReply *)&pd->msg)->bytecount = 1;
	inst->writer = NULL;
	Addready( pd );
      }
    else
      {
        ConsoleXmitReady = 1;
      }
  } /* ConsoleXmit */


/* Device Routines */
SystemCode ConsoleCreate( pd, inst )  
    Process *pd;
    DeviceInstance *inst;
  {
  /*
   * Create instances for the console.  The instance returned is
   *  used for console output.  Instance id + 1 is used for console
   *  input.  Its parameters can be obtained by doing a QueryInstance.
   */
    CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
    register DeviceInstance *oldInst;

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    if( (oldInst = ConsoleInInstance) != NULL )
      {
	if( MapPid( oldInst->owner ) )  return (BUSY);
	ConsoleInInstance = 0;
	oldInst->owner = 0;
      }

    if( (oldInst = ConsoleOutInstance) != NULL )
      {
	if( MapPid( oldInst->owner ) )  return (BUSY);
	ConsoleOutInstance = 0;
	oldInst->owner = 0;
      }

    /*
     * Initialize the device instance descriptors, can assume all fields
     * are zero except for id and first instance's owner.
     */

    /* Output descriptor */
    inst->readfunc = NotSupported;
    inst->writefunc = ConsoleWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = NotSupported;
    inst->releasefunc = ScreenRelease;

    inst->type = (WRITEABLE+STREAM);
    inst->blocksize = sizeof(char);  /* character by character for now */
    inst->lastblock = -1;	     /* indicate file is empty */
    inst->lastbytes = inst->blocksize;
    ConsoleOutInstance = inst;

    /* Input descriptor */
    inst++;
    inst->owner = pd->pid;
    inst->readfunc = ConsoleRead;
    inst->writefunc = ConsoleWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = NotSupported;
    inst->releasefunc = KeyboardRelease;

    inst->type = (READABLE+STREAM);
    inst->blocksize = sizeof(char);  /* assume character by character for now */
    inst->lastblock = -1;	     /* indicate file is empty */
    inst->lastbytes = inst->blocksize;
    ConsoleInInstance = inst; /* Record the instance for interrupt routine */

    CbuffFlush(&ConsoleInBuff);	/* Flush any buffered characters */
    CbuffFlush(&ConsoleOutBuff);
    ConsoleXmitReady = 1;

    return( OK );

  }

SystemCode ConsoleRead( pd, inst )
    Process		*pd;
    DeviceInstance	*inst;

   /* Handle a read instance request for the console
    */
  {
    register IoRequest *req = (IoRequest *) &(pd->msg);
    register IoReply *reply = (IoReply *) &(pd->msg);
    register unsigned bytecount;
    register int c;
    Process *reader;

    bytecount = req->bytecount;
    reply->bytecount = 0;
    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading. 
     */
    if( inst->reader && MAP_TO_RPD(reader, inst->reader) != NULL) 
	return( RETRY );

    if( bytecount > inst->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
    /*
    * Disabled to enable multiple processes to access the console
    * if( req->blocknumber != inst->nextblock ) return( BAD_BLOCK_NO );
    */

    disable;
    /* Try to get characters from the buffer */
    if ( (c = CbuffGet(&ConsoleInBuff)) != CBUFF_EMPTY ) {
	reply->replycode = OK;
	reply->shortbuffer[0] = c;
	reply->bytecount = 1;
	inst->nextblock++;
	inst->reader = 0;
	return( OK );
    }
    else {
	inst->reader = Active->pid;
	Active->state = AWAITING_INT;
	return( NO_REPLY );
    }
  }

SystemCode ConsoleWrite( pd, inst )
    register Process *pd;
    register DeviceInstance	*inst;

   /* Handle a write instance request to a serial line
    */
  {
    extern unsigned short FramebufferType;
    register IoRequest *req = (IoRequest *) &(pd->msg);
    register IoReply *reply = (IoReply *) &(pd->msg);
    register int c = req->shortbuffer[0];
    unsigned bytecount;
    unsigned bufsize;
    unsigned newline;

    bytecount = req->bytecount;
    reply->bytecount = 0;

    if( bytecount > inst->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
/*  Disabled to simplify multi-process access to the console.
    if( req->blocknumber != inst->lastblock+1 ) return( BAD_BLOCK_NO );
*/
    disable;
    
    /*
     * Here is where we deal with the Qvss console -- It is not interrupt
     * driven, and cannot beep with ^G.
     */
    if (FramebufferType == PRF_FRAMEBUFFER_QVSS)
      {
	if (c == 7)
	  {
	    if (ConsoleBellRoutine)
		(*ConsoleBellRoutine)();
	  }
	else
	    K_putchar( c );
	inst->lastblock++;
	reply->bytecount = 1;
	reply->replycode = OK;
	return( OK );
      }
    
    bufsize = CbuffSize(&ConsoleOutBuff);
    newline = c == '\n';
    /* falling through this if => return OK */
    if ( ConsoleXmitReady ) {
        /* bufsize is zero */
        if ( newline ) {
	    ConsolePutchar( '\r' );
	    CbuffPut( &ConsoleOutBuff, '\n' );
	}
	else {
	    ConsolePutchar( c );
	}
	ConsoleXmitReady = 0;
    }
    else if ( bufsize < CBUFF_SIZE-2 || (bufsize == CBUFF_SIZE-2 && !newline))
      {
        if ( newline ) {
	    CbuffPut( &ConsoleOutBuff, '\r' );
	}
	CbuffPut( &ConsoleOutBuff, c );
      }
    else if ( bufsize == CBUFF_SIZE-2 ) {
	CbuffPut( &ConsoleOutBuff, '\r' );
	inst->writer = Active->pid;
	Active->state = AWAITING_INT;
	return( NO_REPLY );
    }
    else if (inst->writer == NULL) { /* cbuff full, wait for int before xmit */
	if ( newline ) {
	    return( RETRY );
	}
	else {
	    inst->writer = Active->pid;
	    Active->state = AWAITING_INT;
	    return( NO_REPLY );
	}
    }
    else {
	return( RETRY );
    }
    inst->lastblock++;
    reply->bytecount = 1;
    reply->replycode = OK;
    return( OK );
  } /* ConsoleWrite */

static SystemCode ScreenRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release VAX screen (console) instance
   */
  {
    inst->owner = 0;
    return( OK );
  }

static SystemCode KeyboardRelease( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release VAX keyboard (console) instance
   */
  {
    IoReply *reply;
    register Process *rwpd;

    if( rwpd = MapPid( inst->reader) ) 
				/* Unblock reader process */
      {
	inst->reader = 0;
	reply = (IoReply *) &rwpd->msg;
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( rwpd );
      }
    inst->owner = 0;
    return( OK );
  }


CreateStdIo( pd )
    Process *pd;
  /* Create a console device instance for the root team's standard i/o */
  {
    Message msg;
    register CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
    register CreateInstanceReply *reply = (CreateInstanceReply *) &pd->msg;
    register RootMessage *rmsg = (RootMessage *) msg;
    extern Process_id KernelServerPid;
    extern SystemCode CreateDeviceInstance();

    req->requestcode = CREATE_INSTANCE;
    req->filemode = FCREATE;
    req->filename = "console";
    req->filenamelen = 7;
    req->filenameindex = 0;
    req->contextid = DEFAULT_CONTEXT;

    if (CreateDeviceInstance( pd ) != OK) 
	Kabort("Error creating std i/o");

    ConsoleInInstance->owner = pd->pid;
    ConsoleOutInstance->owner = pd->pid;

    rmsg->replycode = OK;
    rmsg->rootflags = STDOUT_APPEND | STDERR_APPEND;
    rmsg->stdinserver = rmsg->stdoutserver = rmsg->stderrserver = 
    	reply->fileserver;
    rmsg->stdinfile = reply->fileid + 1;
    rmsg->stdoutfile = rmsg->stderrfile = reply->fileid;
    rmsg->envblock = NULL;
 
    Copy_msg(&pd->msg, rmsg);
  }
