/*
 * V Kernel - Copyright (c) 1981 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 * Copyright (c) 1982 Stanford University.
 *
 * Kernel Process and Team Management Routines
 *
 */

#include "process.h"

/* Global data declarations */

extern Process_id Device_Server_Pid;
extern ProcessId KernelServerPid;
extern SyncQueue Readyq;
extern ProcessBlock ProcBlock;
extern Process *AllocatePd();
extern Process *FindSuspendPd();
extern Team *TdFreeList, *FirstTeam;
extern Process Idle_process;
extern LhnRec *FindLhn();

/* Kernel operations */

SystemCode CreateProcess( active )
    register Process *active;

  /* Create a new process on the same
   * team as the creator and return the new process id.
   * The process is assigned priority, stack and entry
   * specified in active->msg;
   */
  {
    register KernelRequest *req;
    unsigned short priority; 
    register char *entry, *stack;
    register Process *pd;
    register int index, loopstop;
    ProcessId newPid;
    unsigned lhn;
    LhnRec *ld;

    req = (KernelRequest *) &active->msg;
    if( AlienProcess(active) ) return( NO_PERMISSION );

    newPid = req->pid; /* One can request a particular process id. */
     /* If already in use or nonlocal we choose another one. */
    if(  (PidLhn(newPid) == 0) || MapPid(newPid) || !Local(newPid) )
	newPid = 0;
    priority = req->unspecified[0];
    entry = (char *) req->unspecified[1];
    stack = (char *) req->unspecified[2];
    lhn = (unsigned) req->unspecified[3];
    lhn = PidLhn(lhn);		/* Make sure the low order 16 bits are 0 -
				   this allows passing in a regular pid to
				   specify the lhn. */
    if ((lhn == 0) || (lhn == active->team->lhn->lhn))
      {
	ld = active->team->lhn;
	lhn = ld->lhn;
      }
    else
      {
	ld = FindLhn(lhn);
      }
    if( priority > MAX_PROCESS_PRIORITY ) return( BAD_PROCESS_PRIORITY );

    if( (pd = AllocatePd()) == NULL ) return( NO_PDS );

    /* Initialize the fields of the allocated PD */
    pd->blocked_on = active->pid;
    pd->state = AWAITING_REPLY;
    index = ld->pidGen;
    if( newPid != 0 )
      { /* Use suggested newPid */
	index = newPid;
	pd->pid = newPid;
	goto initPd;
      }
    /* Make sure that index doesn't collide with an already existing pd. */
    loopstop = active->team->lhn->pidGen + ( PID_HASH_MASK + 1 );
    while ( ( Pd_bundle[ index & PID_HASH_MASK ] != &Idle_process ) &&
	    ( index < loopstop ) )
      {
	index++;
	if ((PidLocalPart(index) == LOCAL_KERNEL_PROCESS_PID) ||
	    (PidLocalPart(index) == LOCAL_DEVICE_SERVER_PID))
	  {
	    index++;
	  }
      }
    ld->pidGen = index + 1;
    pd->pid = PidLocalPart(index) | PidLhn(lhn);
initPd:
    disable;
    pd->nextPd = Pd_bundle[ index & PID_HASH_MASK ];
    Pd_bundle[ index & PID_HASH_MASK ] = pd;
    enable;
    /* Avoid creating a process with the remote alias process bit on. */
    pd->pid &= (~REMOTE_ALIAS_PROCESS);
    pd->localPid = pd->pid;
    pd->team = active->team;
    pd->priority = ( priority & PROCESS_PRIORITY_MASK ) | 
    		   ( pd->team->team_priority & TEAM_PRIORITY_MASK );
    pd->finish_up = NULL;
    pd->returnMessage = NULL; /* in case Forward sets finish_up to GetReply */
    pd->pdFlags = MSG_FLAG;  /* Return the root message only */
    pd->proc_state.perProcessLoc = active->proc_state.perProcessLoc;
    
    /* Link into the tree of processes */
    pd->father = active;
    pd->brother = active->son;
    active->son = pd;
    pd->son = NULL;
    pd->link = NULL;
    pd->userNumber = active->userNumber;
    pd->processorTime = 0;
    pd->decayRate = 0;
    pd->decayTime = 0;
    pd->next_sender = NULL;
    pd->msgq.head = NULL;
    pd->msgq.tail = (Process *) &(pd->msgq.head);
    pd->replyq.head = NULL;
    pd->replyq.tail = (Process *) &(pd->replyq.head);
    if (pd->extendPd != NULL) printx("zeroing an extendPd\n");
    pd->extendPd = NULL;

    /* Machine-dependent initialization of PD */
    Ms_CreateProcess( pd, entry, stack );
    /* Link into head of the active's msg queue
     * - since awaiting reply from it. */
    Lockq( &(active->msgq) );
    if( (pd->link = active->msgq.head) == NULL ) /* End of queue. */
	active->msgq.tail = pd;
    active->msgq.head = pd;
    pd->queuePtr = &(active->msgq);
    Unlockq( &(active->msgq) );
    req->pid = pd->pid;
    return( OK );
  }

SystemCode DestroyProcess( active )
   register Process *active;
  /* Destroy the process specified by active->req->pid
   * and all its descendants.
   * If a group-id is specified, destroy all members of the group.
   */
  {
    extern ProcessId GetGroupMember();
    extern SystemCode DestroyOneProcess();
    register KernelRequest *req;
    Process_id forwarder;
    register GroupId gid;
    register unsigned i, oks;
    SystemCode tmpr, r;

    req = (KernelRequest *) &active->msg;
    gid = req->pid;
    /* Are we destroying a single process ? */
    if( !IsGroupId( gid ) ) return( DestroyOneProcess( active ) );

    /* This opaque bit of code is designed to return OK if:
     * (a) There was at least one group member.
     * (b) All group members were destroyed.
     * A return code of NONEXISTENT_PROCESS from DestroyOneProcess
     * isn't an error; Dead processes may still be recorded as
     * members of a group due to lazy reclaimation of group
     * descriptors.
     */
    forwarder = active->forwarder;
    oks = 0;
    r = NONEXISTENT_PROCESS;
    for ( i = 0; ( (req->pid = GetGroupMember( gid, i )) != 0 ); i++ )
      {
	tmpr = DestroyOneProcess( active );
	if( tmpr == OK ) ++oks;
	else if( tmpr != NONEXISTENT_PROCESS ) r = tmpr;
      }
    if( (r == NONEXISTENT_PROCESS) && oks ) r = OK;
    if( IsGroupId(forwarder) && (r == NONEXISTENT_PROCESS) )
	r = DISCARD_REPLY;

    return( r );    
  }

SystemCode DestroyOneProcess( active )
Process *active;
    /* Destroy the specified process assuming the userno and
     * alien status (is or is not) are acceptable and return
     * the appropriate error code.
     *
     * The case of destroying a migrating process (indicated by the FROZEN
     * bit being set) is handled by calling DestroyMigratingProcess().
     * If any significant changes are made to DestroyOneProcess then the
     * corresponding changes must also be made to DestroyMigratingProcess.
     */
  {
    register Process *pd, *prev;
    Process *pd1;
    register ProcessId pid;
    register ExtendPd sp;
    unsigned userno;
    KernelRequest *req;
    
    req = (KernelRequest *) &active->msg;
    pid = req->pid;

    if ( !MAP_TO_RPD(pd, pid) ) return( NONEXISTENT_PROCESS );

    userno = active->userNumber;
    if( (userno != pd->userNumber) && 
        (userno != SUPER_USER) &&
	(userno != SYSTEM_USER) &&
	(pd->father != active) )
	return( NO_PERMISSION );

    /* Check if this process is a team root and deallocate the team if so. */
    if (pd->team->team_root == pid) DeallocateTeam(pd->team);

    /* Remove from process tree */
    prev = pd->father;
    if( prev->son == pd ) prev->son = pd->brother;
    else
      {
        prev = prev->son;
        while( prev->brother != pd ) prev = prev->brother;
        prev->brother = pd->brother;
      }
    pd->brother = pd->father = 0;

    while( pd->son != NULL ) pd = pd->son;

    while( pd != NULL )
      {
        if (pd->pdFlags & FROZEN)
	  {
	    DestroyMigratingProcess(pd);
	  }
	else
	  {
	    /* Unqueue from kernel queues if any */
	    RemoveQueue( pd );
	    /* Unqueue all blocked senders. */
	    Lockq( &(pd->msgq) );
	    while( (prev = pd->msgq.head) != NULL )
	      {
		prev->queuePtr = NULL;
		pd->msgq.head = prev->link;
		prev->msg.sysCode = NONEXISTENT_PROCESS;
		Addready( prev ); /* May be an alien - see alien.c */
	      }
	    Unlockq( &(pd->msgq) );
	    /* Unqueue all qroup replies. */
	    if (pd->replyq.head != NULL) FreeReplyBuffers(pd);
	    /* Destroy any local aliens that aren't replies */
	    if (pd->pdFlags & LOCAL_ALIENS_PRESENT) ScavengeAliens( pd->pid );
	    /* Deallocate any extendPds. */
	    while (pd->extendPd != NULL)
	      {
		sp = pd->extendPd;
		DeleteExtendPd(pd, sp->eType);
	      }
	    /* Deallocate any suspend PDs associated with this PD. */
	    pd1 = FindSuspendPd(pd->pid);
	    if (pd1 != NULL)
	      {
		RemoveQueue(pd1);
		FreePd(pd1, pd->pid);
	      }
	    pd->pdFlags &= ~SUSPEND_FLAG;
	    /* Deallocate the pd. */
	    FreePd( pd, pd->pid );
	  }

        /* Find next process to remove */
        if( pd->brother != NULL )
          {
            pd = pd->brother;
            while( pd->son != NULL ) pd = pd->son;
          }
        else pd = pd->father;
      }
    return( OK );
  }

SystemCode QueryProcessState( active )
  register Process *active;
  /* General query about process descriptor 
   */
  {
    KernelRequest *req;
    register Process_id pid;
    register Process *pd;
    register ProcessBlock *pb;

    req = (KernelRequest *) &active->msg;
    pid = req->pid;
    if( !(pd = MapPid(pid)) ) 
      {
	if( pid == Device_Server_Pid ) return( OK );
	return( NONEXISTENT_PROCESS );
      }
 
    pb = &ProcBlock;
    pb->team_root = pd->team->team_root;
    pb->team_priority = pd->team->team_priority;
    pb->team_user = 0; /* Meaningless. */
    pb->team_size = (char *)pd->team->team_space.size;
    if( pd->link != NULL )
  	pb->link = pd->link->pid;
    else
	pb->link = 0;
    if( pd->father != NULL )
	pb->father = pd->father->pid;
    else
	pb->father = 0;
    if( pd->brother != NULL )
	pb->brother = pd->brother->pid;
    else
	pb->brother = 0;
    if( pd->son != NULL )
	pb->son = pd->son->pid;
    else
	pb->son = 0;
    pb->state = pd->state;
    pb->priority = pd->priority;
    pb->blocked_on = pd->blocked_on;
    pb->forwardedby = (Process_id) pd->forwarder;
    if( pd->msgq.head != NULL )
	pb->msg_queue = pd->msgq.head->pid;
    else
	pb->msg_queue = 0;
    if( pd->msgq.head != NULL && pd->msgq.tail != NULL )
	pb->msg_queue_end = pd->msgq.tail->pid;
    else
	pb->msg_queue_end = 0;

    CopyOutProcessorState(&pb->ps.proc_state, &pd->proc_state);

    if( pd == active ||
         pd->state == AWAITING_REPLY && pd->blocked_on == active->pid )
	Copy_msg(pb->msg, &pd->msg);
    else
	clear(pb->msg, sizeof(MsgStruct)); /* clear instead of Zero is
					    * more efficient
					    */
    return( OK );
  }

SystemCode WriteProcessState( active )
   register Process *active;
  /*
   * Write the process's state as described in state.
   *   Must be awaiting reply from the active process.
   * The active process may write its own state, but
   *   this only affects the per-process area information
   *   kept in the state record, not the actual processor
   *   registers.
   */
  {
    register KernelRequest *req;
    register Process_id pid;
    register Processor_state *state;
    register Process *pd;
    register SystemCode error;

    req = (KernelRequest *) &active->msg;
    pid = req->pid;
    state = (Processor_state *) req->segment;
    if( !(pd = MapPid(pid)) ) return( NONEXISTENT_PROCESS );

    if( pd != active && 
        (pd->state != AWAITING_REPLY || pd->blocked_on != active->pid) )
	return( NOT_AWAITINGREPLY );

    return(CopyInProcessorState(&pd->proc_state, state, 
					(active->pid == pd->pid)));
  }

SystemCode SetProcessPriority( pd )
Process *pd;
  {
    register KernelRequest *req;
    register Process *p;
    register unsigned short priority;
    
    req = (KernelRequest *) &pd->msg;
    if (!(p = MapPid(req->pid))) return( NONEXISTENT_PROCESS );
    if (pd->userNumber != p->userNumber && pd->userNumber != SUPER_USER) 
        return( NO_PERMISSION );
    req->unspecified[2] = p->priority;
    req->unspecified[3] = p->decayRate;
    priority = req->unspecified[0] & PROCESS_PRIORITY_MASK;
    if ( priority == MAX_PROCESS_PRIORITY )
	priority |= STOPPED_TEAM_PRIORITY; /* Set to a non-runnable priority */
    else
	priority |= p->team->team_priority & TEAM_PRIORITY_MASK;
    p->priority = req->unspecified[0] = priority;
    p->decayRate = req->unspecified[1];
/*%%%*/
    if (req->unspecified[1])
	printx("%x setting decay rate of %x to %d.\n",
	    pd->pid, p->pid, p->decayRate);
    return( OK );
  }

SystemCode QueryProcessPriority( pd )
Process *pd;
  {
    register KernelRequest *req;
    register Process *p;
    
    req = (KernelRequest *) &pd->msg;
    if (!(p = MapPid(req->pid))) return( NONEXISTENT_PROCESS );
    req->unspecified[0] = p->priority;
    req->unspecified[1] = p->decayRate;
    return( OK );
  }

SystemCode SetTeamPriority( active ) 
    register Process *active;
  /*
   * Set the specified team's priority.
   *   A team is specified by the pid of any process on it.
   */
  {
    register KernelRequest *req;
    register Process *pd;
    register Team *td;
    register unsigned short newpriority;
    register int i;

    req = (KernelRequest *) &active->msg;
    if (active->team != FirstTeam) return( NO_PERMISSION );

    if( !(pd = MapPid(req->pid)) ) return( NONEXISTENT_PROCESS );

    td = pd->team;
    newpriority = req->unspecified[0] & TEAM_PRIORITY_MASK;
    if (newpriority == STOPPED_TEAM_PRIORITY) newpriority--;
    td->team_priority = newpriority;

    for(i = 0; i <= PID_HASH_MASK; i++)
      {
        for (pd = Pd_bundle[i]; pd != &Idle_process; pd = pd->nextPd)
	  {
	    if (pd->team != td) continue;

	    /* Don't reset the priority of the kernel server. */
	    if (pd->pid == KernelServerPid) continue;

	    /* Don't reset the team priority of a process that has been set
	     * to be non-runnable.
	     */
	    if ((pd->priority&TEAM_PRIORITY_MASK) == STOPPED_TEAM_PRIORITY)
	      {
		continue;
	      }

	    disable;
	    pd->priority &= PROCESS_PRIORITY_MASK;
	    pd->priority |= newpriority;
		
	    if( pd->queuePtr == &Readyq )
	      {
		RemoveQueue( pd );
		Addready( pd );
	      }
	    enable;
	  }
      }
    return( OK );
  }

SystemCode QueryProcessorUsage( active ) 
    register Process *active;
  /*
   * Determine the use of the processor by the specified
   * process and team (if requested).
   */
  {
    register KernelRequest *req;
    register Process *pd;
    register Team *td;
    register unsigned i, team_usage;

    req = (KernelRequest *) &active->msg;
    req->opcode = OK;
    
    if( req->pid == 0 )
      {
	req->pid = active->pid;
	pd = active;
      }
    else if (PidLocalPart(req->pid) == 0)
      {
        req->length =IdleProcessPtr->processorTime;
	req->unspecified[0] = 0;
	return( OK ) ;
      }
    else if( !(pd = MapPid(req->pid)) ) return( NONEXISTENT_PROCESS );

    req->length = pd->processorTime;
    if( req->unspecified[0] == 0 ) return( OK );
    td = pd->team;
    /* Otherwise- calculate team processor usage. */
    team_usage = 0;
    for(i = 0; i <= PID_HASH_MASK; i++)
      {
        for (pd = Pd_bundle[i]; pd != &Idle_process; pd = pd->nextPd)
	  {
	    if (pd->team != td) continue;
	    team_usage += pd->processorTime;		
	  }
      }
    req->unspecified[0] = team_usage;
    return( OK );
  }

SystemCode CreateTeam( active )
    register Process *active;

  /* Create a new team, creating a new process as the root process
   * with the specified priority, entry  and stack and return its pid.
   * priority, entry and stack are specified in active->msg.
   * Return 0 if unsuccessful.
   */
  {
    register KernelRequest *req;
    short priority;
    SystemCode r;
    register Process *pd;
    register Team *td;
    register ProcessId pid;
    unsigned lhn;
    register LhnRec *lp;

    req = (KernelRequest *) &active->msg;
    priority = req->unspecified[0];
    if( AlienProcess(active) ) return( NO_PERMISSION );

    /* Find a free team descriptor */
    if ( (td = TdFreeList) == NULL) return( NO_TDS );

    TdFreeList = td->next;
    /* Attach td to relevant logical host's team chain. */
    lhn = req->unspecified[3];
    lhn = PidLhn(lhn);		/* Make sure the low order 16 bits are 0 -
				   this allows passing in a regular pid to
				   specify the lhn. */
    if ((lhn == 0) || (lhn == active->team->lhn->lhn))
      {
        lp = active->team->lhn;
	lhn = lp->lhn;
      }
    else
      {
        lp = FindLhn(lhn);
	if (lp == NULL) 
	  {
	    return(BAD_ARGS);
	  }
      }
    td->next = lp->teams;
    td->prev = NULL;
    lp->teams = td;
    if (td->next != NULL) td->next->prev = td;

    if( (r=CreateProcess(active)) == OK )
      {
        /* Initialize team descriptor */
	pid = req->pid;
        pd = MapPid( pid );
        td->team_root = pid;
	td->owner = active->pid;
	td->team_priority = DEFAULT_TEAM_PRIORITY;
	td->lhn = lp;
	pd->priority = (priority & PROCESS_PRIORITY_MASK) | td->team_priority;
	pd->proc_state.perProcessLoc = NULL;
        pd->team = td;
        Ms_CreateTeam( td, pd ); /* Machine-dependent portion */
      }
     else
      {
	/* Unlink team descriptor. */
	lp->teams = td->next;
	if( td->next != NULL ) td->next->prev = NULL;
	td->next = TdFreeList;
	TdFreeList = td;
      }
    return( r );
  }


DeallocateTeam(td)
    register Team *td;
  {
    /* Take the team off its logical host list. */
    if (td->prev == NULL)
      {
	td->lhn->teams = td->next;
      }
    else 
      {
	td->prev->next = td->next;
      }
    if (td->next != NULL)
      {
	td->next->prev = td->prev;
      }
    /* Check if the logical host is empty and deallocate if so. */
    if (td->lhn->teams == NULL)
      {
	DeallocateLhn(td->lhn);
      }
    td->next = TdFreeList;
    TdFreeList = td;
    td->team_root = 0;
    ReclaimMemory(td);
  }

SystemCode SetObjectOwner( pd )
register Process *pd;
  {
    register KernelRequest *req = (KernelRequest *) &pd->msg;
    register Process *object;
    register Team *td;
    
    if ( !MAP_TO_RPD( object, req->pid)) return( NONEXISTENT_PROCESS );
    
    td = object->team;
    req->unspecified[1] = td->owner;
    /* Should do some permission checking here - Lance */
    if (req->opcode == SET_OBJECT_OWNER)
	td->owner = req->unspecified[0];
    return( OK );
  }
