/*
 * Copyright (c) 1982 Stanford University.
 * Machine-independent mouse device routines.
 */


#include "Vmouse.h"
#include "process.h"
#include "dm.h"
#include "interrupt.h"
#include "mouse.h"


/* Imports */
extern SystemCode NotSupported();
extern DeviceInstance *GetDeviceInstance();

/* Exports */
extern SystemCode MouseCreate();
extern SystemCode MouseRead();
extern SystemCode MouseWrite();
extern SystemCode MouseRelease();



SystemCode MouseCreate( pd, inst )
	Process *pd; DeviceInstance *inst;
  /* Create an instance for the Mouse.
   */
 {
    CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
    register DeviceInstance *mouse;

    if (req->filemode != FCREATE) 
      {
        return (MODE_NOT_SUPPORTED);
      }

    if( (mouse = GetDeviceInstance(MouseInstance)) != NULL )
      {
	if( MapPid( mouse->owner ) ) 
	  {
	    return (BUSY);
	  }
	MouseInstance = 0;
	mouse->owner = 0;
      }
    /*
     * Initialize the device instance descriptor, can assume all fields
     * are zero except for owner and id.
     */

    inst->readfunc = MouseRead;
    inst->writefunc = MouseWrite;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = NotSupported;
    inst->releasefunc = MouseRelease;

    inst->type = (READABLE+WRITEABLE+FIXED_LENGTH);
    inst->blocksize = sizeof(MouseBlock);
    inst->lastbytes = inst->blocksize;

    MouseEvent = 0; /* No outstanding mouse event */
    MouseInstance = inst->id; /* Record the instance for interrupt routine */
    MouseX = 0;
    MouseY = 0;
    MouseButtons = 7;	/* all up */

    /* Initialize the interface */

    StartMouse();

    return( OK );
 }

SystemCode MouseRelease( pd, inst )
    register Process *pd;
    DeviceInstance *inst;
  /*
   * Release mouse instance 
   */
  {
    IoReply *reply;

    StopMouse();

    if ( pd = MapPid(inst->reader) ) 
				/* Unblock reader process */
      {
	inst->reader = 0;
	reply = (IoReply *) &pd->msg;
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( pd );
      }
    inst->owner = 0;
    return( OK );
  }

static SystemCode MouseGetEvent( req, inst )
    IoRequest		*req;
    DeviceInstance	*inst;
  {
    register IoReply	*reply = (IoReply *) req;
    register MouseBlock	*buffer;

    if( MouseEvent == 0 ) return( NO_REPLY ); /* Wait until mouse event */

    reply->replycode = OK;
    buffer = (MouseBlock *) reply->shortbuffer;
    buffer->buttons = 7 & ~MouseButtons; /* Invert mouse buttons */
    buffer->xcoordinate = (int) MouseX;
    buffer->ycoordinate = (int) MouseY;
    inst->reader = 0;
    MouseEvent = 0;

    return( OK );
  }

SystemCode MouseRead( pd, inst )
    Process *pd;
    DeviceInstance	*inst;

   /* Handle a read instance request for the Mouse interface.
    */
  {
    IoRequest *req = (IoRequest *) &pd->msg;
    register Process *reader;
    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading.
     */
    if( inst->reader ) 
        if( MAP_TO_RPD(reader, inst->reader) ) 
	    return (RETRY);
    if( req->bytecount != inst->blocksize ) return( BAD_BYTE_COUNT );
    if( req->blocknumber != 0 ) return( END_OF_FILE );

    /* Check if event is waiting for the reader */
    disable;
    if( MouseGetEvent(req, inst) == OK ) return (OK);

    /* Wait for event to occur */
    inst->reader = pd->pid;
    pd->state = AWAITING_INT;

    return( NO_REPLY );
  }

SystemCode MouseWrite( pd, inst )
    Process *pd;
    DeviceInstance	*inst;

	/*	 
	 *	Handle a write instance request to the Mouse
	 *
	 *	When writing to the mouse, only update my (the kernel's)
	 *	value for the absolute position of the mouse. Do not update
	 *	the button state since the buttons reflect an absolute physical
	 *	state and we can't have somebody mucking with physics.
 	 *	
	 *	This procedure shouldn't be here since the kernel should 
	 *	only return mouse deltas.  Maybe later...
	 */
  {
    IoRequest *req = (IoRequest *) &pd->msg;
    register MouseBlock *buffer;

    if( req->bytecount != inst->blocksize ) return( BAD_BYTE_COUNT );
    if( req->blocknumber != 0 ) return( END_OF_FILE );

    buffer = (MouseBlock *) (req->shortbuffer); /* pointer to X and Y value */

    MouseX = (short) buffer->xcoordinate;
    MouseY = (short) buffer->ycoordinate;
    MouseEvent = 1;

    return( OK );
  }

CheckMouseReader()
  /*
   * Checks if a reader is waiting for the mouse and replies to him.
   */
  {
    register Process *pd;
    register DeviceInstance *inst;

    if (((inst = GetDeviceInstance(MouseInstance)) != NULL) &&
    	    (MAP_TO_RPD(pd, inst->reader)))
      {
        if (MouseGetEvent(&pd->msg, inst) == OK)
	    Addready(pd);
      }
  }
