/*
 * Kernel Logical Host Migration Routines, part 2
 */


#include "Vgroupids.h"
#include "migrate.h"
#include "ipc.h"


extern Process Idle_process;
extern Logical_id_entry Logical_id_map[];
extern ProcessId KernelServerPid;
extern LhnRec *FindLhn();
extern Process *AllocatePd();
extern Process *Alienate();
extern LhnRec *AllocateLhn();
extern GroupMember *Free_group_descs;
extern GroupMember *Group_bundle[GROUP_MASK+1];
extern ProcessId CheckFrozenSender();
extern Process *FindSuspendPd();
extern SystemCode doCopyFrom(), doCopyTo();
extern SystemCode JoinGroup();

SystemCode CopyInLHost();
char *CopyInProcess();
Process *RemapProcessDesc();

extern Unspec *FinishUpFcnTable[];
extern Unspec *TimeoutFcnTable[];


/**************************************************************************
 External routines
 *************************************************************************/


/*
 * TransferHost:
 */

SystemCode TransferHost(active)
    Process *active;
  {
    register Team *oldteam;
    SystemCode status;

    oldteam = GetAddressableTeam();
    SetAddressableTeam(active->team);
    status = CopyInLHost(active);
    SetAddressableTeam(oldteam);
    return(status);
  }


/**************************************************************************
 Internal routines
 *************************************************************************/


/**** Internal routines for TransferHost ****/


SystemCode CopyInLHost(active)
    Process *active;
  {
    KernelRequest *req = (KernelRequest *) &active->msg;
    char *dest;
    unsigned lhost;
    Process *pd;
    Team *utd, *td;
    LhnRec *uld, *oldLd, *ld;
    MHdr *hdr;
    int nP, nT;
    int i;
    ProcessId pid;
    int index;
    ProcessId tmpPid, tmpPid1;
    unsigned tmp;

    dest = (char *) req->segment;
    lhost = PidLhn(req->pid);
    hdr = (MHdr *) dest;
    nP = hdr->nPds;
    nT = hdr->nTds;

    /* Check whether there are enough resources available. */
    /* ??? */

    /* Find the log. host to install things into. */
    oldLd = FindLhn(lhost);
    if (oldLd ==  NULL)
      {
	return(BAD_ARGS);
      }

    dest += sizeof(MHdr);

    /* Initialize the lhost desc. */
    uld = (LhnRec *) dest;
    ld = AllocateLhn(uld->lhn);
    ld->pidGen = uld->pidGen;
    ld->flags = uld->flags;
    ld->teams = oldLd->teams;
    /* The teams field should be pointing to a list of team descriptors that
       will be used to transfer the new l.host's teams into. */
    DeallocateLhn(oldLd);
    dest += sizeof(LhnRec);

    /* Initialize team descriptors for all teams.  We assume that the order
       of the teams in the transfer descriptor is the same as the order of
       the teams to use on the local l.host's teams list. 
       Also delete the root (and only) process of each team at this time. */
    for (i = 0, utd = (Team *) dest, td = ld->teams; 
         i < nT; 
	 i++, utd++, td = td->next)
      {
        MAP_TO_RPD(pd, td->team_root);
	td->team_root = utd->team_root;
	req->pid = pd->pid;
	DestroyProcess(active);	/* NOTE: This HAS to follow the changing of
				   team_root in order to avoid deallocating 
				   the team. */
	td->owner = utd->owner;
	td->team_priority = utd->team_priority;
	td->min_priority = utd->min_priority;
	td->lhn = ld;
      }

    /* Initialize process descriptors and their associated data structures
       for all processes. */
    for (dest = (char *) utd, i = 0; i < nP; i++)
      {
	dest = CopyInProcess(dest, ld);
      }

    /* Do a second pass to link the father, brother, and son fields now
       that all the PDs have been allocated.  
       Also adjust the state of the PD according to what queue it should be
       on.  (This can't be done earlier because the queue may be the msg queue
       of a PD on the same l.host. */
    for (td = ld->teams; td != NULL; td = td->next)
      {
	MAP_TO_RPD(pd, td->team_root);

	/* Set the father link of the root process to active. */
	pd->brother = active->son;
	active->son = pd;
	pd->father = active;
	SetPdState(pd);

	pd->son = MapPid(pd->son);
	if (pd->son != 0)	/* NOTE: We assume that all teams are created 
				   by the team server, which resides on the 
				   system lhost, which is never frozen.
				   Hence, all sons of the root process must 
				   be on the same team. */
	  {
	    SetFBS(pd->son);
	  }
      }

    /* Join the local kernel server group for the newly transferred l.host. */
    tmpPid = active->forwarder;
    tmp = req->unspecified[0];
    tmpPid1 = req->pid;
    active->forwarder = KernelServerPid;
    req->opcode = OK;
    req->unspecified[0] = PidLhn(ld->lhn) | LKERNEL_SERVER_GROUP;
    req->pid = KernelServerPid;
    if( JoinGroup(active) != OK )
      {
	Kabort("Kernel server failed to join LKERNEL_SERVER_GROUP");
      }
    active->forwarder = tmpPid;
    req->unspecified[0] = tmp;
    req->pid = tmpPid1;

    return(OK);
  }


char *CopyInProcess(upd, ld)
    Process *upd;
    LhnRec *ld;
  {
    Process *pd;
    int suspendFlag, nExtend, nMsgq, nGrpMem, nLid;
    Process *qpd, *spd;
    ExtendPd epd;
    int i, index, *iPtr;
    char *dest;
    ProcessId *pPtr, localPid, head, tail;
    ProcessId groupId;
    GroupMember *grpmem;
    kPacket *kp;

    suspendFlag = (int) upd->nextPd;
    nExtend = (int) upd->extendPd;
    nMsgq = (int) upd->msgq.head;
    nGrpMem = (int) upd->replyq.head;
    nLid = (int) upd->replyq.tail;

    /* Fix up the process descriptor. */
    pd = RemapProcessDesc(upd, ld);

    /* Check for suspend descriptor. */
    if (suspendFlag)
      {
        upd++;
/*printx("Suspend PD: %x\n", upd->pid);*/
	spd = RemapProcessDesc(upd, ld);
      }
    upd++;

    /* Allocate extendPd's. */
    pd->extendPd = NULL;
    for (i = 0; i < nExtend; i++, upd++)
      {
	epd = (ExtendPd) AllocatePd();
	Copy(epd, upd, sizeof(Process));
	epd->extendPd = NULL;
	LinkExtendPd(pd, epd);
      }

    /* Link up the msgq. */
    pd->msgq.head = NULL;
    pd->msgq.tail = (Process *) &(pd->msgq.head);
    for (i = 0, dest = (char *) upd;
         i < nMsgq;
         i++, dest += sizeof(kPacket) + 3 * sizeof(ProcessId))
      {
        pPtr = (ProcessId *) dest;
	localPid = *pPtr++;
	head = *pPtr++;
	tail = *pPtr++;
        kp = (kPacket *) pPtr;
	if (PidLhn(pd->pid) == PidLhn(kp->srcPid))
	  {
	    /* On the same l.host.  Will get taken care of later. */
	    continue;
	  }
	else
	  {
	    /* Not on the same l.host. */
	    if (MAP_TO_RPD(qpd, kp->srcPid) &&
	        !IsGroupId(qpd->blocked_on))
	      {
		/* Sender is now local.  Link to its PD. */
		RemoveQueue(qpd); /* Take it off the delay queue.  It will be
				     put in the msgq further down. */
	      }
	    else
	      {
		/* Sender is remote or doing multicast. */
		qpd = Alienate(kp);
		if (localPid == ALIEN_PROCESS)
		  {
		    /* Was remote.  We use the cached forwarder and dstPid
		       fields. */
		    qpd->msgq.head = (Process *) head;
		    qpd->msgq.tail = (Process *) tail;
		  }
		else
		  {
		    /* Was local.  We use the forwarder and dstPid field
		       values in the message itself. */
		    qpd->msgq.head = (Process *) kp->dstPid;
		    qpd->msgq.tail = (Process *) kp->forwarder;
		  }
	      }
	  }
	Lockq(&pd->msgq);
	pd->msgq.tail = (pd->msgq.tail)->link = qpd;
	qpd->queuePtr = &pd->msgq;
	Unlockq(&pd->msgq);
      }
    pd->next_sender = NULL;	/* We only bring along msgs already 
				   received. */

    pd->replyq.head = NULL;
    pd->replyq.tail = (Process *) &(pd->replyq.head);

    /* Set up group memberships. */
    pPtr = (ProcessId *) dest;
    for (i = 0; i < nGrpMem; i++)
      {
        groupId = *pPtr++;
	grpmem = Free_group_descs;
	Free_group_descs = grpmem->link;
	grpmem->groupId = groupId;
	grpmem->pid = pd->pid;
	index = groupId & GROUP_MASK;
	grpmem->link = Group_bundle[index];
	Group_bundle[index] = grpmem;
      }
    dest = (char *) pPtr;

    /* Set up logical id registrations. */
    iPtr = (int *) dest;
    for (i = 0; i < nLid; i++)
      {
        index = *iPtr++;
	Logical_id_map[index].pid = pd->pid;
	Logical_id_map[index].scope = *iPtr++;
      }
    dest = (char *) iPtr;

    return(dest);
  }


Process *RemapProcessDesc(upd, ld)
    Process *upd;
    LhnRec *ld;
  {
    Process *pd, *pd1;
    Team *td;
    int index;

    /* Allocate a process descriptor. */
    pd = AllocatePd();		/* We assume that there are enough resources
				   to accomodate the new l.host. */

    Copy(pd, upd, sizeof(Process));

    index = upd->pid & PID_HASH_MASK;
    disable;
    /* We add the pd at the end of the hash chain in order to give local
       processes faster access. */
    pd->nextPd = &Idle_process;
    pd1 = Pd_bundle[index];
    if (pd1 == &Idle_process)
      {
	Pd_bundle[index] = pd;
      }
    else
      {
	for (; pd1->nextPd != &Idle_process; pd1 = pd1->nextPd)
	  {
	    ;
	  }
	pd1->nextPd = pd;
      }
    enable;

    pd->finish_up = (Unspec (*)()) 
			FinishUpFcnTable[((unsigned) upd->finish_up)];
    pd->timeout_func = (Unspec (*)()) 
			TimeoutFcnTable[((unsigned) upd->timeout_func)];
    for (td = ld->teams; 
    	 td->team_root != ((ProcessId) upd->team); 
	 td = td->next)
      {
	;
      }
    pd->team = td;
    return(pd);
  }


SetPdState(pd)
    Process *pd;
  {
    Process *receiver, *alien, *pd1;
    int queue;

    if (pd->localPid != SUSPEND_PROCESS)
      {
	pd1 = FindSuspendPd(pd->pid);
	if (pd1 != NULL)
	  {
	    SetPdState(pd1);
	  }
      }

    queue = (int) pd->queuePtr;
    pd->queuePtr = NULL;	/* We're not on any queue yet. */
    switch (pd->state)
      {
        case READY:
/*printx("RDY\n");*/
	    Addready(pd);
	    break;

	case RECEIVE_BLKED:
/*printx("RCV-BLK\n");*/
	    DelayProcess(pd);
	    /* We rely on Retransmit fixing up any local sends to this PD. */
	    break;

	case AWAITING_REPLY:
	    /* If trying to create a process group then force a retry. */
	    if (pd->msg.sysCode == CREATE_GROUP)
	      {
		pd->msg.sysCode = RETRY;
		Addready(pd);
		break;
	      }

	    /* The cases that must be handled differently are if the receiver 
	       is now suddenly local rather than remote and if it is now 
	       suddenly remote rather than local. 
	       Group sends just need to be put back on the delay queue. */
	    if (MAP_TO_RPD(receiver, pd->blocked_on))
	      {
	        /* Receiver is local. */
	        if (PidLhn(pd->pid) != PidLhn(pd->blocked_on))
	          {
/*printx("A-R r->l\n");*/
		    /* Receiver now local instead of remote.
		       Replace alien with PD in receiver's msgq and destroy
		       alien. */
		    MAP_TO_APD(alien, pd->pid);
		    pd->queuePtr = alien->queuePtr;
		    alien->queuePtr = NULL;
		    Lockq(&receiver->msgq);
		    for (pd1 = (Process *) &(receiver->msgq.head);
			 pd1->link != alien;
			 pd1 = pd1->link)
		      {
			;
		      }
		    pd->link = alien->link;
		    pd1->link = pd;
		    if (receiver->msgq.tail == alien)
		      {
			receiver->msgq.tail = pd;
		      }
		    if (receiver->next_sender == alien)
		      {
			receiver->next_sender = pd;
		      }
		    Unlockq(&receiver->msgq);
		    DestroyAlien(alien);
		  }
		else
		  {
/*printx("A-R l->l\n");*/
		    /* Receiver was on same l.host.  Link to its msg queue. */
		    if (queue == MSG_QUEUE)
		      {
			/* We go on or after the next_sender ptr. */
			EnqueueMsg(pd, receiver, pd1);
		      }
		    else
		      {
			/* We go before the next_sender ptr. */
			EnqueueMsgAtFront(pd, receiver);
		      }
		  }
	      }
	    else 
	      {
	        /* Receiver is remote or this is a group send. */
	        if (queue == MSG_QUEUE)
		  {
/*printx("A-R l->r\n");*/
		    /* Was local and is now remote.  Must put on delay queue 
		       and set the timeout function. */
		    NonLocalSend(pd);
		  }
		else
		  {
/*printx("A-R r->r\n");*/
		    /* Receiver was and is remote or we have a group send.
		       Put him back on the delay queue for retransmissions. */
		    DelayProcess(pd);
		  }
	      }
	    break;

	case DELAYING:
/*printx("DLY: %x\n", pd->timeout_count);*/
	    DelayProcess(pd);
	    break;

	case MOVEFROM_BLKED:
	case MOVETO_BLKED:
/*printx("MV-BLK\n");*/
	    /* The only case we have to do something special is when a formerly
	       remote sender is now local.  Note that a process can only
	       be MOVE-BLKED on a frozen host if the sender isn't on the same 
	       l.host. */
	    if (MAP_TO_RPD(pd1, pd->blocked_on))
	      {
		/* Sender is local now.  Sender's PD will get put on our
		   msgq.  We must get put back on delay queue checking for
		   dead sender. */
		pd->timeout_func = (Unspec (*)()) CheckFrozenSender;
		pd->timeout_count = FROZEN_SENDER_CHECK_PERIOD;
		DelayProcess(pd);
	      }
	    else if (pd->state == MOVEFROM_BLKED)
	      {
		NonLocalCopyFrom(pd);
	      }
	    else
	      {
		NonLocalCopyTo(pd);
	      }
	    break;

	case AWAITING_INT:
	default:
	    Kabort("TransferHost got a bogus descriptor to install!");
	    break;
      }
  }


SetFBS(pd)
    Process *pd;
  {
    pd->father = MapPid(pd->father);
    pd->brother = MapPid(pd->brother);
    pd->son = MapPid(pd->son);
    SetPdState(pd);
    if (pd->son != NULL)
      {
	SetFBS(pd->son);
      }
    for (pd = pd->brother; pd != NULL; pd = pd->brother)
      {
	pd->father = MapPid(pd->father);
	pd->brother = MapPid(pd->brother);
	pd->son = MapPid(pd->son);
	SetPdState(pd);
	if (pd->son != NULL)
	  {
	    SetFBS(pd->son);
	  }
      }
  }




/**** Additional internal routines ****/


/*
 * SendToLocalMigratedHost:
 * Send a message to a process that used to be remote and has now migrated
 * onto the local machine.
 */

SendToLocalMigratedHost(pd)
    Process *pd;
  {
    Process *sender;
    SystemCode replycode;

    switch (pd->state)
      {
	case RECEIVE_BLKED:
	    Kabort(
	      "SendToLocalMigratedHost called with state == RECEIVER_BLKED");
	    break;

	case AWAITING_REPLY:
printx("SLMH: A-R\n");
	    /* We only get here if the message hasn't already been received
	       because otherwise the act of migration will have already fixed 
	       up the PDs.  Since unreceived messages are discarded on 
	       migration, we can simply reexecute the KSend operation to make 
	       things happen correctly. */
	    KSend(pd);
	    break;

	case MOVEFROM_BLKED:
	case MOVETO_BLKED:
printx("SLMH: MBLK\n");
	    CheckFrozenSender(pd);	/* Must worry about whether still 
					   frozen. */
	    break;

	case READY:
	case AWAITING_INT:
	case DELAYING:
	default:
	    /* Should never happen. */
	    Kabort("SendToLocalMigratedHost called from a bad kernel state!");
	    break;
      }
  }


DestroyMigratingProcess(pd)
    register Process *pd;
  {
  /* We never execute a destroy on a process that is frozen except when 
   * invoked by DestroyHost, which is only used for migrating l.hosts.
   */
    register Process *prev;
    Process *pd1, *pd2;
    register ProcessId pid, tmp;
    register ExtendPd sp;
    unsigned userno;
    KernelRequest *req;

    /* Unqueue all blocked senders. */
    Lockq( &(pd->msgq) );
    while( (prev = pd->msgq.head) != NULL )
      {
	prev->queuePtr = NULL;
	pd->msgq.head = prev->link;
	if (AlienProcess(prev))
	  {
/*printx("DOP: 1\n");*/
	    if (MAP_TO_RPD(pd2, prev->pid))
	      {
		/* Local group send.  Forward it since it won't get 
		   retransmitted. */
		DirectNonLocalForward(prev);
	      }
	    /* It will get recreated through retransmission. */
	    DestroyAlien(prev);
	  }
	else if (PidLhn(prev->pid) != PidLhn(pd->pid))
	  {
/*printx("DOP: 2\n");*/
	    /* Local sender from another l.host.  Force a resend. */
	    NonLocalSend(prev);
	  }
	else
	  {
/*printx("DOP: 3\n");*/
	    /* Local sender on the same l.host.  He'll get killed shortly. */
	    prev->queuePtr = NULL;
				/* This isn't really necessary, but ... */
	  }
      }
    Unlockq( &(pd->msgq) );
    /* Forward all group replies. */
    Lockq( &(pd->replyq) );
    while( (prev = pd->replyq.head) != NULL )
      {
	pd->replyq.head = prev->link;
	prev->queuePtr = NULL;
	if (AlienProcess(prev))
	  {
/*printx("F-1");*/
	    DirectNonLocalForward(prev);
	    DestroyAlien(prev);
	  }
	else if (PidLhn(prev->pid) != PidLhn(pd->pid))
	  {
/*printx("F-2");*/
	    NonLocalSend(prev);
	  }
	else
	  {
/*printx("F-3");*/
	    /* Local sender on the same l.host.  He'll get killed shortly. */
	    prev->queuePtr = NULL;
				/* This isn't really necessary, but ... */
	  }
      }
    Unlockq( &(pd->replyq) );
    /* Destroy any local aliens that aren't replies */
    if (pd->pdFlags & LOCAL_ALIENS_PRESENT) ScavengeAliens( pd->pid );
    /* Deallocate any extendPds. */
    while (pd->extendPd != NULL)
      {
	sp = pd->extendPd;
	DeleteExtendPd(pd, sp->eType);
      }
    /* Deallocate any suspend PDs associated with this PD. */
    pd1 = FindSuspendPd(pd->pid);
    if (pd1 != NULL)
      {
        if (pd1->queuePtr->type == MSG_QUEUE)
	  {
	    InitializeAlien_1(pd1);
	  }
	else
	  {
	    RemoveQueue(pd1);
	    FreePd(pd1, pd->pid);
	  }
      }
    pd->pdFlags &= ~SUSPEND_FLAG;
    /* Remove PD from any queue it is on and replace with an alien if it was
       on a MSG_QUEUE that is not for a PD on the same l.host. */
    if ((pd->queuePtr->type == MSG_QUEUE) &&
        (PidLhn(pd->pid) != PidLhn(pd->blocked_on)))
      {
	/* Turn the PD into an alien and just leave it in place. */
	InitializeAlien_1(pd);
      }
    else
      {
	RemoveQueue( pd );
	FreePd( pd, pd->pid );
      }
  }


DirectNonLocalForward(pd)
    Process *pd;
  {
    ProcessId tmp;
    Process *pd1;

    /* WriteKPacket swaps pid and blocked_on for aliens.  We don't 
       want that in this case since we don't want the packet to go 
       back to its original destination.
       Just changing the PD to not be an alien also doesn't work 
       because then the DestroyAlien timeout function won't work 
       correctly.  
       Swapping pid and blocked_on here also doesn't work because 
       FreePd won't be able to find the PD in the Pd_bundle chain then.
       So we create a second alien that has all the fields and links
       correctly set and use it. */
    tmp = pd->pid;
    pd->pid = pd->blocked_on;
    pd->blocked_on = tmp;
    pd1 = Alienate(&pd->packetType);
    if (pd1 != NULL)
      {
	WriteKPacket( pd1 );
      }
    tmp = pd->pid;
    pd->pid = pd->blocked_on;
    pd->blocked_on = tmp;
  }
