/*
 * V Kernel - Copyright (c) 1984 by Stanford University.
 *
 * Kernel Logical Host Management Routines.
 */


#include "Vikc.h"
#include "Vquerykernel.h"
#include "process.h"
#include "ikc.h"

extern Process Idle_process;
extern Process *KernelServerPd;
extern ProcessId KernelServerPid;
extern ProcessId CreateTeam();
extern int Retransmit();
extern int InitFinishUpFcn();
extern SystemCode JoinGroup();

/*
 * Logical host number data structures.
 */
#define MAX_LHNS 32
#define LHN_HASH_MASK 15
#define PID_GEN_START 10	/* Start numbering processes on a l.host from
				   here to avoid running into the permanently 
				   existing OS ones. */
LhnRec LhnDescriptors[MAX_LHNS];
LhnRec *Lh_bundle[LHN_HASH_MASK+1];
				/* Hash table for logical host numbers. 
				   Points to lists of LhnRecs which hash
				   to the same value. */
LhnRec *LhnFreeList;		/* Head of list of free logical host number 
				   entries. */

unsigned LogicalHostNumberSet;	/* Set of preallocated logical host numbers. */
int LhnCnt;			/* Count of number of preallocated lhns
				   already used minus one. */
int SetLhnFlag;			/* True if LogicalHostNumberSet should be 
				   reset. */
#define MAX_LHN_COUNT 7		/* Number of logical host numbers we 
				   preallocate at a time minus one. */
#define LHN_SET_MASK 0xff8fffff /* Location of preallocated number sets in the
				   logical host id. */
#define LHN_SET_SHIFT 20	/* Shift required to shift a number into the
				   preallocated number set. */

#define HOST_HASH_SHIFT 24	/* Shift used to get to random bits in a
				   logical host number. 
				   *** NOTE ***
				   Has to leave enough bits for both the
				   Lh_bundle and HostCache hash tables. */

#define OLD_LHN_CACHE_SIZE 8
#define EMPTY_OLD_LHN_CACHE_ENTRY -1
				/* Don't use 0 because that's used as a l.host
				   number during start-up. */
unsigned OldLhnCache[OLD_LHN_CACHE_SIZE];
				/* Cache of old, previously used logical host 
				   numbers.  We use this to detect references 
				   to deceased local processes. */
int OldLhnCachePtr;		/* Ptr to last entry made in the old lhn 
				   cache. */

SystemCode NewLhn(), FinishLhn();

int Local( pid ) 
    Process_id pid;
  /* Return 1 if local, etc. 0 */
  {
    register unsigned host;
    register LhnRec *lhnIndex;
    register int i, j;
    unsigned index;

    host = PidLhn(pid);
    index = host >> HOST_HASH_SHIFT;
    for (lhnIndex = Lh_bundle[index & LHN_HASH_MASK]; 
         lhnIndex != NULL; 
	 lhnIndex = lhnIndex->next)
      {
	if (host == lhnIndex->lhn) return( 1 );
      }
    j = OldLhnCachePtr;
    for (i = 0; i < OLD_LHN_CACHE_SIZE; i++)
      {
	if (OldLhnCache[j] == host) return(1);
	if (j == 0) 
	    j = OLD_LHN_CACHE_SIZE - 1;
	else 
	    j--;
      }
    if( pid == 0 ) return( 1 );
    return( 0 );
  }

LocalGroup( gid )
    register GroupId gid;
    /* Return 1 if gid is a groupId local to this workstation,
     * independent of whether there are any members or not.
     */
  {
    if( !IsLocalGroupId(gid) ) return( 0 );
    return( Local(gid) );
  }


/*
 * Logical host management routines
 */

InitLhn()
  {
    /*
     * Initialize the logical host number records and list pointers.
     */
    register int i;
    register LhnRec *ld;

    LhnFreeList = NULL;

    LhnCnt = MAX_LHN_COUNT;	/* Make sure we get a new logical host number 
				   the first time through. */

    OldLhnCachePtr = 0;
    for (i = 0; i < OLD_LHN_CACHE_SIZE; i++)
      {
	OldLhnCache[i] = EMPTY_OLD_LHN_CACHE_ENTRY;
      }

    for (i = 0; i < LHN_HASH_MASK + 1; i++) Lh_bundle[i] = NULL;

    for (i = 0, ld = LhnDescriptors; i < MAX_LHNS; i++, ld++)
      {
        ld->next = LhnFreeList;
	LhnFreeList = ld;
	ld->lhn = 0;
	ld->pidGen = 1;
	ld->teams = NULL;
	ld->requestQ.type  = MSG_QUEUE;
	ld->requestQ.head = NULL;
	ld->requestQ.tail = (Process *) &(ld->requestQ.head);
      }
  }


SystemCode CreateHost( active )
    register Process *active;
  {
    /*
     * Create a new logical host, creating a new team with the specified
     * priority, entry and stack and return the pid of the new host's
     * root process.
     */

    /* Create a new logical host number for this team */
    NewLhn(active);
    /* A new team will be created in the finish-up function. */
    return( NO_REPLY );
  }


DeallocateLhn(ld)
    register LhnRec *ld;
  {
    /*
     * Deallocate a logical host's record.  Assumes that all associated
     * team descriptors have already been deallocated. 
     */
    register int index;

    if (OldLhnCachePtr == (OLD_LHN_CACHE_SIZE - 1))
      {
	OldLhnCachePtr = 0;
      }
    else
      {
	OldLhnCachePtr++;
      }
    OldLhnCache[OldLhnCachePtr] = ld->lhn;

    if (ld->prev == NULL)
      {
	index = (ld->lhn >> HOST_HASH_SHIFT) & LHN_HASH_MASK;
	Lh_bundle[index] = ld->next;
      }
    else
	ld->prev->next = ld->next;
    if (ld->next != NULL) ld->next->prev = ld->prev;
    ld->next = LhnFreeList;
    LhnFreeList = ld;

    /* Forward any deferred kernel operations on this l. host since we 
       usually only destroy frozen logical hosts upon migration.  If it is
       being destroyed without migration then the forwarded ops will
       simply time out.  Note: we do this AFTER the lhost descr. has been
       deallocated so that nonlocal forwarding will work. */
    ExecuteDeferredKernelOps(ld);
  }
  
LhnRec *AllocateLhn(lhn)
    register unsigned lhn;
  {
    /* Create a lhn rec for the specified lhn. Assumes that the
     * lhn doesn't collide with any other local lhns.
     */
    register LhnRec *lhnrec;
    unsigned index;

    lhnrec = LhnFreeList;
    LhnFreeList = LhnFreeList->next;
    lhnrec->lhn = lhn;
    lhnrec->pidGen = PID_GEN_START;	/* Get over the permanently existing
					   OS processes. */

    /* Add it to the hash table */
    index = (lhn >> HOST_HASH_SHIFT) & LHN_HASH_MASK;
    lhnrec->next = Lh_bundle[index];
    Lh_bundle[index] = lhnrec;
    lhnrec->prev = NULL;
    if (lhnrec->next != NULL) lhnrec->next->prev = lhnrec;
    return(lhnrec);
  }

LhnRec *FindLhn(lhn)
    register unsigned lhn;
  {
    /* Find the LhnRec for the specified lhn. Assumes
     * that the lhn exists and is local.
     */
    register LhnRec *lhnrec;
    unsigned index;
    
    index = (lhn >> HOST_HASH_SHIFT) & LHN_HASH_MASK;
    lhnrec = Lh_bundle[index];
    while(lhnrec != NULL)
      {
	if (lhnrec->lhn == lhn) return(lhnrec);
	lhnrec = lhnrec->next;
      }
    return(NULL);
  }

SystemCode NewLhn(active)
    Process *active;
  {
    /* Assign a new logical host number to a team. Make sure
     * that it doesn't collide with an existing lhn in the
     * V domain.
     */
    extern NetAddress EnetHostNumber;
    register unsigned short logicalhost;
    register unsigned logicalHostNumber;
    KernelRequest *msg;

    if (LhnCnt < MAX_LHN_COUNT)
      {
	/* 
	 * NOTE: Will need locking eventually here. 
	 */
        logicalHostNumber = LogicalHostNumberSet;
	LhnCnt++;		/* NOTE: we don't incr. in the if statement
				   because other processes must still see the
				   current state of affairs. */
	logicalHostNumber |= (LhnCnt << LHN_SET_SHIFT);

	AllocateLhn(logicalHostNumber);

	/* Set things up so that we can use the finish-up function just as if 
	   we had just now checked the logical host number. */
	msg = (KernelRequest *) &(active->msg);
	msg->pid = logicalHostNumber;
	msg->opcode = KERNEL_TIMEOUT;
	return(FinishLhn(active));
      }
    else
      {
	SetLhnFlag = 1;
	/* Generate a logical host number. The new host isn't a group. */
	do
	  {
#ifdef ENET10MEG
	    logicalhost = GenerateRandomNumber() + EnetHostNumber.addrlow;
#else ENET3MEG
	    logicalhost = GenerateRandomNumber() + EnetHostNumber;
#endif
	    logicalhost &= ~(GROUP_ID_BIT >> 16);
	    /* Encode byte ordering into the logical host number */
#ifdef LITTLE_ENDIAN
	    logicalhost |= (LITTLE_ENDIAN_HOST >> 16);
#else
	    logicalhost &= ~(LITTLE_ENDIAN_HOST >> 16);
#endif
	    logicalhost &= ~REPLY_SEGMENT_BIT;
	    logicalHostNumber = ((unsigned)logicalhost) << 16;
	  }
	while (logicalhost == 0 || Local(logicalHostNumber) ||
	       (logicalhost == V_BOOT_LHN));
				/*Valid & not local*/
	logicalHostNumber &= LHN_SET_MASK;

	AllocateLhn(logicalHostNumber);

	/* Check for collisions with the other kernels in the domain. */
	if ( (PidLhn(active->pid & ~LITTLE_ENDIAN_HOST) >> 16) == V_BOOT_LHN)
	    /* The init finish_up will reset the pd's lhn if this succeeds. */
	    active->finish_up = (Unspec (*)()) InitFinishUpFcn;
	else
	    active->finish_up = (Unspec (*)()) FinishLhn;
	msg = (KernelRequest *) &(active->msg);
	msg->opcode = QUERY_KERNEL;
	((QueryKernelRequest *) msg)->groupSelect =  LOGICAL_HOST_QUERY;
	((QueryKernelRequest *) msg)->pid =  logicalHostNumber;
	active->blocked_on = VKERNEL_SERVER_GROUP;
	active->forwarder = active->pid;
	KSend(active);
	return( NO_REPLY );
      }
  }

SystemCode FinishLhn( active )
    register Process *active;
  {
    /*
     * Finish-up function for logical host number generation.
     * Adds the new logical host number to the set of valid lhns.
     * Adjusts fields of the root process of the new logical host.
     */
    register KernelRequest *msg;
    unsigned logicalHostNumber;
    SystemCode r;
    ProcessId tmpPid, tmpPid1;
    unsigned tmp;

    msg = (KernelRequest *) &(active->msg);
    logicalHostNumber = (unsigned) msg->pid;
    if (msg->opcode != KERNEL_TIMEOUT)
      {
	DeallocateLhn(FindLhn(logicalHostNumber));
	NewLhn(active);
	active->finish_up = (Unspec (*)()) FinishLhn;
	return( NO_REPLY );
      }

    if (SetLhnFlag)
      {
        SetLhnSet(logicalHostNumber);
      }

    msg->unspecified[3] = logicalHostNumber;
    msg->pid = 0;		/* Don't specify a pid to use. */
    r = CreateTeam(active);
    if (r != OK)
      {
	Addready(active);
	return(r);
      }

    /* Fix the return message. */
    msg->opcode = OK;

    /* Join the local kernel server group for the newly created l.host. */
    tmpPid = active->forwarder;
    tmp = msg->unspecified[0];
    tmpPid1 = msg->pid;
    active->forwarder = KernelServerPid;
    msg->unspecified[0] = PidLhn(logicalHostNumber) | LKERNEL_SERVER_GROUP;
    msg->pid = KernelServerPid;
    if( JoinGroup(active) != OK )
      {
	Kabort("Kernel server failed to join LKERNEL_SERVER_GROUP");
      }
    active->forwarder = tmpPid;
    msg->unspecified[0] = tmp;
    msg->pid = tmpPid1;

    Addready( active );
    return(OK);
  }


SetLhnSet(logicalHostNumber)
    unsigned logicalHostNumber;
  {
    /* 
     * NOTE: Will need locking eventually here. 
     */
    LogicalHostNumberSet = logicalHostNumber;
    LhnCnt = 0;
    SetLhnFlag = 0;
  }


SystemCode FreezeHost(active)
    register Process *active;
  {
    KernelRequest *req;
    ProcessId pid;
    register Process *pd;
    register int i;
    ProcessId lhost;

    req = (KernelRequest *) &active->msg;
    pid = req->pid;
    if (!MAP_TO_RPD(pd, pid))
      {
	return(NONEXISTENT_PROCESS);
      }
/*    if( ((active->userNumber != pd->userNumber) &&
         (active->userNumber != 0)) || 
	(pid == KernelServerPid) )
      {
	return(NO_PERMISSION);
      }*/
    if (pd->pdFlags & FROZEN) 
      {
        return;
      }

    pd->team->lhn->flags |= FROZEN;
    lhost = PidLhn(pid);

    for(i = 0; i <= PID_HASH_MASK; i++)
      {
        for (pd = Pd_bundle[i]; pd != &Idle_process; pd = pd->nextPd)
	  {
	    if ((PidLhn(pd->pid) != lhost) || (pd->localPid == ALIEN_PROCESS))
	      {
	        continue;
	      }
	    FreezeProcess(pd);
	  }
      }
    return( OK );
  }


FreezeProcess(pd)
    register Process *pd;
  {
    pd->pdFlags |= FROZEN;
    switch (pd->state)
      {
	case AWAITING_INT:
	    /* We just let it return to state READY and treat it as such.
	       Thus, freezing a process that is in this state will be deferred
	       until the interrupt is completed. */
	    pd->priority |= STOPPED_TEAM_PRIORITY;
	    break;
	case READY:
	    pd->priority |= STOPPED_TEAM_PRIORITY;
	    RemoveQueue(pd);
	    Addready(pd);
	    break;
	case RECEIVE_BLKED:
	case AWAITING_REPLY:
	    /* Don't need to do anything other than set the FROZEN bit. */
	    break;
	case DELAYING:
	    /* Freeze the delay and re-execute for the remaining time later. */
	    RemoveQueue(pd);
	    pd->length = pd->timeout_count;
				/* Store the remaining time to delay here. */
	    pd->timeout_count = FOREVER;
	    DelayProcess(pd);
	    break;
	case MOVEFROM_BLKED:
	case MOVETO_BLKED:
	    RemoveQueue(pd);
	    pd->timeout_count = FOREVER;
	    DelayProcess(pd);
	    break;
	default:
	    printx("ForceException called on a process in state %d\n",
	    		pd->state);
	    Kabort("Should never have gotten here!");
      }
  }


SystemCode UnfreezeHost(active)
    register Process *active;
  {
    KernelRequest *req;
    ProcessId pid;
    register Process *pd;
    register int i;
    ProcessId lhost;
    LhnRec *lhn;

    req = (KernelRequest *) &active->msg;
    pid = req->pid;
    if (!MAP_TO_RPD(pd, pid))
      {
	return(NONEXISTENT_PROCESS);
      }
/*    if( ((active->userNumber != pd->userNumber) &&
         (active->userNumber != 0)) || 
	(pid == KernelServerPid) )
      {
	return(NO_PERMISSION);
      }*/
    if (!(pd->pdFlags & FROZEN))
      {
        return;
      }

    lhn = pd->team->lhn;
    lhn->flags &= ~FROZEN;
    lhost = PidLhn(pid);

    for(i = 0; i <= PID_HASH_MASK; i++)
      {
        for (pd = Pd_bundle[i]; pd != &Idle_process; pd = pd->nextPd)
	  {
	    if ((PidLhn(pd->pid) != lhost) || (pd->localPid == ALIEN_PROCESS))
	      {
	        continue;
	      }
	    UnfreezeProcess(pd);
	  }
      }

    /* Check for queued kernel server operations to perform. */
    ExecuteDeferredKernelOps(lhn);

    return( OK );
  }


ExecuteDeferredKernelOps(lhn)
    LhnRec *lhn;
  {
    Process *prev;
    
    while( (prev = lhn->requestQ.head) != NULL )
      {
	prev->queuePtr = NULL;
	Lockq(&lhn->requestQ);
	lhn->requestQ.head = prev->link;
	Unlockq(&lhn->requestQ);
	DeliverMsg(KernelServerPd, prev);
      }
  }


UnfreezeProcess(pd)
    register Process *pd;
  {
    Process *receiver;
    ProcessId receiverPid;
    int i;

    pd->pdFlags &= ~FROZEN;
    switch (pd->state)
      {
	case AWAITING_INT:
	    pd->priority = pd->team->team_priority | 
	    		(pd->priority & PROCESS_PRIORITY_MASK);
	    break;
	case READY:
	    pd->priority = pd->team->team_priority | 
	    		(pd->priority & PROCESS_PRIORITY_MASK);
	    RemoveQueue(pd);
	    Addready(pd);
	    break;
	case RECEIVE_BLKED:
	    /* Check if there are queued messages available. */
	    RemoveQueue(pd);
	    KReceiveSpecific( pd );
	    break;
	case AWAITING_REPLY:
	    /* We don't need to check for deferred copy operations because 
	       their timeout functions will cause them to happen 
	       automatically. */

	    /* Check for queued replies. */
	    if (pd->replyq.head != NULL)
	      {
		KGetReply(pd);	/* NOTE: We don't have to set the 
				   timeout_count parameter because we 
				   know that there is a reply queued. */
	      }
	    break;
	case DELAYING:
	    RemoveQueue(pd);
	    pd->timeout_count = pd->length;
	    DelayProcess(pd);
	    break;
	case MOVEFROM_BLKED:
	case MOVETO_BLKED:
	    RemoveQueue(pd);
	    pd->timeout_count = COPY_RETRANS_PERIOD;
	    DelayProcess(pd);
	    break;
	default:
	    printx("ForceException called on a process in state %d\n",
	    		pd->state);
	    Kabort("Should never have gotten here!");
      }
  }


ExecuteDeferredCopyOp(receiver)
    Process *receiver;
  {
    Team *oldteam;
    extern Process *Active;
    Process *saveActive;
    SystemCode status;

    RemoveQueue(receiver);
    saveActive = Active;
    oldteam = GetAddressableTeam();
    Active = receiver;
    SetAddressableTeam(receiver->team);
    /* Local ipc operation.  Check the receiver to see if 
       it is blocked on a deferred copy operation. */
    if (receiver->state == MOVEFROM_BLKED)
      {
	/* Execute the deferred copyfrom operation. */
	status = CopyFrom(receiver->blocked_on, 
		 receiver->dataSegmentPtr,
		 receiver->remoteSegmentPtr, 
		 receiver->dataSegmentSize);
      }
    else
      {
	/* Execute the deferred copyto operation. */
	status = CopyTo(receiver->blocked_on, 
		 receiver->remoteSegmentPtr, 
		 receiver->dataSegmentPtr,
		 receiver->dataSegmentSize);
      }
    receiver->msg.sysCode = OK;
    Active = saveActive;
    SetAddressableTeam(oldteam);
  }


/*
 * DestroyHost:
 * Destroys a host, whether or not it is frozen.  This allows us to 
 * circumvent the deferred kernel op checks in the kernel serve for the
 * regular DestroyProcess and appropriately order operations so that will
 * things will get forwarded properly for migration.
 */

SystemCode DestroyHost(active)
    Process *active;
  {
    KernelRequest *req = (KernelRequest *) &active->msg;
    ProcessId lhost;
    LhnRec *ld;
    Team *td;
    SystemCode status;

    lhost = req->pid;
    ld = FindLhn(PidLhn(lhost));
    if (ld == NULL)
      {
	return(BAD_ARGS);
      }
    /* Deallocate teams from the back because the prev ptr isn't touched
       by team deallocation. */
    for (td = ld->teams; td->next != NULL; td = td->next)
      {
	;
      }
    for (; td != NULL; td = td->prev)
      {
	req->pid = td->team_root;
	status = DestroyProcess(active);
	if (status != OK)
	  {
	    return(status);
	  }
      }

    /* We have now deallocated the l.host.  But we have to remove its id from 
       the list of deceased local l.hosts in order for messages to be
       properly directed. */
    OldLhnCache[OldLhnCachePtr] = EMPTY_OLD_LHN_CACHE_ENTRY;
    if (OldLhnCachePtr == 0)
	OldLhnCachePtr = OLD_LHN_CACHE_SIZE - 1;
    else
	OldLhnCachePtr--;

    return(OK);
  }
