/*
 * Distributed V Kernel - Copyright (c) 1982 by Stanford University
 *
 * Inter kernel communication routines
 *
 */

#include "process.h"
#include "naming.h"

#include "Vikc.h"
#include "ikc.h"

int numRetrans = 0;
extern Unspec Retransmit();
extern int Local();
extern ProcessId KernelServerPid;
extern FreeAlien();
	
int NonLocal( pid ) 
    Process_id pid;
  /*
   * A pid is local if its first 16 bits are equal to those of a logical
   * host number. 
   */
  {
    return( !Local(pid) );
  }    

Unspec LocalGroupTimeout( pd )
register Process *pd;
  {
    register Process *alien;
    Lockq(&pd->replyq);
    if ((alien = pd->replyq.head) != NULL) 
      {
	if( (pd->replyq.head = alien->link) == NULL )
	    pd->replyq.tail = (Process *) &(pd->replyq.head);
        alien->queuePtr = NULL;
	Unlockq( &(pd->replyq) );
	ReturnPid( pd, alien->pid );
	ReturnSegmentSize( pd, 0 );
	Copy_msg( &pd->msg, &alien->msg );
	FreeAlien( alien );
	Addready( pd );
	return;
      }
    Unlockq(&pd->replyq);
    if (!MAP_TO_APD(alien, pd->pid))
      {
        /* If all of the local aliens have gone away then there
	 * isn't going to be a reply.
	 */
	pd->msg.sysCode = KERNEL_TIMEOUT;
        Addready ( pd );
	return;
      }
    pd->timeout_count = SEND_ACKED_RETRANS_PERIOD;
    DelayProcess( pd );
  }

NonLocalSend( active )
register Process *active;

    /* Cause the active process to do a remote send to the
     * process specified by active->blocked_on.
     */
  {
    extern ProcessId Device_Server_Pid;
    register MsgStruct *msg;
    int nGm;

    msg = (MsgStruct *) &active->msg;

    if( active->blocked_on == Device_Server_Pid)
      {
        SendDevice( active );
        return(Device_Server_Pid); 
      }

    if( active->blocked_on & GROUP_ID_BIT )
      { /* Send to local group members. */
	GenericGid(active->blocked_on);
	if( LocalGroup(active->blocked_on) )
	  {  /* Send to a local group */

	    nGm = SendGroupMembers(active);
	    if( nGm == 0 ) /* No group members. */
	      {
		msg->sysCode = KERNEL_TIMEOUT;
		Addready( active );
	      }
	    else if (nGm > 1)
	      {
		if ( msg->sysCode & DATAGRAM_SEND_BIT )
		    Addready( active );
		else
		  {
		    active->numTrans = 1;
		    active->timeout_count = SEND_RETRANS_PERIOD;
		    active->timeout_func = LocalGroupTimeout;
		    DelayProcess( active );
		  }
	      }
	    return;
	  }
	SendGroupMembers( active );
      }
    else if ( (active->blocked_on == 0) || Local(active->blocked_on) )
      {
	msg->sysCode = NONEXISTENT_PROCESS;
	Addready( active );
	return;
      }
    
    /* Setup process descriptor for transmission. */

    active->pdFlags &= ~SEND_ACKED;
    active->packetType = remoteSend;
    if( msg->sysCode & DATAGRAM_SEND_BIT )
	   active->state = READY; /* No reply, continue immediately. */
    else
     {  /* Setup for time outs and retransmissions. */
	active->numTrans = 1;
	active->timeout_count = SEND_RETRANS_PERIOD;
	active->timeout_func = Retransmit;
      }
    WriteKPacket( active );
  }


NonLocalReceiveSpecific( active )
    register Process *active; 
  {
    extern ProcessId KReceiveSpecific();
    extern Process Idle_process;

    if( active->numTrans > MAX_RETRANSMISSIONS )
      {
	ReturnPid(active, 0);
	active->msg.sysCode = KERNEL_TIMEOUT;
	Addready( active );
	return( 0 );
      }
    /* Fix up packet information for retransmission. */
    if (active->numTrans > 2)
      {
	/* We've sent this packet twice.  
	   Perhaps the destination l.host has migrated.
	   Shouldn't have to check whether it is now local since we come into
	   here through KReceiveSpecific, which already checks for local
	   receivers. */

	/* Invalidate the host cache entry and resort to multicast to get 
	   the packet to its destination. */
	HostCacheDelete((active->blocked_on & ~GROUP_ID_BIT)>>16);
      }
    /* Adjust process state */
    active->segmentSize = 0;
    /* Send kernel packet */
    active->packetType = remoteReceiveSpecific;
    ++active->numTrans; /* Set to zero in original ReceiveSpecific call. */
    active->timeout_count = SEND_RETRANS_PERIOD;
    active->timeout_func = (Unspec (*)()) KReceiveSpecific;
    WriteKPacket( active );
  }

GroupReplyDelay( pd )
register Process *pd;
  {
    pd->timeout_count = 2*SEND_ACKED_RETRANS_PERIOD;
    pd->timeout_func = (Unspec (*)()) FreeAlien;
    WriteKPacket( pd );
  }

NonLocalReplyWithSegment( senderalien, active )
    register Process *senderalien, *active;

    /* Reply to senderalien (an alien) or a suspended process
     * using active to specify the segment if any to be appended.
     */
  {
    register Process *sender, *prev;
    register immediateReply;
    MsgStruct *msg;
    Process *AllocatePd();

    immediateReply = active->msg.sysCode & IMMEDIATE_REPLY_BIT;
    active->msg.sysCode &= ~IMMEDIATE_REPLY_BIT;
    if( Local(senderalien->pid) ) /* Reply to local group send or suspended 
				     process. */
      {
        if (senderalien->pdFlags & FROZEN)
	  {
	    sender = senderalien;
	    goto queue_reply;
	  }
	/* If 	(a) the sender has died
	 *   or (b) the sender has initiated a new msg transaction
	 *   or (c) the reply code is DISCARD_REPLY
	 * then this reply can be ignored.
	 *
	 * We shouldn't look at the sender's sequence # field
	 * without having a lock on his pd.
	 */
	MAP_TO_RPD(sender,senderalien->pid);
	if ( (sender == NULL) ||
	     (sender->oldSeqNo != senderalien->seqNo) ||
	     (active->msg.sysCode == DISCARD_REPLY) )
	  {
#ifdef DEBUG
	    printx("remoteReply: old reply message discarded\n" );
#endif
	    FreeAlien( senderalien );
	    Addready( active );
	    return;
	  }
	if( RemoveDelayQueue(sender) )
	  {
	    if( sender->state != AWAITING_REPLY )
	      {
		DelayProcess( sender );
		goto queue_reply;
	      }
	    /* Reply as in local case. */
	    Copy_msg( &sender->msg, &active->msg );
	    if( active->segmentSize != 0 )
	      {
		/* This restriction will be removed with redesign of IPC */
		if( (active->segmentSize > MAX_APPENDED_SEGMENT) ||
			(CopyToSegment(active, sender) != OK) )
		    sender->msg.sysCode = BAD_REPLY_SEGMENT;
	      }
	    ReturnPid( sender, active->pid );
	    ReturnSegmentSize( sender, active->segmentSize );
	    Addready( sender );
	    FreeAlien( senderalien );
	  }
	else /* Not the first reply; queue in reply queue. */
	  {
queue_reply:
            Lockq( &(sender->replyq) );
            /* Create and initialize an alien to stand in the reply queue. */
            if( senderalien->pdFlags & FROZEN )
	      {
	        if ( (senderalien = AllocatePd()) == NULL)
		  {
		    printx("Discarding local reply\n");
		    Unlockq( &sender->replyq );
		    return;
		  }
	      }
	    else
		/* Remove alien from AlienMap to use a reply buffer. */
		Dealienate( senderalien, senderalien->pid );

	    Copy_msg( &senderalien->msg, &active->msg );
	    senderalien->pid = active->pid;
	    if ( (senderalien->link = sender->replyq.head) == NULL )
		sender->replyq.tail = senderalien;
	    sender->replyq.head = senderalien;
	    senderalien->queuePtr = &(sender->replyq);
	    senderalien->state = REPLIED_TO;
            Unlockq( &(sender->replyq) );
	  }
	Addready( active );
	return;
      }
    /* This is a reply to a remote sender. */
    if ( (active->msg.sysCode == DISCARD_REPLY) )
      {
	senderalien->msg.sysCode = DISCARD_REPLY;
	senderalien->state = REPLIED_TO;
	senderalien->timeout_count =
	    2 * MAX_RETRANSMISSIONS * SEND_ACKED_RETRANS_PERIOD;
	senderalien->timeout_func = (Unspec (*)()) FreeAlien;
	Addready( active );
	DelayProcess( senderalien );
	return;
      }
    if( (active->segmentSize != 0) )
      {
	/* Assume this is an idempotent reply */
	if( active->segmentSize > MAX_APPENDED_SEGMENT )
	  {
	    active->segmentSize = 0;
	    msg = (MsgStruct *) &(active->msg);
	    msg->sysCode = BAD_REPLY_SEGMENT;
	  }
	active->seqNo = senderalien->seqNo;
	active->forwarder = senderalien->forwarder;
	FreeAlien( senderalien );
	senderalien = active;
	if ( ( senderalien->forwarder & GROUP_ID_BIT ) && !immediateReply )
	  {
	    /* Delay for up to three clicks before replying */
	    senderalien->timeout_count =  GenerateRandomNumber() & 3;
	    senderalien->timeout_func = (Unspec (*)()) WriteKPacket;
	  }
      }
    else
      {
	/* This reply is non-idempotent. Delay sending replies to group 
	 * sends.  Keep an alien around to suppress duplicate sends. Setup
	 * the alien so that it will timeout in due course.
	 */
	senderalien->state = REPLIED_TO;
	if ( ( senderalien->forwarder & GROUP_ID_BIT ) && !immediateReply )
	  {
	    /* Delay sending replies to group sends to keep from swamping
	     * the sender under a sea of simultaneous packets. This should
	     * be fixed in hardware.
	     */
	    senderalien->timeout_count = GenerateRandomNumber() & 3;
	    senderalien->timeout_func = (Unspec (*)()) GroupReplyDelay;
	  }
	else
	  {	    
	    senderalien->timeout_count = 2*SEND_ACKED_RETRANS_PERIOD;
	    senderalien->timeout_func = (Unspec (*)()) FreeAlien;
	  }

	/* Setup for packet transmission. */
	Copy_msg( &senderalien->msg, &active->msg );
	senderalien->segmentSize = 0;
	Addready( active );
      }
    senderalien->packetType = remoteReply;
    if ( ( senderalien->forwarder & GROUP_ID_BIT ) && !immediateReply )
	DelayProcess( senderalien );
    else
        WriteKPacket( senderalien );
  }


NonLocalForward( pd )
    register Process *pd; 
  {
    /*
     * pd is an alien to be forwarded to pd->forwarder.
     */
    register ProcessId to_pid;

    to_pid = pd->forwarder;
    pd->state = FORWARDED_TO;
    pd->timeout_func = (Unspec (*)()) FreeAlien;
    /* Set the timeout to 0 if idempotent. */
    pd->timeout_count = 2*SEND_ACKED_RETRANS_PERIOD;
    /* Assume blocked_on is set to the forwarder, pd->forwarder to
     * the process to be forwarded to.
     */
    pd->packetType = remoteForward;
    pd->segmentSize = 0;
    WriteKPacket( pd );
    return( to_pid );
  }

Process *FindSender( active, pid )
    register Process *active;
    register ProcessId pid;
  {
    register Process *prev, *sender;

    /* Return the alien representing the sender whose pid is specified */
    Lockq( &(active->msgq) );
    prev = (Process *) &(active->msgq.head);
    while( ((sender = prev->link) != active->next_sender) )
      {
	if( sender->pid == pid ) 
	  {
    	    Unlockq( &(active->msgq) );
	    return (sender);
	  }
	prev = sender;
      }
    Unlockq( &(active->msgq) );
    return( 0 ); /* Sender not found */
  }

NonLocalCopyFrom ( active )
    register Process *active;
  {
    register Process *sender;
    
    if ( (sender = FindSender(active, active->blocked_on)) == 0 ) 
      {
        Addready( active );
	return( NOT_AWAITINGREPLY );
      }

    if( active->dataSegmentSize == 0 )
      {
	Addready( active );
	return( OK );
      }

    active->segmentPtr = (Unspec *) active->dataSegmentPtr;
    active->state = MOVEFROM_BLKED;
    active->packetType = remoteMoveFromReq;
    active->segmentSize = 0;
    active->length = active->dataSegmentSize;
    active->localaddress = active->dataSegmentPtr;
    active->remoteaddress = active->remoteSegmentPtr;
    active->seqNo = ++active->oldSeqNo;
    active->msg.unspecified[0] = sender->seqNo;
    active->numTrans = 1;
    active->timeout_count = COPY_RETRANS_PERIOD;
    active->timeout_func = Retransmit;
    WriteKPacket( active );
    return( NO_REPLY );
  }

  
NonLocalCopyTo( active )
    register Process *active; 
    /* segmentSize, segmentPtr, dataSegmentSize and dataSegmentPtr  have
     * all been set by CopyTo.
     */
  {
    register Process *sender;
    
    if ( (sender = FindSender(active, active->blocked_on)) == 0 ) 
      {
        Addready( active );
	return( NOT_AWAITINGREPLY );
      }

    active->state = MOVETO_BLKED;
    active->packetType = remoteMoveToReq;
    active->localaddress = (Unspec *) active->segmentSize; /* Hack! */
    /* copy parameters segmentPtr and segmentSize set in CopyTo */
    active->msg.segmentPtr = (char *) active->segmentPtr;
    active->msg.segmentSize = active->segmentSize;
    active->seqNo = ++active->oldSeqNo;
    active->msg.unspecified[0] = sender->seqNo;
    active->numTrans = 1;
    active->timeout_count = COPY_RETRANS_PERIOD;
    active->timeout_func = Retransmit;
    WriteKPacket( active );
    return( NO_REPLY );
  }


Unspec Retransmit ( pd ) 
    Process *pd;
  {
    register MsgStruct *msg = &pd->msg;
    extern int GroupMsgExists();

    if ( pd->numTrans > MAX_RETRANSMISSIONS )
      {
	/* We've retransmitted as many times as we're going to, but we don't
	 * time out until all of the local group members have discarded the
	 * their reply.
	 */
	if ( IsGroupId( pd->blocked_on ) &&
	     GroupMsgExists( pd->pid, pd->blocked_on, pd->seqNo ) )
	  {
	    pd->timeout_count = SEND_ACKED_RETRANS_PERIOD;
	    pd->timeout_func = (Unspec (*)()) Retransmit;
	    DelayProcess( pd );
	    return;
	  }
        /* Timed out */
	msg->sysCode = KERNEL_TIMEOUT;
        Addready ( pd );
	return;
      }
    /* Fix up packet information for retransmission. */
    if (pd->numTrans > 2)
      {
	/* We've sent this packet twice.  
	   Perhaps the destination l.host has migrated.
	   Check whether it is now local. */
	if (MapPid(pd->blocked_on) || LocalGroup(pd->blocked_on))
	  {
	    /* Destination is now local! */
	    SendToLocalMigratedHost(pd);
	    return;
	  }
	/* Invalidate the host cache entry and resort to multicast to get 
	   the packet to its destination. */
	HostCacheDelete((pd->blocked_on & ~GROUP_ID_BIT)>>16);
      }
    switch( pd->state )
      {
	case AWAITING_REPLY:
	    pd->seqNo = pd->oldSeqNo;
	    /* The following  is needed because a Copy operation
	     *  may have changed the field values. */
	    /*  Should we be doing something different after a process
             * has been moveto or from'ed but before a reply? */
	    pd->packetType = remoteSend;
	    if( (msg->sysCode&READ_BIT) && (msg->segmentSize != 0 ) )
	      {
		pd->segmentPtr = (Unspec *) msg->segmentPtr;
		pd->segmentSize = min(msg->segmentSize,
						MAX_APPENDED_SEGMENT);
	      }
	    else
	      {
		pd->segmentSize = 0;
	      }
	    pd->pdFlags &= ~MULTI_PACKET_REPLY;
	    if( pd->pdFlags & SEND_ACKED )
		pd->timeout_count = SEND_ACKED_RETRANS_PERIOD;
	    else
		pd->timeout_count = SEND_RETRANS_PERIOD;
	    pd->remoteSegmentPtr = (Unspec *) MAXUNSIGNED;
	    break;
	case MOVETO_BLKED:
	   /* Reset the copy parameters, changed by WriteKPacket */
	    pd->segmentPtr = pd->dataSegmentPtr;
	    pd->segmentSize = pd->dataSegmentSize;
	    /* Now share code with CopyFrom to reset timeout_count */
	case MOVEFROM_BLKED:
	    pd->timeout_count = COPY_RETRANS_PERIOD;
	    break;
	default:
	    if( pd->pdFlags & SEND_ACKED )
		pd->timeout_count = SEND_ACKED_RETRANS_PERIOD;
	    else
		pd->timeout_count = SEND_RETRANS_PERIOD;
      }
    /* Now retransmit packet. */
    pd->numTrans++;
    numRetrans++;  /* Global system count. */
    pd->timeout_func = (Unspec (*)()) Retransmit;
    WriteKPacket( pd );
  }

