/*
 * Distributed V Kernel - Copyright (c) 1982 by Stanford University.
 *
 * Group IPC Communication
 *
 */

/* #define DEBUG 1 */
/* #define DEBUG_LEAVE 1 */
/* #define DEBUG_SCAVENGE 1 */

#include "process.h"
#include "Vikc.h"
#include "groupipc.h"

/* Imports */
extern ProcessId KernelServerPid;
extern SystemCode AddLogicalHostGroup(), DeleteLogicalHostGroup();

/* Exports */
extern SystemCode JoinGroup();
extern SystemCode QueryGroup();
extern SystemCode LeaveGroup();
extern unsigned SendGroupMembers();
extern Process *Alienate();

/* Memory for group descriptors */
GroupMember Group_free_list[MAX_GROUP_DESC];
GroupMember *Free_group_descs;		/* Free list */
GroupMember *Group_bundle[GROUP_MASK+1];/* Edge vector for group desc. */
GroupMember *groupsInCreation;

/* Kludge to allow remote alias processes to join groups. */
int DeliverToEnetReader;

InitGroupIpc()
    /* Allocate group member descriptors and initialize */
  {
    register unsigned i;
    
#ifdef DEBUG
printx("Initializing group descriptors.\n");
#endif
    for (i = 0; i < (MAX_GROUP_DESC-1); i++)
        Group_free_list[i].link = &(Group_free_list[i+1]);
    Group_free_list[MAX_GROUP_DESC].link = NULL;
    Free_group_descs = Group_free_list;
    
    for (i = 0; i < (GROUP_MASK+1); i++) Group_bundle[i] = NULL;
    groupsInCreation = NULL;
  }

ProcessId GetGroupMember( gid, i )
    register GroupId gid; register unsigned i;
  /* Return the pid of the i-th local group member of the group
   * specified by gid if there is one, else 0.
   * This is used/intended to be used to implement random operations
   * on groups, like process destruction.
   */
  {
    register GroupMember *grpmem, *prev;

    prev = (GroupMember *) &Group_bundle[gid & GROUP_MASK];

    while( (grpmem = prev->link) != NULL )
      {
	if( grpmem->groupId == gid )
	  {
	    if( i == 0 ) return( grpmem->pid );
	     --i;
	  }
	 prev = grpmem;
       }
    return( 0 );
  }

int IsGroupMember( pid, gid )
register ProcessId pid;
register GroupId gid;
  /* Return zero if pid isn't a member of gid.
   */
  {
    register GroupMember *grpmem;

    for ( grpmem = Group_bundle[gid & GROUP_MASK];
          grpmem != NULL; grpmem = grpmem->link )
	if (( grpmem->groupId == gid ) && ( grpmem->pid == pid )) return( 1 );
    return( 0 );
  }

ScavengeGroupDescriptors()
  {
    /* Look through all of the group descriptors in use and reclaim those
     * that have no processes associated with them. Have lower level
     * routines reclaim groupIds that have no more local members. 
     */
    register GroupMember *grpmem, *tstmem, *prev;
    register unsigned index;
    Process *pd;

#ifdef DEBUG_SCAVENGE
printx("Scavenging group descriptors\n");
#endif

    for( index = 0; index < GROUP_MASK+1; index++ )
      {
	prev = (GroupMember *) &Group_bundle[index];
	while ( (grpmem = prev->link) != NULL )
	  {
	    if ( (grpmem->pid & REMOTE_ALIAS_PROCESS) )
		{ prev = grpmem; continue; }

	    if ( MAP_TO_RPD( pd, grpmem->pid ) &&
	           ( !(grpmem->groupId & LOCAL_GROUP_BIT) ||
		    FindLhn( grpmem->groupId & LOGICAL_HOST_PART ) ) )
		{ prev = grpmem; continue; }

#ifdef DEBUG_SCAVENGE
printx("Freeing group descriptor %x\n", grpmem->groupId);
#endif
	    /* Free descriptor. */
	    prev->link = grpmem->link;
		
	    for(tstmem = Group_bundle[index]; 
	        (tstmem != NULL) && (tstmem->groupId != grpmem->groupId);
	    	tstmem = tstmem->link );

            if ( tstmem == NULL )
	      {
#ifdef DEBUG_SCAVENGE
printx("Deleting LHG %x\n", grpmem->groupId);
#endif
	        if (DeleteLogicalHostGroup( grpmem->groupId ) != OK)
	            printx("LHG %x not deleted\n", grpmem->groupId);
	      }

	    grpmem->link = Free_group_descs;
	    Free_group_descs = grpmem;
	  }
      }

    /* If a process is killed while trying to create a group then it leaves
     * a bogus entry in the groupsInCreation chain. This prevents a group
     * from being formed with that group id, but that's no big deal.
     */
    prev = (GroupMember *) &groupsInCreation;
    while ( grpmem = prev->link )
      {
	if ( !MapPid( grpmem->pid ) )
	  {
	    prev->link = grpmem->link;
	    grpmem->link = Free_group_descs;
	    Free_group_descs = grpmem;
	  }
	else prev = grpmem;
      }
  }

SystemCode AddGroupMember( pd )
register Process *pd;
  {
    register KernelRequest *req = (KernelRequest *) &pd->msg;
    register GroupId groupId = ( GroupId ) req->unspecified[0];
    register GroupMember *grpmem, *prev;
    register Process *jpd;
    register unsigned index;
 
    prev = (GroupMember *) &groupsInCreation;
    while ( grpmem = prev->link )
      {
	if ( grpmem->pid == pd->pid ) 
	  {
	    prev->link = grpmem->link;
	    grpmem->link = Free_group_descs;
	    Free_group_descs = grpmem;
	  }
	else prev = grpmem;
      }
    
    if( !Local(req->pid) && ( (req->pid & REMOTE_ALIAS_PROCESS) == 0 ) )
      {
	pd->blocked_on = PidLhn(req->pid) | LKERNEL_SERVER_GROUP;
	pd->forwarder = KernelServerPid;
	if( AlienProcess( pd ) ) NonLocalForward( pd );
	else NonLocalSend( pd );
	return( NO_REPLY );
      }

    /* Search for a current member of the group in order to prevent
     * allocating multiple LHG decriptors for a single groupId.
     */
    index = groupId & GROUP_MASK;
    for (grpmem = Group_bundle[index];
         (grpmem != NULL) && (grpmem->groupId != groupId);
	 grpmem = grpmem->link);

    if ((Free_group_descs == NULL) || 
        ( (grpmem == NULL) && (AddLogicalHostGroup(groupId) != OK) ))
      {
	ScavengeGroupDescriptors();

	if ((Free_group_descs == NULL) || 
	    ( (grpmem == NULL) && (AddLogicalHostGroup(groupId) != OK) ))
	  {
#ifdef DEBUG
	    printx("AddGroupMember failed: %s\n", (Free_group_descs == NULL) ?
				"No group descriptors" :
				"Couldn't add logical host");
#endif
	    return( NO_GROUP_DESC );
	  }
      }

    grpmem = Free_group_descs;
    Free_group_descs = grpmem->link;

#ifdef DEBUG
    printx("AddGroupMember: Joining %x to %x\n", pd->pid, groupId );
#endif
    /* Initialize descriptor. */
    grpmem->groupId = groupId;
    grpmem->pid = req->pid;
/* KLUDGE KLUDGE KLUDGE: CreateGroup() looks at this field to determine
 * if this group already exists. This alogorithm doesn't detect collisions
 * because when joining an unrestricted group we never query the rest of
 * the kernel server group.
 */
    req->unspecified[1] = 0;
    /* Link into hash table. */
    grpmem->link = Group_bundle[index];
    Group_bundle[index] = grpmem;
    return( OK );
  }

JoinGroupFinishup( pd )
register Process *pd;
    /* Finish up after checking with other kernel servers if we are
     * allowed to stay in the group.
     */
  {
    register KernelRequest *req = (KernelRequest *) &(pd->msg);

#ifdef DEBUG
printx("JoinGroupFinishup: %x joining %x to %x [%x]\n", pd->pid, req->pid,
	req->unspecified[0], pd->msg.sysCode );
#endif

    if ( req->opcode != KERNEL_TIMEOUT )	/* Refused from some reason. */
      {
	Addready( pd );
	return;
      }

    /* Nobody authorized us and nobody objected. Ask the kernel process to
     * join us to the group. The forward field is used as a capability.
     */
    pd->blocked_on = PidLhn(req->pid) | LKERNEL_SERVER_GROUP;
    pd->forwarder = KernelServerPid;
    req->opcode = JOIN_GROUP;

    KSend( pd );
    return;
  }

SystemCode CreateGroup( pd )
/* Create a new group id and ad the specified pid to the new group. */
register Process *pd;
  {
    /* When creating a group, all that we have to be careful about is that
     * the group id doesn't collide with another existing group id somewhere
     * in the V-domain. Local groups can be handled directly by the local
     * (relative to the joinee pid) kernel. Global groups require querying
     * the kernel server group. When querying we are only interested in the
     * existence of the group, not in what may be contained in it. Once
     * uniqueness has been determined the pid can be joined; no additional
     * security checks are necessary.
     */
    register KernelRequest *req = (KernelRequest *) &pd->msg;
    register ProcessId pid = req->pid;
    register GroupId gid = (GroupId) req->unspecified[0];
    register GroupMember *ptr;

#ifdef DEBUG
printx("CreateGroup: Create %x with first member %x\n", gid, pid );
#endif
    if ( !IsGroupId( gid ) ) return( BAD_ARGS );
    
    if ( gid & LOCAL_GROUP_BIT )
      {
	if ( PidLhn( req->pid ) != PidLhn( gid ) )
	    return( NO_PERMISSION );
	if ( !Local( req->pid ) ) return( AddGroupMember( pd ) );
#ifdef DEBUG
printx("CreateGroup: Checking local group membership.\n");
#endif
	if ( GetGroupMember( gid, 0 ) ) return( RETRY );
#ifdef DEBUG
printx("CreateGroup: Adding %x to group %x\n", pid, gid);
#endif
	return( AddGroupMember( pd ) );
      }

    if ( IsGroupId( pd->forwarder ) )     /* Is another kernel querying us ? */
      {
#ifdef DEBUG
printx("CreateGroup: Query of %x\n", gid );
#endif
	if ( GetGroupMember( gid, 0 ) ) return( RETRY );

	for ( ptr = groupsInCreation; ptr != NULL; ptr = ptr->link )
	    if ((ptr->groupId == gid)&&(ptr->pid != pd->pid)) return( RETRY );
	
	return( DISCARD_REPLY );
      }

    if ( AlienProcess( pd ) ) return( NO_PERMISSION );
    
#ifdef DEBUG
printx("CreateGroup: forwarding to the kernel servers\n");
#endif
    if ( Free_group_descs == NULL ) ScavengeGroupDescriptors();
    if ( Free_group_descs == NULL ) return( NO_GROUP_DESC );
    ptr = Free_group_descs;
    Free_group_descs = ptr->link;
    ptr->groupId = gid;
    ptr->pid = pd->pid;
    ptr->link = groupsInCreation;
    groupsInCreation = ptr;
    pd->blocked_on = VKERNEL_SERVER_GROUP;
    pd->finish_up = (Unspec (*)()) JoinGroupFinishup;
    NonLocalSend( pd );
    return( NO_REPLY );
  }

SystemCode JoinGroup( pd )
register Process *pd;
    /* The process of joining a group has three phases:
     * (1) Authorization: If we can determine locally that the join
     * can proceed then join him immediatley.
     * (2) Network querying: If we aren't sure about this join we have
     * to ask the group of kernel servers what they think about it.
     * (3) Joining: If either of the two preceeding cases determine
     * that the join is legal then the join actually takes place.
     */

    /* A process is allowed to join a group if:
     *    (1) The process performing the join is a group member.
     * or (2) The group is unrestricted.
     * or (3) The new member has the same user number of all previous members.
     * or (4) The joiner is the SUPER_USER or SYSTEM_USER.
     * or (5) The joinee is the first global member of the group.
     * or (6) The joiner is the King of Spain.
     *
     * Non-local processes are not allowed to join a local group.
     *
     * Processes whose forwarder is a kernel server pid are treated as if
     * they have already gone through the authorization process, except
     * in the local group case.
     *
     * Requests to join local groups are immediately forwarded to the
     * joinee's kernel server, where all authorization must take place.
     * The forwarder field is not treated as a special key.
     */
  {
    register KernelRequest *req = (KernelRequest *) &pd->msg;
    register GroupId groupId = ( GroupId ) req->unspecified[0];
    register SystemCode r;

    if ( !IsGroupId( groupId ) ) return( BAD_ARGS );
    
    GenericGid(groupId);
    
    req->unspecified[0] = groupId;

    if ( groupId & LOCAL_GROUP_BIT )
      {
	if ( ( PidLhn( req->pid ) != PidLhn( groupId )) &&
	     ( Local( groupId ) && !Local( req->pid ) ) )
	  {
	    return( NO_PERMISSION );
	  }
	if ( !Local( req->pid ) ) return( AddGroupMember( pd ) );
	r = QueryGroup( pd );
	if ( (r == OK) || (r == NOT_FOUND) ) return( AddGroupMember( pd ) );
	return( r );
      }

   if ( ( groupId & UNRESTRICTED_GROUP_BIT ) ||
	 ( PidLocalPart( pd->forwarder ) == LOCAL_KERNEL_PROCESS_PID ) ||
	 ( pd->userNumber == SUPER_USER ) ||
	 ( pd->userNumber == SYSTEM_USER ) ||
	 ( IsGroupMember( pd->pid, groupId ) ) ||
	 ( req->pid & REMOTE_ALIAS_PROCESS ) ||
	 ( pd->pdFlags & KING_OF_SPAIN ) )
      {
	if ( Local( req->pid ) && ( IsGroupMember( req->pid, groupId ) ) )
	  {
#ifdef DEBUG
printx("JoinGroup: Duplicate Name %x -> %x\n", req->pid, groupId );
#endif
	    return( DUPLICATE_NAME );
	  }
#ifdef DEBUG
printx("JoinGroup: Immediately %x -> %x by %x\n", req->pid, groupId, pd->pid );
#endif
	return( AddGroupMember( pd ) );
      }
	
    if ( AlienProcess( pd ) ) return( NO_PERMISSION );

#ifdef DEBUG
printx("JoinGroup: Querying %x -> %x by %x\n", req->pid, groupId, pd->pid);
#endif
    /* Ask the group of kernel servers for their two cents worth. */
    req->opcode = QUERY_GROUP;
    pd->blocked_on = VKERNEL_SERVER_GROUP;
    pd->finish_up = (Unspec (*)()) JoinGroupFinishup;
    NonLocalSend( pd );
    return( NO_REPLY );
  }


SystemCode QueryGroup( active )
    register Process *active;
  /* Determine whether req->pid would be allowed to join the 
   * group specified by req->unspecified[0].
   */
  {
    register Process *pd;
    register ProcessId pid;
    register GroupMember *grpmem, *prev;
    register unsigned members, umembers;
    register SystemCode r;
    KernelRequest *req = (KernelRequest *) &active->msg;
    register GroupId groupId = (GroupId) req->unspecified[0];

    if( !IsGroupId( groupId ) ) return( BAD_ARGS );

    GenericGid(groupId);
    members = 0; /* Count of members in group. */
    umembers = 0; /* Count of member in group with same userNumber. */

    pid = req->pid;

#ifdef DEBUG
    printx("QueryGroup of group %x, process %x from process %x\n",
		groupId, pid, active->pid );
#endif
    r = NOT_FOUND;
    if ( ( groupId & UNRESTRICTED_GROUP_BIT ) ||
	 ( pd->userNumber == SUPER_USER ) ||
	 ( pd->userNumber == SYSTEM_USER ) ||
	 ( IsGroupMember( pd->pid, groupId ) ) ||
	 ( pd->pdFlags & KING_OF_SPAIN ) )
	r = OK;

    prev = (GroupMember *) &Group_bundle[ groupId & GROUP_MASK ];
    while( ( grpmem = prev->link ) != NULL )
      {
	if( ( grpmem->groupId == groupId ) && ( pd = MapPid( grpmem->pid ) ) )
	  {
	    ++members;
	    if( grpmem->pid == pid ) r = DUPLICATE_NAME;
	    if( pd->userNumber == active->userNumber ) ++umembers;
	  }
	prev = grpmem;
      }

    req->unspecified[1] = members;

    if ( r == DUPLICATE_NAME ) return( DUPLICATE_NAME );

    /* If there some of the local group members have different user numbers
     * we return NO_PERMISSION. There may be cases where this group could be
     * added, such as when a group member performs the add. We don't catch
     * that case here if the requestor is non-local.
     */
    if ( ( umembers != members ) && ( r != OK ) )return( NO_PERMISSION );

    if( ( groupId & LOCAL_GROUP_BIT ) &&
	( !Local( groupId ) || !Local( pid ) ) )
	  return( NO_PERMISSION );

#ifdef DEBUG
   printx("QueryGroup by %x, group %x, %d members, %d umembers, return %d\n",
	active->pid, groupId, members, umembers, r );
   printx( "Group id bit = %d\n", active->forwarder & GROUP_ID_BIT );
#endif

    if ( IsGroupId( active->forwarder ) && (members == 0) )
	return( DISCARD_REPLY );
    return( r );
  }
     
SystemCode LeaveGroup( active )
    register Process *active;
  {
    KernelRequest *req;
    register GroupId groupId;
    register ProcessId pid;
    register GroupMember *grpmem, *prev;
    register Process *pd, *prevpd;
    unsigned int GroupCount = 0;

    req = (KernelRequest *) &active->msg;
    groupId = (ProcessId) req->unspecified[0];

#ifdef DEBUG_LEAVE
printx("%x is having %x leave group %x\n", active->pid, req->pid,
req->unspecified[0]);
#endif

    if( req->pid == 0 ) /* Remove the active process. */
      {
	pid = active->pid;
	pd = active;
      }
    else
      {
	if ((req->pid & REMOTE_ALIAS_PROCESS) == 0)
	  {
	    MAP_TO_RPD(pd, req->pid);
	    if (!pd) return(NONEXISTENT_PROCESS );
	    if( (active->userNumber != pd->userNumber) &&
		(active->userNumber != SUPER_USER) &&
		(active->userNumber != SYSTEM_USER) )
	      {
#ifdef DEBUG
printx( "LeaveGroup: permission denied pid:%x %x -> %x\n",
	active->pid, active->userNumber, pd->userNumber );
#endif
		return( NO_PERMISSION );
	      }
	  }
	pid = req->pid;
      }

    /* Search through the hash table bucket for this groupId and pull
     * it out wherever it is found. We count up the remaining group
     * members because lower level softaware has to be notified when
     * a group is deleted.
     */

#ifdef DEBUG_LEAVE
printx("%x is leaving group %x\n", pid, groupId);
#endif

    GenericGid(groupId);
    prev = (GroupMember *) &Group_bundle[ groupId & GROUP_MASK ];
    while( ( grpmem = prev->link ) != NULL )
      {
	if (grpmem->groupId != groupId) prev = grpmem;
	else if (grpmem->pid != pid) { prev = grpmem; GroupCount++; }
	else
	  {
	    prev->link = grpmem->link;
	    grpmem->link = Free_group_descs;
	    Free_group_descs = grpmem;
	  }
      }

#ifdef DEBUG_LEAVE
if (GroupCount == 0) printx("Freeing LHN %x\n", groupId);
#endif

    if ( GroupCount == 0 ) DeleteLogicalHostGroup( groupId );

    /* Now discard all messages queued for this process that were sent to
     * to this group (only if there are unreceived messages).
     */
    if( (pid & REMOTE_ALIAS_PROCESS) == 0 && pd->next_sender )
      {
	active = pd;
	Lockq( &(active->msgq) );
	/* Locate previous to next sender to aid dequeueing. */
	prevpd = (Process *) &(active->msgq.head);
	while( prevpd->link != active->next_sender ) prevpd = prevpd->link;
	while( (pd=prevpd->link) != NULL )
	  {
	    if( pd->forwarder == groupId ) /* Delete this alien. */
	      {
#ifdef DEBUG
	printx( "LeaveGroup: discarding message from %x\n", pd->pid );
#endif
		prevpd->link = pd->link; /* Unlink from message queue. */
		if( active->next_sender == pd )
		     active->next_sender = pd->link;
		pd->queuePtr = NULL;
		DestroyAlien( pd );
	      }
	    else
		prevpd = pd;
	  }
	Unlockq( &(active->msgq) );
      }
    return( OK );
  }

/*
 * SendGroupMembers:
 * Send the message in pd->msg to all local members of the group
 * specified by pd->blocked.
 */

unsigned SendGroupMembers( pd )
    register Process *pd;
  {
    register short len;
    register Process *alien, *receiver;
    Process *first_receiver;
    register GroupMember *grpmem, *prev;
    Process *AllocatePd();
    register unsigned receivers;

    pd->packetType = 0; /* Breaks Alienate if the ALIEN_EMBEDED_BIT is set */
first_receiver = NULL;
    receivers = 0; /* Count of number of receivers; */
    prev = (GroupMember *) &Group_bundle[ pd->blocked_on & GROUP_MASK ];

#ifdef DEBUG
    printx("Delivering messages to group %x\n", pd->blocked_on );
#endif
    /* Transmit the message to each local member of the group. */
    while(  ( grpmem = prev->link ) != NULL )
      {
	if( (grpmem->groupId != pd->blocked_on) || (grpmem->pid == pd->pid) )
	  { /* Skip over other group descriptors and dont send to self. */
	     prev = grpmem;
	     continue;
	  }
	if ( grpmem->pid & REMOTE_ALIAS_PROCESS )
	  {
	    prev = grpmem;
	    DeliverToEnetReader = 1;
	    continue;
	  }
	MAP_TO_RPD(receiver, grpmem->pid);
	if ( !receiver )
	  { /* Free descriptor for dead process. */
	    prev->link = grpmem->link;
	    grpmem->link = Free_group_descs;
	    Free_group_descs = grpmem;
	    continue;
	  }
	prev = grpmem; /* Save for next itteration */
	++receivers;
#ifdef DEBUG
	printx("Delivering message to %x\n", receiver->pid );
#endif	
	/* Have an alien pd stand in the receiver's message queue. If
	 * the sending pd is an alien use it for the first receiver,
	 * but delay its delivery until last to maintain its state.
	 */
 	if(receivers == 1)
	  {
	    first_receiver = receiver;
	    continue;
	  }
	/* Initialize alien PD */
	if( (alien = Alienate(&pd->packetType)) == NULL ) break;
	alien->state = AWAITING_REPLY;
	alien->forwarder = pd->blocked_on;
	alien->blocked_on = receiver->pid;
	/* Remember the original receiver pid and forwarder for retransmits. */
	alien->msgq.head = (Process *) pd->blocked_on;
	alien->msgq.tail = (Process *) pd->forwarder;
	
	/* Hacks for CopyFromSegment */
	if (!Local(pd->pid))
	  {
	    /* remoteSegmentPtr points to the kernel packet */
	    alien->remoteSegmentPtr = pd->remoteSegmentPtr;
	    ((kPacket *)(alien->remoteSegmentPtr))->remoteaddress =
	     (Unspec *) receiver->dataSegmentPtr;
	  }
	else
	  {
	    alien->team = pd->team;
	    alien->dataSegmentPro = pd->dataSegmentPro;
	    alien->dataSegmentPtr = pd->dataSegmentPtr;
	    alien->dataSegmentSize = pd->dataSegmentSize;
	    alien->segmentPtr = pd->segmentPtr;
	    alien->segmentSize = pd->segmentSize;
	  }

	DeliverMsg( receiver, alien );
      }
    if( first_receiver )
      {
        if (AlienProcess(pd))
	  {
	    /* Use the existing alien instead of duplicating it. */
	    pd->state = AWAITING_REPLY;
	    pd->forwarder = pd->blocked_on;
	    alien = pd;
	  }
	else if ((receivers == 1) && IsLocalGroupId(pd->blocked_on))
	  {
	    /* Exactly one receiver.  Make this a regular send. */
	    alien = pd;
	  }
	else
	  {
	    /* Create an alien stand-in. */
	    if( (alien = Alienate(&pd->packetType)) == NULL ) goto done;
	    alien->state = AWAITING_REPLY;
	    alien->forwarder = pd->blocked_on;
	    /* Remember the original receiver pid and forwarder for 
	       retransmits. */
	    alien->msgq.head = (Process *) pd->blocked_on;
	    alien->msgq.tail = (Process *) pd->forwarder;

	  }
	alien->blocked_on = first_receiver->pid;

	/* Hacks for CopyFromSegment */
	if (!Local(pd->pid))
	  {
	    /* remoteSegmentPtr points to the kernel packet */
	    alien->remoteSegmentPtr = pd->remoteSegmentPtr;
	    ((kPacket *)(alien->remoteSegmentPtr))->remoteaddress =
	     (Unspec *) first_receiver->dataSegmentPtr;
	  }
	else
	  {
	    alien->team = pd->team;
	    alien->dataSegmentPro = pd->dataSegmentPro;
	    alien->dataSegmentPtr = pd->dataSegmentPtr;
	    alien->dataSegmentSize = pd->dataSegmentSize;
	    alien->segmentPtr = pd->segmentPtr;
	    alien->segmentSize = pd->segmentSize;
	  }

	DeliverMsg( first_receiver, alien );
      }

done:
    if ( receivers > 1 ) pd->pdFlags |= LOCAL_ALIENS_PRESENT;
        
#ifdef DEBUG
    printx( "Delivery completed to %d receivers\n", receivers );
#endif
    return( receivers );
  }

int GroupMsgExists( pid, gid, seqno )
/* Return 1 if there is still an alien representing this message outstanding
 * on this machine. This is used when timing out group sends.
 */
register ProcessId pid;
register GroupId gid;
register short seqno;
  {
    register GroupMember *grpmem;
    register Process *pd, *msgalien;
    
    grpmem = Group_bundle[gid & GROUP_MASK];
    while ( grpmem != NULL )
      {
	if ( grpmem->groupId == gid )
	  {
	    MAP_TO_RPD( pd, grpmem->pid );
	    if (pd != NULL)
	      {
		Lockq( &pd->msgq );
		msgalien = pd->msgq.head;
		while ( msgalien != NULL )
	  	  {
		    if ( (msgalien->pid == pid) && (msgalien->seqNo == seqno) )
		      {
	    		Unlockq( &pd->msgq );
			return( 1 );
		      }
		    msgalien = msgalien->link;
	  	  }
		Unlockq( &pd->msgq );
	      }
	  }
	grpmem = grpmem->link;
      }
    return( 0 );
  }
