/*
 * V Kernel - Copyright (c) 1982 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 *
 * Kernel device management
 */

#include "Vio.h"
#include "Vnaming.h"
#include "Vdirectory.h"
#include "process.h"
#include "dm.h"
#include "config.h"

/* Imports */
extern KernelDeviceDescriptor DeviceDirectory[];

/* Exports */
extern InitDeviceServer();
extern ProcessId SendDevice();
extern SystemCode CreateDeviceInstance();
extern SystemCode QueryDeviceInstance();
extern SystemCode ReadDeviceInstance();
extern SystemCode WriteDeviceInstance();
extern SystemCode ReleaseDeviceInstance();
extern SystemCode SetDeviceInstanceOwner();
extern SystemCode ModifyDevice();
extern SystemCode QueryDevice();
extern SystemCode NModifyDevice();
extern SystemCode NQueryDevice();
extern SystemCode ReadDeviceDescriptor();
extern SystemCode DeviceDirCreate();
extern SystemCode DeviceDirRead();
extern SystemCode DeviceDirRelease();
extern DeviceInstance *AllocDeviceInstance();
extern SystemCode NotReadable();
extern SystemCode NotWriteable();
extern SystemCode NotSupported();
extern DeviceInstance *DeviceInstanceTable[];
extern DeviceInstance *GetDeviceInstance();

DeviceInstance *DeviceInstanceTable[];

Process_id Device_Server_Pid; 	/* Kernel device server pid */

#define DFREE_LIST (sizeof(DeviceInstance)*MAX_DEVICES)
char Device_free_list[DFREE_LIST]; /* Raw kernel memory for Device descs */

InitDeviceServer()
  {
    register DeviceInstance *di;
    register unsigned i;

    Device_Server_Pid = 0;
    /* Initialize the device descriptors */
    di = ( DeviceInstance * ) &Device_free_list[0];
    for ( i = 0; i < MAX_DEVICES; ++i )
      {
	DeviceInstanceTable[i] = di;
	di->id = i + MAX_DEVICES;	/* make sure 0 is invalid */
	di->owner = 0;
	di->reader = 0;
	di->writer = 0;
	++di;
      }
  }

/* Note: we need to change all of these functions to take a pointer
 * to the "effectively" active process descriptor as a parameter,
 * plus change the device routines that are invoked by these routines. */

ProcessId SendDevice( pd )
    register Process *pd;

  /* Implement the kernel device server, providing access to kernel
   * devices according to the standard I/O protocol.
   */
  {
    extern Process_id Device_Server_Pid;
    register IoRequest *req;
    register IoReply *replymsg;
    register SystemCode reply;

    req = (IoRequest *)&(pd->msg);
    replymsg = (IoReply *)&(pd->msg);
    if( AlienProcess(pd) )
      {
	/* Requests from remote processes not supported. */
	replymsg->replycode = NO_PERMISSION;
	return( Device_Server_Pid );
      }

    switch( req->requestcode )
      {
	case CREATE_INSTANCE:
	  {
	    reply = CreateDeviceInstance( pd );
            break;
	  }
  	case QUERY_INSTANCE:
	  {
	    reply = QueryDeviceInstance( pd );
	    break;
	  }
	case READ_INSTANCE:
	  {
	    reply = ReadDeviceInstance( pd );
	    break;
	  }
	case WRITE_INSTANCE:
	case WRITESHORT_INSTANCE:
	  {
	    reply = WriteDeviceInstance( pd );
	    break;
	  }
	case RELEASE_INSTANCE:
	  {
	    reply = ReleaseDeviceInstance( pd );
	    break;
	  }
	case SET_INSTANCE_OWNER:
	  {
	    reply = SetDeviceInstanceOwner( pd );
	    break;
	  }
	case MODIFY_FILE:
	  {
	     reply = ModifyDevice( pd );
	     break;
	  }
	case QUERY_FILE:
	  {
	     reply = QueryDevice( pd );
	     break;
	  }
	case NMODIFY_FILE:
	  {
	     reply = NModifyDevice( pd );
	     break;
	  }
	case NQUERY_FILE:
	  {
	     reply = NQueryDevice( pd );
	     break;
	  }
	case READ_DESCRIPTOR:
	case NREAD_DESCRIPTOR:
	  {
	    reply = ReadDeviceDescriptor( pd );
	    break;
	  }
	case WRITE_DESCRIPTOR:
	case NWRITE_DESCRIPTOR:
	  {
	    reply = REQUEST_NOT_SUPPORTED;
	  }
	default:
	  {
  	     reply = ILLEGAL_REQUEST;
 	  }
      }
    if( reply == NO_REPLY )
      {
	/* Suspend process in state AWAITING_REPLY */
	pd->blocked_on = Device_Server_Pid;
      }
    else
      {
	replymsg->replycode = reply;
	Addready( pd );
      }
    return( Device_Server_Pid );
  }

SystemCode CreateDeviceInstance( act )
    register Process *act;

  /* Create an instance of the specified device.
   * Most of the work is performed by a device-dependent routine.
   */
  {
    register DeviceInstance *inst;
    register CreateInstanceRequest *req = (CreateInstanceRequest *) &act->msg;
    register CreateInstanceReply *reply = (CreateInstanceReply *) &act->msg;
    register SystemCode (*func)(), r;
    short unsigned dirIndex;
    short unsigned LookupDeviceName();

    if( (dirIndex = LookupDeviceName(act, req)) > DEV_DIR_SIZE )
      {
	return( NOT_FOUND );
      }

    if( (inst=AllocDeviceInstance(act)) == NULL ) 
	return( NO_SERVER_RESOURCES );

    func = DeviceDirectory[dirIndex].CreateFunc;
    inst->dirIndex = dirIndex;
    inst->unit = DeviceDirectory[dirIndex].d.unit;

    if( (r = (*func)(act, inst)) != OK )
      {
	inst->owner = 0;
	return( r );
      }
    reply->fileid = inst->id;

    return( QueryDeviceInstance( act ) );
  }


short unsigned LookupDeviceName( pd, req )
    register Process *pd;
    register NameRequest *req;
  {
    /* Look up the name passed in a request in the device directory */
    register short i, size;
    register char *p, *q;
    Team *oldteam;

    /* Special case: a null name matches the zeroth entry, which
     *   should always be the descriptor for the device directory
     *   itself */
    oldteam = GetAddressableTeam();
    SetAddressableTeam(pd->team);
    if (req->namelength - req->nameindex == 0 ||
	req->nameptr[req->nameindex] == '\0')
      {
	SetAddressableTeam(oldteam);
	return (0);
      }
    for ( i = 0; i < DEV_DIR_SIZE; i++ )
      {
	p = DeviceDirectory[i].d.name;
    	if (*p == '\0')
	  {
	    SetAddressableTeam(oldteam);
	    return( DEV_DIR_SIZE+1 );  /* not found */
	  }
	q = &req->nameptr[req->nameindex];
	size = req->namelength - req->nameindex;
	if (size > MAX_NAME_LENGTH)
	    size = MAX_NAME_LENGTH;
	while (size > 0)
	  {
	    if (*p != *q) break;	/* no match here */
	    if (*p == '\0') break;	/* end of device's name */
	    p++;
	    q++;
	    size--;			/* may need to look further */
	  }
	if (*p == '\0' && *q == '\0')
	  {
	    SetAddressableTeam(oldteam);
	    return(i);			/* match */
	  }
      }
    SetAddressableTeam(oldteam);
    return( DEV_DIR_SIZE+1 );		/* not found */
  }

SystemCode QueryDeviceInstance( act )
   Process *act;
  /* Satisfy a query instance request from a process.
   */
  {
    extern Process_id Device_Server_Pid;
    register CreateInstanceReply *reply = (CreateInstanceReply *) &act->msg;
    register DeviceInstance *inst;

    if( (inst = GetDeviceInstance(reply->fileid)) == NULL ) 
	return( INVALID_FILE_ID );

    reply->fileserver = Device_Server_Pid;
    reply->blocksize = inst->blocksize;
    reply->filelastbytes = inst->lastbytes;
    reply->filelastblock = inst->lastblock;
    reply->filenextblock = inst->nextblock;
    reply->filetype = inst->type;

    return( OK );
  }


SystemCode ReadDeviceInstance( act )
    Process *act;

  /* Handle a read instance request to a kernel device.
   */
  {
    IoRequest *req = (IoRequest *) &act->msg;
    register DeviceInstance *inst;

    if( (inst = GetDeviceInstance(req->fileid)) == NULL ) 
	return( INVALID_FILE_ID );

    return( (*inst->readfunc)( act, inst ) );
  }

SystemCode WriteDeviceInstance( act ) 
   Process *act;

  /* Handle a write instance request to a kernel device.
   */
  {
    IoRequest *req = (IoRequest *) &act->msg;
    register DeviceInstance *inst;

    if( (inst = GetDeviceInstance(req->fileid)) == NULL ) 
	return( INVALID_FILE_ID );

    return( (*inst->writefunc)( act, inst ) );
  }


SystemCode ModifyDevice( act ) 
    Process *act;

  /* Handle a modify file request to a kernel device.
   */
  {
    IoRequest *req = (IoRequest *) &act->msg;
    register DeviceInstance *inst;

    if( (inst = GetDeviceInstance(req->fileid)) == NULL ) 
	return( INVALID_FILE_ID );

    return( (*inst->modifyfunc)( act, inst, inst->dirIndex ) );
  }

SystemCode QueryDevice( act )
    Process *act;

  /* Handle a query file request to a kernel device.
   */
  {
    IoRequest *req = (IoRequest *) &act->msg;
    register DeviceInstance *inst;

    if( (inst = GetDeviceInstance(req->fileid)) == NULL ) 
	return( INVALID_FILE_ID );

    return( (*inst->queryfunc)( act, inst, inst->dirIndex ) );
  }

SystemCode NModifyDevice( act ) 
    Process *act;
  /* Handle a named modify file request to a kernel device.
   */
  {
    NameRequest *req = (NameRequest *) &act->msg;
    short unsigned dirIndex;

    if( (dirIndex = LookupDeviceName(act,req)) > DEV_DIR_SIZE )
	return( NOT_FOUND );

    return( (*DeviceDirectory[dirIndex].NModifyFunc)( act, NULL, dirIndex ) );
  }

SystemCode NQueryDevice( act ) 
    Process *act;

  /* Handle a named query file request to a kernel device.
   */
  {
    NameRequest *req = (NameRequest *) &act->msg;
    short unsigned dirIndex;

    if( (dirIndex = LookupDeviceName(act,req)) > DEV_DIR_SIZE )
	return( NOT_FOUND );

    return( (*DeviceDirectory[dirIndex].NQueryFunc)( act, NULL, dirIndex ) );
  }

SystemCode ReleaseDeviceInstance( act )
    register Process *act;

  /* Handle a release instance request to a kernel device.
   */
  {
    IoRequest *req = (IoRequest *) &act->msg;
    register DeviceInstance *inst;
    register Process *pd;

    if( (inst = GetDeviceInstance(req->fileid)) == NULL )
	return( INVALID_FILE_ID );
    if( !(pd = MapPid(inst->owner)) || 
        (pd->team != act->team) )
      {
	return( NO_PERMISSION );
      }

    return( (*inst->releasefunc)( act, inst ) );
  }

SystemCode SetDeviceInstanceOwner( act )
    register Process *act;

  /* Handle a set instance owner request to a kernel device.
   */
  {
    register IoRequest *req = (IoRequest *) &act->msg;
    register DeviceInstance *inst;
    register Process *pd;

    if( (inst = GetDeviceInstance(req->fileid)) == NULL ) 
	return( INVALID_FILE_ID );
    if( !(pd = MapPid(inst->owner)) || 
        (pd->team != act->team) ) 
      {
	return( NO_PERMISSION );
      }

    inst->owner = req->instanceowner;

    return( OK );
  }

SystemCode ReadDeviceDescriptor( act )

    Process *act;
  /* Handle a read-descriptor or nread-descriptor request to a kernel device.
   */
  {
    register DescriptorRequest *req = (DescriptorRequest *) &act->msg;
    register DeviceInstance *inst;
    short unsigned dirIndex;
    char *buffer;

    if (req->requestcode == READ_DESCRIPTOR)
      {
	if( (inst = GetDeviceInstance(req->fileid)) == NULL ) 
	    return( INVALID_FILE_ID );
	dirIndex = inst->dirIndex;
      }
    else /* req->requestcode == NREAD_DESCRIPTOR */
      {
	if( (dirIndex = LookupDeviceName(act,req)) > DEV_DIR_SIZE )
	    return( NOT_FOUND );
      }

    buffer = req->segmentptr + req->dataindex;

    if ( (buffer < req->segmentptr) ||
	 (buffer + sizeof(DeviceDescriptor) > 
	    req->segmentptr + req->segmentlen) )
	return(BAD_BUFFER);

    /* Copy descriptor to user space */
    act->remoteSegmentPtr =(Unspec *) buffer;
    act->segmentPtr = (Unspec *) &DeviceDirectory[dirIndex].d;
    act->segmentSize = sizeof(DeviceDescriptor);
    CopyToSegment( act, act );

    return (OK);
  }

SystemCode NotSupported()
  /* Dummy function for requests not supported by device.
   */
  {
    return( REQUEST_NOT_SUPPORTED );
  }

SystemCode NotReadable()
  /* Dummy read function for non-readable devices.
   */
  {
    return( NOT_READABLE );
  }

SystemCode NotWriteable()
  /* Dummy write function for non-writeable devices.
   */
  {
    return( NOT_WRITEABLE );
  }

DeviceInstance *DeviceInstanceTable[MAX_DEVICES];

DeviceInstance *AllocDeviceInstance( act )
    register Process * act;
  /* Allocate a pair of consecutive free device instance
   * descriptors from the instance table and return a pointer
   * to the first one. The instance ids are set to the next
   * "generation" for these descriptors.
   */
  {
    extern DeviceInstance *DeviceInstanceTable[];
    InstanceId	id;
    static int index = -2;
    register DeviceInstance *inst;
    register char *p;
    register unsigned i;

    for( i = 0; i < MAX_DEVICES; i += 2 )
      {
	index += 2;
	if( index >= MAX_DEVICES ) index = 0;

	inst = DeviceInstanceTable[index];

        if( !MapPid(inst->owner) &&
	    !MapPid((inst+1)->owner) )
	  {
	    id = inst->id + MAX_DEVICES; /* Next generation */
	    p = (char *) inst;	/* Now zero the descriptors */
	    for( i = 0; i < sizeof(DeviceInstance)*2; ++i ) *p++ = 0;
	    inst->id = id;
	    (inst+1)->id = id+1;
	    inst->owner = act->pid;  /* Only set owner for first inst */
	    return (inst);
	  }
      }
    return( NULL );
  }

DeviceInstance *GetDeviceInstance( id )
    register InstanceId id;

  /* Locate the kernel device instance descriptor with the corresponding
   * instance id if any and return a pointer to this instance.
   * Return NULL if no such instance.
   */
  {
    extern DeviceInstance *DeviceInstanceTable[];
    register DeviceInstance *inst;

    inst = DeviceInstanceTable[id & (MAX_DEVICES-1)];

    return( (inst->id == id) ? inst : NULL );
  }

/*
 * Routines to handle reading the device directory using
 *   the I/O protocol
 */
SystemCode DeviceDirCreate( act, inst )
    Process *act;
    DeviceInstance *inst;
  {
    CreateInstanceRequest *req = (CreateInstanceRequest *)&act->msg;

    if (req->filemode != FREAD+FDIRECTORY) 
        return (MODE_NOT_SUPPORTED);

    inst->readfunc = DeviceDirRead;
    inst->writefunc = NotSupported;
    inst->modifyfunc = NotSupported;
    inst->queryfunc = NotSupported;
    inst->releasefunc = DeviceDirRelease;
    
    inst->type = READABLE+FIXED_LENGTH;
    inst->blocksize = sizeof(DeviceDescriptor);
    inst->lastblock = DEV_DIR_SIZE-1;
    inst->lastbytes = sizeof(DeviceDescriptor);

    return (OK);
  }


SystemCode DeviceDirRelease( act, inst )
    Process *act;
    DeviceInstance *inst;
  {

    inst->owner = 0;
    return (OK);
  }

SystemCode DeviceDirRead( act, inst )
    Process *act;
    register DeviceInstance *inst;
  {
    IoRequest *req  = (IoRequest *) &act->msg;
    register int bytes;
    
    bytes = req->bytecount;
    req->bytecount = 0;

    if (bytes > inst->blocksize) return (BAD_BYTE_COUNT);
    if ( req->blocknumber > DEV_DIR_SIZE-1 ) return (END_OF_FILE);

    /* Copy descriptor to user space */
    act->remoteSegmentPtr = (Unspec *) req->bufferptr;
    act->segmentPtr = (Unspec *) &DeviceDirectory[req->blocknumber].d;
    act->segmentSize = bytes;
    CopyToSegment( act, act );

    req->bytecount = bytes;

    if (req->blocknumber == DEV_DIR_SIZE-1 && 
	    bytes == sizeof(DeviceDescriptor))
      {
	return (END_OF_FILE);	/* successful read to EOF */
      }
    else
      {
	return (OK);
      }
  }
