/*
 * Kernel Debugging and Diagnostics Routines.
 */


#include "process.h"

extern ProcessId KernelServerPid;
extern Process *Active;
extern SyncQueue Delayq;
extern Process *ProcessDescriptors;
extern Process *EndPds;

char *PrintState(), *PrintFinishUpFcn(), *PrintTimeoutFcn();

int Debug1 = 0, Debug2 = 0, Debug3 = 0;


/*
 * DebugOperation:
 */

SystemCode DebugOperation(pd)
    Process *pd;
  {
    register KernelRequest *req = (KernelRequest *) &pd->msg;

    switch(req->unspecified[0])
      {
	case 0:
	    printx("**** DebugOperation ??? ****\n");
	    break;
	case 1:
	    break;
	default:
	    return(BAD_ARGS);
      }
    return(OK);
  }


/*
 * PrintPdDesc:
 * Print out a process descriptor.
 */

PrintPdDesc(pid)
    ProcessId pid;
  {
    register Process *pd;
    unsigned *m;

    pd = MapPid(pid);
    if (pd == NULL)
      {
	return;
      }
    printx("PD for %x\n", pid);
    printx("link: %x,  nextPd: %x\n", 
    		(pd->link == NULL) ? 0 : pd->link->pid,
		(pd->nextPd == NULL) ? 0 : pd->nextPd->pid);
    printx("father: %x,  brother: %x,  son: %x\n",
		pd->father->pid, 
		(pd->brother == NULL) ? 0 : pd->brother->pid, 
		(pd->son == NULL) ? 0 : pd->son->pid);
    printx("localPid: %x,  pdFlags: %x,  state: %s,  priority: %x\n",
		pd->localPid, pd->pdFlags, PrintState(pd->state), pd->priority);
    printx("team: %x,  next_sender: %x,  msgq: %x,  replyq: %x\n",
		pd->team->team_root, 
		(pd->next_sender == NULL) ? 0 : pd->next_sender->pid,
		pd->msgq.head, pd->replyq.head);
    printx("msgq queue: "); PrintMsgQueue(pd->msgq.head);
    printx("replyq queue: "); PrintMsgQueue(pd->replyq.head);
    printx("oldSeqNo: %x,  numTrans: %x\n",
		pd->oldSeqNo, pd->numTrans);
    printx("dataSegmentPro: %x,  dataSegmentPtr: %x,  dataSegmentSize: %x\n",
		pd->dataSegmentPro, pd->dataSegmentPtr, pd->dataSegmentSize);
    printx("dataExpected: %x,  remoteSegmentPtr: %x\n",
		pd->dataExpected, pd->remoteSegmentPtr);
    printx("processorTime: %d,  ", pd->processorTime);
    PrintQueueType(pd->queuePtr);
    printx("recBufSizePtr: %x,  finish_up: %s,  returnMessage: %x\n",
		pd->recBufSizePtr, PrintFinishUpFcn(pd->finish_up),
		pd->returnMessage);
    printx("segmentPtr: %x,  segmentSize: %x,  currentSegmentPtr: %x\n",
		pd->segmentPtr, pd->segmentSize, pd->currentSegmentPtr);
    printx("packetType: %x,  seqNo: %x,  pid: %x,  blocked_on: %x\n",
		pd->packetType, pd->seqNo, pd->pid, pd->blocked_on);
    printx("forwarder: %x,  userNumber: %x,  length: %x\n",
		pd->forwarder, pd->userNumber, pd->length);
    printx("localaddress: %x,  remoteaddress: %x\n",
		pd->localaddress, pd->remoteaddress);
    m = (unsigned *) &pd->msg;
    printx("msg: %x %x %x %x %x %x %x %x\n", m[0], m[1], m[2], m[3], m[4],
		m[5], m[6], m[7]);
    PrintExtendPd(pd->extendPd);
    printx("timeout_func: %s, timeout_count: %d\n",
    		PrintTimeoutFcn(pd->timeout_func), pd->timeout_count);
    PrintProcState(&(pd->proc_state));
  }


/*
 * PrintProcState:
 */

PrintProcState(ps)
    KProcessor_state *ps;
  {
    int i;
    
#ifndef VAX
    printx("Processor state:\n");
    printx("perProcess: %x,  perProcessLoc: %x\n", 
		ps->perProcess, ps->perProcessLoc);
    printx("regs: d0: %x  d1: %x    a0: %x  a1: %x\n",
		ps->regs[0], ps->regs[1], ps->regs[8], ps->regs[9]);
    printx("      d2: %x  d3: %x    a2: %x  a3: %x\n",
		ps->regs[2], ps->regs[3], ps->regs[10], ps->regs[11]);
    printx("      d4: %x  d5: %x    a4: %x  a5: %x\n",
		ps->regs[4], ps->regs[5], ps->regs[12], ps->regs[13]);
    printx("      d6: %x  d7: %x    fp: %x  sp: %x\n",
		ps->regs[6], ps->regs[7], ps->regs[14], ps->regs[15]);
#else VAX
    DumpProcState(ps);
#endif
  }


/*
 * PrintMsgQueue:
 * Prints the designated message queue of a process.
 */

PrintMsgQueue(pd)
    Process *pd;
  {
    register Process *m;

    for (m = pd; m != NULL; m = m->link)
      {
	printx("%x  ", m->pid);
      }
    printx("\n");
  }


/*
 * PrintState:
 */

char *PrintState(state)
    int state;
  {
    char *s;

    switch (state)
      {
	case 1:
	    s = "READY";
	    break;
	case 2:
	    s = "RECEIVE_BLKED";
	    break;
	case 3:
	    s = "AWAITING_REPLY";
	    break;
	case 5:
	    s = "AWAITING_INT";
	    break;
	case 6:
	    s = "DELAYING";
	    break;
	case 8:
	    s = "MOVEFROM_BLKED";
	    break;
	case 9:
	    s = "MOVETOBLKED";
	    break;
	case 10:
	    s = "REPLIED_TO";
	    break;
	case 11:
	    s = "FORWARDED_TO";
	    break;
	case 13:
	    s = "XMIT_BLKED";
	    break;
	default:
	    s = "***UNKNOWN_STATE***";
	    break;
      }
    return(s);
  }


/*
 * PrintQueueType:
 */

PrintQueueType(queue)
    SyncQueue *queue;
  {
    extern SyncQueue Readyq, Delayq, IkpOutq;
    extern Process Idle_process;
    Process *tmp_pd;

    printx("queuePtr: ");
    if (queue == NULL)
      {
	printx("NULL");
      }
    else if (queue == &Readyq)
      {
	printx("READYQ");
      }
    else if (queue == &Delayq)
      {
	printx("DELAYQ");
      }
    else if (queue == &IkpOutq)
      {
	printx("IKPOUTQ");
      }
    else if (queue->type == MSG_QUEUE)
      {
	printx("MSG_QUEUE - ");
	/* Calculate the address of the pd containing the queue. */
	tmp_pd = (Process *) ( (char *) queue -
		((char *)&Idle_process.msgq - (char *) &Idle_process));
	printx("%x", tmp_pd->pid);
      }
    else
      {
	printx("Unknown queue!");
      }
    printx("\n");
  }


/*
 * PrintFinishUpFcn:
 */

char *PrintFinishUpFcn(fcn)
    unsigned fcn;
  {
    extern int SendnAck(), KReceiveSpecific(), KernelServer();
    extern InitFinishUpFcn();

    if (fcn == 0)
	return("NULL");
    else if (fcn == (unsigned) SendnAck)
	return("SendnAck");
    else if (fcn == (unsigned) KReceiveSpecific)
	return("KReceiveSpecific");
    else if (fcn == (unsigned) KernelServer)
	return("KernelServer");
    else if (fcn == (unsigned) InitFinishUpFcn)
	return("InitFinishUpFcn");
    else
	return("UNKNOWN");
  }


/*
 * PrintExtendPd:
 */

PrintExtendPd(p)
    ExtendPd p;
  {
    if (p != NULL)
      {
	printx("extendPd: ");
	if (p->eType == FORCE_EXCEPTION_TYPE)
	  {
	    printx("eType: FORCE_EXCEPTION_TYPE");
	  }
	else
	  {
	    printx("eType: UNKNOWN");
	  }
	printx(", v0: %x, v1: %x, v2: %x, v3: %x\n", 
				p->v0, p->v1, p->v2, p->v3);
	if (p->extendPd != NULL)
	  {
	    PrintExtendPd(p->extendPd);
	  }
      }
  }


/* 
 * PrintTimeoutFcn:
 */

char *PrintTimeoutFcn(fcn)
    unsigned fcn;
  {
    extern int DestroyAlien(), Addready(), LocalGroupTimeout(), Retransmit();
    extern int KReceiveSpecific(), WriteKPacket(), GroupReplyDelay();

    if (fcn == NULL) return("NULL");
    if (fcn == (unsigned) DestroyAlien) return("DestroyAlien");
    if (fcn == (unsigned) Addready) return("Addready");
    if (fcn == (unsigned) LocalGroupTimeout) return("LocalGroupTimeout");
    if (fcn == (unsigned) Retransmit) return("Retransmit");
    if (fcn == (unsigned) KReceiveSpecific) return("KReceiveSpecific");
    if (fcn == (unsigned) WriteKPacket) return("WriteKPacket");
    if (fcn == (unsigned) GroupReplyDelay) return("GroupReplyDelay");
    return("UNKNOWN");
  }


#ifdef UNDEF


CheckMsgQ(pd)
    Process *pd;
  {
    Process *pd1, *badPd;
    int flag = 0;

    for (pd1 = pd->msgq.head; pd1 != NULL; pd1 = pd1->link)
      {
	if (!CheckValidPd(pd1, "CheckMsgQ", 0))
	  {
	    flag = 1;
	    badPd = pd1;
	    break;
	  }
      }
    if (flag)
      {
        printx("msgq:  ");
	for (pd1 = pd->msgq.head; pd1 != badPd; pd1 = pd1->link)
	  {
	    printx("%x(%x)  ", pd1, pd1->pid);
	  }
	printx("%x\n", badPd);
      }
  }


SanityCheck(pd)
    Process *pd;
  {
    if (!Debug3) return;
    if (pd == &Idle_process) return;
    CheckValidPd(pd, "SanityCheck", 1);
    PrintIdleQ(4);
    if (pd->localPid == ALIEN_PROCESS || pd->localPid == SUSPEND_PROCESS ||
					        pd->localPid == pd->pid)
	return;
    printx("SanityCheck fails for PD - pd: %x, pid: %x, pT: %x\n", 
    		pd, pd->pid, pd->packetType);
    printx("Active: %x(%x),  Readyq.head: %x(%x),  Delayq.head: %x(%x)\n", 
		Active, Active->pid, Readyq.head, Readyq.head->pid, 
		Delayq.head, Delayq.head->pid);
    Kabort("");
  }

PrintPid(pd)
    Process *pd;
  {
if (Debug2)
    printx("xx  ");
  }

int CheckValidPd(pd, str, abortFlag)
    Process *pd;
    char *str;
    int abortFlag;
  {
    Process *pD;
    int flag = 0;

    if (pd == &Idle_process)
      {
	return(1);
      }
    for (pD = ProcessDescriptors; pD != EndPds; pD++)
      {
	if (pD == pd)
	  {
	    flag = 1;
	    break;
	  }
	else if (pD > pd)
	  {
	    flag = 0;
	    pD--;
	    break;
	  }
      }
    if (!flag)
      {
        disable;
	printx("Bogus pd: %x!!!    %s\n", pd, str);
        printx("Active: %x(%x),  Readyq.head: %x(%x),  Delayq.head: %x(%x)\n", 
		Active, Active->pid, Readyq.head, Readyq.head->pid, 
		Delayq.head, Delayq.head->pid);
	printx("Bracketing pds: %x, %x\r\n", pD, (pD+1));
	if (abortFlag)
	    Kabort("");
	else
	    return(0);
      }
    return(1);
  }


PrintIdleQ(n)
    int n;
  {
    Process *prev;

    if (Idle_process.msgq.head == NULL) return;

    printx("(%d) Idle.q: ", n);
    for (prev = (Process *) Idle_process.msgq.head; 
         prev != NULL; 
	 prev = prev->link)
      {
	printx("%x(%x) ", prev, prev->pid);
      }
    printx("\n");
    Kabort("");
  }

int QueueLength(p)
register Process *p;
  {
    register int i = 0;
    while (p != NULL)
      {
        i++;
	p = p->link;
      }
    return(i);
  }

PrintQueues()
  {
    register Process *pd, *head;
    extern Process *ProcessDescriptors, *EndPds;
    
    for (head = ProcessDescriptors; head++; head != EndPds)
      {
	if (head->pid == head->localPid)
	  {
	    printx("%x (%d,%d): [", pd->pid, QueueLength(head->msgq.head),
					     QueueLength(head->replyq.head));
	    pd = head->msgq.head;
	    while (pd != NULL)
	      {
		printx("%x,", pd->pid);
		pd = pd->link;
	      }
	    printx("]{");
	    pd = head->replyq.head;
	    while (pd != NULL)
	      {
		printx("%x,", pd->pid);
		pd = pd->link;
	      }
	    printx("}\n");
	  }
      }
  }
#endif UNDEF

#ifdef MC68020

#include "Venviron.h"
#include "Vexceptions.h"
#include "config.h"
#include "process.h"
#include "machine.h"

extern SyncQueue Readyq, DelayQ, IkpOutq;
extern Process Idle_process;

char *state(pd)
Process *pd;
  {
    switch (pd->state)
      {
	case READY:		return("Ready");
	case RECEIVE_BLKED:	return("Receive blocked");
	case AWAITING_REPLY:	return("Awaiting reply");
	case SEND_BLKED:	return("Send blocked !");
	case AWAITING_INT:	return("Awaiting interrupt");
	case DELAYING:		return("Delaying");
	case GETPID_BLKED:	return("GetPid blocked !");
	case MOVEFROM_BLKED:    return("Remote MoveFrom");
	case MOVETO_BLKED:	return("Remote MoveTo");
	case REPLIED_TO:	return("Replied to");
	case FORWARDED_TO:	return("Forwarded to");
	case ACTIVE:		return("<<< Active Process >>>");
	default:		return("Unknown state");
      }
  }

char *queue(pd)
Process *pd;
  {
    if (pd->queuePtr == NULL) return("(null)");
    switch(pd->queuePtr->type)
      {
	case READY_QUEUE:	return("ready_queue");
	case MSG_QUEUE:		return("msg_queue");
	case TIMEOUT_QUEUE: 	return("timeout_queue");
	case IKP_OUT_QUEUE:	return("ikp_out_queue");
	case REPLY_QUEUE:	return("reply_queue");
	case DISK_QUEUE:	return("disk_queue");
	default:		return("unknown queue type");
      }
  }

Dump()
  {
    Process *queue;
    ExceptionRequest *req;
    int i;

    /* Print pids of processes in ready queue */
    disable;
    printx("Ready queue: ");
    queue = Readyq.head;
    while (queue != NULL)
      {
	printx("%x ",queue->pid);
	queue = queue->link;
      }
    printx("\n");
    
    printx("Delay queue: ");
    queue = Delayq.head;
    while (queue != NULL)
      {
	printx("%x ",queue->pid);
	queue = queue->link;
      }
    printx("\n");
    
    printx("IkpOutq: ");
    queue = IkpOutq.head;
    while (queue != NULL)
      {
	printx("%x ",queue->pid);
	queue = queue->link;
      }
    printx("\n");
    Processes();
  }

Processes()
  {
    int i;
    Process *ptr;
    ExceptionRequest *req;

    disable;
    printx("Process %x is active\n", Active->pid);

    printx("Idle process:\n");
    ptr = &Idle_process;
    info(ptr);
    req = (ExceptionRequest *) &ptr->msg;
    if ( req->requestcode == EXCEPTION_REQUEST ) PrintExcept(req);

    for (i=0; i < (PID_HASH_MASK+1); i++)
      {
	for (ptr = Pd_bundle[i]; ptr->pid != 0; ptr = ptr->nextPd)
	  {
	    info(ptr);
	    req = (ExceptionRequest *) &ptr->msg;
	    if ( req->requestcode == EXCEPTION_REQUEST )
		PrintExcept(req);
	  }
      }
  }

Aliens()
  {
    Process *ptr;

    disable;
    for (ptr = ProcessDescriptors; ptr < EndPds; ptr++)
	if (ptr->localPid == ALIEN_PROCESS)
	    printx("[ %x ] %x -> %x ^%x (%x) %s %s\n", ptr,
		ptr->pid, ptr->blocked_on, ptr->forwarder,
		ptr->msg.sysCode, queue(ptr), state(ptr));
  }

/* info:
 *
 * Prints information about a process descriptor, if it represents a
 *  live process.  The information printed includes
 *	- The pid.
 *	- The current state (ready/blocked/etc.)
 *	- If blocked sending or receiving, the process blocked on.
 *	- The parent's pid.
 *	- The team root's pid.
 */

info(proc)
Process *proc;
  {
    register Process *p;
    
    if ( LocalProcess(proc) || (proc == P_IdleProcess))
      {
        printx("%x prior %x parent %x team %xx): ",proc->pid, proc->priority,
			proc->father->pid, proc->team->team_root);
      }
    else if ( AlienProcess(proc) )
      {
	printx("%x  (ALIEN): ", proc->pid);
      }
    else
      {
	return;			/* Invalid process descr. */
      }

    switch (proc->state)
      {
	case READY:
	    printx("Ready");
	    break;

	case RECEIVE_BLKED:
	    printx("Receive blocked");
	    if (proc->blocked_on != 0)
	        printx(" %x",proc->blocked_on);
	    break;

	case AWAITING_REPLY:
	    printx("Awaiting reply %x",proc->blocked_on);
	    break;

	case SEND_BLKED:
	    printx("Send blocked !");
	    break;

	case AWAITING_INT:
	    printx("Awaiting interrupt");
	    break;

	case DELAYING:
	    printx("Delaying, %d clicks",proc->timeout_count);
	    break;

	case GETPID_BLKED:
	    printx("GetPid blocked !");
	    break;

	case MOVEFROM_BLKED:
	    printx("Remote MoveFrom, from %x",proc->blocked_on);
	    break;

	case MOVETO_BLKED:
	    printx("Remote MoveTo, to %x",proc->blocked_on);
	    break;

	case REPLIED_TO:
	    printx("Alien, replied to by %x", proc->blocked_on);
	    break;

	case FORWARDED_TO:
	    printx("Alien, forwarded to %x", proc->blocked_on);
	    break;

	case LHN_COLLISION_BLKED:
	    printx("Blocked on LHN collision");
	    break;

	case XMIT_BLKED:
	    printx("Blocked waiting for network transmission");
	    break;

	case ACTIVE:
	    printx("<<< Active Process >>>");

	default:
	    printx("Unknown state %d", proc->state);
	    break;

      }

    printx("\n");
    if (AlienProcess(proc)) return;
    
    if ((p = proc->msgq.head) != NULL)
      {
	printx("Msgq: ");
	while (p != NULL)
	  {
	    if ( p == proc->next_sender ) printx("* ");
	    printx("%x ", p->pid);
	    p = p->link;
	  }
	printx("\n");
      }
    if ((p = proc->replyq.head) != NULL)
      {
	printx("Replyq: ");
	while (p != NULL)
	  {
	    printx("%x ", p->pid);
	    p = p->link;
	  }
	printx("\n");
      }

  }


/*
 * PrintExcept:
 *
 * Interpret an exception message
 */
PrintExcept(req)
ExceptionRequest *req;
  {
    switch (req->type)
      {
      case BUSERROR:
	printx("Bus error ");
	switch (req->buserrortype)
	  {
	  case OUT_OF_RANGE:
	    printx("(out of range)");
	    break;

	  case SPURIOUS:
	    printx("(spurious)");
	    break;

	  case SYSTEM_SPACE:
	    printx("(system space)");
	    break;

	  case PROTECTION:
	    printx("(protection)");
	    break;

	  case PAGE_INVALID:
	    printx("(page invalid)");
	    break;

	  case SEG_INVALID:
	    printx("(segment invalid)");
	    break;

	  case MB_TIMEOUT:
	    printx("(multibus timeout)");
	    break;

	  case PARITY:
	    printx("(parity)");
	    break;
	  }
	printx(", code %x, access addr %x, instr %x",
		req->code, req->accaddr, req->instruction);
	break;

      case ADDRERROR:
	printx("Address error");
	printx(", code %x, access addr %x, instr %x",
		req->code, req->accaddr, req->instruction);
	break;

      case ILLINST:  case EMU1010:  case EMU1111:
	printx("Illegal instruction");
	break;

      case ZERODIV:
        printx("Division by zero");
	break;

      case CHKINST:
        printx("CHK instruction");
	break;

      case TRAPVINST:
        printx("Overflow trap");
	break;

      case PRIVVIOL:
        printx("Privileged instruction");
	break;

      case TRACETRAP:
        printx("Trace trap");
	break;

      case FORMATERR:
	printx("Format error");
	break;

      case UNINITINT:
	printx("Uninitialized interrupt vector");
	break;

      case SPURINT:
        printx("Spurious interrupt");
	break;

      case INT1:  case INT2:  case INT3:  case INT4:
      case INT5:  case INT6:  case INT7:
	printx("Interrupt level %d", (req->type - SPURINT)>>2);
        break;

      case TRAP0:  case TRAP1:  case TRAP2:  case TRAP3:
      case TRAP4:  case TRAP5:  case TRAP6:  case TRAP7:
      case TRAP8:  case TRAP9:  case TRAP10: case TRAP11:
      case TRAP12: case TRAP13: case TRAP14: case TRAP15:
	printx("Trap instruction %d", (req->type - TRAP0)>>2);
	break;
      }
    printx(", statreg %x, pc %x\n", req->status, req->errpc);
  }

#endif
