/*
 * V Kernel - Copyright (c) 1982 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 *
 * Basic Process Implementation
 *
 * Provides routines which implement chained hashing for process descriptors
 * and provides operations to add and remove processes from
 * the ready queue:  Addready(pd) and RemoveQueue( pd ).
 * RemoveQueue is a general-purpose routine that removes a process
 * from any queue it happens to be in, as indicated by the pd->queuePtr field.
 *
 * Kernel synchronization is achieved by only switching processes
 * upon exit from the kernel.  Exit from a kernel trap can
 * always cause a process switch; exit from an interrupt service
 * routine can only cause a switch if the
 * preempted process was not running in kernel mode.
 * Interrupt routines are assumed to only access local data
 * structures plus add processes to the ready queues.
 * The latter is synchronized by disabling interrupts.
 * There is assumed to be an idle process that is always ready so
 * that the ready queue is never empty.
 */

#include "process.h"

Process *Pd_bundle[ PID_HASH_MASK + 1 ]; 
				/* Hash table for mapping pids to PD's. */
Process *Active;		/* This will have to go in multi-p land */
Process *ProcessDescriptors;	/* Pointer to array of process descriptors */
Process *EndPds;		/* Ptr to the end of the process descriptors */
unsigned long MaxProcesses;

/* Free PD list needs to be synchronized for multiple processor because
 * PDs are allocated and freed as aliens in the IKC code.
 */
SyncQueue FreePdq;		/* Queue of free PDs */
Process *PdIndex;		/* Ptr to next pd to check on timer int. */

/* Allocation of team descriptors is not synchronized because only handled
 * through the kernel server - is that sufficient? DRC. */

Team *TeamDescriptors;
unsigned long MaxTeams;
Team *TdFreeList;		/* Td free list header. */

extern Process Idle_process;
extern SyncQueue Delayq;

/* Ready queue head and active process pointer */
SyncQueue Readyq;
unsigned FreePds;		/* Count of number of free PD's */

InitPDsandTDs()
  {
    register Process *pd;
    register Team *td;
    register unsigned i;

    /* Initialize team descriptors */
    td = TeamDescriptors;
    TdFreeList = NULL;
    for( i = 0; i < MaxTeams; ++i )
      {
	td->next = TdFreeList;
	TdFreeList = td;
	td->team_root = 0;
	Ms_td_init( td, (MaxTeams - 1) - i );
				/* We do it this way because the first team
				   expects to run in context 0. */
	++td;
      }

    /* Initialize process descriptors. */
    IdleProcessPtr = &Idle_process;
    pd = ProcessDescriptors;
    PdIndex = pd;
    FreePdq.head = NULL;
    FreePdq.tail = pd;

    for ( i = 0; i < (PID_HASH_MASK+1); i++)
	Pd_bundle[i] = AlienMap[i] = &Idle_process;
    for ( i = 0; i < MaxProcesses; ++i )
      {
	pd->link = NULL;
	pd->pid = 0;
	pd->localPid = INVALID_PROCESS;
	pd->pdFlags = 0;
	pd->extendPd = NULL;
	pd->nextPd = FreePdq.head;
	pd->msgq.type  = MSG_QUEUE;
	pd->msgq.head = NULL;
	pd->msgq.tail = (Process *) &(pd->msgq.head);
	pd->replyq.type = REPLY_QUEUE;
	pd->replyq.head = NULL;
	pd->replyq.tail = (Process *) &(pd->replyq.head);
	pd->next_sender = NULL;
	FreePdq.head = pd;
	Ms_pd_init( pd, i );
	++pd;
      }
    FreePds = MaxProcesses;
  }
/*
 * Chained hashing routines for process descriptors
 */

Process *MapPid(pid)
    ProcessId pid;
   /* Map pid to the corresponding (local) process descriptor
    * if any, and return a pointer to the PD, else NULL.
    */
  {
    register Process *pd;

    pd = Pd_bundle[ pid & PID_HASH_MASK ];
    while (pd != &Idle_process)
      {
        if (pd->localPid == pid)
	  {
	    return(pd);
	  }
	pd = pd->nextPd;
      }
    return(NULL);
  }

Process *RPdChain(pd, pid)
    register Process *pd;
    ProcessId pid;
  /*
   * Returns a ptr to the process descriptor corresponding to the real 
   * process' process-id pid.
   * Returns NULL if none exists.
   */
  {
    while (pd != IdleProcessPtr)
      {
        if (pd->localPid == pid)
	  {
	    return(pd);
	  }
	pd = pd->nextPd;
      }
    return(NULL);
  }


Process *APdChain(pd, pid)
    register Process *pd;
    ProcessId pid;
  /*
   * Returns a ptr to the process descriptor corresponding to the alien 
   * process' process-id pid.
   * Returns NULL if none exists.
   */
  {
    while (pd != &Idle_process)
      {
        if (pd->pid == pid) return(pd);
	pd = pd->nextPd;
      }
    return(NULL);
  }

Process *AllocatePd()
   /* Allocate a process descriptor and return a pointer to same
    * if one is available, else return NULL.
    * This needs to be synchronized.
    */
  {
    register Process *pd;

    Lockq( &FreePdq );
    if( (pd = FreePdq.head) == NULL )
      {
#ifdef LANCE
PrintQueues();
#endif
	Unlockq( &FreePdq );
	return( NULL );
      }
    FreePdq.head = pd->nextPd;
    pd->nextPd = NULL;
    --FreePds;
    Unlockq( &FreePdq );

    return( pd );
  }

FreePd(pd, pid)
    Process *pd;
    ProcessId pid;
  {
    /*
     * Free up the process descriptor by removing it from its hashing chain
     * (if necessary) and setting its pid field to 0.
     * Note: pid value is passed so that it can be invalidated earlier in
     * the process descriptor if necessary - as in DestroyProcess.
     */
    extern unsigned FreePds;
    register Process *pd1;
    register int index = pid & PID_HASH_MASK;

    Lockq( &FreePdq );
    if ((pd1 = Pd_bundle[index]) == pd)
      {
	Pd_bundle[index] = pd->nextPd;
      }
    else
      {
	while (pd1->nextPd != pd)
	  {
	    if( pd1 == IdleProcessPtr ) 
	      {
	        goto freePD;
	      }
	    pd1 = pd1->nextPd;
	  }
	pd1->nextPd = pd->nextPd;
      }
freePD:
    ++FreePds;
    pd->nextPd = FreePdq.head;
    FreePdq.head = pd;
    pd->pid = 0;
    pd->localPid = INVALID_PROCESS;
    Unlockq( &FreePdq );
  }


ExtendPd FindExtendPd(pd, eType)
    Process *pd;
    short eType;
  /*
   * Locate an extend PD with type eType.
   */
  {
    ExtendPd sp;

    for (sp = pd->extendPd; sp != NULL; sp = sp->extendPd)
      {
	if (sp->eType == eType)
	  {
	    break;
	  }
      }
    return(sp);
  }


LinkExtendPd(pd, sp)
    Process *pd;
    ExtendPd sp;
  /*
   * Link an extend PD into pd's save area chain.
   */
  {
    register ExtendPd p;

    p = pd->extendPd;
    pd->extendPd = sp;
    sp->extendPd = p;
  }


int DeleteExtendPd(pd, eType)
    Process *pd;
    short eType;
  /*
   * Returns the specified extend PD to the free list.  Returns 1 if
   * successful, 0 otherwise.
   */
  {
    register ExtendPd sp, sp1;
    register Process *pd1;

    sp1 = pd->extendPd;
    if (sp1 == NULL)
      {
	return(0);
      }
    if (sp1->eType == eType)
      {
	pd->extendPd = sp1->extendPd;
	goto addFreeList;
      }
    for (sp = pd->extendPd, sp1 = sp->extendPd;
         sp1 != NULL; 
         sp = sp->extendPd, sp1 = sp1->extendPd)
      {
	if (sp1->eType == eType)
	  {
	    sp->extendPd = sp1->extendPd;
	    goto addFreeList;
	  }
      }
    return(0);

addFreeList:
    pd1 = (Process *) sp1;
    ++FreePds;
    pd1->pid = 0;
    pd1->localPid = INVALID_PROCESS;
    Lockq( &FreePdq );
    pd1->nextPd = FreePdq.head;
    FreePdq.head = pd1;
    Unlockq( &FreePdq );
    return(1);
  }

/*
 * Ready queue manipulation.
 */

Addready( pd ) 
    register  Process *pd;

  /* Add the specified process to the ready queue.
   * Note: this must restore interrupts to the same level as
   * the level when it was called.
   * Note: aliens are setup to send back a nAck and self-destruct
   * if they are added to the ready queue.
   */
  {
    register Process *curr_pd, *tmp_pd;
    register unsigned short prio;
    register Unspec (*finish_up)();
    extern Process Idle_process;

    /* Check if already in ready queue - for kernel server. */
    if( pd->queuePtr == &Readyq ) return;
/*%%%*/if (pd->queuePtr != NULL)
	{
	  printx("Addready: process %x already on queue %x\n", 
		pd->pid, pd->queuePtr);
	  StackDump();
#ifndef VAX
	  ;asm("trap #14");;
#else
	  ;asm("mtpr $0xf05, $0x23");;
#endif
        }
/*%%%*/if (pd->localPid == INVALID_PROCESS && pd != &Idle_process)
      {
	printx("Addready: readying an invalid process (%x,%x)\n", pd,pd->pid);
	StackDump();
#ifndef VAX
	  ;asm("trap #14");;
#else
	  ;asm("mtpr $0xf05, $0x23");;
#endif
      }
      
    if (AlienProcess(pd))
      {
	finish_up = pd->finish_up;
        pd->finish_up = NULL;
        (*finish_up)(pd);
	return;
      }
      
    if ( pd->decayRate )
      {
/*%%%*/ printx("Decaying %x.\n", pd->pid);
/*%%%*/ if (pd->localPid == ALIEN_PROCESS)
	    Kabort("Decaying alien");
	while ( pd->processorTime - pd->decayTime >= pd->decayRate )
	  {
	    if ((pd->priority & TEAM_PRIORITY_MASK)!=STOPPED_TEAM_PRIORITY)
	      {
      		pd->priority++;
		if ((pd->priority & (PROCESS_PRIORITY_MASK-1)) ==
						(PROCESS_PRIORITY_MASK - 1) )
		  {
		    pd->priority = 
		    		STOPPED_TEAM_PRIORITY | PROCESS_PRIORITY_MASK;
		    pd->decayTime = pd->processorTime;
		    break;
		  }
	      }
	    pd->decayTime += pd->decayRate;
	    
	  }
      }
    prio = pd->priority;

    Lockq( &Readyq );
    pd->state = READY;
    pd->queuePtr = &Readyq;
    curr_pd = (Process *) &(Readyq.head);
    tmp_pd = Readyq.head;

    while( (tmp_pd != NULL) && (tmp_pd->priority <= prio) ) 
      {
/*%%%*/if (tmp_pd == pd)
	 { 
	    printx("Addready: process %x already in ready queue\n", pd);
	    StackDump();
	    Unlockq( &Readyq );
	    return;
	 }

	curr_pd = tmp_pd;
	tmp_pd = tmp_pd->link;
      }

    pd->link = tmp_pd;
    curr_pd->link = pd;
    Unlockq( &Readyq );
  }

SyncQueue *RemoveQueue( pd )
    register Process *pd;
  /* Remove the specified process from any synchronized queues it
   * might be linked into.  Returns pointer to queue, if any on
   * success.
   * Note: removing from the Delayq is handled as a special case.
   * Also, it is assumed there is no harm in setting the tail pointer
   * even for queues that dont normally maintain a tail pointer.
   */
  {
    register SyncQueue *q;
    register unsigned count = 0;
    register Process *tmp_pd;

    if( (q=pd->queuePtr) == NULL ) 
        return( NULL );

    Lockq( q );
    pd->queuePtr = NULL;
    tmp_pd = (Process *) &(q->head);
    while( tmp_pd->link != pd )
      {
	if( (tmp_pd = tmp_pd->link) == NULL )
	  {
	    Unlockq( q );
/*%%%*/     printx("Removeq: process %x not found in queue %x\n", pd, q);
/*%%%*/     StackDump();
#ifndef VAX
	  ;asm("trap #14");;
#else
	  ;asm("mtpr $0xf05, $0x23");;
#endif
	    return( NULL ); /* Not found in the queue. */
	  }
	count += tmp_pd->timeout_count;  /* In case this is the Delayq */
      }
    if( (tmp_pd->link = pd->link) == NULL )
      {
	q->tail = tmp_pd;
      }
    else
	if( q == &Delayq ) /* Adjust timeout_count for next PD */
	     pd->link->timeout_count += pd->timeout_count;

    if( q->type == TIMEOUT_QUEUE ) pd->timeout_count += count;
    else if( q->type == MSG_QUEUE )
      {
	/* Calculate the address of the pd containing the queue. */
	tmp_pd = (Process *) ( (char *) q -
		((char *)&Idle_process.msgq - (char *) &Idle_process));
	if( tmp_pd->next_sender == pd )
	  {
	    tmp_pd->next_sender = pd->link;
	  }
      }
    Unlockq( q );
    return( q );
  }
