/*
 * Distributed V Kernel - Copyright (c) 1982 by Stanford University.
 *
 * Alien handling routines
 *
 */

#include "Venviron.h"
#include "Vikc.h"
#include "process.h"

extern Process *AllocatePd();
extern SyncQueue FreePdq;
extern unsigned FreePds;
extern Process Idle_process;

unsigned DestroyAlien();
unsigned FreeAlien(), Dealienate(), SendnAck();

Process *AlienMap[PID_HASH_MASK+1];

InitializeAlien( alien )
register Process *alien;
    /* Initialize the specified process descriptor as an alien.
     * assumes that alien->pid is already set.
     */
  {
    register unsigned index;
    register Process *p;

    /* Attach alien to the appropriate collision list and initialize. */
    /* Aliens are added to the front so newest aliens occur first. */
    index = alien->pid & PID_HASH_MASK;
    disable;
    alien->nextPd = AlienMap[index];
    AlienMap[index] = alien;
    enable;
    alien->link = NULL;
    alien->queuePtr = NULL;
    InitializeAlien_1(alien);
  }

InitializeAlien_1(alien)
    register Process *alien;
  {
  /* Initialize alien that is already in the Pd_bundle and may be on a queue.
   */
    alien->team = NULL;
    alien->localPid = ALIEN_PROCESS;
    alien->pdFlags = 0;
    alien->state = 0;  /* Must not be READY */
    alien->priority = DEFAULT_ALIEN_PRIORITY;
    alien->finish_up = (Unspec (*)()) SendnAck;
    alien->timeout_func = (Unspec (*)()) FreeAlien; 
    alien->timeout_count = 0;
    alien->next_sender = NULL;
    alien->msgq.head = NULL;
    alien->msgq.tail = (Process *) &(alien->msgq.head);
    alien->replyq.head = NULL;
    alien->replyq.tail = (Process *) &(alien->replyq.head);
    alien->decayRate = 0;
    alien->segmentSize = 0;
  }

Process *Alienate( kp )
register kPacket *kp;
    /* Take a kernel packet and return an alien process descriptor
     * initialized with the contents of the kpacket header.
     * The network driver may signal that this is already contained in
     * in an alien by turning on the high-order bit of the packetType
     * field, in which case this function simply calculates the address
     * of the containing alien, turns off the bit - to indicate it is
     * going to be used by the IKC level and returns a pointer to same.
     */
  {
    register short i;
    register unsigned *ptr1, *ptr2;
    register Process *alien;

    if( (alien = AllocatePd()) == NULL )
      {
#ifdef too_obnoxious_to_print_all_the_time
        printx("remote op %x failed to get alien PD\n", kp->packetType );
#endif
        return( NULL ); /* No PD's */
      }
    /* Copy into alien descriptor. */
    ptr1 = (unsigned *) &(alien->packetType);
    ptr2 = (unsigned *) kp;

    i = WORDS_PER_PACKET - 1;
    do { *ptr1++ = *ptr2++; } while( --i != -1 );

    InitializeAlien( alien );

    return( alien );
  }

unsigned Dealienate( alien, pid )
    register Process *alien; ProcessId pid;
  /* Remove alien from AlienMap, for converting an alien to a reply buffer.
   * Called by NonLocalReply.
   * Also called by FreeAlien and DestroyAlien.
   */
  {
    register unsigned index;
    register Process *p;

    index = pid & PID_HASH_MASK;
    disable;
    if( (p = AlienMap[index]) == alien ) AlienMap[index] = alien->nextPd;
    else if( p == &Idle_process ) goto dealienateError;
    else
      {
	while( p->nextPd != alien )
	    if( (p = p->nextPd) == &Idle_process ) goto dealienateError;
        p->nextPd = alien->nextPd;
     }
    alien->nextPd = NULL;
    enable;
    return;

dealienateError:
    printx("Alien %x (blocked on %x) not in AlienMap\n", 
	alien->pid, alien->blocked_on );
    Kabort("Dealienating a nonalien");
  }

unsigned DestroyAlien( alien ) 
    Process *alien;
    /* This only succeeds if alien is in some queue and can be removed. */
    /* Returns 0 if fails, else non-zero. */
  {
    ProcessId pid;

/*%%%*/ if( !AlienProcess(alien) )
      {
        printx("DestroyAlien: process %x not an alien\n", alien );
        StackDump();
#ifndef VAX
          ;asm("trap #14");;
#else
          ;asm("mtpr $0xf05, $0x23");;
#endif
      }

    if( RemoveQueue(alien) == NULL )
{
Kabort("Alien not on queue");
return( 0 );
}
    /* Invalidate pid */
    pid = alien->pid;
    alien->pid = INVALID_PROCESS;

    Dealienate( alien, pid );
    SimpleFreePd( alien );
  }


unsigned FreeAlien( alien ) 
    Process *alien;
    /* Free an alien descriptor, assuming it is not in a queue, but "held"
     * by the invoker of this procedure.
     */
  {
    ProcessId pid;

/*%%%*/ if( !AlienProcess(alien) )
      {
        printx("FreeAlien: process %x not an alien\n", alien );
        StackDump();
#ifndef VAX
          ;asm("trap #14");;
#else
          ;asm("mtpr $0xf05, $0x23");;
#endif
      }
/*%%%*/ if (alien->queuePtr != NULL) printx("Queued alien being freed!\n");
    pid = alien->pid;
    alien->pid = INVALID_PROCESS;
    Dealienate( alien, pid );
    SimpleFreePd( alien );
  }

unsigned SendnAck( alien )
register Process *alien;
  /* Cause the specified alien to send an nAck back to
   * the real process.  This is used as a finishup function
   * for the case when the local process the alien is blocked on
   * is destroyed.
   */
  {
    alien->timeout_count = 0; /* Ensure immediate timeout. */
    alien->state = 0; /* Not ready, of course. */
    /* timeout func is set by InitializeAlien */
/*%%%*/
    if (alien->timeout_func != (Unspec (*)()) FreeAlien)
	printx("SendnAck: %x (blocked on %x fails sanity check.\n",
		alien->pid, alien->blocked_on);
/*  alien->timeout_func = (Unspec (*)()) FreeAlien; */
    alien->packetType = nAck;
    alien->segmentSize = 0;	/* In case it was set earlier. */
    WriteKPacket( alien );
  }

ScavengeAliens( pid )
    register ProcessId pid;
    /* Attempt to destroy any aliens representing this pid except replies.
     * This fails to destroy any aliens currently not queued.
     */
  {
    register Process *alien, *savealien;

    alien = AlienMap[ pid & PID_HASH_MASK ];
    while ( alien != &Idle_process )
        if(  alien->pid != pid ) alien = alien->nextPd;
        else
          {
            savealien = alien->nextPd;
            DestroyAlien( alien );
            alien = savealien;
          }
  }

/* This routine might be better in bpi.c? */
SimpleFreePd( pd )
    Process *pd;
    /* Free a process descriptor used as a reply buffer or control packet
     * buffer and thus not linked into the AlienMap or other structure.
     */
  {
    Lockq( &FreePdq );
    pd->pid = 0;
    pd->localPid = INVALID_PROCESS;
    pd->nextPd = FreePdq.head;
    ++FreePds;
    FreePdq.head = pd;
    Unlockq( &FreePdq );
  }

FreeReplyBuffers( pd )
register Process *pd;
    /* Return the pd's reply buffers to the free pd list.
     * Called from KSend and DestroyProcess.
     */
  {
    register unsigned count;
    register Process *pdtail, *pdhead, *p;

    /* Remove from pd reply queue. */
    Lockq( &(pd->replyq.head) );
    pdtail = pd->replyq.tail;
    pdhead = pd->replyq.head;
    pd->replyq.head = NULL;
    pd->replyq.tail = (Process *) &(pd->replyq.head);
    Unlockq( &(pd->replyq) );
    /* Count and clean up PD's */
    count = 0;
    p = pdhead;
/*%%%*/ if (p == NULL) Kabort("FreeReplyBuffers called with no reply buffers");

    do
      {
	count++;
	p->nextPd = p->link;	/* FreePdq list is linked via nextPd */
	p->link = NULL;
	p->queuePtr = NULL;
	p->pid = INVALID_PROCESS;
	p->localPid = INVALID_PROCESS;
	p = p->nextPd;
      }
    while( p != NULL );

    Lockq( &FreePdq );
    FreePds += count;
    pdtail->nextPd = FreePdq.head;
    FreePdq.head = pdhead;
    Unlockq( &FreePdq );
  }
