/*
 * V Kernel - Copyright (c) 1984 by Stanford University.
 *
 * Zilog 8530 Serial Communication Controller driver
 * Tim Mann, 7-24-84
 */

#include "z8530scc.h"
#include "memory.h"
#include "dm.h"
#include "interrupt.h"
#include "Vexceptions.h"
#include "Vquerykernel.h"

/* Imports */
extern SystemCode NotSupported();
extern short FramebufferType;

/* Exports */
extern void SccPowerup();
extern SystemCode SccCreate();
extern SystemCode SccRead();
extern SystemCode SccWrite();
extern SystemCode SccQuery();
extern SystemCode SccModify();
extern SystemCode SccRelease();
extern SystemCode SccBellKludge();

/* Private */
static void SccInterruptHandler();
static void SccInitChannel();
static void SccSetDefaults();
static Unspec SccBellKludge2();

/* Chip needs 1.6 usec settling time between commands */
#ifdef SUN3
SccSnooze() { int i = 5; while(i) i--; }
#else
#define SccSnooze() { int i, j; i = j; j = i; }
#endif SUN3


unsigned char ReadRegister0(scc) 
  SccChannel *scc;
  {
    unsigned char value;

    DISABLE;
    SccSnooze();
    value = (scc)->control;
    ENABLE;
    return value;
  }

WriteRegister0(value, scc)
  unsigned char value;
  SccChannel *scc;
  {
    DISABLE;
    SccSnooze();
    (scc)->control = (value);
    ENABLE;
  }

unsigned char ReadRegister(reg, scc)
  int reg;
  SccChannel *scc;
  {
    register unsigned char value;

    DISABLE;
    SccSnooze(); 
    (scc)->control = (reg);
    SccSnooze();
    value = (scc)->control;
    ENABLE;
    return value;
  }

unsigned char ReadDataRegister(scc)
  SccChannel *scc;
  {
    register unsigned char value;

    DISABLE;
    SccSnooze();
    value = (scc)->data;
    ENABLE;
    return value;
  }

WriteRegister(reg, value, scc)
  int reg;
  unsigned char value;
  SccChannel *scc;
  {
    DISABLE;
    SccSnooze();
    (scc)->control = (reg);
    SccSnooze();
    (scc)->control = (value);
    ENABLE;
  }

WriteDataRegister(value, scc)
  unsigned char value;
  SccChannel *scc;
  {
    DISABLE;
    SccSnooze();
    (scc)->data = (value);
    ENABLE;
  }

/* List of channel status structures */
SccChannelStatus KeyboardSccStatus =
  {
    /* link 	*/  NULL,
    /* channel 	*/  &( ((SccChip *) V_KBD_MOUSE_UART)->a ),
    /* ab 	*/  SCC_CHANNEL_A,
  };

SccChannelStatus MouseSccStatus =
  {
#ifdef KEYBOARD_DRIVER	/* we are still letting the PROM handle the keyboard */
    /* link 	*/  &KeyboardSccStatus,
#else
    /* link 	*/  NULL,
#endif
    /* channel 	*/  &( ((SccChip *) V_KBD_MOUSE_UART)->b ),
    /* ab 	*/  SCC_CHANNEL_B,
  };

SccChannelStatus SerialSccStatus[2] =
  {
     {
	/* link */	&MouseSccStatus,
	/* channel */	&( ((SccChip *) V_SERIAL_PORT)->a ),
	/* ab */	SCC_CHANNEL_A
      },
      {
	/* link */	&SerialSccStatus[0],
	/* channel */	&( ((SccChip *) V_SERIAL_PORT)->b ),
	/* ab */	SCC_CHANNEL_B,
      }
  };

SccChannelStatus *SccChannelList = &SerialSccStatus[1];

/*
 * Powerup and initialize the Sun-2 SCC's.
 */
void SccPowerup()
  {
    register SccChannelStatus *s;
    register int (**vector)() = (int(**)()) (INT6 + VECTOR_BASE);
    extern int Asm_SccInterruptHandler();

    /* Plug interrupt vector */
    *vector = Asm_SccInterruptHandler;

    /* Reset SCCs */
#ifdef KEYBOARD_DRIVER	/* we are still letting the PROM handle the keyboard */
    if ((FramebufferType == PRF_FRAMEBUFFER_SMI120) ||
        (FramebufferType == PRF_FRAMEBUFFER_SUN3))
      {
	/* The keyboard/mouse UART is on the Model 120 framebuffer board */
	WriteRegister0(SccNoOperation, KeyboardSccStatus.channel);
	WriteRegister(9, SccForceHardwareReset, KeyboardSccStatus.channel);
      }
#endif
    WriteRegister0(SccNoOperation, SerialSccStatus[0].channel);
    WriteRegister(9, SccForceHardwareReset, SerialSccStatus[0].channel);
    
    /* Initialize status structures */
    SccSetDefaults(&KeyboardSccStatus);
    SccSetDefaults(&MouseSccStatus);
    MouseSccStatus.dataRate = 1200;
    SccSetDefaults(&SerialSccStatus[0]);
    SccSetDefaults(&SerialSccStatus[1]);

#ifdef SUN3
    /* The Sun-3 uses vectored interrupts instead of autovectored. We
     * set the scc vector number to point at the level-6 autovector
     * location for easier compatibility.
     */
    WriteRegister(2, (INT6/4), SerialSccStatus[0].channel);
#endif

    /* Unlink keyboard/mouse status structures if device not present */
    if ( (FramebufferType != PRF_FRAMEBUFFER_SMI120) &&
	 (FramebufferType != PRF_FRAMEBUFFER_SUN3) )
	SerialSccStatus[0].link = NULL;
#ifdef SUN3
    else
	WriteRegister(2, (INT6/4), MouseSccStatus.channel);
#endif

    /* Initialize channels from status structures */
    for (s = SccChannelList; s != NULL; s = s->link)
      {
	SccInitChannel(s);
      }

    /* Enable chip interrupts */
    if ((FramebufferType == PRF_FRAMEBUFFER_SMI120) ||
        (FramebufferType == PRF_FRAMEBUFFER_SUN3))
      {
#ifndef KEYBOARD_DRIVER		/* disable keyboard interrupts */
        WriteRegister(15, 0, KeyboardSccStatus.channel);
        WriteRegister(1, 0, KeyboardSccStatus.channel);
#endif
#ifdef SUN3
        WriteRegister(9,  SccMasterIntEnable, KeyboardSccStatus.channel);
#else
        WriteRegister(9,  SccMasterIntEnable|
		          SccNoVector, KeyboardSccStatus.channel);
#endif
      }
#ifdef SUN3
    WriteRegister(9,  SccMasterIntEnable, SerialSccStatus[0].channel);
#else
    WriteRegister(9,  SccMasterIntEnable|
		      SccNoVector, SerialSccStatus[0].channel);
#endif
  }


/* Initialize a status structure with default values */
static void SccSetDefaults(s)
    SccChannelStatus *s;
  {
    s->dataRate = 9600;
    s->charWidth = 8;
    s->parity = PARITY_NONE;
    s->stopBits = STOP_BITS_1;
    s->dtr = 1;
    s->rts = 0;
    s->brk = 0;
  }


/* Initialize a channel from its status structure */
static void SccInitChannel(s)
    SccChannelStatus *s;
  {
    register int timeconst;

    /* Set clock divisor, parity, and stop bits.  The chip seems to
     *  work much better in X16 mode than in X1 mode.  Fortunately,
     *  the baud rate generator has no trouble generating high enough
     *  frequencies. */
    WriteRegister(4,  SccX16Clock|
		      SccStopBits(s->stopBits)|
		      SccParity(s->parity), s->channel);
		  
    /* Set data rate */
    WriteRegister(14, 0, s->channel);  /* temp disable baud rate gen */
    timeconst = (int) ((SCC_CLOCK_RATE*10/2/16) / s->dataRate - 20 + 5)/10;
    WriteRegister(12, timeconst, s->channel);
    WriteRegister(13, timeconst>>8, s->channel);
    WriteRegister(14, SccBrGenEnable|
    		      SccBrGenSourcePclk, s->channel);	/*?*/
		  
    /* Establish baud rate generator as clock source */
    WriteRegister(11, SccRxClockBrGen|
		      SccTxClockBrGen|
		      SccTRxCOutput|
		      SccTRxCOutputTxClock, s->channel);
		  
    /* Disable all external/status interrupts */
    WriteRegister(15, 0, s->channel);

    /* Establish Tx parameters and enable transmitter */
    WriteRegister(5,  (s->dtr ? SccDtr : 0)|
		      (s->rts ? SccRts : 0)|
		      SccTxBitsPerChar(s->charWidth)|
		      (s->brk ? SccSendBreak : 0)|
		      SccTxEnable, s->channel);
		  
    /* Establish Rx parameters and enable receiver */
    WriteRegister(3,  SccRxBitsPerChar(s->charWidth)|
		      SccRxEnable, s->channel);
			  
    /* Enable Rx and Tx interrupts */
    WriteRegister(1,  SccRxIntAllChars|
		      SccTxIntEnable, s->channel);

  }



SystemCode SccCreate( pd, inst )
    Process *pd;
    DeviceInstance *inst;
  /*
   * Create instances for a serial line.  The instance returned
   *  is used for output to the line.  The instance with the next
   *  larger id is used for input; its parameters may be obtained
   *  by doing a QueryInstance.
   */
  {
    CreateInstanceRequest *req = (CreateInstanceRequest *) &pd->msg;
    register SccChannelStatus *s;

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    /* Find SccStatus structure for the requested unit number */
    s = &(SerialSccStatus[inst->unit]);

    /* These are single-user devices; check for conflict */
    if( s->inputInst != NULL )
      {
	if( MapPid( s->inputInst->owner ) )  return (BUSY);
	s->inputInst->owner = 0;	/* old owner is dead */
	s->inputInst = NULL;
      }

    if( s->outputInst != NULL )
      {
	if( MapPid( s->outputInst->owner ) )  return (BUSY);
	s->outputInst->owner = 0;	/* old owner is dead */
	s->outputInst = NULL;
      }

    /*
     * Initialize the device instance descriptors, can assume all fields
     * are zero except for id and first instance's owner and unit number.
     */

    /* Output descriptor */
    inst->readfunc = NotSupported;
    inst->writefunc = SccWrite;
    inst->modifyfunc = SccModify;
    inst->queryfunc = SccQuery;
    inst->releasefunc = SccRelease;

    inst->type = WRITEABLE+STREAM+VARIABLE_BLOCK;
    inst->blocksize = IO_MSG_BUFFER;
    inst->lastblock = 0;
    inst->lastbytes = inst->blocksize;
    s->outputInst = inst;

    /* Input descriptor */
    inst++;
    inst->owner = pd->pid;
    inst->unit = (inst-1)->unit;
    inst->readfunc = SccRead;
    inst->writefunc = NotSupported;
    inst->modifyfunc = SccModify;
    inst->queryfunc = SccQuery;
    inst->releasefunc = SccRelease;

    inst->type = READABLE+STREAM+VARIABLE_BLOCK;
    inst->blocksize = IO_MSG_BUFFER;
    inst->lastblock = MAXBLOCKS;
    inst->lastbytes = MAXBYTES;
    s->inputInst = inst;
		 /* Record the instance for interrupt routine */

    CbuffFlush(&(s->cbuff));

    /* Initialize the device */
    SccSetDefaults(s);
    SccInitChannel(s);

    return( OK );
  }


SystemCode SccRead( pd, inst )
    Process *pd;
    DeviceInstance	*inst;
   /*
    * Handle a read instance request for a serial line
    */
  {
    IoRequest *req = (IoRequest *) &(pd->msg);
    register unsigned bytecount;
    register int c, i;
    register IoReply *reply = (IoReply *) &(pd->msg);
    register char *sbuff = &reply->shortbuffer[0];
    Process *reader;
    CircularBuffer *cbuff;

    bytecount = req->bytecount;
    reply->bytecount = 0;
    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading. 
     */
    if( inst->reader ) 
	if( MAP_TO_RPD(reader, inst->reader) ) 
	    return( RETRY );

    if( bytecount > inst->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
    if( req->blocknumber != inst->nextblock ) return( BAD_BLOCK_NO );

    disable;
    /* Try to get characters from the buffer */
    cbuff = &(SerialSccStatus[inst->unit].cbuff);
    for (i = 0; i < bytecount; i++)
      {
	if ( (c = CbuffGet(cbuff)) == CBUFF_EMPTY ) break;
	*sbuff++ = c;
      }

    if (i != 0) 
      {
	reply->replycode = OK;
	reply->bytecount = i;
	inst->nextblock++;
	return (OK);  /* One or more characters returned */
      }

    /* Wait for a character to come in */
    inst->reader = pd->pid;
    pd->state = AWAITING_INT;

    return( NO_REPLY );
  }


SystemCode SccWrite( pd, inst )
    Process *pd;
    DeviceInstance	*inst;
   /*
    * Handle a write instance request to a serial line
    */
  {
    register IoRequest	*req = (IoRequest *) &(pd->msg);
    IoReply *reply = (IoReply *) req;
    register SccChannelStatus *s;
    register SccChannel *channel;

    if( req->bytecount > inst->blocksize ) 
      {
	req->bytecount = 0;
	return( BAD_BYTE_COUNT );
      }

    if( req->bytecount == 0 ) return ( OK );

/*  Disabled to simplify multi-process access to the console.
 *  if( req->blocknumber != inst->lastblock+1 ) 
 *    {
 *	req->bytecount = 0;
 *	return( BAD_BLOCK_NO );
 *    }
 */

    s = &SerialSccStatus[inst->unit];
    s->outputPtr = req->shortbuffer;
    s->outputCount = req->bytecount;
    channel = s->channel;

    disable;
    /* Output a byte if we can */
    if (ReadRegister0(channel) & SccTxBufferEmpty)
      {
	WriteDataRegister(*s->outputPtr++, channel);
	if (--(s->outputCount) == 0)
	  {
	    inst->lastblock++;
	    inst->writer = NULL;
	    return (OK);
	  }
      }
    inst->writer = pd->pid;	/* Wait until UART is ready for another char */
    pd->state = AWAITING_INT;

    return (NO_REPLY);
  }


/* This macro creates an assembly-language interrupt service routine
 *  that saves the necessary registers and calls SccInterruptHandler.
 */
Call_inthandler(SccInterruptHandler);

static void SccInterruptHandler()
  {
    register SccChannelStatus *s;
    register DeviceInstance *inst;
    register Process *pd;
    register IoReply *reply;
    register char rr0;
    register char c;

    /* Examine each channel to see which interrupted */
    for (s = SccChannelList; s != NULL; s = s->link)
      {
	rr0 = ReadRegister0(s->channel);

	/* First check receiver interrupt */
	if (rr0 & SccRxCharacterAvailable)
	  {
	    /* Read the received character */
	    c = ReadDataRegister(s->channel);

	    /* Is there an instance for this channel? */
	    if ( (inst = s->inputInst) != NULL )
	      {
		/* Is there a reader waiting? */
	        if ( MAP_TO_RPD(pd, inst->reader) )
		  {
		    /* Reader waiting -- give it the character */
		    reply = (IoReply *) &(pd->msg);
		    reply->shortbuffer[0] = c;
		    reply->bytecount = 1;
		    reply->replycode = OK;
		    inst->nextblock++;
		    inst->reader = NULL;
		    Addready( pd );
		  }
		else
		  {
		    /* Reader not waiting -- queue the character */
		    CbuffPut(&(s->cbuff), c);
		  }
	      }
	    else
	      {
		/* No instance -- check if this is the mouse */
		if (s == &MouseSccStatus)
		  {
		    /* Yes -- handle squeak */
		    SmiMouseSqueak(c);
		  }
	      }
	  }

	/* Next check if transmitter buffer is empty.  Even if it is,
	 *  it may not have caused the interrupt, but there is no harm
	 *  in executing this code anyway. */
	if (rr0 & SccTxBufferEmpty)
	  {
	    /* Is there an writer waiting? */
	    if ( (inst = s->outputInst) != NULL &&
	          MAP_TO_RPD(pd, inst->writer) )
	      {
		/* Writer waiting -- write character(s) */
		reply = (IoReply *) &(pd->msg);
		WriteDataRegister(*s->outputPtr++, s->channel);
		if (--(s->outputCount) == 0)
		  {
		    inst->lastblock++;
		    inst->writer = NULL;
		    reply->replycode = OK;
		    Addready( pd );
		  }
	      }
	    else
	      {
		/* No writer waiting -- reset the interrupt */
		WriteRegister0(SccResetTxIntPending, s->channel);
	      }
	  }
	WriteRegister0(SccResetHighestIus, s->channel);
      }
  }


/*
 * Query the device-dependent parameters of an Scc channel
 */
SystemCode SccQuery(pd, inst, dirIndex)
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QuerySerialDeviceReply *reply = (QuerySerialDeviceReply *)&(pd->msg);
    register SccChannelStatus *s;

    s = &SerialSccStatus[inst->unit];
    reply->charWidth = s->charWidth;
    reply->parity = s->parity;
    reply->dtr = s->dtr;
    reply->rts = s->rts;
    reply->brk = s->brk;
    reply->dataRate = s->dataRate;
    reply->stopBits = s->stopBits;

    return (OK);
  }


/*
 * Modify the device-dependent parameters of an Scc channel
 */
SystemCode SccModify(pd, inst, dirIndex)
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    ModifySerialDeviceRequest *req = (ModifySerialDeviceRequest *)&(pd->msg);
    register SccChannelStatus *s;

    s = &SerialSccStatus[inst->unit];
    s->charWidth = req->charWidth;
    s->parity = req->parity;
    s->dtr = req->dtr;
    s->rts = req->rts;
    s->brk = req->brk;
    s->dataRate = req->dataRate;
    s->stopBits = req->stopBits;

    SccInitChannel(s);

    return (OK);
  }


SystemCode SccRelease(pd, inst)
    Process *pd;
    DeviceInstance *inst;
  /*
   * Release serial line instance
   */
  {
    IoRequest  *req = (IoRequest *) &(pd->msg);
    IoReply *reply;
    register Process *rwpd;

    if( rwpd = MapPid(inst->reader) ) 
				/* Unblock reader process */
      {
	inst->reader = 0;
	reply = (IoReply *) &(rwpd->msg);
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( rwpd );
      }

    if( rwpd = MapPid(inst->writer) ) 
				/* Unblock writer process */
      {
	inst->writer = 0;
	reply = (IoReply *) &(rwpd->msg);
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( rwpd );
      }

    inst->owner = 0;
    return( OK );
  }


/* Sound the bell on a Sun-2 keyboard.  This requires sending out a
 *   "bell-on" character, delaying a while, and then sending a 
 *   "bell-off."
 */
SystemCode SccBellKludge( pd )
    Process *pd;
  {
    /* If it's ready, turn on the bell, else wait and try again */
    if (ReadRegister0(KeyboardSccStatus.channel) & SccTxBufferEmpty)
      {
	WriteDataRegister('\2', KeyboardSccStatus.channel);
        pd->timeout_func = SccBellKludge2;
        pd->timeout_count = (CLICKS_PER_SEC/10);
      }
    else
      {
	/* This probably never happens anyway, but just in case */
	pd->timeout_func = (Unspec (*)()) SccBellKludge;
	pd->timeout_count = 1;
      }
    DelayProcess( pd );
    return (NO_REPLY);
  }

static Unspec SccBellKludge2( pd )
    Process *pd;
  {
    /* Now turn off the bell.  We punt if it's busy, since that means
     *  either someone is turning it off, in which case we are happy,
     *  or else they are turning it on, in which case they can worry
     *  about turning it off, and we are still happy.
     */
    
    if (ReadRegister0(KeyboardSccStatus.channel) & SccTxBufferEmpty)
      {
	WriteDataRegister('\3', KeyboardSccStatus.channel);
      }

    ((IoReply *)&(pd->msg))->replycode = OK;
    Addready(pd);
  }
